/*******************************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.java.internal.junit.editor;

import org.eclipse.core.resources.IResource;
import org.eclipse.jdt.core.IPackageFragment;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.StyledText;
import org.eclipse.swt.custom.VerifyKeyListener;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.ui.dialogs.ElementTreeSelectionDialog;

import org.eclipse.hyades.models.common.facades.behavioral.IImplementor;
import org.eclipse.hyades.models.common.facades.behavioral.ITest;
import org.eclipse.hyades.test.java.TestJavaPlugin;
import org.eclipse.hyades.test.java.internal.util.TestJavaUtil;
import org.eclipse.hyades.test.ui.internal.editor.form.util.WidgetFactory;
import org.eclipse.hyades.ui.internal.util.GridDataUtil;

/**
 * @author marcelop
 * @since 0.3.0
 */
public class SourceInfoViewer 
implements ModifyListener, SelectionListener
{	 
	private StyledText packageText;
	private StyledText classNameText;
	private Button packageButton;
	
	private IImplementor implementor;
	private VerifyKeyListener verifyKeyListener;

	/**
	 * @see org.eclipse.update.ui.forms.internal.FormSection#dispose()
	 */
	public void dispose()
	{
		implementor = null;
	}
	
	protected boolean isReadOnly()
	{
		return false;
	}

	protected IImplementor getImplementor()
	{
		return implementor;
	}

	public Composite createClient(Composite parent, VerifyKeyListener verifyKeyListener, WidgetFactory widgetFactory)
	{
		this.verifyKeyListener = verifyKeyListener;
		
		Composite composite = null;
		if(widgetFactory != null)
			composite = widgetFactory.createComposite(parent);
		else
			composite = new Composite(parent, SWT.NONE);
			
		composite.setLayout(new GridLayout());
		composite.setLayoutData(GridDataUtil.createFill());
				
		if(widgetFactory != null)
			widgetFactory.createLabel(composite, TestJavaPlugin.getString("STR_WB_PACK"));
		else
			new Label(composite, SWT.NONE).setText(TestJavaPlugin.getString("STR_WB_PACK"));

		if(widgetFactory != null)
			packageText = widgetFactory.createStyledText(composite, SWT.FULL_SELECTION | SWT.SINGLE);
		else
			packageText =  new StyledText(composite, SWT.FULL_SELECTION | SWT.SINGLE | SWT.BORDER);
		packageText.setLayoutData(GridDataUtil.createHorizontalFill());
		packageText.setEditable(false);

		if(widgetFactory != null)
		{
			packageButton = widgetFactory.createButton(composite, TestJavaPlugin.getString("STR_WB_PACK_BTN"), SWT.PUSH);
		}
		else
		{
			packageButton = new Button(composite, SWT.PUSH);
			packageButton.setText(TestJavaPlugin.getString("STR_WB_PACK_BTN"));
		}
		
		packageButton.setLayoutData(new GridData(GridData.HORIZONTAL_ALIGN_END));
		packageButton.addSelectionListener(this);

		if(widgetFactory != null)
			widgetFactory.createLabel(composite, TestJavaPlugin.getString("STR_WB_CLASS"));
		else
			new Label(composite, SWT.NONE).setText(TestJavaPlugin.getString("STR_WB_CLASS"));

		if(widgetFactory != null)
			classNameText = widgetFactory.createStyledText(composite, SWT.FULL_SELECTION | SWT.SINGLE);
		else
			classNameText =  new StyledText(composite, SWT.FULL_SELECTION | SWT.SINGLE | SWT.BORDER);
		classNameText.setLayoutData(GridDataUtil.createHorizontalFill());
		classNameText.setEditable(false);
			
		if(widgetFactory != null)			
			widgetFactory.paintBordersFor(composite);
		return composite;
	}
	
	/**
	 * @see org.eclipse.hyades.test.ui.internal.editor.form.base.BaseEditorSection#setInput(java.lang.Object)
	 */
	public void setInput(Object object)
	{
		implementor = null;

		if(object instanceof IStructuredSelection)
		{
			IStructuredSelection structuredSelection = (IStructuredSelection)object;
			if(structuredSelection.size() == 1)
				object = structuredSelection.getFirstElement();
		}
		
		if(object instanceof ITest)
			object = ((ITest)object).getImplementor();

		if((object != null) && (object instanceof IImplementor))
		{
			implementor = (IImplementor)object;
			
			adjustTexts();
			
			packageText.setEditable(true);
			classNameText.setEditable(true);
		}
		else
		{
			packageText.setText("");
			classNameText.setText("");

			packageText.setEditable(false);
			classNameText.setEditable(false);
		}
		
	}
	
	protected void adjustTexts()
	{
		if(verifyKeyListener != null)
		{
			packageText.removeVerifyKeyListener(verifyKeyListener);
			classNameText.removeVerifyKeyListener(verifyKeyListener);
		}
		packageText.removeModifyListener(this);
		classNameText.removeModifyListener(this);

		String className = "";
		String packageName = "";
		String value = getImplementor().getResource();
		if(value != null)
		{
			int index = value.lastIndexOf('.');
			if(index >= 0)
				packageName = value.substring(0, index);
			else
				index = -1;
				
			if(index != (value.length()-1))
				className = value.substring(index+1);
		}
			
		packageText.setText(packageName);
		classNameText.setText(className);		

		if(verifyKeyListener != null)
		{
			packageText.addVerifyKeyListener(verifyKeyListener);
			classNameText.addVerifyKeyListener(verifyKeyListener);
		}
		packageText.addModifyListener(this);
		classNameText.addModifyListener(this);
	}

	/**
	 * @see org.eclipse.hyades.test.ui.internal.editor.form.base.BaseEditorSection#getInput()
	 */
	public Object getInput()
	{
		return getImplementor();
	}

	/**
	 * @see org.eclipse.hyades.test.ui.internal.editor.form.base.BaseEditorSection#getStructuredSelection()
	 */
	public IStructuredSelection getStructuredSelection()
	{
		if(getImplementor() == null)
			return StructuredSelection.EMPTY;
		return new StructuredSelection(getImplementor());
	}
	
	/**
	 * @see org.eclipse.ui.part.ISetSelectionTarget#selectReveal(org.eclipse.jface.viewers.ISelection)
	 */
	public void selectReveal(ISelection selection)
	{

	}

	/**
	 * @see org.eclipse.swt.events.ModifyListener#modifyText(org.eclipse.swt.events.ModifyEvent)
	 */
	public void modifyText(ModifyEvent e)
	{
		if((e.widget == classNameText) || (e.widget == packageText))
		{
			String pack = getPackage();
			String cls = getClassName();
			
			if((pack.length() > 0) && (cls.length() > 0))
				getImplementor().setResource(pack + "." + cls);
			else
				getImplementor().setResource(pack + cls);
		}
	}
	
	/**
	 * @see org.eclipse.swt.events.SelectionListener#widgetDefaultSelected(org.eclipse.swt.events.SelectionEvent)
	 */
	public void widgetDefaultSelected(SelectionEvent e)
	{
		e.doit = (!isReadOnly());
	}

	/**
	 * @see org.eclipse.swt.events.SelectionListener#widgetSelected(org.eclipse.swt.events.SelectionEvent)
	 */
	public void widgetSelected(SelectionEvent e)
	{
		if(isReadOnly())
		{
			e.doit = false;
		}
		else
		{
			if(e.widget == packageButton)
			{
				ElementTreeSelectionDialog dialog = TestJavaUtil.createPackageDialog(packageButton.getShell(), false);
				dialog.setTitle(TestJavaPlugin.getString("TITLE_PACK_SELECT"));
				dialog.setMessage(TestJavaPlugin.getString("STR_PACK_SELECT"));
				dialog.setEmptyListMessage(TestJavaPlugin.getString("NO_PACK"));
				
				if(dialog.open() == IDialogConstants.OK_ID)
				{
					IPackageFragment packageFragment = (IPackageFragment)dialog.getResult()[0];
					
					IResource resource = null;
					if(packageFragment.getParent() == null)
						resource = packageFragment.getParent().getJavaProject().getProject();
					else
						resource = packageFragment.getParent().getResource();
					if(resource != null)
						getImplementor().setLocation(resource.getFullPath().toString());
					packageText.setText(packageFragment.getElementName());
				}
			}
		}
	}
	
	public String getPackage()
	{
		return packageText.getText().trim();
	}
	
	public String getClassName()
	{
		return classNameText.getText().trim();
	}
	
	public void setInitialFocus()
	{
		if(packageText != null)
			packageText.setFocus();
	}
}