/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.analysis.engine;

public interface IAnalysisEngine {

	/**
	 * Reloads the base symptom database from the base XML file.
	 * 
	 * @return true if the symptom database has been successfully reloaded
	 */
	public boolean reloadSymptomDatabase();

	/**
	 * Loads a new symptom database XML file denoted by the parameter path
	 * and merges any unique records with the base symptom database. 
	 * 
	 * @param  symptomDbPath the path to the merged symptom database XML file
	 * @return true if the new symptom database has been successfully merged 
	 */
	public boolean mergeSymptomDatabase(String symptomDbPath);

	/**
	 * Removes the base symptom database.
	 * <p>
	 * <b>NOTE:</b> A base symptom database may be added using 
	 * <code>replaceSymptomDatabase(symptomDbPath)</code>.
	 * 
	 * @return true if the symptom database has been successfully removed
	 */
	public boolean removeSymptomDatabase();

	/**
	 * A new symptom database is created denoted by the parameter XML file 
	 * path.
	 * <p>
	 * If the new symptom database XML file was successfully loaded, it
	 * replaces the base symptom database. 
	 * 
	 * @param  symptomDbPath the path to the symptom database XML file
	 * @return true if the new symptom database has been successfully replaced 
	 */
	public boolean replaceSymptomDatabase(String symptomDbPath);

        /**
         * Returns the XML file path of the base symptom database.
         * <p>
         * If no base symptom database XML file is loaded, null is returned.
         * 
         * @return the XML file path of the base symptom database, otherwise null
         */
        public String getSymptomDatabasePath();

	/**
	 * The current symptom database is searched for <code>Incident</code> 
	 * matches.
	 * <p>
	 * All <code>Directive</code>(s) from successful <code>Incident</code> 
	 * matches are returned in an array structure.
	 * <p>
	 * <code>Incident</code> matches are based on a lexicographical String
	 * comparison between the <code>messageId</code> and <code>rawData</code>
	 * array in the parameter <code>Incident</code> and the <code>value</code> 
	 * attribute of the <code>matchPattern</code> Node in the base symptom 
	 * database.
	 * 
	 * @param  incident the search criteria used for possible matches
	 * @return an array of <code>Directive</code> from successful <code>Incident</code> matches 
	 * @see    Object
	 * @see    Incident
	 */
	public Object[] analyze(Incident incident);

	/**
	 * The current symptom database is searched for <code>Incident</code> 
	 * matches.
	 * <p>
	 * All <code>Solution</code>(s) from successful <code>Incident</code> 
	 * matches are returned in an array structure.
	 * <p>
	 * <code>Incident</code> matches are based on a lexicographical String
	 * comparison between the <code>messageId</code> and <code>rawData</code>
	 * array in the parameter <code>Incident</code> and the <code>value</code> 
	 * attribute of the <code>matchPattern</code> Node in the base symptom 
	 * database.
	 * 
	 * @param  incident the search criteria used for possible matches
	 * @return an array of <code>Solution</code> from successful <code>Incident</code> matches 
	 * @see    Solution
	 * @see    Incident
	 */
	public Solution[] analyzeForSolutions(Incident incident);
}