/*******************************************************************************
 * Copyright (c) 2003 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.models.common.facades.behavioral.test;

import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import junit.framework.Test;
import junit.framework.TestCase;
import junit.framework.TestSuite;

import org.eclipse.hyades.models.common.facades.behavioral.IAction;
import org.eclipse.hyades.models.common.facades.behavioral.IDecision;
import org.eclipse.hyades.models.common.facades.behavioral.IImplementor;
import org.eclipse.hyades.models.common.facades.behavioral.ILoop;
import org.eclipse.hyades.models.common.facades.behavioral.IStub;
import org.eclipse.hyades.models.common.facades.behavioral.ISystemUnderTest;
import org.eclipse.hyades.models.common.facades.behavioral.ITargetInvocation;
import org.eclipse.hyades.models.common.facades.behavioral.ITest;
import org.eclipse.hyades.models.common.facades.behavioral.ITestCase;
import org.eclipse.hyades.models.common.facades.behavioral.ITestInvocation;
import org.eclipse.hyades.models.common.facades.behavioral.ITestSuite;
import org.eclipse.hyades.models.common.facades.behavioral.IVerificationPoint;

/*
 * Mandatory not null?
 * 
 * Questions:
 * 	- decision.getTarget()
 *  - loop.getTarget()
 */


/**
 * JUnit test case created to validate the completeness of an instantiation of
 * the facade.  This test validates if:
 * <OL>
 * <LI>The required attributes are set</LI>
 * <LI>The bidirectional associations are respected</LI>
 * <LI>The expected conditions are respected</LI>
 * </OL>
 * 
 * <p>The validation is performed based on the {@link #facadeTestSuite} which must be 
 * set by overwriting the {@link #getFacadeTestSuite()} method.  If this test suite invokes
 * another test suite - directly or through one of the test cases of the second test 
 * suite - then this test suite is also validated. 
 * 
 * @author marcelop
 * @since 0.0.1
 */
public class ValidateFacade
extends TestCase
{
	/**
	 * Facade's test suite that is tested by this class.  Subclasses are 
	 * supposed to set this attribute by overwriting the {@link TestCase#setUp()}
	 * method.
	 */
	protected static ITestSuite facadeTestSuite;
	
	/**
	 * Set of the test suites that were already validated.  This set is used by
	 * the {@link #validate(ITestSuite)} method to ensure that a test suite is not 
	 * validate twice.
	 */
	protected static Set validatedTestSuites;
	
	/**
	 * Constructor for ITestSuiteTest
	 * @param name
	 */
	public ValidateFacade(String name)
	{
		super(name);
	}

	/**
	 * Returns the JUnit Test Suite
	 * @return Test
	 */
	public static Test suite()
	{
		TestSuite junitTestSuite = new TestSuite("Facade Validation");
		
		junitTestSuite.addTest(new ValidateFacade("suiteSetUp"));
		junitTestSuite.addTest(new ValidateFacade("testTestSuite"));
		junitTestSuite.addTest(new ValidateFacade("testTestCases"));
		junitTestSuite.addTest(new ValidateFacade("testSystemsUnderTest"));
		junitTestSuite.addTest(new ValidateFacade("testTestComponents"));
		junitTestSuite.addTest(new ValidateFacade("suiteTearDown"));
		
		return junitTestSuite;
	}
	
	/**
	 * JUnit test suite's set up 
	 */
	public void suiteSetUp()
	{ 
		validatedTestSuites = new HashSet();
		facadeTestSuite = getFacadeTestSuite();
	}
		
	/**
	 * JUnit test suite's tear down 
	 */
	public void suiteTearDown()
	{ 
		validatedTestSuites.clear();
		facadeTestSuite = null;
	}

	/**
	 * Returns the facade's test suite that is used by the JUnit test methods.  This
	 * method is invoke only once for each execution of the suite returned by
	 * {@link #suite()}. 
	 * @return ITestSuite
	 */
	protected ITestSuite getFacadeTestSuite()
	{
		return null;
	}

	/**
	 * Tests the {@link #facadeTestSuite} test suite.
	 * @throws Exception
	 * @see #validate(ITestSuite)
	 */
	public void testTestSuite()
	throws Exception
	{
		assertNotNull(facadeTestSuite);
		validate(facadeTestSuite);
	}
	
	/**
	 * Tests all the test cases available at the 
	 * {@link #facadeTestSuite} test suite.
	 * @throws Exception
	 * @see #validate(ITestCase)
	 */
	public void testTestCases()
	throws Exception
	{
		assertNotNull(facadeTestSuite);
		List testCases = facadeTestSuite.getITestCases();
		if(testCases.isEmpty())
			return;

		for (Iterator i = testCases.iterator(); i.hasNext();)
			validate((ITestCase)i.next());
	}
		
	/**
	 * Tests all the systems under test available at the 
	 * {@link #facadeTestSuite} test suite.
	 * @throws Exception
	 * @see #validate(ISystemUnderTest)
	 */
	public void testSystemsUnderTest()
	throws Exception
	{
		assertNotNull(facadeTestSuite);
		List systemsUnderTest = facadeTestSuite.getISystemsUnderTest();
		if(systemsUnderTest.isEmpty())
			return;
		
		for (Iterator i = systemsUnderTest.iterator(); i.hasNext();)
			validate((ISystemUnderTest)i.next());
	}

	/**
	 * Tests all the test component available at the 
	 * {@link #facadeTestSuite} test suite.
	 * @throws Exception
	 * @see #validate(ITestComponent)
	 */
	public void testStubs()
	throws Exception
	{
		assertNotNull(facadeTestSuite);
		List testStubs = facadeTestSuite.getIStubs();
		if(testStubs.isEmpty())
			return;
		
		for (Iterator i = testStubs.iterator(); i.hasNext();)
			validate((IStub)i.next());
	}

	/**
	 * Implementation of the validation of a facade's test suite.
	 * @param testSuite
	 * @throws Exception
	 */
	protected void validate(ITestSuite testSuite)
	throws Exception
	{
		if(validatedTestSuites.contains(testSuite))
			return;
		validatedTestSuites.add(testSuite);
		
		assertNotNull(testSuite.getName());
		validate(testSuite.getImplementor());
		assertNotNull(testSuite.getITestCases());
		assertNotNull(testSuite.getISystemsUnderTest());
		assertNotNull(testSuite.getIStubs());
	}

	/**
	 * Implementation of the validation of a facade's test case.
	 * @param testCase
	 * @throws Exception
	 */
	protected void validate(ITestCase testCase)
	throws Exception
	{
		assertNotNull(testCase);
		assertNotNull(testCase.getName());
		assertEquals(testCase.getOwner(), facadeTestSuite);
		validate(testCase.getImplementor());
	}

	/**
	 * Implementation of the validation of a facade's implementor.
	 * @param implementor
	 * @throws Exception
	 */
	protected void validate(IImplementor implementor)
	throws Exception
	{
		assertNotNull(implementor);
		if(implementor.isExternalImplementor())		
		{
			IImplementor externalImplementor = (IImplementor)implementor;
			assertNotNull(externalImplementor.getResource());
		}
		else if(implementor instanceof IImplementor)
		{
			IImplementor impl = (IImplementor)implementor;
			assertEquals(impl.getOwner(), facadeTestSuite);
		}
	}

	/**
	 * Implementation of the validation of a facade's system under test.
	 * @param systemUnderTest
	 * @throws Exception
	 */
	protected void validate(ISystemUnderTest systemUnderTest)
	throws Exception
	{
		assertNotNull(systemUnderTest);
		assertNotNull(systemUnderTest.getName());
		assertEquals(systemUnderTest.getOwner(), facadeTestSuite);
	}

	/**
	 * Implementation of the validation of a facade's test component.
	 * @param testComponent
	 * @throws Exception
	 */
	protected void validate(IStub stub)
	throws Exception
	{
		assertNotNull(stub.getName());
		assertEquals(stub.getOwner(), facadeTestSuite);
	}
	
	/**
	 * Implementation of the validation of a facade's action.
	 * 
	 * <p>If the action is an instance of an <code>ITestInvocation</code>
	 * then the invoked test is validated.
	 * 
	 * @param action
	 * @throws Exception
	 */
	protected void validate(IAction action)
	throws Exception
	{
		assertNotNull(action.getActionProperties());
		
		if(action instanceof IDecision)
		{
			IDecision decision = (IDecision)action;
			assertNotNull(decision.getCondition());
			assertNotNull(decision.getCondition().getConstraint());
			assertNotNull(decision.getSuccessBlock());
			assertNotNull(decision.getFailureBlock());
		}
		else if(action instanceof ILoop)
		{
			ILoop loop = (ILoop)action;
			assertNotNull(loop.getCondition());
			assertNotNull(loop.getCondition().getConstraint());
			assertNotNull(loop.getBlock());
		}
		else if(action instanceof IVerificationPoint)
		{
		}
		else if(action instanceof ITargetInvocation)
		{
			ITargetInvocation targetInvocation = (ITargetInvocation)action;
			assertNotNull(targetInvocation.getTarget());
		}
		else if(action instanceof ITestInvocation)
		{
			ITestInvocation testInvocation = (ITestInvocation)action;
			
			ITest test = testInvocation.getInvokedTest();
			assertNotNull(test);

			ITestSuite invokedTestSuite = null;
			if(test instanceof ITestSuite)
			{
				invokedTestSuite = (ITestSuite)test;
			}
			else if(test instanceof ITestCase)
			{
				invokedTestSuite  = ((ITestCase)test).getOwner();
			}
			assertNotNull(invokedTestSuite);
			validate(invokedTestSuite);

			IImplementor implementor = testInvocation.getInvokedTest().getImplementor();
			assertNotNull(implementor);
		}
	}	
}