/*******************************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.ui.internal.navigator;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.Platform;
import org.eclipse.jface.action.IMenuListener;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.dialogs.IDialogSettings;
import org.eclipse.jface.viewers.IOpenListener;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.OpenEvent;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.StructuredViewer;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.KeyListener;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IElementFactory;
import org.eclipse.ui.IMemento;
import org.eclipse.ui.IPartListener;
import org.eclipse.ui.IPersistableElement;
import org.eclipse.ui.ISelectionListener;
import org.eclipse.ui.IViewSite;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.IWorkbenchPreferenceConstants;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.actions.ActionContext;
import org.eclipse.ui.actions.ActionGroup;
import org.eclipse.ui.help.WorkbenchHelp;
import org.eclipse.ui.part.ViewPart;

import org.eclipse.hyades.ui.HyadesUIPlugin;
import org.eclipse.hyades.ui.internal.navigator.action.NavigatorActionGroup;
import org.eclipse.hyades.ui.internal.util.CoreUtil;
import org.eclipse.hyades.ui.internal.util.UIUtil;

/**
 * Hyades navigator base class.
 * 
 * <p>Clients are not supposed to manipulate the navigator preferences and UI before the 
 * end of the {@link #init(IViewSite, IMemento)} method execution.
 * 
 * <p>This implementation is based on the 
 * {@link org.eclipse.ui.views.navigator.ResourceNavigator} class.
 * 
 * @author marcelop
 * @since 0.0.1
 */
public abstract class Navigator 
extends ViewPart implements	INavigator, 
							IPartListener, ISelectionChangedListener, KeyListener, 
							IMenuListener, IOpenListener, ISelectionListener
{
	/*
	 * SET_* constants are used in the IDialogSettings
	 */
	protected static final String SET_LINK_TO_EDITOR = IWorkbenchPreferenceConstants.LINK_NAVIGATOR_TO_EDITOR;
	
	/*
	 * TAG* constants are used in the IMemento
	 */
	protected static final String TAG_SELECTION = "selection";
	protected static final String TAG_ELEMENT = "element";
	protected static final String TAG_STORE_VALUE = "storeValue";
	protected static final String TAG_FACTORY_ID = "factoryID";
	
	private static final String ID_POPUP_MENU = "#NavigatorPopupMenu";
	
	private StructuredViewer structuredViewer;
	private Object initialViewerInput;
	
	private IDialogSettings settings;
	private IMemento memento;
	
	private NavigatorActionGroup actionGroup;
	
	private boolean linkingEnabled;

	/**
	 * Constructor for Navigator
	 */
	public Navigator()
	{
		super();
	}	

	/**
	 * @see org.eclipse.ui.IWorkbenchPart#dispose()
	 */
	public void dispose()
	{
		getViewer().removeOpenListener(this);
		getViewer().removeSelectionChangedListener(this);
		getViewSite().getPage().removePartListener(this);

		if(!getViewer().getControl().isDisposed())
			getViewer().getControl().removeKeyListener(this);
		
		if (getActionGroup() != null)
			getActionGroup().dispose();

		initialViewerInput = null;
		structuredViewer = null;
		settings = null;
		actionGroup = null;
		
		super.dispose();
	}
	
	/**
	 * @see org.eclipse.ui.IViewPart#init(org.eclipse.ui.IViewSite, org.eclipse.ui.IMemento)
	 */
	public void init(IViewSite site, IMemento memento)
	throws PartInitException
	{
		super.init(site, memento);
		this.memento = memento;
		site.getPage().addPostSelectionListener(this);

		IDialogSettings viewsSettings = getPlugin().getDialogSettings();
		settings = viewsSettings.getSection(getStoreSection());
		if(settings == null)
		{
			settings = viewsSettings.addNewSection(getStoreSection());
			initSettings();
		}
		else
		{		
			loadSettings();
		}
	}
	
	/**
	 * This method is invocated when the dialog settings has just been
	 * initialized. 
	 */
	protected void initSettings()
	{
	}
	
	/**
	 * @see org.eclipse.hyades.ui.internal.navigator.INavigator#isInitialized()
	 */
	public boolean isInitialized()
	{
		return (getViewSite() != null);
	}
	
	/**
	 * Returns this navigator's settings.
	 * @return IDialogSettings
	 */
	protected IDialogSettings getSettings()
	{
		return settings;
	}

	/**
	 * Loads the settings from the dialog settings buffer.  This class' setting keys
	 * corresponds to the "SET_" string constants.
	 * @see #getSettings()
	 */
	protected void loadSettings()
	{
		setLinkingEnabled(getSettings().getBoolean(SET_LINK_TO_EDITOR));
	}

	/**
	 * Returns the store section string that is used to retrieve this navigator's
	 * settings from the memento and dialog settings.
	 * @return String.  This method should not return null.
	 */
	abstract public String getStoreSection();

	/**
	 * Returns the action group.
	 * @return NavigatorActionGroup
	 */
	protected NavigatorActionGroup getActionGroup()
	{
		return actionGroup;
	}

	/**
	 * Sets the action group.
	 * @param actionGroup
	 */
	protected void setActionGroup(NavigatorActionGroup actionGroup)
	{
		this.actionGroup = actionGroup;
	}
	
	/**
	 * Subclasses should create the actions in this method.
	 */
	abstract protected void createActions();

	/**
	 * @see org.eclipse.ui.IWorkbenchPart#createPartControl(org.eclipse.swt.widgets.Composite)
	 */
	public void createPartControl(Composite parent)
	{
		structuredViewer = createViewer(parent);
		
		getViewer().addSelectionChangedListener(this);
		getViewer().addOpenListener(this);
		getViewer().getControl().addKeyListener(this);
		initContextMenu();
		
		getViewer().setInput(createViewerInput());
		initialViewerInput = getViewer().getInput();
		
		createActions();
		if(getActionGroup() != null)
			getActionGroup().fillActionBars(getViewSite().getActionBars());
		updateActionBars(getStructuredSelection());

		getSite().setSelectionProvider(getViewer());
		getSite().getPage().addPartListener(this);
		
		String helpContextId = getHelpContextId();
		if(helpContextId != null)
			WorkbenchHelp.setHelp(getViewer().getControl(), helpContextId);
		
		if(memento != null)
		{
			loadMemento(memento);
			memento = null;
		}
	}
	
	/**
	 * Returns the help context id for this navigator or <code>null</code> if
	 * no help is provided.
	 * @return String
	 */
	protected String getHelpContextId()
	{
		return null;
	}
	
	/**
	 * Loads the information persisted in the memento.  This method is
	 * invoked after the creation of the UI controls.
	 * @param memento
	 */
	protected void loadMemento(IMemento memento)
	{
		loadSelection(memento);
	}
	
	/**
	 * Loads the previous selection from the memento.
	 * @param memento
	 */
	protected void loadSelection(IMemento memento)
	{
		IMemento selectionMemento = memento.getChild(TAG_SELECTION);
		if(selectionMemento == null)
			return;
			
		List selection = loadElements(selectionMemento);
		if(!selection.isEmpty())
			selectReveal(new StructuredSelection(selection));
	}
	
	/**
	 * Initializes and registers the context menu.
	 */
	protected void initContextMenu()
	{
		MenuManager menuMgr = new MenuManager(ID_POPUP_MENU);
		menuMgr.setRemoveAllWhenShown(true);
		menuMgr.addMenuListener(this);

		Menu menu = menuMgr.createContextMenu(getViewer().getControl());
		getViewer().getControl().setMenu(menu);
	}

	/**
	 * Creates the object that is used as the input for this navigator's
	 * viewer.
	 * @return Object
	 */
	abstract protected Object createViewerInput();	
	
	/**
	 * Returns the input returned by {@link #createViewerInput()}
	 * @return Object
	 */
	protected Object getInitialViewerInput()
	{
		return initialViewerInput;
	}
	
	/**
	 * Creates the viewer that is displayed by this navigator.  Subclasses are 
	 * responsible for setting the content and label provider and also to ensure
	 * that this method returns a valid value.
	 * @param parent
	 * @return StructuredViewer
	 */
	abstract protected StructuredViewer createViewer(Composite parent);
	
	/**
	 * @see org.eclipse.hyades.ui.internal.navigator.INavigator#getViewer()
	 */
	public StructuredViewer getViewer()
	{
		return structuredViewer;
	}

	/**
	 * @see org.eclipse.ui.IWorkbenchPart#setFocus()
	 */
	public void setFocus()
	{
		getViewer().getControl().setFocus();
	}

	/**
	 * @see org.eclipse.ui.part.ISetSelectionTarget#selectReveal(org.eclipse.jface.viewers.ISelection)
	 */
	public void selectReveal(ISelection selection)
	{
		IStructuredSelection structuredSelection = convertToViewer(selection);
		if(structuredSelection.isEmpty())
			return;
			
		if((getViewer() == null) || (getViewer().getControl() == null) || getViewer().getControl().isDisposed())
			return;
			
		if(!UIUtil.areEquals(getStructuredSelection(), structuredSelection))
			getViewer().setSelection(structuredSelection, true);
	}
	
	/**
	 * Converts the viewer's selection into a form usable by the application.
	 * @param selection.
	 * @param StructuredSelection.  Cannot be <code>null</code>.
	 */
	protected IStructuredSelection convertFromViewer(ISelection selection)
	{
		if((selection != null) && (selection instanceof IStructuredSelection))
			return (IStructuredSelection)selection;
			
		return StructuredSelection.EMPTY; 
	}
	
	/**
	 * Converts a selection into a form usable by the viewer.
	 * @param selection.
	 * @param StructuredSelection.  Cannot be <code>null</code>.
	 */
	protected IStructuredSelection convertToViewer(ISelection selection)
	{
		if((selection != null) && (selection instanceof IStructuredSelection))
			return (IStructuredSelection)selection;
			
		return StructuredSelection.EMPTY; 
	}
	
	/**
	 * @see org.eclipse.hyades.ui.internal.navigator.INavigator#getStructuredSelection()
	 */
	public IStructuredSelection getStructuredSelection()
	{
		if(getViewer() == null)
			return StructuredSelection.EMPTY;
			
		return convertFromViewer(getViewer().getSelection());
	}
	
	/**
	 * @see org.eclipse.hyades.ui.internal.navigator.INavigator#isLinkingEnabled()
	 */
	public boolean isLinkingEnabled()
	{
		return linkingEnabled;
	}
	
	/**
	 * @see org.eclipse.hyades.ui.internal.navigator.INavigator#setLinkingEnabled(boolean)
	 */
	public void setLinkingEnabled(boolean enabled)
	{
		linkingEnabled = enabled;

		// remember the last setting in the dialog settings		
		getSettings().put(SET_LINK_TO_EDITOR, enabled);
		
		// if turning linking on, update the selection to correspond to the active editor
		if(enabled)
		{
			IEditorPart editor = getSite().getPage().getActiveEditor();
			if (editor != null)
				editorActivated(editor);
		}
	}

	/**
	 * @see org.eclipse.ui.IPartListener#partActivated(org.eclipse.ui.IWorkbenchPart)
	 */
	public void partActivated(IWorkbenchPart part)
	{
		if(isLinkingEnabled() &&  (part instanceof IEditorPart))
			editorActivated((IEditorPart) part);
	}

	/**
	 * @see org.eclipse.ui.IPartListener#partBroughtToTop(org.eclipse.ui.IWorkbenchPart)
	 */
	public void partBroughtToTop(IWorkbenchPart part)
	{
	}

	/**
	 * @see org.eclipse.ui.IPartListener#partClosed(org.eclipse.ui.IWorkbenchPart)
	 */
	public void partClosed(IWorkbenchPart part)
	{
	}

	/**
	 * @see org.eclipse.ui.IPartListener#partDeactivated(org.eclipse.ui.IWorkbenchPart)
	 */
	public void partDeactivated(IWorkbenchPart part)
	{
	}

	/**
	 * @see org.eclipse.ui.IPartListener#partOpened(org.eclipse.ui.IWorkbenchPart)
	 */
	public void partOpened(IWorkbenchPart part)
	{
	}
	
	/**
	 * An editor has been activated.  Sets the selection in this navigator
	 * to be the editor's input, if linking is enabled.
	 * 
	 * <p>Subclasses are don't need to test if the linking is enabled.
	 * 
	 * <p>This method may be executed before the creation of the UI controls. 
	 * 
	 * @param editor the active editor
	 * @return <code>true</code> if the appropriate action was taken or 
	 * <code>false</code> otherwise.
	 */
	abstract protected boolean editorActivated(IEditorPart editor);

	/**
	 * Activates the editor if the selected resource is open.
	 * @param selection
	 */
	abstract protected void linkToEditor(IStructuredSelection selection);
	
	/**
	 * @see org.eclipse.hyades.ui.internal.navigator.INavigator#updateTitle()
	 */
	public void updateTitle()
	{
		String viewName = getConfigurationElement().getAttribute("name");
		setTitle(viewName);
		setTitleToolTip("");
	}

	/**
	 * @see org.eclipse.jface.viewers.ISelectionChangedListener#selectionChanged(org.eclipse.jface.viewers.SelectionChangedEvent)
	 */
	public void selectionChanged(SelectionChangedEvent event)
	{
		if(event.getSelectionProvider() == structuredViewer)
		{
			IStructuredSelection structuredSelection = convertFromViewer(event.getSelection());
				
			updateActionBars(structuredSelection);
			updateStatusLine(structuredSelection);
			
			if((!structuredSelection.isEmpty()) && (isLinkingEnabled()))
				linkToEditor(structuredSelection);
		}
	}
		
	/**
	 * Updates the action bar actions.
	 * @param selection the current selection
	 */
	protected void updateActionBars(IStructuredSelection structuredSelection)
	{
		ActionGroup group = getActionGroup();
		if(group != null)
		{
			group.setContext(new ActionContext(structuredSelection));
			group.updateActionBars();
		}
	}
	
	/**
	 * Updates the message shown in the status line.
	 * @param selection the current selection
	 */
	protected void updateStatusLine(IStructuredSelection structuredSelection)
	{
		String msg = getStatusLineMessage(structuredSelection);
		getViewSite().getActionBars().getStatusLineManager().setMessage(msg);
	}
	
	/**
	 * Returns the message to show in the status line.  If this method
	 * returns null the status line won't present any message.
	 * 
	 * <p>If a subclass requires a more sophisticated use of the 
	 * status line, it should manipulate the status line direct.  The
	 * status line can be obtained by 
	 * <code>getViewSite().getActionBars().getStatusLineManager()</code>.
	 *
	 * @param selection the current selection
	 * @return the status line message
	 */
	protected String getStatusLineMessage(IStructuredSelection structuredSelection)
	{
		int size = structuredSelection.size();
		if(size > 1)
			return HyadesUIPlugin.getString("STS_LNE_MULTI", Integer.toString(size));

		return null;
	}
	

	/**
	 * This class' memento keys corresponds to the "TAG_" string constants.
	 * @see org.eclipse.ui.IViewPart#saveState(org.eclipse.ui.IMemento)
	 */
	public void saveState(IMemento memento)
	{
		super.saveState(memento);		
		saveSelection(memento);
	}
	
	/**
	 * Saves the current selection in the memento.
	 * @param memento
	 */
	protected void saveSelection(IMemento memento)
	{
		IStructuredSelection structuredSelection = convertFromViewer(getViewer().getSelection());
		if(structuredSelection.isEmpty())
			return;
		
		IMemento selectionMemento = memento.createChild(TAG_SELECTION);
		for (Iterator i = structuredSelection.iterator(); i.hasNext();)
			saveElement(selectionMemento, i.next());
	}
	
	/**
	 * Saves an element in the given memento.  This method expects the
	 * element to be an instance of <code>IPersistableElement</code> or to be
	 * an adaptable to this interface.
	 * @param memento
	 * @param element
	 */
	protected void saveElement(IMemento memento, Object element)
	{
		IPersistableElement persistableElement = null;
		if(element instanceof IPersistableElement)
			persistableElement = (IPersistableElement)element;
		else if(element instanceof IAdaptable)
			persistableElement = (IPersistableElement)((IAdaptable)element).getAdapter(IPersistableElement.class);
		else
			persistableElement = (IPersistableElement)Platform.getAdapterManager().getAdapter(element, IPersistableElement.class);
			
		if(persistableElement == null)
			return;

		IMemento elementMemento = memento.createChild(TAG_ELEMENT);
		elementMemento.putString(TAG_FACTORY_ID, persistableElement.getFactoryId());
		persistableElement.saveState(elementMemento);
	}
	
	
	/**
	 * Loads the elements marked with the TAG_ELEMENT from the given memento.  
	 * @param memento
	 * @param element
	 * @see #saveElement(IMemento, Object)
	 */
	protected List loadElements(IMemento memento)
	{
		List elements = new ArrayList();
		IMemento[] elementMementos = memento.getChildren(TAG_ELEMENT);
		for(int i = 0, maxi = elementMementos.length; i < maxi; i++)
		{
			String factoryId = elementMementos[i].getString(TAG_FACTORY_ID);
			if(factoryId == null)
				continue;
				
			IElementFactory elementFactory = CoreUtil.getElementFactory(factoryId);
			if(elementFactory == null)
				continue;
				
			Object element = elementFactory.createElement(elementMementos[i]);
			if(element != null)
				elements.add(element);
		}
		
		return elements;
	}
	
	/**
	 * @see org.eclipse.swt.events.KeyListener#keyPressed(org.eclipse.swt.events.KeyEvent)
	 */
	public void keyPressed(KeyEvent event)
	{
		if(event.widget == getViewer().getControl())
		{
			NavigatorActionGroup group = getActionGroup();
			if(group != null)
				group.handleKeyPressed(event);
		}			
	}

	/**
	 * @see org.eclipse.swt.events.KeyListener#keyReleased(org.eclipse.swt.events.KeyEvent)
	 */
	public void keyReleased(KeyEvent event)
	{
	}
	
	/**
	 * @see org.eclipse.jface.action.IMenuListener#menuAboutToShow(org.eclipse.jface.action.IMenuManager)
	 */
	public void menuAboutToShow(IMenuManager manager)
	{
		IStructuredSelection selection = convertFromViewer(getViewer().getSelection());
	
		ActionGroup group = getActionGroup();
		if(group != null)
		{
			group.setContext(new ActionContext(selection));
			group.fillContextMenu(manager);
		}
	}
	
	/**
	 * @see org.eclipse.jface.viewers.IOpenListener#open(org.eclipse.jface.viewers.OpenEvent)
	 */
	public void open(OpenEvent event)
	{
		if(event.getViewer() == getViewer())
		{
			IStructuredSelection structuredSelection = convertFromViewer(event.getSelection());
			NavigatorActionGroup group = getActionGroup();
			if(group != null)
				getActionGroup().runDefaultAction(structuredSelection);
		}
	}
	
	/**
	 * @see org.eclipse.hyades.ui.internal.navigator.INavigator#handleChange(int)
	 */
	public void handleChange(int type)
	{

	}
	
	/**
	 * @see org.eclipse.ui.ISelectionListener#selectionChanged(org.eclipse.ui.IWorkbenchPart, org.eclipse.jface.viewers.ISelection)
	 */
	public void selectionChanged(IWorkbenchPart part, ISelection selection)
	{
		if(part == this)
			return;
			
		if((part instanceof IEditorPart) && (!isLinkingEnabled()))
			return;
			
		selectReveal(selection);
	}
}
