/*******************************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.ui.internal.util;

import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.List;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.runtime.IPluginDescriptor;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredViewer;
import org.eclipse.jface.viewers.ViewerFilter;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.VerifyEvent;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IEditorDescriptor;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IEditorReference;
import org.eclipse.ui.IFileEditorInput;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.PlatformUI;

import org.eclipse.hyades.ui.HyadesUIPlugin;

/**
 * Contains UI utility methods.
 * 
 * @author marcelop
 * @author popescu
 * @since 0.0.1
 */
public class UIUtil
{
	/**
	 * Returns an ImageDescriptor whose path relative to the plugin described 
	 * by <code>pluginDescriptor</code> is <code>subdirectoryAndFilename</code>.
	 * Returns <code>null</code>if no image could be found.
	 *
	 * <p>This method is convenience and only intended for use by the workbench 
	 * because it explicitly uses the workbench's registry for caching/retrieving 
	 * images from other extensions -- other plugins must user their own registry. 
	 * This convenience method is subject to removal.
	 *
	 * <p>Note: subdirectoryAndFilename must not have any leading "." or path 
	 * separators / or \ ISV's should use icons/mysample.gif and not 
	 * ./icons/mysample.gif.
	 *
	 * <p>Note:  This consults the plugin for extension and obtains its installation 
	 * location.  All requested images are assumed to be in a directory below and 
	 * relative to that plugins installation directory.
	 */
	public static ImageDescriptor getImageDescriptorFromPlugin(IPluginDescriptor pluginDescriptor, String subdirectoryAndFilename)
	{
		URL path = pluginDescriptor.getInstallURL();
		URL fullPathString = null;
		try
		{
			fullPathString = new URL(path,subdirectoryAndFilename);
			return ImageDescriptor.createFromURL(fullPathString);
		}
		catch (MalformedURLException e)
		{
			HyadesUIPlugin.logError(e);
		}
		
		return null;
	}
	
	/**
	 * Fires the current selection of a given structured viewer.
	 * @param structuredViewer
	 * @return <code>true</code> if the selection was fired or <code>false</code>
	 * otherwise. 
	 */
	public static boolean fireCurrentSelection(StructuredViewer structuredViewer)
	{
		SelectionChangedEvent event = new SelectionChangedEvent(structuredViewer, structuredViewer.getSelection());
		try
		{
			CoreUtil.invokeObjectMethod(structuredViewer, "fireSelectionChanged", new Object[]{event}, true);
		}
		catch(RuntimeException e)
		{
			HyadesUIPlugin.logError(e);
			return false;
		}
		
		return true;
	}
	
	/**
	 * Calculates the size of one line of text in a composite. 
	 * @param parent
	 * @return int
	 */
	public static int availableRows(Composite parent)
	{
		int fontHeight = (parent.getFont().getFontData())[0].getHeight();
		int displayHeight = parent.getDisplay().getClientArea().height;

		return displayHeight / fontHeight;
	}
	
	/**
	 * Returns the opened editor associated with a given IFile.
	 * @param inputFile
	 * @return IEditorPart
	 */
	public static IEditorPart getOpenEditor(IFile inputFile)
	{
		IWorkbenchPage activePage = getActiveWorkbenchPage();
		if(activePage == null)
			return null;
			
		IEditorReference[] editorReferences = activePage.getEditorReferences();
		for(int i=0, length=editorReferences.length; i<length; i++)
		{
			IEditorPart editorPart = editorReferences[i].getEditor(false);
			if(editorPart == null)
				continue;
				
			Object input = editorPart.getEditorInput();
			if((input != null) && (input instanceof IFileEditorInput))
			{
				IFile file = ((IFileEditorInput)input).getFile();
				if((file != null) && (inputFile != null) && (file.equals(inputFile)))
					return editorPart;
				
				if((file == null) && (inputFile == null))
					return editorPart;
			}
		}

		return null;
	}
	
	/**
	 * Returns an active workbench page.
	 * @return IWorkbenchPage
	 */
	public static IWorkbenchPage getActiveWorkbenchPage()
	{
		IWorkbench workbench = PlatformUI.getWorkbench();
		IWorkbenchPage activePage = null;
		if(workbench.getActiveWorkbenchWindow() != null)
		{
			activePage = workbench.getActiveWorkbenchWindow().getActivePage();
		}
		else if(workbench.getWorkbenchWindows().length > 0)
		{
			activePage = workbench.getWorkbenchWindows()[0].getActivePage();
		}
		
		return activePage;
	}
	
	/**
	 * Opens the editor for the specified file.
	 * 
	 * <p>If <code>editorId</code> is null the default editor is opened.  
	 * If substring is true then the <code>editorId</code> value is considered to be 
	 * first part of the editor id.
	 * 
	 * @param file;
	 * @param editorId;
	 * @param substring;
	 * @return The opened editor part or <code>null</code> if this method was not
	 * able to open the editor.
	 */
	public static IEditorPart openEditor(IFile file, String editorId, boolean substring)
	{
		if(file == null)
			return null;
			
		IWorkbenchPage activePage = getActiveWorkbenchPage();
		if(activePage == null)
			return null;
		
		if("".equals(editorId))
			editorId = null;
		
		try
		{
			if(editorId == null)
			{
				return activePage.openEditor(file);
			}
			else
			{
				IWorkbench workbench = PlatformUI.getWorkbench();
				IEditorDescriptor[] editors = workbench.getEditorRegistry().getEditors(file);
				if(editors != null)
				{
					boolean found = false;
					for(int i=0, length=editors.length; i<length; i++)
					{
						if(substring)
							found = editors[i].getId().startsWith(editorId);
						else
							found = editorId.equals(editors[i].getId());
							
						if(found)
							return activePage.openEditor(file, editors[i].getId());
					}
				}				
			}
		}
		catch(Exception e)
		{
			HyadesUIPlugin.logError(e);
		}
	
		return null;
	}
	
	/**
	 * Shows an error dialog to present to the user why a file saving attempt
	 * has failed.  
	 * @param filePath
	 * @param throwable
	 */
	public static void openSaveFileErrorDialog(Shell shell, String filePath, Throwable throwable)
	{
		if(filePath == null)
			filePath = "";
		
		
		String error = "";
		if(throwable != null)
		{
			if(throwable.getLocalizedMessage() != null)
				error = throwable.getLocalizedMessage();
			else if(throwable.getMessage() != null)
				error = throwable.getMessage();
		}
		
		String message = HyadesUIPlugin.getString("_ERROR_SAVING", new String[]{filePath, error});
		MessageDialog.openError(shell, HyadesUIPlugin.getString("W_ERROR"), message);
	}
	
	
	/**
	 * Applies all the filters of the specified structured viewer to the element
	 * array.
	 * @param structuredViewer
	 * @param elements
	 * @return Object[]
	 */
	public static Object[] applyFilters(StructuredViewer structuredViewer, Object parent, Object[] elements)
	{
		if(elements.length == 0)
			return elements;
			
		ViewerFilter[] filters = structuredViewer.getFilters();
		for(int i = 0, maxi = filters.length; i < maxi; i++)
		{
			elements = filters[i].filter(structuredViewer, parent, elements);
			if(elements.length == 0)
				return elements;
		}
		return elements;
	}
	
	/**
	 * Returns whether the two selections have the same elements.
	 * @param selection1
	 * @param selection2
	 * @return boolean
	 */
	public static boolean areEquals(ISelection selection1, ISelection selection2)
	{
		if(selection1 == selection2)
			return true;
			
		if((selection1 instanceof IStructuredSelection) && (selection2 instanceof IStructuredSelection))
		{
			List list1 = new ArrayList(((IStructuredSelection)selection1).toList());
			List list2 = new ArrayList(((IStructuredSelection)selection2).toList());
			
			if(list1.size() == list2.size())
			{
				list1.removeAll(list2);
				return list1.isEmpty();
			}
		}
		
		return false;	
	}
	
	public static void integerStyledTextVerifyKey(VerifyEvent event)
	{
		if(event.doit)
		{
			switch(event.keyCode)
			{
				case SWT.ARROW_DOWN:			
				case SWT.ARROW_UP:
				case SWT.ARROW_LEFT:
				case SWT.ARROW_RIGHT:
				case SWT.HOME:
				case SWT.END:
					return;
			}
			
			switch(event.character)
			{
				case '0':
				case '1':
				case '2':
				case '3':
				case '4':
				case '5':
				case '6':
				case '7':
				case '8':
				case '9':
				case SWT.BS:
				case SWT.DEL:
					return;

				default:
					event.doit = false;
					return;
			}
		}		
	}
}
