/*******************************************************************************
 * Copyright (c) 2003 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.execution.harness;

import java.io.FileNotFoundException;
import java.net.UnknownHostException;
import java.util.HashMap;

import org.eclipse.core.internal.plugins.ConfigurationElement;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IPluginRegistry;
import org.eclipse.core.runtime.Platform;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.hyades.execution.core.ExecutionComponentStateChangeEvent;
import org.eclipse.hyades.execution.core.IEclipseExecutionComponentFactory;
import org.eclipse.hyades.execution.core.IExecutableObject;
import org.eclipse.hyades.execution.core.IExecutionComponent;
import org.eclipse.hyades.execution.core.IExecutionComponentFactory;
import org.eclipse.hyades.execution.core.IExecutionComponentStateChangeListener;
import org.eclipse.hyades.execution.core.IExecutionEnvironment;
import org.eclipse.hyades.execution.core.IExecutor;
import org.eclipse.hyades.execution.core.INode;
import org.eclipse.hyades.execution.core.IRemoteHyadesComponent;
import org.eclipse.hyades.execution.core.ISession;
import org.eclipse.hyades.execution.core.UnknownDaemonException;
import org.eclipse.hyades.execution.harness.util.ISystemUtility;
import org.eclipse.hyades.execution.harness.util.StandaloneExecutionUtilities;
import org.eclipse.hyades.execution.harness.util.StandaloneSystemUtility;
import org.eclipse.hyades.execution.local.EclipseExecutionComponentFactoryImpl;
import org.eclipse.hyades.execution.local.ExecutionComponentFactoryImpl;
import org.eclipse.hyades.execution.local.NodeImpl;
import org.eclipse.hyades.internal.execution.local.control.AgentControllerUnavailableException;
import org.eclipse.hyades.loaders.util.RegistryReader;
import org.eclipse.hyades.models.common.facades.behavioral.ITest;
import org.eclipse.hyades.models.common.facades.behavioral.ITestSuite;
import org.eclipse.hyades.models.common.facades.behavioral.impl.FacadeResourceFactoryImpl;
import org.eclipse.hyades.models.common.facades.behavioral.impl.HyadesFactory;
import org.eclipse.hyades.models.common.testprofile.TPFTest;
import org.eclipse.hyades.models.common.testprofile.TPFTestSuite;
import org.eclipse.hyades.models.common.testprofile.impl.Common_TestprofilePackageImpl;
import org.eclipse.hyades.models.common.util.ICommonConstants;
import org.eclipse.hyades.models.hierarchy.util.StringUtil;


/**
 * <p>This class provides the mechanism for launching Hyades tests.  The 
 * flow in the launchTest method is depicted in the execution use case
 * realization: <a href="http://dev.eclipse.org/viewcvs/indextools.cgi/~checkout~/hyades-home/docs/components/execution_environment/Launch_A_Test_On_A_Specified_Node_And_Send_Control_Events.html"> UC1.1 </a></p>
 * 
 * <p>This class may used to launch tests from within the workbench (from code
 * running inside the eclipse shell), or it may be used to launch tests from outside
 * the workbench (referred to as "standalone".)  When running standalone, you must
 * use the constructor that accepts a string argument, and pass in a configuration directory
 * as that argument.  The configuration directory should be structured like, and may be,
 * an eclipse directory.  The directory is used to determine information such as
 * the execution components and their mappings to test types, the location of resource
 * bundles, etc.  Also remember that when running standalone, you are responsible for 
 * properly setting the classpath of your JVM so that all necessary classes can be loaded.</p>
 * 
 * An example of how to launch a test standalone is shown below.  Be sure to substitute the
 * correct values for standaloneConfigurationFiles and test suite path.</p><br>
 * 
 * <code><pre>
 *	public static void main( String args[] )
 *	{
 *		TestExecutionHarness harness = new TestExecutionHarness("C:\\Program Files\\eclipse 2.1\\eclipse");
 *		
 *		harness.launchTest(
 *			"C:\\Program Files\\eclipse 2.1\\eclipse\\runtime-workspace\\test\\src\\first.testsuite",
 *			null,
 *			"localhost",
 *			"10002",
 *			"c:\\temp",
 *			"executionresult.execution",
 *			true,
 *			true
 *		);				
 *	}
 * </code></pre>
 * 
 */
public class TestExecutionHarness {
	
	private Object agentLock = new Object();
	private boolean bAgentReady = false;
	private static HashMap testTypeMap = null;
	private static boolean bFactoryInitialized = false;
	private static boolean bStandalone = false;

	private IExecutionComponentFactory factory;
	private ISession session;
	private INode node;
	private IExecutionEnvironment exeEnvironment;
	private IExecutableObject executableObject;
	private IExecutor executor;
	private IRemoteHyadesComponent agent;
	private String[] standaloneConfigurationFiles;
	private ISystemUtility systemUtility;

	public static void main( String args[] )
	{
		TestExecutionHarness harness = new TestExecutionHarness("C:\\Hyades\\_0916_0831_workspace");
		
		System.out.println(harness.launchTest(
			"C:\\Program Files\\eclipse 2.1\\eclipse\\runtime-workspace\\test\\src\\first.testsuite",
			null,
			"localhost",
			"10002",
			"c:\\temp",
			"executionresult.execution",
			true,
			true
		));				
	}


	public TestExecutionHarness() {
		super();
		
		try {
			// try to load an eclipse class
			Class tmp = Class.forName("org.eclipse.core.boot.BootLoader");
			
			// if we didn't just throw an exception, then we know that we're
			// running in Eclipse's JVM.
			bStandalone = false;
			systemUtility = ExecutionHarnessPlugin.getDefault();
		}
		catch ( ClassNotFoundException exc)
		{
			// We couldn't load a class that must be present if we're running from
			// eclipse, so we must be running standalone.  But the caller didn't 
			// specify a config file dir in the constructor, and without that 
			// argument, we can't initialize for standalone execution.
			// We also can't localize the error message, since, without eclipse
			// and without a config file directory, we can't locate the resource
			// bundles.
			throw new IllegalArgumentException("Standalone execution requires that " +
				"a configuration file directory is passed to the TestExecutionHarness constructor!"); 
		}
	}
	
	public TestExecutionHarness(String configurationFileDir)
	{
		super();
		
		try {
			// try to load an eclipse class
			Class tmp = Class.forName("org.eclipse.core.boot.BootLoader");
			
			// if we didn't just throw an exception, then we know that we're
			// running in Eclipse's JVM.  Ignore the specified configurationFileDir
			bStandalone = false;
			systemUtility = ExecutionHarnessPlugin.getDefault();
		}
		catch ( ClassNotFoundException exc)
		{
			// We couldn't load a class that must be present if we're running from
			// eclipse, so we must be running standalone.
			bStandalone = true;
			String[] plugins = StandaloneExecutionUtilities.getConfigFilesFromConfigDir(configurationFileDir);
			setStandaloneConfigurationFiles(plugins);
			
			// Configure our utility class for standalone use			
			String harnessDir = StandaloneExecutionUtilities.getHarnessDir(configurationFileDir);
			systemUtility = new StandaloneSystemUtility(harnessDir);
			
			// Configure the model.hierarchy code (containing the model loaders) for
			// for standalone execution.
			RegistryReader.standaloneConfiguration = new RegistryReader.StandaloneConfiguration();
			RegistryReader.standaloneConfiguration.setEclipseFolder(configurationFileDir);
			init();

		}
	}

	private void init()
	{
		// Initialize our EMF models.
		Resource.Factory.Registry.INSTANCE.getExtensionToFactoryMap().put(ICommonConstants.TEST_SUITE_FILE_EXTENSION, new FacadeResourceFactoryImpl());
		Resource.Factory.Registry.INSTANCE.getExtensionToFactoryMap().put(ICommonConstants.DEPLOYMENT_FILE_EXTENSION, new FacadeResourceFactoryImpl());
		Resource.Factory.Registry.INSTANCE.getExtensionToFactoryMap().put(ICommonConstants.EXECUTION_FILE_EXTENSION, new FacadeResourceFactoryImpl());
		Resource.Factory.Registry.INSTANCE.getExtensionToFactoryMap().put("xmi", new FacadeResourceFactoryImpl());		
		Common_TestprofilePackageImpl.init();
	}

	protected void releaseAgentLock() {
		synchronized(agentLock) {
			bAgentReady=true;
			agentLock.notify();
		}
	}

	/**
	 * This method initializes the ExecutionComponentFactory with all registered execution component
	 * types.  It also populates a testTypeMap to allow determination of which execution components
	 * should be used by which test types.
	 * 
	 * @param factory the factory to be initialized
	 * @param bStandalone whether we are running outside the workbench.  If we are, this method delegates
	 * to a method in StandaloneExecutionUtilities
	 * @throws ClassNotFoundException
	 * @throws FileNotFoundException
	 */
	protected void initializeRegisteredExecutionComponents(
		IExecutionComponentFactory factory, boolean bStandalone) throws ClassNotFoundException, FileNotFoundException
	{
		testTypeMap = new HashMap();

		if ( !bStandalone )
		{
			IEclipseExecutionComponentFactory eclipseFactory = (IEclipseExecutionComponentFactory) factory;
			IPluginRegistry reg = Platform.getPluginRegistry();
	
			IConfigurationElement[] execComps =
				Platform.getPluginRegistry().getConfigurationElementsFor("org.eclipse.hyades.execution.harness.RegisteredExecutionComponentImpl");
				
			for ( int i=0; i<execComps.length; i++)
			{
				String type = execComps[i].getAttribute("type");
				
				String executionComponentName = execComps[i].getAttribute("name");
				eclipseFactory.addExecutionComponent(execComps[i]);
				
				Object map = testTypeMap.get(type);
				if ( map == null )
				{
					map = new HashMap();
					testTypeMap.put(type, map);
				}
				HashMap execCompMap = (HashMap)map;
				
				IConfigurationElement[] supportedTestTypes = execComps[i].getChildren("SupportedTestType");
				for ( int j=0; j<supportedTestTypes.length; j++ )
				{
					execCompMap.put(supportedTestTypes[j].getAttribute("name"), executionComponentName);
				}
			}
			
			mapAdapterClasses("org.eclipse.hyades.execution.harness.ExecutionEnvironmentAdapter", "EXECUTION_ENVIRONMENT_ADAPTER", bStandalone);
			mapAdapterClasses("org.eclipse.hyades.execution.harness.ExecutableObjectAdapter", "EXECUTABLE_OBJECT_ADAPTER", bStandalone);
			
		}
		else
		{
			StandaloneExecutionUtilities.initializeRegisteredExecutionComponents(standaloneConfigurationFiles, testTypeMap, factory);
		}
	}
	
	private void mapAdapterClasses(String configElements, String type, boolean bStandalone) {
		
		if ( !bStandalone )
		{
			IConfigurationElement[] execEnvAdapters = 
				Platform.getPluginRegistry().getConfigurationElementsFor(configElements);
				
			for ( int i=0; i<execEnvAdapters.length; i++)
			{
				String adapterClass = execEnvAdapters[i].getAttribute("class");
			
				Object map = testTypeMap.get(type);
				if ( map == null )
				{
					map = new HashMap();
					testTypeMap.put(type, map);
				}
				HashMap execCompMap = (HashMap)map;
				
				IConfigurationElement[] supportedTestTypes = execEnvAdapters[i].getChildren("SupportedTestType");
				for ( int j=0; j<supportedTestTypes.length; j++ )
				{
					execCompMap.put(supportedTestTypes[j].getAttribute("name"), execEnvAdapters[i]);
				}
			}
		}
	}

	/**
	 * This method queries the testTypeMap and determines which execution component should be
	 * used for a given test type and execution component category.
	 * @param executionComponentType the category of execution component being queried
	 * @param testType the test type being queried
	 * @param bStandalone are we running outside the workbench
	 * @return the type of execution component that should be used (which can be passed to the factory)
	 * @throws ClassNotFoundException
	 */
	protected String getExecutionComponentForTestType( String executionComponentType, String testType, boolean bStandalone )
		throws ClassNotFoundException
	{
		Object temp = testTypeMap.get(executionComponentType);
		if ( temp != null && temp instanceof HashMap )
		{
			HashMap execCompMap = (HashMap) temp;
			temp = execCompMap.get(testType);
			if ( temp != null && temp instanceof String )
			{
				return (String) temp;
			}
		}
		
		String msg = systemUtility.getString("EXEC_NOT_FOUND_ERR_");
		msg = StringUtil.change(msg, "%1", executionComponentType);
		msg = StringUtil.change(msg, "%2", testType);
		
		// We couldn't find the right execution component
		throw new ClassNotFoundException(msg);
		  		
	}
	
	/**
	 * 
	 * Launches the specified test on the specified machine.  The flow in the 
	 * launchTest method is depicted in the execution use case realization: 
	 * <a href="http://dev.eclipse.org/viewcvs/indextools.cgi/~checkout~/hyades-home/docs/components/execution_environment/Launch_A_Test_On_A_Specified_Node_And_Send_Control_Events.html"> UC1.1 </a>

	 * @param suite the test suite to be launched
	 * @param theTest the test to be launched (may be the same as suite, or may 
	 *        be a test case contained within suite.
	 * @param machineName the machine name on which the test will be launched 
	 *        (IP address or host name) 
	 * @param port the port on which the RAC is running (usually 10002)
	 * @param executionResultLocation the directory in which to write the execution history file
	 * @param executionResultName the filename for the execution history file
	 * @param bMonitor should we monitor the test
	 * @param bStandalone are we not launching from the workbench
	 * @return a string containing an error message, or null if no error occurred
	 */
	public synchronized String launchTest( ITestSuite suite, ITest theTest , String machineName, String port, String executionResultLocation, String executionResultName, boolean bMonitor, boolean bStandalone )
	{
		// Reinitialize all fields
		factory = null;
		session = null;
		exeEnvironment = null;
		executableObject = null;
		executor = null;
		agent = null;
		node = null;
	
		String testType = theTest.getType();
		String machine = machineName;
		StringBuffer bufError = new StringBuffer();
				
		// Get a session for this Node.		
		session = sessionConnect(port, session, machine, bufError);
		
		if ( session == null )
		{
			return bufError.toString();
		}
		
		try {
			
			// Get the ExecutionComponentFactory for this session 
			if ( bStandalone )
			{		
				factory = ExecutionComponentFactoryImpl.getInstance(session);
			}
			else
			{
				factory = EclipseExecutionComponentFactoryImpl.getInstance(session);
			}
			if ( !bFactoryInitialized )
			{
				initializeRegisteredExecutionComponents(factory, bStandalone);
				// TODO: investigate this -- is the session not being reused??
				//bFactoryInitialized = true;
			}
				
			// Create the appropriate type of execution environment
			String envType = getExecutionComponentForTestType("ENVIRONMENT", testType, bStandalone);
			exeEnvironment = (IExecutionEnvironment)factory.createExecutionComponentByType(envType);
			session.addChild(exeEnvironment);

			// Call out into an adapter class to configure this execution 
			// environment for this test
			if ( !adaptExecutionEnvironment(suite, theTest, bStandalone, testType, bufError))
				return bufError.toString();
			
			// Create the appropriate type of executor
			String executorType = getExecutionComponentForTestType("EXECUTOR", testType, bStandalone);
			executor = (IExecutor)factory.createExecutionComponentByType(executorType);
			exeEnvironment.addChild(executor);

			// Create the controlling agent and add it to the executor
			String agentType = getExecutionComponentForTestType("AGENT", testType, bStandalone);
			agent = (IRemoteHyadesComponent)factory.createExecutionComponentByType(agentType);
			executor.addChild(agent);
			
			// Create a model loader to process incoming XML fragments, and 
			// start monitoring the test. 
			agent.startMonitoring( new XMLExecutionDataProcessor(null, (TPFTestSuite)suite, executionResultLocation, executionResultName));
			
			// Create a state change listener on the Agent.
			agent.addExecutionComponentStateChangeListener(new IExecutionComponentStateChangeListener() {
				public void stateChanged(ExecutionComponentStateChangeEvent event) {
					// When the agent is ready, release the agent lock
					if(event.getState()==IExecutionComponent.READY) {
						releaseAgentLock();
					}	
				}
			});	
			
			
			// create our executable object
			String execObjType = getExecutionComponentForTestType("EXECUTABLEOBJECT", testType, bStandalone);
			executableObject = executor.getCompatibleExecutableObject(execObjType);

			if(executableObject!=null) {

				// Call out into an adapter class to configure this execution 
				// environment for this test
				if ( !adaptExecutableObject(suite, theTest, bStandalone, testType, bufError))
				{
					return bufError.toString();
				}
				
				// Set our executable object -- this will move it to the remote side	
				executor.setExecutableObject(executableObject);
				
				// Wait for the agent to become active
				waitForActiveAgent();
				
				// Start the test
				executor.launch();
			}
			

		}			
		catch(ClassNotFoundException e) {
			
			systemUtility.logError(e);			
			return StringUtil.change(systemUtility.getString("CLASS_NOT_FOUND_ERR_"), "%1", e.getMessage());			
		}
		catch(Throwable t) {
			
			systemUtility.logError(t);			
			return t.getMessage();			
			
		}
		finally {
			session.release();
		}
		
		return null;
	}
	
	
	private void waitForActiveAgent() {
		int temp = executor.getState();
		
		if ( agent.getState() == IExecutionComponent.SUSPENDED)
		{
			// Block and wait for the agent to come up.
			while(!bAgentReady) {
				synchronized(agentLock) {
					try {
						agentLock.wait();
					}
					catch(InterruptedException e) {
						/* We can ignore this */
					}
				}
			}
		}
	}
	
	private Object getAdapterInstance(String executionComponentType, String testType, 
		boolean bStandalone ) throws ClassNotFoundException
	{
		Object temp = testTypeMap.get(executionComponentType);
		if ( temp != null && temp instanceof HashMap )
		{
			HashMap execCompMap = (HashMap) temp;
			temp = execCompMap.get(testType);
			try {
				if ( bStandalone )
				{
					if ( temp != null && temp instanceof String )
					{
						return Class.forName((String)temp).newInstance();
					}
				}
				else
				{
					if ( temp != null && temp instanceof ConfigurationElement)
					{
						return ((ConfigurationElement)temp).createExecutableExtension("class");
					}
				}
			}
			catch ( Exception accessExc )
			{
			}
			
		}
		String msg = systemUtility.getString("EXEC_NOT_FOUND_ERR_");
		msg = StringUtil.change(msg, "%1", executionComponentType);
		msg = StringUtil.change(msg, "%2", testType);
		
		// We couldn't find the right execution component
		throw new ClassNotFoundException(msg);
	}

	private boolean adaptExecutableObject(ITestSuite suite, ITest theTest, boolean bStandalone, String testType, StringBuffer bufError) {
		// Call out into an adapter class to configure this execution 
		// environment for this test
		IExecutableObjectAdapter execObjAdapter = null;
		try {
			execObjAdapter = (IExecutableObjectAdapter) getAdapterInstance("EXECUTABLE_OBJECT_ADAPTER", testType, bStandalone);
		}
		catch ( Exception accessExc )
		{
			systemUtility.logError(accessExc);			
			bufError.append(StringUtil.change(systemUtility.getString("EXEC_ENV_ERR_"), "%1", testType));
			return false;					
		}
		if ( execObjAdapter != null )
			execObjAdapter.setupExecutableObject(executableObject, (TPFTestSuite)suite, (TPFTest)theTest);
			
		return true;
	}

	private boolean adaptExecutionEnvironment(ITestSuite suite, ITest theTest, boolean bStandalone, String testType, StringBuffer bufError) {
		// Call out into an adapter class to configure this execution 
		// environment for this test
		
		IExecutionEnvironmentAdapter envAdapter = null;
		try {
			envAdapter = (IExecutionEnvironmentAdapter)getAdapterInstance("EXECUTION_ENVIRONMENT_ADAPTER", testType, bStandalone);
		}
		catch ( Exception accessExc )
		{
			systemUtility.logError(accessExc);			
			bufError.append(StringUtil.change(systemUtility.getString("EXEC_ENV_ERR_"), "%1", testType));
			return false;				
		}
		if ( envAdapter != null )
			envAdapter.setupExecutionEnvironment(exeEnvironment, (TPFTestSuite)suite, (TPFTest)theTest);
			
		return true;
	}

	private ISession sessionConnect(String port, ISession session, String machine, StringBuffer bufError) {
		INode node;
		bufError.setLength(0);
		
		try {
			node = new NodeImpl(machine);
			session = node.connect(port, null);
		}
		catch (UnknownHostException e) {
			bufError.append(StringUtil.change(systemUtility.getString("TIMEOUT_NODE_ERROR_"), "%1", machine));
		}
		catch (AgentControllerUnavailableException e) {
			bufError.append(StringUtil.change(systemUtility.getString("AGENT_UNAV_ERROR_"), "%1", machine));
		}
		catch (UnknownDaemonException e) {
			
			bufError.append(StringUtil.change(systemUtility.getString("AGENT_UNAV_ERROR_"), "%1", machine));
		}
		catch(Exception e)
		{
			systemUtility.logError(e);			
			bufError.append(StringUtil.change(systemUtility.getString("RUN_FAILED_ERR_"), "%1", machine));
		}
		return session;
	}

	/**
	 * 
	 * Launches the specified test on the specified machine.  The flow in the 
	 * launchTest method is depicted in the execution use case realization: 
	 * <a href="http://dev.eclipse.org/viewcvs/indextools.cgi/~checkout~/hyades-home/docs/components/execution_environment/Launch_A_Test_On_A_Specified_Node_And_Send_Control_Events.html"> UC1.1 </a>

	 * @param rootSuitePath the fully qualified path to the test suite to be launched
	 * @param testID the ID of the test to be launched (may be null, or may 
	 *        be the ID of a test case contained within suite.
	 * @param machineName the machine name on which the test will be launched 
	 *        (IP address or host name) 
	 * @param port the port on which the RAC is running (usually 10002)
	 * @param executionResultLocation the directory in which to write the execution history file
	 * @param executionResultName the filename for the execution history file
	 * @param bMonitor should we monitor the test
	 * @param bStandalone are we not launching from the workbench
	 * @return a string containing an error message, or null if no error occurred
	 */
	public String launchTest( String rootSuitePath, String testID, String machineName, String port, String executionResultLocation, String executionResultName, boolean bMonitor, boolean bStandalone )
	{
		ITest theTest = null;
		ITestSuite suite = null;

		// Load the test suite.
		try {
			suite = HyadesFactory.INSTANCE.loadTestSuite(rootSuitePath);
		}
		catch ( Exception exc )
		{
			systemUtility.logError(exc);
			return StringUtil.change(systemUtility.getString("LOAD_SUITE_ERR_"), "%1", rootSuitePath);
		}
		
		// If we're executing a test case and not a suite, get the test case.
		if ( testID != null && testID.length() != 0 )
		{
			theTest = (ITest) HyadesFactory.INSTANCE.getObjectByID(suite, testID);
		}
		else
		{
			theTest = suite;
		}
		
		return launchTest(suite, theTest, machineName, port, executionResultLocation, executionResultName, bMonitor, bStandalone);
		
	}
	
	/**
	 * @return an array of configuration filenames (plugin.xml files) for use when
	 * we are running standalone.
	 */
	private String[] getStandaloneConfigurationFiles() {
		return standaloneConfigurationFiles;
	}

	/**
	 * Sets the standaloneConfigurationFiles member to an array of strings containing
	 * all registered execution components and adapters.
	 * @param standaloneConfigurationFiles
	 */
	private void setStandaloneConfigurationFiles(String[] standaloneConfigurationFiles) {
		this.standaloneConfigurationFiles = standaloneConfigurationFiles;
	}

}
