package org.eclipse.hyades.internal.logging.core;

/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

import java.awt.EventQueue;
import java.io.ByteArrayOutputStream;
import java.io.PrintWriter;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.util.Collection;
import java.util.Date;
import java.util.Enumeration;
import java.util.Iterator;
import java.util.Map;
import java.util.StringTokenizer;
import java.util.Vector;

import org.eclipse.hyades.logging.core.Guid;
import org.eclipse.hyades.logging.core.XmlUtility;

public final class XmlGenerator {

    /* The OS-independent character(s) corresponding to a line separator */
    private String lineSeparator = null;

    /* String representation of the logging level used  */
    private String loggingLevel = null;

    /* The name of the logger that is used to log the strings generated by this class */
    private String loggerName = null;

    /* Used to determines the format that is to be used to log data */
    private boolean format;

    /* Indicates the current xml level that this generator is in */
    private int currentLevel;

    /* The generator will not go passed this xml level */
    private int maxLevels;

    /* This generator will includes a time stamp attribute with every logged object if 'includeTimeStamp' is set */
    private boolean includeTimeStamp = false;

    private final byte SIMPLE_TYPE = 1;
    private final byte COMPLEX_TYPE = 2;

    public XmlGenerator() {
    }

    /**
     * Constructor: Used to initialize 'loggerName'.  The 'reset()' method may need to 
     * be invoked after the invoking the constructor.
     * 
     * @param loggerName	-	The logger name that is using this generator
     */
    public XmlGenerator(String loggerName) {
        this.loggerName = loggerName;
    }

    /**
     * Captures the stack trace of the parameter throwable (i.e. exception or error) in 
     * a String.
     *
     * @param throwable exception or error which the stack trace is captured.
     * @return String representation of the stack trace of the parameter throwable.
     */
    public static synchronized String getThrowableStackTrace(Throwable throwable) {

        if (throwable == null)
            return "null";

        else {
            try {
                ByteArrayOutputStream stream = new ByteArrayOutputStream();
                PrintWriter writer = new PrintWriter(stream, true);

                throwable.printStackTrace(writer);

                writer.flush();
                writer.close();

                stream.flush();
                stream.close();

                return (stream.toString().trim());
            }
            catch (Exception e) {
                return (throwable.toString());
            }
        }
    }

    /**
     * Captures the stack trace of the parameter thread in a String.
     *
     * @param thread thread which the stack trace is captured.
     * @return String representation of the stack trace of the parameter thread.
     */
    //MUST IMPLEMENT PARAMETER THREAD STACK TRACE INSTEAD OF A STACK TRACE OF THE CURRENT THREAD:
    public static String getThreadStackTrace(Thread thread) {

        if (thread == null)
            return "null";

        else {
            StringBuffer stackTrace = new StringBuffer("Stack trace of ");
            stackTrace.append(thread.toString());
            stackTrace.append(Constants.LINE_SEPARATOR);

            //Start of temporary fix:
            String stack = getThrowableStackTrace(new Throwable());

            //Remove Throwable.toString() header:
            stackTrace.append(stack.substring(stack.indexOf(Constants.LINE_SEPARATOR) + 2));

            return (stackTrace.toString());
        }
    }

    /**
     * Initializes variables that can be changed between writes.
     * 
     * @param	loggingLevel	 -	The logging level that this generator will adapt
     * 			format			 -	Indicates the type of the format that this generator will use
     * 			maxLevels		 -	Indicates the xml level that this generator should not pass
     * 			includeTimeStamp - 	A time stamp attribute is included with each write if this parameter is set
     * 
     */

    public void reset(String loggingLevel, boolean format, int maxLevels, boolean includeTimeStamp) {
        this.loggingLevel = loggingLevel;
        this.format = format;
        this.maxLevels = maxLevels;
        this.includeTimeStamp = includeTimeStamp;
    }

    public void reset(String loggingLevel, boolean format, int maxLevels) {
        this.reset(loggingLevel, format, maxLevels, false);
    }

    /**
     * Generate the XML string that is to be used to log 'object'
     * 
     * @param object	-	The generated XML string corresponds to 'object'
     * @return String	-	The generated XML string of 'object'
     */
    public String objectToXML(Object object) {

        if (object == null)
            return nullObjectToXML(null, null, loggerName, loggingLevel, null);

        if (getType(object.getClass()) == SIMPLE_TYPE)
            return simpleObjectToXML(object, null, loggerName, loggingLevel, null);

        if (format)
            lineSeparator = Constants.LINE_SEPARATOR;
        else
            lineSeparator = "";

        currentLevel = -1;

        Method[] methods = object.getClass().getMethods();
        boolean methodFound = false;
        for (int counter = methods.length - 1; counter >= 0; counter--) {
            if (methods[counter].getName().equals("toCanonicalXMLString") || methods[counter].getName().equals("externalizeCanonicalXmlString")) {
                methodFound = true;
                break;
            }
        }

        /* Attempt to invoke toCanonicalXMLString() of 'object' to generate a customly designed XML string */
        if (methodFound) {
            try {
                return (createCanonicalXMLString(object, null, loggerName, loggingLevel, null));
            }
            catch (Throwable t) {
            }
        }

        /* Assertion: 'object' does not have a valid toCanonicalXMLString().  Begin creating the XML string */
        return (createXMLTag(object, null).toString().trim());

    }

    /**
     * Create the XML string corresponding to 'object'
     * 
     * @param	object		- The XML string generated corresponds to 'object'
     * 		objectsName		- The name of the object
     * 
     * @return StringBuffere	- The XML string corresponding to 'object'
     */

    private StringBuffer createXMLTag(Object object, String objectsName) {

        StringBuffer xmlTag = new StringBuffer();
        xmlTag.append(indention(++currentLevel));

        /* In the case where 'object' is null */
        if (object == null) {
            if (currentLevel == 0)
                xmlTag.append(nullObjectToXML(null, null, loggerName, loggingLevel, objectsName));
            else
                xmlTag.append(nullObjectToXML(null, null, null, null, objectsName));

            xmlTag.append(lineSeparator);
            return xmlTag;
        }

        /* Parameters that will be used to form the XML string corresponding to 'object' */
        Class objectsClass = object.getClass();
        String className;
        String classNameTag;

        if (objectsClass.isArray()) {
            classNameTag = "Array";
            className = classNameTag.concat(getClassName(objectsClass));
        }
        else {
            className = objectsClass.getName().trim();
            classNameTag = removeTagNameChars(className.substring(className.lastIndexOf('.') + 1));
        }

        /* Generate a unique id for the object that will be logged */
        String id = className + "_" + new Guid().toString();

        /* Attempt to invoke toCanonicalXMLString() method of 'object' */
        try {
            if (currentLevel == 0)
                xmlTag.append(createCanonicalXMLString(object, id, loggerName, loggingLevel, objectsName));
            else
                xmlTag.append(createCanonicalXMLString(object, id, null, null, objectsName));

            xmlTag.append(lineSeparator);
        }

        catch (Throwable t) { /* Don't do anything.  Allow the normal flow of the program. */
        }

        /* Assertion: 'object' does not have a toCanonicalXMLString() method.  Begin generating the XML string */
        /* In case that 'object' is of simple type. ( as defined by getType() ) */
        if (getType(objectsClass) == SIMPLE_TYPE) {

            if (currentLevel == 0)
                xmlTag.append(simpleObjectToXML(object, id, loggerName, loggingLevel, objectsName));
            else
                xmlTag.append(simpleObjectToXML(object, id, null, null, objectsName));

            xmlTag.append(lineSeparator);
            return xmlTag;

        }

        /* Assertion: 'object' is of complex type */
        /* Set the common attribute */
        xmlTag.append("<");
        xmlTag.append(classNameTag);
        xmlTag.append(getNameAttribute(objectsName));
        xmlTag.append(getIdAttribute(id));
        xmlTag.append(getTimeStamp());
        if (currentLevel == 0) {
            xmlTag.append(getLoggerNameAttribute(loggerName));
            xmlTag.append(getLevelAttribute(loggingLevel));
        }

        /* Case: Array, Collection, Enumeration, or Iterator */
        if ((objectsClass.isArray()) || (object instanceof Collection) || (object instanceof Enumeration) || (object instanceof Iterator)) {
            if (!objectsClass.isArray()) {
                /* Case: Collection */
                if (object instanceof Collection)
                    object = ((Collection) object).toArray();

                /* Case: Enumeration */
                else if (object instanceof Enumeration) {
                    Vector enumerationElements = new Vector();

                    while (((Enumeration) (object)).hasMoreElements()) {
                        enumerationElements.add(((Enumeration) (object)).nextElement());
                    }

                    object = enumerationElements.toArray();
                }

                /* Case: Iterator */
                else if (object instanceof Iterator) {
                    Vector iteratorElements = new Vector();
                    while (((Iterator) (object)).hasNext()) {
                        iteratorElements.add(((Iterator) (object)).next());
                    }

                    object = iteratorElements.toArray();
                }
            }

            xmlTag.append(">");
            xmlTag.append(lineSeparator);

            if (currentLevel == maxLevels)
                xmlTag.append(indention(currentLevel + 1) + getEndOfNestingTag());
            else
                logEachEntry(object, xmlTag, objectsClass, id, objectsName);

            endTag(classNameTag, xmlTag);

        }

        /* Case: Thread */
        else if (object instanceof Thread) {

            /* Append customized attributes for thread */
            xmlTag.append(" Name=\"");
            xmlTag.append(((Thread) object).getName().trim());
            xmlTag.append("\"");
            xmlTag.append(" Priority=\"");
            xmlTag.append(((Thread) object).getPriority());
            xmlTag.append("\"");
            xmlTag.append(" Group=\"");
            xmlTag.append(((Thread) object).getThreadGroup().getName().trim());
            xmlTag.append("\">");
            xmlTag.append(lineSeparator);

            /* Write the current stack trace */
            xmlTag.append(stackTraceToXML(currentLevel));

            endTag(classNameTag, xmlTag);

        }

        else if (object instanceof Throwable) {
            /* Append customized attributes for thread */
            String message = ((Throwable) object).getMessage();

            if (message != null) {
                xmlTag.append(" Message=\"");
                xmlTag.append(message);
                xmlTag.append("\"");
            }

            xmlTag.append(">");
            xmlTag.append(lineSeparator);

            /* Capture stack trace of the throwable */
            StringTokenizer tokens = new StringTokenizer(getThrowableStackTrace((Throwable) object), Constants.LINE_SEPARATOR);

            //Remove the Throwable.toString() header:
            tokens.nextToken();

            try {
                while (tokens.hasMoreTokens()) {
                    xmlTag.append(indention(currentLevel + 1));
                    xmlTag.append(simpleObjectToXML(tokens.nextToken().trim(), null, null, null, objectsName));
                    xmlTag.append(lineSeparator);
                }
            }

            catch (Exception e) {
                xmlTag.append(indention(currentLevel + 1));
                xmlTag.append(simpleObjectToXML(getThrowableStackTrace((Throwable) object), null, null, null, objectsName));
                xmlTag.append(lineSeparator);
            }

            endTag(classNameTag, xmlTag);

        }

        /* Case: Map */
        else if (object instanceof Map) {
            xmlTag.append(">");
            xmlTag.append(lineSeparator);

            if (currentLevel == maxLevels) {
                xmlTag.append(indention(currentLevel + 1));
                xmlTag.append(getEndOfNestingTag());
            }

            /* Append customized attribute */
            else {
                Iterator keyList = ((Map) (object)).keySet().iterator();

                while (keyList.hasNext()) {
                    Object key = keyList.next();

                    xmlTag.append(indention(currentLevel + 1)).append("<Entry>").append(lineSeparator).append(indention(currentLevel + 2)).append("<Key>").append(lineSeparator).append(indention(currentLevel + 3)).append(createXMLTag(key, null).toString().trim());
                    currentLevel--;

                    xmlTag.append(indention(currentLevel + 2)).append("</Key>").append(lineSeparator).append(indention(currentLevel + 2)).append("<Value>").append(lineSeparator).append(indention(currentLevel + 3)).append(createXMLTag(((Map) (object)).get(key), null).toString().trim());
                    currentLevel--;

                    xmlTag.append(indention(currentLevel + 2)).append("</Value>").append(lineSeparator).append(indention(currentLevel + 1)).append("</Entry>").append(lineSeparator);
                }
            }

            endTag(classNameTag, xmlTag);

        }

        /* Case: EventQueue */
        else if (object instanceof EventQueue) {
            xmlTag.append("/>");
            xmlTag.append(lineSeparator);
        }

        /* Case: Class */
        else if (object instanceof Class) {
            /* Begin appending customized attributes */
            xmlTag.append(" Name=\"").append(getClassName((Class) (object))).append("\"").append(" Type=\"");

            if (isPrimitive((Class) object))
                xmlTag.append("primitive");
            else if (((Class) (object)).isArray())
                xmlTag.append("array");
            else if (((Class) (object)).isInterface())
                xmlTag.append("interface");
            else
                xmlTag.append("class");

            xmlTag.append("\"");

            /* Append the package attribute */

            xmlTag.append(" Package=\"");
            try {
                xmlTag.append(((Class) (object)).getPackage().getName());
            }
            catch (Exception e) {
                xmlTag.append("null");
            }

            xmlTag.append("\"");

            /* Append the modifiers attribute */
            xmlTag.append(" Modifers=\"").append(Modifier.toString(((Class) (object)).getModifiers())).append("\"");

            /* Append the super class attribute */
            xmlTag.append(" Superclass=\"");
            try {
                xmlTag.append(getClassName(((Class) (object)).getSuperclass()));
            }
            catch (Exception e) {
                xmlTag.append("null");
            }

            xmlTag.append("\"/>").append(lineSeparator);

        }

        /* Case: Default */
        else {
            /* Used to store simple getter methods */
            Vector simpleGetterMethods = new Vector();

            /* Used to store complex getter methods */
            Vector complexGetterMethods = new Vector();
            Method getterMethod;
            String methodName;
            Object returnObject;
            Method[] objectsMethods = new Method[0];

            try {
                objectsMethods = objectsClass.getMethods();
            }
            catch (Exception e) {
            }

            /* Iterate through all the methods of 'object' and find simpleGetterMethods and complexGetterMethods */
            for (int counter = objectsMethods.length - 1; counter >= 0; counter--) {
                methodName = objectsMethods[counter].getName().trim();
                if ((methodName.length() > 3) && (methodName.startsWith("get")) && (objectsMethods[counter].getParameterTypes().length == 0)) {
                    if (getType(objectsMethods[counter].getReturnType()) == SIMPLE_TYPE)
                        simpleGetterMethods.addElement(objectsMethods[counter]);
                    else
                        complexGetterMethods.addElement(objectsMethods[counter]);
                }
            }

            /* End 'xmlTag' if no getter methods were found */
            if ((simpleGetterMethods.size() + complexGetterMethods.size()) == 0) {
                xmlTag.append("/>");
                xmlTag.append(lineSeparator);
                return xmlTag;
            }

            /* Iterate through each simple getter method and append the method name followed 
             * by its returned value to 'xmlTag' */
            for (int counter = simpleGetterMethods.size() - 1; counter >= 0; counter--) {
                getterMethod = (Method) simpleGetterMethods.elementAt(counter);
                methodName = getterMethod.getName().trim().substring(3);

                xmlTag.append(" ");
                xmlTag.append(methodName);
                xmlTag.append("=\"");

                try {
                    returnObject = getterMethod.invoke(object, null);
                    if (returnObject != null)
                        xmlTag.append(XmlUtility.normalize(returnObject.toString().trim()));
                    else
                        throw new Exception();
                }
                /* Failed to invoke the getter method */
                catch (Exception e) {
                    xmlTag.append("null");
                }

                xmlTag.append("\"");
            }

            if (complexGetterMethods.size() == 0) {
                xmlTag.append("/>");
                xmlTag.append(lineSeparator);

                return xmlTag;
            }

            xmlTag.append(">").append(lineSeparator);

            if (currentLevel == maxLevels) {
                xmlTag.append(indention(currentLevel + 1));
                xmlTag.append(getEndOfNestingTag());
            }

            else {
                /* Iterate through each complex getter method and append the xml string of 
                 * its returned object */
                for (int counter = complexGetterMethods.size() - 1; counter >= 0; counter--) {
                    getterMethod = (Method) complexGetterMethods.elementAt(counter);

                    try {
                        returnObject = getterMethod.invoke(object, null);
                        if (returnObject != null) {
                            xmlTag.append(createXMLTag(returnObject, getterMethod.getName().trim().substring(3)).toString());
                            currentLevel--;
                        }
                        else
                            throw new Exception();
                    }

                    /* In case there are problems of invoking the method, append the xml string
                     * of a null object */
                    catch (Exception e) {
                        xmlTag.append(indention(currentLevel + 1));

                        Class returnTypeClass = getterMethod.getReturnType();
                        String returnTypeName;

                        if (returnTypeClass.isArray())
                            returnTypeName = "Array";

                        else {
                            returnTypeName = returnTypeClass.getName().trim();
                            returnTypeName = removeTagNameChars(returnTypeName.substring(returnTypeName.lastIndexOf('.') + 1));
                        }

                        xmlTag.append(nullObjectToXML(returnTypeName, null, null, null, getterMethod.getName().trim().substring(3)));
                        xmlTag.append(lineSeparator);
                    }
                }
            }

            endTag(classNameTag, xmlTag);
        }
        return xmlTag;
    }

    /**
     * A helper method used to end the 'classNameTag'
     *
     * @param classNameTag	- The class name tag that is terminated
     *        xmlTag		- The string buffer that this termination tag is appended to
     */
    private void endTag(String classNameTag, StringBuffer xmlTag) {
        xmlTag.append(indention(currentLevel)).append("</").append(classNameTag).append(">").append(lineSeparator);
    }

    /**
     * A helper method used to log each entry of an array, collection, enumeration, or an iterator
     *
     * @param object	- The data structure which is used to log each of its entries
     * 	  	   xmlTag	- The string buffer that is used to append the xml string for each entry
     */
    private void logEachEntry(Object object, StringBuffer xmlTag, Class objectsClass, String id, String objectsName) {
        /* Iterate through each element and append their xml string to xmlTag */
        /* The for-loops are designed to decrement instead of incrementing becuase the JVM is optimized to compare integers between 0 to +5 */
        /* Case: boolean array */
        if (object instanceof boolean[]) {
            int size = ((boolean[]) object).length - 1;
            for (int counter = size; counter >= 0; counter--)
                xmlTag.append(indention(currentLevel + 1)).append(primitiveToXML(((boolean[]) object)[Math.abs(counter - size)])).append(lineSeparator);
        }

        /* Case: char array */
        else if (object instanceof char[]) {
            int size = ((char[]) object).length - 1;
            for (int counter = size; counter >= 0; counter--)
                xmlTag.append(indention(currentLevel + 1)).append(primitiveToXML(((char[]) object)[Math.abs(counter - size)])).append(lineSeparator);
        }

        /* Case: byte array */
        else if (object instanceof byte[]) {
            int size = ((byte[]) object).length - 1;
            for (int counter = size; counter >= 0; counter--)
                xmlTag.append(indention(currentLevel + 1)).append(primitiveToXML(((byte[]) object)[Math.abs(counter - size)])).append(lineSeparator);
        }

        /* Case: short array */
        else if (object instanceof short[]) {
            int size = ((short[]) object).length - 1;
            for (int counter = size; counter >= 0; counter--)
                xmlTag.append(indention(currentLevel + 1)).append(primitiveToXML(((short[]) object)[Math.abs(counter - size)])).append(lineSeparator);
        }

        /* Case: int array */
        else if (object instanceof int[]) {
            int size = ((int[]) object).length - 1;
            for (int counter = size; counter >= 0; counter--)
                xmlTag.append(indention(currentLevel + 1)).append(primitiveToXML(((int[]) object)[Math.abs(counter - size)])).append(lineSeparator);
        }

        /* Case: long array */
        else if (object instanceof long[]) {
            int size = ((long[]) object).length - 1;
            for (int counter = size; counter >= 0; counter--)
                xmlTag.append(indention(currentLevel + 1)).append(primitiveToXML(((long[]) object)[Math.abs(counter - size)])).append(lineSeparator);
        }

        /* Case: float array */
        else if (object instanceof float[]) {
            int size = ((float[]) object).length - 1;
            for (int counter = size; counter >= 0; counter--)
                xmlTag.append(indention(currentLevel + 1)).append(primitiveToXML(((float[]) object)[Math.abs(counter - size)])).append(lineSeparator);
        }

        /* Case: double array */
        else if (object instanceof double[]) {
            int size = ((double[]) object).length - 1;
            for (int counter = size; counter >= 0; counter--)
                xmlTag.append(indention(currentLevel + 1)).append(primitiveToXML(((double[]) object)[Math.abs(counter - size)])).append(lineSeparator);
        }

        /* Case: Default (includes complex array, collection, enumeration, and iterator) */
        else {
            if (currentLevel == maxLevels) {
                xmlTag.append(indention(currentLevel + 1));
                xmlTag.append(getEndOfNestingTag());
            }

            else {
                /* Iterate through each element of the array and append their xml string
                 * to xmlTag */
                int counter2 = 0;
                for (int counter = ((Object[]) object).length - 1; counter >= 0; counter--) {
                    /* Case: The entry is null */
                    if (((Object[]) object)[counter2] == null) {
                        xmlTag.append(indention(currentLevel + 1));
                        if ((currentLevel + 1) == 0)
                            xmlTag.append(nullObjectToXML(objectsClass.getComponentType().getName().trim(), id, loggerName, loggingLevel, objectsName));
                        else
                            xmlTag.append(nullObjectToXML(objectsClass.getComponentType().getName().trim(), id, null, null, objectsName));
                        xmlTag.append(lineSeparator);
                    }
                    /* Case: The entry is not null */
                    else {
                        xmlTag.append(createXMLTag(((Object[]) object)[counter2], null).toString());
                        currentLevel--;
                    }
                    counter2++;
                }
            }
        }
    }

    /**
     * Wrap a boolean value in a XML empty tag.
     *
     * @param valueAttribute 	- Boolean attribute value of the tag.
     *	  msgLoggerName	- The logger name of the message logger
     *	  msgLoggingLevel - The logging level of this message
     *	  timeStamp		- Indicates whether a time stamp should/should not be included
     *
     * @return String 		- Representation of the boolean XML empty tag fragment.
     */
    public static String primitiveToXML(boolean valueAttribute, String msgLoggerName, String msgLoggingLevel, boolean timeStamp) {
        return ("<boolean" + (timeStamp ? " time_stamp=\"" + new Date(System.currentTimeMillis()).toString() + "\"" : "") + getLoggerNameAttribute(msgLoggerName) + getLevelAttribute(msgLoggingLevel) + " Value=\"" + String.valueOf(valueAttribute) + "\"/>");
    }

    public static String primitiveToXML(boolean valueAttribute) {
        return (primitiveToXML(valueAttribute, null, null, false));
    }

    public static String primitiveToXML(boolean valueAttribute, String msgLoggerName) {
        return (primitiveToXML(valueAttribute, msgLoggerName, null, false));
    }

    /**
     * Wrap a char value in a XML empty tag.
     *
     * @param valueAttribute 	- Char attribute value of the tag.
     *	  msgLoggerName	- The logger name of the message logger
     *	  msgLoggingLevel - The logging level of this message
     *	  timeStamp		- Indicates whether a time stamp should/should not be included
     *
     * @return String 		- Representation of the char XML empty tag fragment.
     */
    public static String primitiveToXML(char valueAttribute, String msgLoggerName, String msgLoggingLevel, boolean timeStamp) {
        return ("<char" + (timeStamp ? " time_stamp=\"" + new Date(System.currentTimeMillis()).toString() + "\"" : "") + getLoggerNameAttribute(msgLoggerName) + getLevelAttribute(msgLoggingLevel) + " Value=\"" + XmlUtility.normalize(String.valueOf(valueAttribute)) + "\"/>");
    }

    public static String primitiveToXML(char valueAttribute) {
        return (primitiveToXML(valueAttribute, null, null, false));
    }

    public static String primitiveToXML(char valueAttribute, String msgLoggerName) {
        return (primitiveToXML(valueAttribute, msgLoggerName, null, false));
    }

    /**
     * Wrap a byte value in a XML empty tag.
     *
     * @param valueAttribute 	- Byte attribute value of the tag.
     *	  msgLoggerName	- The logger name of the message logger
     *	  msgLoggingLevel - The logging level of this message
     *	  timeStamp		- Indicates whether a time stamp should/should not be included
     *
     * @return String 		- Representation of the byte XML empty tag fragment.
     */
    public static String primitiveToXML(byte valueAttribute, String msgLoggerName, String msgLoggingLevel, boolean timeStamp) {
        return ("<byte" + (timeStamp ? " time_stamp=\"" + new Date(System.currentTimeMillis()).toString() + "\"" : "") + getLoggerNameAttribute(msgLoggerName) + getLevelAttribute(msgLoggingLevel) + " Value=\"" + String.valueOf(valueAttribute) + "\"/>");
    }

    public static String primitiveToXML(byte valueAttribute) {
        return (primitiveToXML(valueAttribute, null, null, false));
    }

    public static String primitiveToXML(byte valueAttribute, String msgLoggerName) {
        return (primitiveToXML(valueAttribute, msgLoggerName, null, false));
    }

    /**
     * Wrap a short value in a XML empty tag.
     *
     * @param valueAttribute 	- Short attribute value of the tag.
     *	  msgLoggerName	- The logger name of the message logger
     *	  msgLoggingLevel - The logging level of this message
     *	  timeStamp		- Indicates whether a time stamp should/should not be included
     *
     * @return String 		- Representation of the short XML empty tag fragment.
     */
    public static String primitiveToXML(short valueAttribute, String msgLoggerName, String msgLoggingLevel, boolean timeStamp) {
        return ("<short" + (timeStamp ? " time_stamp=\"" + new Date(System.currentTimeMillis()).toString() + "\"" : "") + getLoggerNameAttribute(msgLoggerName) + getLevelAttribute(msgLoggingLevel) + " Value=\"" + String.valueOf(valueAttribute) + "\"/>");
    }

    public static String primitiveToXML(short valueAttribute) {
        return (primitiveToXML(valueAttribute, null, null, false));
    }

    public static String primitiveToXML(short valueAttribute, String msgLoggerName) {
        return (primitiveToXML(valueAttribute, msgLoggerName, null, false));
    }

    /**
     * Wrap an int value in a XML empty tag.
     *
     * @param valueAttribute 	- Int attribute value of the tag.
     *	  msgLoggerName	- The logger name of the message logger
     *	  msgLoggingLevel - The logging level of this message
     *	  timeStamp		- Indicates whether a time stamp should/should not be included
     *
     * @return String 		- Representation of the int XML empty tag fragment.
     */
    public static String primitiveToXML(int valueAttribute, String msgLoggerName, String msgLoggingLevel, boolean timeStamp) {
        return ("<int" + (timeStamp ? " time_stamp=\"" + new Date(System.currentTimeMillis()).toString() + "\"" : "") + getLoggerNameAttribute(msgLoggerName) + getLevelAttribute(msgLoggingLevel) + " Value=\"" + String.valueOf(valueAttribute) + "\"/>");
    }

    public static String primitiveToXML(int valueAttribute) {
        return (primitiveToXML(valueAttribute, null, null, false));
    }

    public static String primitiveToXML(int valueAttribute, String msgLoggerName) {
        return (primitiveToXML(valueAttribute, msgLoggerName, null, false));
    }

    /**
     * Wrap a long value in a XML empty tag.
     *
     * @param valueAttribute 	- Long attribute value of the tag.
     *	  msgLoggerName	- The logger name of the message logger
     *	  msgLoggingLevel - The logging level of this message
     *	  timeStamp		- Indicates whether a time stamp should/should not be included
     *
     * @return String representation of the long XML empty tag fragment.
     */
    public static String primitiveToXML(long valueAttribute, String msgLoggerName, String msgLoggingLevel, boolean timeStamp) {
        return ("<long" + (timeStamp ? " time_stamp=\"" + new Date(System.currentTimeMillis()).toString() + "\"" : "") + getLoggerNameAttribute(msgLoggerName) + getLevelAttribute(msgLoggingLevel) + " Value=\"" + String.valueOf(valueAttribute) + "\"/>");
    }

    public static String primitiveToXML(long valueAttribute) {
        return (primitiveToXML(valueAttribute, null, null, false));
    }

    public static String primitiveToXML(long valueAttribute, String msgLoggerName) {
        return (primitiveToXML(valueAttribute, msgLoggerName, null, false));
    }

    /**
     * Wrap a float value in a XML empty tag.
     *
     * @param valueAttribute 	- Float attribute value of the tag.
     *	  msgLoggerName	- The logger name of the message logger
     *	  msgLoggingLevel - The logging level of this message
     *	  timeStamp		- Indicates whether a time stamp should/should not be included
     *
     * @return String representation of the float XML empty tag fragment.
     */
    public static String primitiveToXML(float valueAttribute, String msgLoggerName, String msgLoggingLevel, boolean timeStamp) {
        return ("<float" + (timeStamp ? " time_stamp=\"" + new Date(System.currentTimeMillis()).toString() + "\"" : "") + getLoggerNameAttribute(msgLoggerName) + getLevelAttribute(msgLoggingLevel) + " Value=\"" + String.valueOf(valueAttribute) + "\"/>");
    }

    public static String primitiveToXML(float valueAttribute) {
        return (primitiveToXML(valueAttribute, null, null, false));
    }

    public static String primitiveToXML(float valueAttribute, String msgLoggerName) {
        return (primitiveToXML(valueAttribute, msgLoggerName, null, false));
    }

    /**
     * Wrap a double value in a XML empty tag.
     *
     * @param valueAttribute 	- Double attribute value of the tag.
     * 	  msgLoggerName	- The logger name of the message logger
     *	  msgLoggingLevel - The logging level of this message
     *	  timeStamp		- Indicates whether a time stamp should/should not be included
     *
     * @return String representation of the double XML empty tag fragment.
     */
    public static String primitiveToXML(double valueAttribute, String msgLoggerName, String msgLoggingLevel, boolean timeStamp) {
        return ("<double" + (timeStamp ? " time_stamp=\"" + new Date(System.currentTimeMillis()).toString() + "\"" : "") + getLoggerNameAttribute(msgLoggerName) + getLevelAttribute(msgLoggingLevel) + " Value=\"" + String.valueOf(valueAttribute) + "\"/>");
    }

    public static String primitiveToXML(double valueAttribute) {
        return (primitiveToXML(valueAttribute, null, null, false));
    }

    public static String primitiveToXML(double valueAttribute, String msgLoggerName) {
        return (primitiveToXML(valueAttribute, msgLoggerName, null, false));
    }

    /**
     * Returns the level attribute and its value only if 'level' is a string with length bigger than zero;
     * otherwise "" is return.
     *
     * @param level	- The string representation of the level that will be included as the value of the attribute
     * @return String	- The string representing of the level attribute followed by its value
     */
    private static String getLevelAttribute(String level) {
        if ((level != null) && (level.trim().length() != 0))
            return (" " + Constants.LOGGING_UTIL_LOGGER_LEVEL + "=\"" + level.trim() + "\"");
        return "";
    }

    /**
     * Returns the logger name attribute and its value only if 'name' is a string with length bigger than zero;
     * otherwise "" is returned
     *
     * @param name	- The name of the logger
     * @return String	- The string representation of the logger name attribute followed by its value
     */
    private static String getLoggerNameAttribute(String name) {
        if ((name != null) && (name.trim().length() != 0))
            return (" " + Constants.LOGGING_UTIL_AGENT_IDREF + "=\"" + name.trim() + "\"");
        return "";
    }

    /**
     * Returns the ID attribute and its value only if 'id' is a string with length bigger than zero;
     * otherwise "" is returned
     *
     * @param id		- The id of the object whose xml string is generated
     * @return String	- The string representation of the ID attribute followed by its value
     */
    private String getIdAttribute(String id) {
        if ((id != null) && (id.trim().length() != 0))
            return (" " + Constants.LOGGING_UTIL_ID + "=\"" + id + "\"");
        return "";
    }

    /**
     * Returns the name attribute and its value only if 'objectsName' is a string with length bigger than zero;
     * otherwise "" is returned
     *
     * @param objectName	- The name of the object
     * @return String		- The string representation of the object name attribute followed by its value
     */
    private String getNameAttribute(String objectsName) {
        if ((objectsName != null) && (objectsName.trim().length() != 0))
            return (" Instance_Name=\"" + objectsName.trim() + "\"");
        return "";
    }

    /**
     * The tag that is used when the maximum xml level has been reached
     *
     * @return String	- The string representation of the end tag
     */
    private String getEndOfNestingTag() {
        return ("<EndOfNesting Level=\"" + maxLevels + "\"/>" + lineSeparator);
    }

    /**
     * Returns the time stamp if 'includeTimeStamp' is set; otherwise "" is returned
     *
     * @return String - The time stamp or ""
     */
    private String getTimeStamp() {
        return (includeTimeStamp ? " time_stamp=\"" + new Date(System.currentTimeMillis()).toString() + "\"" : "");
    }

    /**
     *  Returns a string representing the canonical XML string of the 'object' as returned by its implemented toCanonicalXMLString () method.
     * The returned string has some of the required attributes concatenated with the attributes of the canonical XML string 
     *
     * @param object		- The object that is used to generate its xml string
     *	  id			- The id of the object
     *	  msgLoggerName	- The name of the logger that will be used to log the generated xml string
     *	  msgLoggingLevel	- The logging level that will be used to generate the xml string for the object
     *	  objectsName	- The name of the object
     */

    /* 
     * Returns a string representing the canonical XML string of the 'object' as returned by its implemented toCanonicalXMLString() or externalizeCanonicalXmlString() methods.
     * The returned string has some of the required attributes concatenated with the attributes of the canonical XML string.
     */
    private String createCanonicalXMLString(Object object, String id, String msgLoggerName, String msgLoggingLevel, String objectsName) throws Throwable {

        if (object == null)
            return nullObjectToXML(null, id, msgLoggerName, msgLoggingLevel, objectsName);

        //Attempt to invoke the toCanonicalXMLString() API if one exists on the parameter object:
        String canonicalXMLString = null;

        try {
            canonicalXMLString = ((String) (object.getClass().getMethod("toCanonicalXMLString", null).invoke(object, null))).trim();
        }
        catch (Throwable t) {
            canonicalXMLString = ((String) (object.getClass().getMethod("externalizeCanonicalXmlString", null).invoke(object, null))).trim();
        }

        int firstSlashIndex = canonicalXMLString.indexOf('/');

        //Confirm if a well-formed XML fragment: 
        if ((!canonicalXMLString.startsWith("<")) || (!canonicalXMLString.endsWith(">")) || (firstSlashIndex == -1))
            throw new IllegalArgumentException("Malformed canonical XML string");

        //Find the index for the starting position of the first possible attribute in the parent element:
        int firstSpaceIndex = canonicalXMLString.indexOf(' ');
        int firstAttributeIndex;

        if (firstSpaceIndex != -1)
            firstAttributeIndex = Math.min(Math.min(firstSpaceIndex, firstSlashIndex), canonicalXMLString.indexOf('>'));
        else
            firstAttributeIndex = Math.min(firstSlashIndex, canonicalXMLString.indexOf('>'));

        //Insert any possible logging.util attributes:
        canonicalXMLString = canonicalXMLString.substring(0, firstAttributeIndex).concat(getNameAttribute(objectsName)).concat(getIdAttribute(id)).concat(getLoggerNameAttribute(msgLoggerName)).concat(getLevelAttribute(msgLoggingLevel)).concat(getTimeStamp()).concat(canonicalXMLString.substring(firstAttributeIndex));

        return canonicalXMLString;
    }

    /**
     * A helper method used to generate the xml string for simple objects as defined by getType()
     *
     * @param object		- The object that is used to generate its xml string
     *	  id			- The id of the object
     *  	  msgLoggerName	- The name of the logger that will be used to log the generated xml string
     *  	  msgLoggingLevel	- The logging level that will be used to generate the xml string for the object
     *  	  objectsName	- The name of the object
     *
     * @return String	- The xml string of the simple object
     */
    private String simpleObjectToXML(Object object, String id, String msgLoggerName, String msgLoggingLevel, String objectsName) {

        if (object == null)
            return nullObjectToXML(null, id, msgLoggerName, msgLoggingLevel, objectsName);

        String className = object.getClass().getName().trim();

        return ("<" + removeTagNameChars(className.substring(className.lastIndexOf('.') + 1)) + getNameAttribute(objectsName) + getIdAttribute(id) + getTimeStamp() + getLoggerNameAttribute(msgLoggerName) + getLevelAttribute(msgLoggingLevel) + " Value=\"" + XmlUtility.normalize(object.toString().trim()) + "\"/>");
    }

    /**
     * A helper method used to generate the xml string for null objects
     *
     * @param object		- The object that is used to generate its xml string
     *	  id			- The id of the object
     *  	  msgLoggerName	- The name of the logger that will be used to log the generated xml string
     *  	  msgLoggingLevel	- The logging level that will be used to generate the xml string for the object
     *  	  objectsName	- The name of the object
     *
     * @return String	- The xml string of the null object
     */
    private String nullObjectToXML(String objectType, String id, String msgLoggerName, String msgLoggingLevel, String objectsName) {
        if (objectType == null)
            return ("<null" + getIdAttribute(id) + getLoggerNameAttribute(msgLoggerName) + getLevelAttribute(msgLoggingLevel) + "/>");
        else
            return ("<" + removeTagNameChars(objectType) + getNameAttribute(objectsName) + getIdAttribute(id) + getLoggerNameAttribute(msgLoggerName) + getLevelAttribute(msgLoggingLevel) + " Value=\"null\"/>");
    }

    /**
     * Returns the value of 'Constans.SimpleType' if 'testClass' is either primitive, StringBuffer, String, BigDecimal, or BigInteger;
     * otherwise the value of 'Constants.ComplexType' is returned
     *
     * @param testClass	- The class whose type is checked
     * @return byte	- Indicates the type of 'testClass'
     */
    private byte getType(Class testClass) {
        if (isPrimitive(testClass))
            return SIMPLE_TYPE;

        return COMPLEX_TYPE;
    }

    /**
     * If 'tagName' is not null then it is with every occurance of '$' replaced with '_'; 
     * otherwise 'null' is returned 
     *
     * @param tagName	- The tag name
     * @return String	- Either 'null' or the value of tagName with every '$' replaced by '_'
     */
    private String removeTagNameChars(String tagName) {
        if (tagName == null)
            return "null";
        return (tagName.replace('$', '_'));
    }

    /**
     * Returns a string with a specified number of spaces if 'format' is set; otherwise "" is returned
     *
     * @param spaces	- The number of spaces
     * @return String	- The number of spaces specified
     */
    private String indention(int spaces) {
        if (format) {
            StringBuffer indentSpaces = new StringBuffer();
            for (int counter = 0; counter < spaces; counter++)
                indentSpaces.append(' ');
            return indentSpaces.toString();
        }

        return "";
    }

    /**
     * Generate the XML string used to log the stack trace of the current thread
     * 
     * @param currLevel - The current level
     * @return A XML string representing the stack trace that will be logged
     */
    public String stackTraceToXML(int currLevel) {
        String stackTrace = getThreadStackTrace(Thread.currentThread());

        if (stackTrace == null)
            return "";

        /* Initialize the current level */
        currentLevel = currLevel;

        /* The unwanted element of the stack */
        final String UNWANTED_COMP = "com.ibm.etools.logging.util";

        StringBuffer xmlTag = new StringBuffer();
        xmlTag.append(indention(currentLevel + 1));

        /* The full class name and the class name tag */
        String className = "StackTrace"; //stackTrace.getClass().getName().trim();
        String classNameTag = "StackTrace"; //removeTagNameChars(className.substring(className.lastIndexOf('.') + 1));

        String id = className + "_" + new Guid().toString();

        /* Standard string generation */
        xmlTag.append("<");
        xmlTag.append(classNameTag);
        xmlTag.append(getIdAttribute(id));

        if (currentLevel == 0) {
            xmlTag.append(getLoggerNameAttribute(loggerName));
            xmlTag.append(getLevelAttribute(loggingLevel));
        }

        /* Customized attributes */
        xmlTag.append(" Name=\"");
        xmlTag.append("Stack Trace");
        xmlTag.append("\"");
        xmlTag.append(" CurrentThread=\"");
        xmlTag.append(Thread.currentThread().getName().trim());
        xmlTag.append("\">");
        xmlTag.append(Constants.LINE_SEPARATOR);

        /* Output the string values */
        StringTokenizer tokens = new StringTokenizer(stackTrace, Constants.LINE_SEPARATOR);
        String currentToken;
        tokens.nextToken();

        try {
            while (tokens.hasMoreTokens()) {
                currentToken = tokens.nextToken().trim();

                /* Exclude the unwanted elements of the stack */
                if (currentToken.indexOf(UNWANTED_COMP) != -1)
                    continue;

                xmlTag.append(indention(currentLevel + 2));
                xmlTag.append(simpleObjectToXML(currentToken, null, null, null, null));
                xmlTag.append(Constants.LINE_SEPARATOR);
            }

        }

        catch (Exception e) {
            xmlTag.append(indention(currentLevel + 1));
            xmlTag.append(simpleObjectToXML(getThreadStackTrace(Thread.currentThread()), null, null, null, null));
            xmlTag.append(Constants.LINE_SEPARATOR);
        }

        xmlTag.append(indention(currentLevel + 1));

        xmlTag.append("</");
        xmlTag.append(classNameTag);
        xmlTag.append(">");
        xmlTag.append(Constants.LINE_SEPARATOR);

        return xmlTag.toString();

    }

    public String stackTraceToXML() {
        return stackTraceToXML(-1);
    }

    /**
     * The following helper method is used to determine if a 'object' is "primitive".
     * 'object' must be a class representation of one of the following in order to be
     * considered as "primitive":
     * 		primitive class (e.g. int, boolean, etc...), Boolean, Character, 
     * 		Byte, Short, Integer, Long, Float, Double, BigDecimal, BigInteger, 
     * 		String, or StringBuffer.
     * 
     * @param 	object	- the 'object' to be tested
     * @return 	true if 'object' is "primitive" as defined in the comments; false otherwise 
     */
    private boolean isPrimitive(Class object) {
        String className = object.getName().trim();

        if (object.isPrimitive() || className.equals("java.lang.Boolean") || className.equals("java.lang.Character") || className.equals("java.lang.Byte") || className.equals("java.lang.Short") || className.equals("java.lang.Integer") || className.equals("java.lang.Long") || className.equals("java.lang.Float") || className.equals("java.lang.Double") || className.equals("java.math.BigDecimal") || className.equals("java.math.BigInteger") || className.equals("java.lang.String") || className.equals("java.lang.StringBuffer"))
            return true;

        return false;
    }

    /**
     * Returns the class name of 'theClass'
     *
     * @param theClass	- The class
     * @return String	- The class name
     */
    public static String getClassName(Class theClass) {
        /* In case 'theClass' is null */
        if (theClass == null)
            return "null";

        String name = theClass.getName().trim();

        /* In case 'theClass' is an array */
        if (theClass.isArray()) {
            String depth = name.substring(0, (name.lastIndexOf('[') + 1));
            String encoding = name.substring(name.lastIndexOf('[') + 1);
            String firstChar = encoding.substring(0, 1);
            String type = "";

            /* Determine the type of 'theClass' */
            if (firstChar.equals("L") && encoding.endsWith(";"))
                type = encoding.substring(1, (encoding.length() - 1));
            else if (firstChar.equals("B"))
                type = "byte";
            else if (firstChar.equals("C"))
                type = "char";
            else if (firstChar.equals("D"))
                type = "double";
            else if (firstChar.equals("F"))
                type = "float";
            else if (firstChar.equals("I"))
                type = "int";
            else if (firstChar.equals("J"))
                type = "long";
            else if (firstChar.equals("S"))
                type = "short";
            else if (firstChar.equals("Z"))
                type = "boolean";

            return (depth.concat(type).concat(depth.replace('[', ']')));
        }

        return name;
    }
}