/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.trace.internal.ui;

import java.util.Iterator;

import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.hyades.models.hierarchy.TRCAgentProxy;
import org.eclipse.hyades.models.hierarchy.TRCMonitor;
import org.eclipse.hyades.models.hierarchy.TRCNode;
import org.eclipse.hyades.models.hierarchy.TRCProcessProxy;
import org.eclipse.hyades.models.hierarchy.util.HierarchyResourceSetImpl;
import org.eclipse.hyades.models.hierarchy.util.SaveUtil;
import org.eclipse.hyades.trace.ui.IProfileEventListener;
import org.eclipse.hyades.trace.ui.ProfileEvent;
import org.eclipse.hyades.trace.ui.UIPlugin;
import org.eclipse.hyades.trace.ui.internal.actions.TraceNavigatorActionGroup;
import org.eclipse.hyades.trace.ui.internal.core.TraceAssociationManager;
import org.eclipse.hyades.trace.ui.internal.launcher.AttachTraceAction;
import org.eclipse.hyades.trace.ui.internal.launcher.LaunchTraceAction;
import org.eclipse.hyades.trace.ui.internal.util.UnresolvedResourceHelper;
import org.eclipse.jface.action.IMenuListener;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.dialogs.IDialogSettings;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.viewers.DoubleClickEvent;
import org.eclipse.jface.viewers.IDoubleClickListener;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.KeyListener;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.TreeItem;
import org.eclipse.ui.IMemento;
import org.eclipse.ui.IPropertyListener;
import org.eclipse.ui.IViewPart;
import org.eclipse.ui.IViewSite;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.actions.ActionContext;
import org.eclipse.ui.part.ViewPart;

public class PDProjectExplorer extends ViewPart  implements IProfileEventListener
												  , IPropertyListener
{
	public static final String SET_LINK_TO_VIEW = "SET_LINK_TO_VIEW";
	public static final String TRACE_SECTION = "TraceNavigator";
	
	protected TraceNavigatorActionGroup actionGroup;

	private PDProjectViewer structuredViewer;
	private IDialogSettings settings;	
	private boolean linkingEnabled;	
	
    class RefreshUI extends Thread {
	  public RefreshUI() {
		  super("Profile_Refresh");
	  }

	  public void run() {

          while(structuredViewer != null)
          {
          	try {
          		
				IPreferenceStore store = UIPlugin.getDefault().getPreferenceStore();
				int refreshType = store.getInt(TraceConstants.REFRESH_TYPE);
				Display display = Display.getDefault();
				
				if(refreshType == 1
				    && display != null && !display.isDisposed())//automatic refresh
				{
					display.asyncExec(new Runnable() {
							public void run() {
								
								TreeItem item = structuredViewer.getTreeSelection();	   
								if(item != null && (item.getData() instanceof EObject))
								{
									EObject data = (EObject)item.getData();
									
									if(isModified(data))
									{
										ProfileEvent event = UIPlugin.getDefault().getUpdateModelEvent(data);
										UIPlugin.getDefault().notifyProfileEventListener(event); 										
									}
								}	  	  	 
							}
						});
					
				}
				 				
				sleep(store.getInt(TraceConstants.REFRESH_INTERVAL)*1000);				
			  }          		
		      catch (InterruptedException exc) {
		      } 
          }          
 	  }
 	  
	  private boolean isModified(Object selObject)
	  {
		  if(selObject != null)
		  {
			  if(selObject instanceof TRCAgentProxy)
			  {
				  return isAgentModified((TRCAgentProxy)selObject);
			  }
			
			  if(selObject instanceof TRCProcessProxy)
			  {
				  Object[] agents = ((TRCProcessProxy)selObject).getAgentProxies().toArray();
				  for(int idx=0; idx<agents.length; idx++)
				  {
				  	  boolean isModified = isAgentModified((TRCAgentProxy)agents[idx]);
				  	  if(isModified)
				  	    return true;	
				  }							
				  return false;
			  }
			
			  if(selObject instanceof TRCNode)
			  {
				  Object[] processes = ((TRCNode)selObject).getProcessProxies().toArray();
				  for(int idx=0; idx<processes.length; idx++)
				  {
					  Object[] agents = ((TRCProcessProxy)processes[idx]).getAgentProxies().toArray();
					  for(int i=0; i<agents.length; i++)
					  {
						  boolean isModified = isAgentModified((TRCAgentProxy) agents[i]);
						  if(isModified)
						    return true;						
					  }					
				  }
				  return false;								
			  }
			
			  if(selObject instanceof TRCMonitor)
			  {
				  Object[] nodes = ((TRCMonitor)selObject).getNodes().toArray();
				
				  for(int k=0; k<nodes.length; k++)
				  {
					  Object[] processes = ((TRCNode)nodes[k]).getProcessProxies().toArray();
					  for(int idx=0; idx<processes.length; idx++)
					  {
						  Object[] agents = ((TRCProcessProxy)processes[idx]).getAgentProxies().toArray();
						  for(int i=0; i<agents.length; i++)
						  {
							boolean isModified = isAgentModified((TRCAgentProxy) agents[i]);
							if(isModified)
							  return true;						
						  }						  						
					  }
				  }
				  return false;								
			  }
		  }
		  
		  return false;
    	
	  }
    
	  private boolean isAgentModified(TRCAgentProxy agent)
	  {
	  	 if(agent.getType().equals(TraceConstants.PROFILE_AGENT_TYPE))
	  	 {
	  	 	if(agent.isCollectionData()) 
				return true;
				
			if(agent.getAgent() != null && agent.getAgent().eResource() != null &&
				agent.getAgent().eResource().isModified())
			{
				agent.getAgent().eResource().setModified(false);
				return true;
			}

	  	 	  
	  	 	return false;  
	  	 }
	  	 else
	  	 {
	  	 	if(agent.getAgent() != null && agent.getAgent().eResource() != null &&
	  	 	   agent.getAgent().eResource().isModified())
	  	 	   {
				   agent.getAgent().eResource().setModified(false);
				   return true;				   
	  	 	   }
	  	 	
	  	 	return false;
	  	 }
	  	 
	  }	
  }

	
	public PDProjectExplorer() {
		super();
		
		/* set project explorer to resource helper such that it can be notified dirty for save */
		Object ue = HierarchyResourceSetImpl.getInstance().getUnresolvedException();
		if ((ue != null) && (ue instanceof UnresolvedResourceHelper))
			 ((UnresolvedResourceHelper) ue).setPDProjectExplorer(this);
	}
	/**
	 * Creates the SWT controls for a part.  
	 *
	 * TIP : This method must call hookFocus() on some control (most likely
	 * the top-level control), within the view.  This translates OS-level
	 * focus to part focus within the desktop.
	 */
	public void createPartControl(Composite parent) {
	
	    TraceAssociationManager.getTraceViewMappingRegistry().addCommitChangeListener(this);
	    
		structuredViewer = new PDProjectViewer(parent);
	
		structuredViewer.setContentProvider(new PDContentProvider(this));
		structuredViewer.setLabelProvider(new PDLabelProvider());
		structuredViewer.setInput(ResourcesPlugin.getWorkspace().getRoot());
			
		initContextMenu();
		initListeners();

		makeActions();
		// Fill the action bars and update the global action handlers'
		// enabled state to match the current selection.
		getActionGroup().fillActionBars(getViewSite().getActionBars());
		updateActionBars((IStructuredSelection) structuredViewer.getSelection());

		// register this viewer as the debug UI selection provider
		UIPlugin.getDefault().addSelectionProvider(structuredViewer, this);		
		UIPlugin.getDefault().addProfileEventListener(this);
		
		(new RefreshUI()).start();
	}

	/**
	 * Updates the action bar actions.
	 * 
	 * @param selection the current selection
	 * @since 2.0
	 */
	protected void updateActionBars(IStructuredSelection selection) {
		TraceNavigatorActionGroup group = getActionGroup();
		if (group != null) {
			group.setContext(new ActionContext(selection));
			group.updateActionBars();
		}
	}
	
	/**
	 * Updates the action bar actions.
	 * 
	 * @param selection the current selection
	 * @since 2.0
	 */
	public void updateActionBars() {
		TraceNavigatorActionGroup group = getActionGroup();
		if (group != null) {
			group.updateActionBars();
		}
	}
	

	/**
	 * @see IWorkbenchPart
	 */
	public void dispose() {

		
		TraceAssociationManager.getTraceViewMappingRegistry().removeCommitChangeListener(this);
		settings = null;
		
		if (getActionGroup() != null) {
			getActionGroup().dispose();
		}
		
		UIPlugin.getDefault().removeProfileEventListener(this);		
		
		if (structuredViewer != null)
			UIPlugin.getDefault().removeSelectionProvider(structuredViewer, this);

 		try {
 			
 			boolean modified = false;
			if (HierarchyResourceSetImpl.getInstance().getResources().size() > 0) {
				
				Iterator docs = HierarchyResourceSetImpl.getInstance().getResources().iterator();
				while(docs.hasNext())
				{
					org.eclipse.emf.ecore.resource.Resource res = (org.eclipse.emf.ecore.resource.Resource) docs.next();
					if(res != null && res.isLoaded() && res.isModified())
					{
						modified = true;
						break;
					}
				}
				
				if(modified)
				{
					String title = UIPlugin.getResourceString("TRACE_MSG");
	
					String msg = UIPlugin.getResourceString("SAVE_PROJECT_SQ");
	
					if (MessageDialog.openQuestion(null, title, msg)) {
						SaveUtil.saveDocuments();
	
						IResource res = UIPlugin.getPluginWorkbench().getRoot();
						res.refreshLocal(IResource.DEPTH_INFINITE, null);
					}
				}
				
//				//clear documents list
//				SaveUtil.getDocuments().clear();
				HierarchyResourceSetImpl.getInstance().getResources().clear();

			}
		} catch (Exception exc) {
			exc.printStackTrace();
			//do nothing
		}


        structuredViewer.dispose();
        structuredViewer = null; 

		ProfileEvent event = new ProfileEvent();
		event.setSource(null);
		event.setType(ProfileEvent.CLEAN_UP);
		UIPlugin.getDefault().notifyProfileEventListener(event);

        TraceUIManager.getTraceUIManager().dispose();

		super.dispose();
	}
	
	public final boolean hasFilter(int filter){
		return structuredViewer.hasFilter(filter);		
	}

	public Control getControl(){
		return structuredViewer.getControl();
	}
	
	public void refresh(){
		structuredViewer.refresh();
	}	
	private boolean areAllProcessActive(Object[] selections) {
		for (int idx = 0; idx < selections.length; idx++) {
			if (!((TRCProcessProxy) selections[idx]).isActive())
				return false;
		}
		return true;
	}

	private boolean anyProcessActive(Object[] selections) {
		for (int idx = 0; idx < selections.length; idx++) {
			if (((TRCProcessProxy) selections[idx]).isActive())
				return true;
		}
		return false;
	}

	private boolean areSameClass(Object[] selection) {
		Class objClass = selection[0].getClass();
		for (int idx = 1; idx < selection.length; idx++) {
			if (!(objClass.equals(selection[idx].getClass()))) {
				return false;
			}
			objClass = selection[idx].getClass();
		}
		return true;
	}

	/**
	 * Initializes and registers the context menu.
	 * 
	 */
	protected void initContextMenu() {
		MenuManager menuMgr = new MenuManager("#PopupMenu"); //$NON-NLS-1$
		menuMgr.setRemoveAllWhenShown(true);
		menuMgr.addMenuListener(new IMenuListener() {
			public void menuAboutToShow(IMenuManager manager) {
				PDProjectExplorer.this.fillContextMenu(manager);
			}
		});
		Menu menu = menuMgr.createContextMenu(structuredViewer.getTree());
		structuredViewer.getTree().setMenu(menu);
		getSite().registerContextMenu(menuMgr, structuredViewer);
	}

	/**
	 * Adds the listeners to the viewer.
	 * 
	 */
	protected void initListeners() {
		
		structuredViewer.addDoubleClickListener(new IDoubleClickListener() {
			public void doubleClick(DoubleClickEvent event) {
				handleDoubleClick(event);
			}
		});
	
		getSite().setSelectionProvider(structuredViewer);
		
		/*
		viewer.addSelectionChangedListener(new ISelectionChangedListener() {
			public void selectionChanged(SelectionChangedEvent event) {
				handleSelectionChanged(event);
			}
		});
		viewer.addOpenListener(new IOpenListener() {
			public void open(OpenEvent event) {
				handleOpen(event);
			}
		});
		*/
		structuredViewer.getControl().addKeyListener(new KeyListener() {
			public void keyPressed(KeyEvent event) {
				handleKeyPressed(event);
			}
			public void keyReleased(KeyEvent event) {
				handleKeyReleased(event);
			}
		});
	}

    /**
     * Initialize and register context menu
     * @param menu
     */
	public void fillContextMenu(IMenuManager menu) {
		
		IStructuredSelection selection =
			(IStructuredSelection) getViewer().getSelection();
		getActionGroup().setContext(new ActionContext(selection));
		getActionGroup().fillContextMenu(menu);
	}

	/**
	 * Returns the trace explorer part of the active perspective. If 
	 * there isn't any trace explorer part <code>null</code> is returned.
	 */
	public static PDProjectExplorer getFromActivePerspective() {
		
		try {
			IViewPart view =
				UIPlugin.getActivePage().showView(PDPerspective.ID_PD_NAVIGATOR_VIEW);
			if (view instanceof PDProjectExplorer)
				return (PDProjectExplorer) view;
	
		}
		catch(PartInitException exc)
		{
			exc.printStackTrace();
			return null;
		}
		
		return null;
	}
	/**
	 * Insert the method's description here.
	 * Creation date: (10/02/2000 1:19:12 PM)
	 * @return com.ibm.itp.core.api.resources.IFolder
	 * @param selection java.lang.Object
	 */
	public IResource getSelectionFolder() {
		return structuredViewer.getSelectionFolder();
	}

	public IResource getSelectionFolder(TreeItem selection) {
		return structuredViewer.getSelectionFolder(selection);
	}
	/**
	 * Returns the shell to use for opening dialogs.
	 * Used in this class, and in the actions.
	 */
	public Shell getShell() {
		return structuredViewer.getTree().getShell();
	}
	/**
	 * Returns the tool tip text for the view.
	 */
	protected String getToolTipText() {
		return UIPlugin.getResourceString("PDPROEXPTOOLTIP");
	}
	
	/**
	 * Insert the method's description here.
	 * Creation date: (1/18/2001 5:10:11 PM)
	 * @return com.ibm.etools.pd.core.PDProjectViewer
	 */
	public PDProjectViewer getViewer() {
		return structuredViewer;
	}
	
	/**
	 * Handles a key press event from the viewer.
	 * 
	 * @param event the key event
	 */
	protected void handleKeyPressed(KeyEvent event) {
		
		getActionGroup().handleKeyPressed(event);
	}

	/**
	 * Returns the action group.
	 * 
	 * @return the action group
	 */
	protected TraceNavigatorActionGroup getActionGroup() {
		return actionGroup;
	}

	/**
	 * Sets the action group.
	 * 
	 * @param actionGroup the action group
	 */
	protected void setActionGroup(TraceNavigatorActionGroup actionGroup) {
		this.actionGroup = actionGroup;
	}

	/**
	 * Handles a key release in the viewer.  Does nothing by default.
	 * 
	 * @param event the key event
	 */
	protected void handleKeyReleased(KeyEvent event) {
	}
	
	/**
	 * Handles double clicks in viewer.
	 * Opens editor if file double-clicked.
	 */
	protected void handleDoubleClick(DoubleClickEvent event) {
		
		getActionGroup().handleDoubleClick(event);
	}
	
	/* (non-Javadoc)
	 * Method declared on IViewPart.
	 */
	public void init(IViewSite site, IMemento memento) throws PartInitException {
		super.init(site, memento);
		
		IDialogSettings viewsSettings = UIPlugin.getDefault().getDialogSettings();
		settings = viewsSettings.getSection(TRACE_SECTION);
		if(settings == null)
		{
			settings = viewsSettings.addNewSection(TRACE_SECTION);
			initSettings();
		}
		else
		{		
			loadSettings();
		}
		
	}
	
	/**
	 * This method is invocated when the dialog settings has just been
	 * initialized. 
	 */
	protected void initSettings()
	{
	}
	
	/**
	 * Loads the settings from the dialog settings buffer.  This class' setting keys
	 * corresponds to the "SET_" string constants.
	 * @see #getSettings()
	 */
	protected void loadSettings()
	{
		setLinkingEnabled(getSettings().getBoolean(SET_LINK_TO_VIEW));
	}
	
	
	/**
	 *  Create self's action objects
	 */
	protected void makeActions() {
		
		setActionGroup(new TraceNavigatorActionGroup(this));
	}

	/**
	 * Insert the method's description here.
	 * Creation date: (10/05/2000 1:19:46 PM)
	 */
	public void refreshView(Object selection) {
		structuredViewer.refresh();

		structuredViewer.selectObject(selection);

	}
	/**
	 * Insert the method's description here.
	 * Creation date: (09/26/2000 12:35:46 PM)
	 * @param obj java.lang.Object
	 */
	public void selectObject(Object item) {
		structuredViewer.selectObject(item);
	}
	/**
	 * Asks the part to take focus within the desktop.
	 */
	public void setFocus() {
		if (structuredViewer != null)
		  structuredViewer.getTree().setFocus();
	}

	private void addLaunchOptions(IMenuManager menu)
	{
		menu.add(new LaunchTraceAction());		
		menu.add(new AttachTraceAction());
	}

    /**
     * 
     */
	public void handleProfileEvent(ProfileEvent event) {
			
		switch(event.getType())
		{
			case ProfileEvent.START_MONITOR:
			
		   	   structuredViewer.selectObject(event.getSource());
			   
			break;
			
			case ProfileEvent.ATTACH:
			
		   	   structuredViewer.selectObject(event.getSource());
			   
			break;
						
			default:
				structuredViewer.refresh(event.getSource());
			
			
		}	
	}

	/**
	 * Indicates that a property has changed.
	 * @see IPropertyListener
	 */
	public void propertyChanged(Object source, int propId)
	{
		//reset trace view actions
		getActionGroup().propertyChanged(source, propId);
	}
  
	/**
	 * Returns this navigator's settings.
	 * @return IDialogSettings
	 */
	protected IDialogSettings getSettings()
	{
		return settings;
	}

    /**
     * 
     * @return
     */
	public boolean isLinkingEnabled()
	{
		return linkingEnabled;
	}

    /**
     * 
     * @param enabled
     */  
	public void setLinkingEnabled(boolean enabled)
	{
		linkingEnabled = enabled;

		// remember the last setting in the dialog settings		
		getSettings().put(SET_LINK_TO_VIEW, enabled);
		
		// if turning linking on, update the selection to correspond to the selection
		if(enabled)
		{
			if(getViewer() != null)
				getViewer().setSelection(getViewer().getSelection());
		}
	}
	
}

