/*******************************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.trace.ui.internal.actions;

import java.net.*;
import java.util.*;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.runtime.Platform;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.hyades.models.hierarchy.*;
import org.eclipse.hyades.trace.internal.ui.*;
import org.eclipse.hyades.trace.ui.UIPlugin;
import org.eclipse.hyades.trace.ui.internal.core.TraceAssociationManager;
import org.eclipse.hyades.trace.ui.internal.launcher.*;
import org.eclipse.hyades.trace.ui.internal.wizard.*;
import org.eclipse.hyades.ui.HyadesUIPlugin;
import org.eclipse.hyades.ui.extension.*;
import org.eclipse.jdt.ui.IContextMenuConstants;
import org.eclipse.jface.action.*;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.viewers.*;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.ui.*;
import org.eclipse.ui.actions.*;
import org.eclipse.ui.dialogs.PropertyDialogAction;
import org.eclipse.ui.plugin.AbstractUIPlugin;

/**
 * Group action for the Test Navigator.
 * 
 * <p>This implementation is based on the 
 * {@link org.eclipse.ui.views.navigator.MainActionGroup} class.
 * 
 */
public class TraceNavigatorActionGroup extends ActionGroup
{
	protected PDProjectExplorer fViewer;
	protected DeleteAction fDeleteAction;
	protected OpenWizardAction fOpenMonitorAction;
	protected OpenWizardAction fOpenNodeAction;
	protected OpenWizardAction fOpenProfAgentAction;
	protected OpenWizardAction fOpenLogAgentAction;
	protected RefreshFromLocalAction fRefreshLocal;
	protected RefreshTreeAction fRefreshTree;
	protected SaveAction fSaveToFile;
	protected StartTraceAction fStartTraceAction;
	protected StopTraceAction fStopTraceAction;
	protected PropertyDialogAction fPropertyDialogAction;
	protected OpenWizardAction fOpenProcessAction;
	protected ToggleLinkingAction toggleLinkingAction;
	
	protected ActionGroup fImportActionGroup;
	protected FilterActionGroup filterActionGroup;
	protected Separator fAttachGrp;
	protected Separator fNewSnapshotGrp;
	protected Separator fGCGroup;	
	protected Separator fMonitorGrp;
	protected Separator fTerminateGrp;
	protected Separator fRefreshGrp;
	protected RelaunchTraceAction fRelaunchAction;

	private final String GROUP_ADD_VIEWS = "group.add.views";
	private final String GROUP_REFRESH = "group.refresh";
	private final String GROUP_DELETE = "group.delete";
	private final String GROUP_SAVE = "group.save";
	private final String GROUP_OPEN_PROPERTY = "group.open.property";
	
	protected Map traceViewsMap;

	class ImportActionGroup extends ActionGroup
	{
	
		 private final String GROUP_IMPORT= "group.import";
		
		 private ImportResourcesAction fImportAction;
		 private ExportResourcesAction fExportAction;
	
		 /**
		  * Creates a new <code>ImportActionGroup</code>. The group 
		  * requires that the selection provided by the part's selection provider 
		  * is of type <code>org.eclipse.jface.viewers.IStructuredSelection</code>.
		  * 
		  * @param part the view part that owns this action group
		  */
		 public ImportActionGroup(IViewPart part) {
			 IWorkbenchWindow workbench = part.getSite().getWorkbenchWindow();
			 fImportAction= new ImportResourcesAction(workbench);
			 fExportAction= new ExportResourcesAction(workbench);	
			PDPluginImages.setImageDescriptors(fImportAction, PDPluginImages.T_LCL, PDPluginImages.IMG_IMPORT);
			PDPluginImages.setImageDescriptors(fExportAction, PDPluginImages.T_LCL, PDPluginImages.IMG_EXPORT);
			 		
		 }
		
		 /* (non-Javadoc)
		  * Method declared in ActionGroup
		  */
		 public void fillContextMenu(IMenuManager menu) {
			 menu.appendToGroup(IContextMenuConstants.GROUP_REORGANIZE, new Separator(GROUP_IMPORT));
			 menu.appendToGroup(GROUP_IMPORT, fImportAction);
			 menu.appendToGroup(GROUP_IMPORT, fExportAction);
			
			 super.fillContextMenu(menu);
		 }	
	 }
	
	/**
	 * Constructor for NavigatorActionGroup
	 * @param testNavigator
	 */
	public TraceNavigatorActionGroup(PDProjectExplorer viewer)
	{
		super();
		
		fViewer = viewer;
		createActions();
	}
	
	/**
	 * @see org.eclipse.ui.actions.ActionGroup#dispose()
	 */
	public void dispose()
	{
		traceViewsMap.clear();
		traceViewsMap = null;
		
		if(fImportActionGroup != null)
		   fImportActionGroup.dispose();
           
		fAttachGrp = null;
		fRefreshGrp = null;
		fTerminateGrp = null;
		fMonitorGrp = null;
		filterActionGroup.dispose();

        if(fSaveToFile != null)				
			fSaveToFile.dispose();
		fDeleteAction.dispose();
		fOpenLogAgentAction.dispose();
		fOpenMonitorAction.dispose();
		fOpenNodeAction.dispose();
		fOpenProcessAction.dispose();
		fPropertyDialogAction.dispose();
		fRefreshLocal.dispose();
		fRefreshTree.dispose();
		fSaveToFile.dispose();
           			
        toggleLinkingAction.dispose();
           				
		super.dispose();
	}

	/**
	 * @see org.eclipse.ui.actions.ActionGroup#setContext(org.eclipse.ui.actions.ActionContext)
	 */
	public void setContext(ActionContext context)
	{
		super.setContext(context);
		/*
		sorterActionGroup.setContext(context);
		newActionGroup.setContext(context);
		workingSetFilterActionGroup.setContext(context);
		frameListActionGroup.setContext(context); 
		*/
	}

	/**
	 * @see org.eclipse.hyades.ui.internal.navigator.action.NavigatorActionGroup#createActions()
	 */
	protected void createActions()
	{
		traceViewsMap = new HashMap();

		toggleLinkingAction = new ToggleLinkingAction(fViewer, UIPlugin.getResourceString("STR_LINK_VIEW")); 
		toggleLinkingAction.setToolTipText(UIPlugin.getResourceString("STR_LINK_VIEW_TIP")); 
		toggleLinkingAction.setImageDescriptor(getImageDescriptor("elcl16/synced.gif"));
		toggleLinkingAction.setHoverImageDescriptor(getImageDescriptor("clcl16/synced.gif"));
		
		fImportActionGroup = new ImportActionGroup(fViewer);
		filterActionGroup = new FilterActionGroup(fViewer.getViewer());
		
		fAttachGrp = new Separator(TraceConstants.PROFILE_ATTACH_GROUP);
		fNewSnapshotGrp = new Separator(TraceConstants.PROFILE_NEW_SNAPSHOT_GROUP);
		fGCGroup = new Separator(TraceConstants.PROFILE_GC_GROUP);
		fRefreshGrp = new Separator(TraceConstants.PROFILE_REFRESH_GROUP);
		fTerminateGrp = new Separator(TraceConstants.PROFILE_TERMINATE_GROUP);
		fMonitorGrp = new Separator(TraceConstants.PROFILE_MONITOR_GROUP);
		
		fRefreshTree = new RefreshTreeAction(fViewer);
	
		fRefreshLocal = new RefreshFromLocalAction(fViewer);
		fSaveToFile = new SaveAction(fViewer);
	
		fOpenMonitorAction =
			new OpenWizardAction(
		fViewer,
				OpenMonitorWizard.class,
				UIPlugin.getResourceString("PROPERTIES"));
		fOpenProfAgentAction =
			new OpenWizardAction(
		fViewer,
				OpenProfAgentWizard.class,
				UIPlugin.getResourceString("PROPERTIES"));
		fOpenLogAgentAction =
			new OpenWizardAction(
		fViewer,
				OpenLogAgentWizard.class,
				UIPlugin.getResourceString("PROPERTIES"));
		fOpenNodeAction =
			new OpenWizardAction(
			fViewer,
				OpenNodeWizard.class,
				UIPlugin.getResourceString("PROPERTIES"));
		fOpenProcessAction =
			new OpenWizardAction(
			fViewer,
				OpenProcessWizard.class,
				UIPlugin.getResourceString("PROPERTIES"));
	
		fDeleteAction = new DeleteAction(fViewer);
		fPropertyDialogAction = new PropertyDialogAction(fViewer.getShell(), fViewer.getViewer());
	
		fRelaunchAction = new RelaunchTraceAction(fViewer);
	}

	/**
	 * @see org.eclipse.ui.actions.ActionGroup#fillContextMenu(org.eclipse.jface.action.IMenuManager)
	 */
	public void fillContextMenu(IMenuManager menu)
	{
		IStructuredSelection selection = (IStructuredSelection)getContext().getSelection();
		boolean sameClass = false;
		boolean multiSelection = (selection.size() > 1);
		if (multiSelection)
			sameClass = areSameClass(selection);
				
		if (selection.isEmpty()) {
			
			menu.add(new LaunchTraceAction());		
			menu.add(new AttachTraceAction());			
			menu.add(new Separator(IContextMenuConstants.GROUP_REORGANIZE));
			menu.add(new Separator(IContextMenuConstants.GROUP_ADDITIONS));
			fImportActionGroup.fillContextMenu(menu);		
			menu.add(new Separator(GROUP_REFRESH));
			menu.appendToGroup(GROUP_REFRESH,fRefreshTree);
			return;
		}
		
		Object sel = selection.iterator().next();		
		if (sel instanceof IContainer)
		{
			menu.add(new LaunchTraceAction());		
			menu.add(new AttachTraceAction());
		}
		else if(sel instanceof TRCMonitor)
		{
			menu.add(new LaunchTraceAction());		
			menu.add(new AttachTraceAction());			
			menu.add(new Separator(GROUP_ADD_VIEWS));		
			addViews(menu, sel);
			menu.add(fRefreshGrp);						    	
		}
		else if(sel instanceof TRCNode)
		{
			menu.add(new LaunchTraceAction());		
			menu.add(new AttachTraceAction());			
			menu.add(new Separator(GROUP_ADD_VIEWS));			
			addViews(menu, sel); 
			menu.add(fRefreshGrp);
					
	    	
		}
		else if(sel instanceof TRCProcessProxy)
		{
			if (!multiSelection) {
				menu.add(fGCGroup);
				menu.add(fTerminateGrp);
				if (((TRCProcessProxy) sel).getLaunchMode() == 1) //launch mode
				{
					menu.add(fRelaunchAction);
				}
				
				menu.add(new Separator(GROUP_ADD_VIEWS));		
				addViews(menu, sel);
				menu.add(fRefreshGrp);
				
			}
	    	
		}
		else if(sel instanceof TRCAgentProxy)
		{
			if (!multiSelection || sameClass) {
				menu.add(fMonitorGrp);
				menu.add(fGCGroup);
				menu.add(fAttachGrp);				
			}
			
			if(!multiSelection)
			{
				menu.add(fNewSnapshotGrp);
				menu.add(fTerminateGrp);
				if (((TRCAgentProxy) sel).getProcessProxy().getLaunchMode() == 1) //launch mode
				{
					menu.add(fRelaunchAction);
				}
				
				menu.add(new Separator(GROUP_ADD_VIEWS));		
				addViews(menu, sel);
				menu.add(fRefreshGrp);
			}
	    	
		}

		menu.add(new Separator(GROUP_DELETE));
		menu.appendToGroup(GROUP_DELETE,fDeleteAction);
	
		menu.add(new Separator(GROUP_SAVE));
		menu.appendToGroup(GROUP_SAVE,fSaveToFile);
	
		menu.add(new Separator(IContextMenuConstants.GROUP_REORGANIZE));

		menu.add(new Separator(GROUP_REFRESH));
		menu.appendToGroup(GROUP_REFRESH,fRefreshTree);
		menu.appendToGroup(GROUP_REFRESH,fRefreshLocal);

		menu.add(new Separator(IContextMenuConstants.GROUP_ADDITIONS));	

		if (!multiSelection) {
			
			menu.add(new Separator(GROUP_OPEN_PROPERTY));
			if (fPropertyDialogAction.isApplicableForSelection())
				menu.appendToGroup(GROUP_OPEN_PROPERTY,fPropertyDialogAction);
	
			if (sel instanceof TRCMonitor) {
				menu.appendToGroup(GROUP_OPEN_PROPERTY,fOpenMonitorAction);
			} else if (sel instanceof TRCNode) {
				menu.appendToGroup(GROUP_OPEN_PROPERTY,fOpenNodeAction);
			} else if (sel instanceof TRCProcessProxy) {
				menu.appendToGroup(GROUP_OPEN_PROPERTY,fOpenProcessAction); 
			} else if (sel instanceof TRCAgentProxy) {
				if (((TRCAgentProxy) sel).getType().equals(TraceConstants.PROFILE_AGENT_TYPE)) {
					menu.appendToGroup(GROUP_OPEN_PROPERTY,fOpenProfAgentAction);
				} else {
						menu.appendToGroup(GROUP_OPEN_PROPERTY,fOpenLogAgentAction);
				}
			}
		}

		fImportActionGroup.fillContextMenu(menu);		
	}
	
	/**
	 * Insert the method's description here.
	 * Creation date: (08/24/2000 2:06:10 PM)
	 * @param menu com.ibm.jface.MenuManager
	 */
	protected void addViews(IMenuManager menu, Object obj) {
		
		if(obj == null || !(obj instanceof EObject))
		  return;

		MenuManager popup = new MenuManager(UIPlugin.getResourceString("MENU_MANAGER"));
		menu.appendToGroup(GROUP_ADD_VIEWS,popup);
		  
		if(obj instanceof TRCMonitor)
		{
			TRCMonitor monitor = (TRCMonitor)obj;
			IAction[] actions = getViewMappingForType(TraceConstants.MONITOR_TYPE);
			for(int idx=0; idx<actions.length; idx++)
			{
				if(actions[idx] != null)
				   popup.add(actions[idx]);
			}
			
			ArrayList typesList = new ArrayList();
			EList nodes = monitor.getNodes();
			if(nodes.size() > 0)
			{ 
				for(int idx=0; idx<nodes.size(); idx++)
				{
					TRCNode node = (TRCNode) nodes.get(idx);
					
					EList processes = node.getProcessProxies();
					for(int i=0; i<processes.size(); i++)
					{
						TRCProcessProxy process = (TRCProcessProxy)processes.get(i);
						EList agents = process.getAgentProxies();
						
						for(int j=0; j<agents.size(); j++)
						{
							TRCAgentProxy agent = (TRCAgentProxy)agents.get(j);
							
							if(typesList.contains(agent.getType()))
							  continue; //actions already added to the popup
							
							typesList.add(agent.getType());
							actions = getViewMappingForType(agent.getType());
							if(actions.length > 0)
							{//add the actions to the popup
								popup.add(new Separator());								
								for(int k=0; k<actions.length; k++)
									popup.add(actions[k]);
							}
						}
					}
				}
			}
		}
		else if(obj instanceof TRCNode)
		{
			IAction[] actions = getViewMappingForType(TraceConstants.HOST_TYPE);
			for(int idx=0; idx<actions.length; idx++)
			{
				if(actions[idx] != null)
				   popup.add(actions[idx]);
			}
			
			ArrayList typesList = new ArrayList();
			EList processes = ((TRCNode) obj).getProcessProxies();
			for(int i=0; i<processes.size(); i++)
			{
				TRCProcessProxy process = (TRCProcessProxy)processes.get(i);
				EList agents = process.getAgentProxies();
				
				for(int j=0; j<agents.size(); j++)
				{
					TRCAgentProxy agent = (TRCAgentProxy)agents.get(j);
					
					if(typesList.contains(agent.getType()))
					  continue; //actions already added to the popup
					
					typesList.add(agent.getType());
					actions = getViewMappingForType(agent.getType());
					if(actions.length > 0)
					{//add the actions to the popup
						popup.add(new Separator());						
						for(int k=0; k<actions.length; k++)
							popup.add(actions[k]);
					}
				}
			}
			
		}
		else if(obj instanceof TRCProcessProxy)
		{
			ArrayList typesList = new ArrayList();
				EList agents = ((TRCProcessProxy)obj).getAgentProxies();
				
			for(int j=0; j<agents.size(); j++)
			{
				TRCAgentProxy agent = (TRCAgentProxy)agents.get(j);
				
				if(typesList.contains(agent.getType()))
				  continue; //actions already added to the popup
				
				typesList.add(agent.getType());
				IAction[] actions = getViewMappingForType(agent.getType());
				if(actions.length > 0)
				{//add the actions to the popup
					popup.add(new Separator());
					for(int k=0; k<actions.length; k++)
						popup.add(actions[k]);
				}
			}			
		}
		else if(obj instanceof TRCAgentProxy)
		{
			IAction[] actions = getViewMappingForType(((TRCAgentProxy)obj).getType());
			for(int idx=0; idx<actions.length; idx++)
			{
				if(actions[idx] != null)
				   popup.add(actions[idx]);
			}
		}
	}

	/**
	 * Returns the list of actions available for this trace type
	 * @param type
	 */
	protected IAction[] getViewMappingForType(String type)
	{
		Object mapp = traceViewsMap.get(type);
    	
		if(mapp != null)
			return ((IAction[])mapp);
    	    
		IAssociationMappingRegistry registry = TraceAssociationManager.getTraceViewMappingRegistry();
		IAssociationMapping mapping = registry.getAssociationMapping(HyadesUIPlugin.EP_ANALYZER_EXTENSIONS);
		
		if(mapping == null)
		{
			IAction[] actions = new IAction[0];
			traceViewsMap.put(type, actions);
			
			return actions;
		}
    		    		  
		IAssociationDescriptor[] array = mapping.getAssociationDescriptors(type);
		IAction[] actions = new IAction[array.length];
		
		for(int idx=0; idx<array.length; idx++)
		{
			IAssociationDescriptor desc = array[idx];
				
			try {
					
				IAction action = (IAction)desc.createImplementationClassInstance();
				
				if(action != null)
				{
					action.setText(desc.getName());
					action.setImageDescriptor(desc.getImageDescriptor());
				
					actions[idx] = action;										
				}								
			} catch (Exception exc) {
				exc.printStackTrace();
			}
								
		}
		
		traceViewsMap.put(type, actions);
		return actions;
    	        	  
	}

	/**
	 * @see org.eclipse.ui.actions.ActionGroup#fillActionBars(org.eclipse.ui.IActionBars)
	 */
	public void fillActionBars(IActionBars actionBars)
	{
		filterActionGroup.fillActionBars(actionBars);
		
		IMenuManager menu = actionBars.getMenuManager();
		menu.add(new Separator());
		menu.add(toggleLinkingAction);
		
		actionBars.updateActionBars();

		IToolBarManager toolbar = actionBars.getToolBarManager();		
		toolbar.add(new Separator());
		toolbar.add(toggleLinkingAction);
		
	}
	
	/**
	 * @see org.eclipse.hyades.ui.internal.navigator.action.NavigatorActionGroup#handleKeyPressed(org.eclipse.swt.events.KeyEvent)
	 */
	public void handleKeyPressed(KeyEvent event)
	{
		if(event.character == SWT.DEL && (event.stateMask == 0))
		{
			if(fDeleteAction.isEnabled())
				fDeleteAction.run();
		}
	}

	/**
	 * Handles double clicks in viewer.
	 * Opens editor if file double-clicked.
	 */
	public void handleDoubleClick(DoubleClickEvent event) {
		
		IStructuredSelection s = (IStructuredSelection) event.getSelection();
		Object obj = s.getFirstElement();

		if(obj == null || !(obj instanceof EObject))
		  return;

		if(obj instanceof TRCMonitor)
		{
			IAction[] actions = getViewMappingForType("Monitor");
			if(actions.length > 0 && actions[0] != null)
				actions[0].run();
		}
		else if(obj instanceof TRCNode)
		{
			IAction[] actions = getViewMappingForType("Host");
			if(actions.length > 0 && actions[0] != null)
				actions[0].run();
		}
		else if(obj instanceof TRCProcessProxy)
		{
			EList agents = ((TRCProcessProxy)obj).getAgentProxies();
			for(int idx=0; idx<agents.size(); idx++)
			{
				TRCAgentProxy agent = ((TRCAgentProxy)agents.get(idx));
				if(agent == null || agent.eIsProxy())
				  continue;
				  
				IAction[] actions = getViewMappingForType(agent.getType());
				if(actions.length > 0 && actions[0] != null)
				{
					actions[0].run();					
					return;
				}				  				 
			}
		}
		else if(obj instanceof TRCAgentProxy)
		{
			IAction[] actions = getViewMappingForType(((TRCAgentProxy)obj).getType());
			if(actions.length > 0 && actions[0] != null)
				actions[0].run();
		}
	}

	/**
	 * Indicates that a property has changed.
	 * @see IPropertyListener
	 */
	public void propertyChanged(Object source, int propId)
	{
		//reset trace view actions
		if(source instanceof IAssociationMappingRegistry)
			traceViewsMap.clear();
	}

	/**
	 * Returns the image descriptor from the eclipse UI plugin with the given relative 
	 * path.
	 * @return ImageDescriptor;
	 */
	protected ImageDescriptor getImageDescriptor(String relativePath)
	{
		String iconPath = "icons/full/";
		try
		{
			AbstractUIPlugin plugin = (AbstractUIPlugin) Platform.getPlugin(PlatformUI.PLUGIN_ID);
			URL installURL = plugin.getDescriptor().getInstallURL();
			URL url = new URL(installURL, iconPath + relativePath);
			return ImageDescriptor.createFromURL(url);
		}
		catch(MalformedURLException e)
		{
			return ImageDescriptor.getMissingImageDescriptor();
		}
	}
	
	/**
	 * Returns true if the selections have the same type
	 * @param selection
	 * @return
	 */
	private boolean areSameClass(IStructuredSelection selection) {
		
		if(selection == null || selection.isEmpty())
		  return true;
		  
		Class objClass = selection.getFirstElement().getClass();  
		Iterator iterator = selection.iterator();  
		while(iterator.hasNext())
		{
			if (!(objClass.equals(iterator.next().getClass()))) {
				return false;
			}
		}
		
		return true;
	}

	public void updateActionBars() {
		
		filterActionGroup.updateActionBars();
	}

}
