/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.trace.ui.internal.util;

import java.util.ArrayList;
import java.util.Iterator;

import org.eclipse.jface.preference.JFacePreferences;
import org.eclipse.jface.resource.JFaceColors;
import org.eclipse.jface.util.IPropertyChangeListener;
import org.eclipse.jface.util.PropertyChangeEvent;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.StyleRange;
import org.eclipse.swt.custom.StyledText;
import org.eclipse.swt.events.*;
import org.eclipse.swt.events.MouseAdapter;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.graphics.*;
import org.eclipse.swt.graphics.Cursor;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.program.Program;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;

/**
 * @author popescu
 *
 * To change the template for this generated type comment go to
 * Window&gt;Preferences&gt;Java&gt;Code Generation&gt;Code and Comments
 */
public class HyperlinkText extends StyledText {

	private ArrayList detailsHyperlinkRanges = new ArrayList();
	private ArrayList analyzeHyperlinkRanges = new ArrayList();	
	private ArrayList actionRanges = new ArrayList();	
	private IPropertyChangeListener colorListener;
	private String linkStart = "http";
	int[][] detailsRanges;
	
	private Cursor handCursor;
	private Cursor busyCursor;
	private boolean mouseDown = false;
	private boolean dragEvent = false;	

	/**
	 * @param parent
	 * @param style
	 */
	public HyperlinkText(Composite parent, int style) {
		super(parent, style);
		
		final Display display = parent.getDisplay();
		handCursor = new Cursor(display, SWT.CURSOR_HAND);
		busyCursor = new Cursor(display, SWT.CURSOR_WAIT);
		
		colorListener = 
			new IPropertyChangeListener(){
				public void propertyChange(PropertyChangeEvent event){
					if(event.getProperty().equals(JFacePreferences.HYPERLINK_COLOR)){
						Color fg = JFaceColors.getHyperlinkText(display);
						Iterator links = detailsHyperlinkRanges.iterator();
						while(links.hasNext()){
							StyleRange range = (StyleRange) links.next();
							range.foreground = fg;
						}						
					}
				}
			};
		
		JFacePreferences.getPreferenceStore().addPropertyChangeListener(colorListener);
		
		Color background = JFaceColors.getBannerBackground(display);
		Color foreground = JFaceColors.getBannerForeground(display);
		JFaceColors.setColors(this,foreground,background);		
		
		addListeners();

	}

	/**
	 * Adds listeners to the given styled text
	 */
	protected void addListeners() {
		
		if (!SWT.getPlatform().equals("win32")) {
			 return;
		}
		
		addMouseListener(new MouseAdapter() {
			public void mouseDown(MouseEvent e) {
				if (e.button != 1) {
					return;
				}
				mouseDown = true;
			}
			public void mouseUp(MouseEvent e) {
				mouseDown = false;
				StyledText text = (StyledText)e.widget;
				int offset = text.getCaretOffset();
				if (dragEvent) {
					dragEvent = false;
					if (isLinkAt(offset)) {
						text.setCursor(handCursor);
					}
				} else if (isLinkAt(offset)) {	
					text.setCursor(busyCursor);
					if (e.button == 1) {
						triggerLinkAt(offset);
						StyleRange selectionRange = getCurrentLink();
						text.setSelectionRange(selectionRange.start, selectionRange.length);
						text.setCursor(null);
					}
				}
			}
		});
	
		addMouseMoveListener(new MouseMoveListener() {
			public void mouseMove(MouseEvent e) {
				// Do not change cursor on drag events
				if (mouseDown) {
					if (!dragEvent) {
						StyledText text = (StyledText)e.widget;
						text.setCursor(null);
					}
					dragEvent = true;
					return;
				}
				StyledText text = (StyledText)e.widget;
				int offset = -1;
				try {
					offset = text.getOffsetAtLocation(new Point(e.x, e.y));
				} catch (IllegalArgumentException ex) {
					// location is not over a character
				}
				if (offset == -1)
					text.setCursor(null);
				else if (isLinkAt(offset)) 
					text.setCursor(handCursor);
				else 
					text.setCursor(null);
			}
		});
			
	}
	
	/**
	 * Returns true is a link (action or help) is present at the given character location
	 */
	private boolean isLinkAt(int offset) {
		
		if(detailsRanges == null)
		  return false;
		  
		// Check if there is a link at the offset
		for (int i = 0; i < detailsRanges.length; i++){
			
			if (offset >= detailsRanges[i][0] && offset < detailsRanges[i][0] + detailsRanges[i][1]) {					
				return true;
			}
		}
		return false;
	}

	public void openBrowser(String url)
	{
		if (url == null || "".equals(url)) {
			return;
		}
		if (SWT.getPlatform().equals("win32")) {
			Program.launch(url);
		}
	}

	/**
	 * Finds the current link of the current selection.
	 */
	protected StyleRange getCurrentLink(){
		StyleRange[] ranges = getStyleRanges();
		int currentSelectionEnd = getSelection().y;
		int currentSelectionStart = getSelection().x;
	
		for (int i = 0; i < ranges.length; i++) {
			if((currentSelectionStart >= ranges[i].start) && 
				(currentSelectionEnd <= (ranges[i].start + ranges[i].length))) {
				return ranges[i];
			}
		}
		return null;
	}

	/**
	 * Triggers the link at the given offset (if there is one)
	 */
	protected void triggerLinkAt(int offset) {

		if(detailsRanges == null)
		  return;

		// Check if there is a link at the offset
		for (int i = 0; i < detailsRanges.length; i++){
			if (offset >= detailsRanges[i][0] && offset < detailsRanges[i][0] + detailsRanges[i][1]) {
				openBrowser(getText().substring(detailsRanges[i][0], detailsRanges[i][0] + detailsRanges[i][1]));
				
				return;
			}
		}
	}

	public void dispose()
	{
		if (busyCursor != null)
			busyCursor.dispose();
		if (handCursor != null)
			handCursor.dispose();
		
		if (colorListener != null) {
			JFacePreferences.getPreferenceStore().
				removePropertyChangeListener(colorListener);
		}
		
		colorListener = null;

	}
	
	/**
	 * Sets the styled text's link (blue) ranges
	 */
	private void setLinkRanges(String text) {
		
		actionRanges.clear();
		detailsHyperlinkRanges.clear();
		
		if(text == null || text.length() == 0)
		   return;
		   
		//parser the string and find the links
		int idx= text.indexOf(linkStart);
		int offset =0;
		while(idx != -1)
		{
			String tmpText = text.substring(offset+idx);
			//find space
			int tmpIdx = tmpText.indexOf(" ");
			if(tmpIdx == -1)
			  tmpIdx = tmpText.indexOf("\n");
			if(tmpIdx == -1)
			{
				actionRanges.add(new int[] {idx+offset, tmpText.length()-idx});
				text ="";				
			}
			else
			{
				actionRanges.add(new int[] {idx+offset, tmpIdx});
				tmpText = tmpText.substring(tmpIdx+1);					
				offset+= tmpIdx+1;						
			}
		
			offset+=idx;
			idx = tmpText.indexOf(linkStart);
		}
		
		Color fg = JFaceColors.getHyperlinkText(getShell().getDisplay());
		
			detailsRanges = (int[][])actionRanges.toArray(new int[actionRanges.size()][2]);		
			
			for (int i = 0; i < detailsRanges.length; i++) {
				StyleRange r = new StyleRange(detailsRanges[i][0], detailsRanges[i][1], fg, null, SWT.NONE);
				setStyleRange(r);
				detailsHyperlinkRanges.add(r);
			}			
	}

    public void setText(String text)
    {
    	super.setText(text);
    	setLinkRanges(text);
    }
}
