/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.trace.ui.internal.util;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.UnknownHostException;
import java.text.DateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.Enumeration;
import java.util.Iterator;
import java.util.Vector;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceStatus;
import org.eclipse.core.resources.IWorkspace;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Status;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.hyades.internal.execution.local.common.CustomCommand;
import org.eclipse.hyades.internal.execution.local.common.Options;
import org.eclipse.hyades.internal.execution.local.control.Agent;
import org.eclipse.hyades.internal.execution.local.control.AgentConfiguration;
import org.eclipse.hyades.internal.execution.local.control.AgentConfigurationEntry;
import org.eclipse.hyades.internal.execution.local.control.AgentFactory;
import org.eclipse.hyades.internal.execution.local.control.InactiveAgentException;
import org.eclipse.hyades.internal.execution.local.control.InactiveProcessException;
import org.eclipse.hyades.internal.execution.local.control.Node;
import org.eclipse.hyades.internal.execution.local.control.NodeFactory;
import org.eclipse.hyades.internal.execution.local.control.NotConnectedException;
import org.eclipse.hyades.internal.execution.local.control.Process;
import org.eclipse.hyades.internal.execution.local.control.ProcessFactory;
import org.eclipse.hyades.internal.execution.local.control.ProcessImpl;
import org.eclipse.hyades.internal.execution.local.control.Variable;
import org.eclipse.hyades.loaders.util.LoadersUtils;
import org.eclipse.hyades.models.hierarchy.HierarchyFactory;
import org.eclipse.hyades.models.hierarchy.TRCAgent;
import org.eclipse.hyades.models.hierarchy.TRCAgentProxy;
import org.eclipse.hyades.models.hierarchy.TRCCollectionMode;
import org.eclipse.hyades.models.hierarchy.TRCConfiguration;
import org.eclipse.hyades.models.hierarchy.TRCFilter;
import org.eclipse.hyades.models.hierarchy.TRCMonitor;
import org.eclipse.hyades.models.hierarchy.TRCNode;
import org.eclipse.hyades.models.hierarchy.TRCOption;
import org.eclipse.hyades.models.hierarchy.TRCProcessProxy;
import org.eclipse.hyades.models.hierarchy.util.HierarchyResourceSetImpl;
import org.eclipse.hyades.security.util.ConnectUtil;
import org.eclipse.hyades.trace.internal.ui.PDContentProvider;
import org.eclipse.hyades.trace.internal.ui.PDPerspective;
import org.eclipse.hyades.trace.internal.ui.TraceConstants;
import org.eclipse.hyades.trace.ui.HyadesConstants;
import org.eclipse.hyades.trace.ui.ProfileEvent;
import org.eclipse.hyades.trace.ui.UIPlugin;
import org.eclipse.hyades.trace.ui.internal.console.ConsoleDataProcessor;
import org.eclipse.hyades.trace.ui.internal.core.EnvTableElement;
import org.eclipse.hyades.trace.ui.internal.piclient.PIAttachListener;
import org.eclipse.hyades.trace.ui.internal.piclient.PIProcessListener;
import org.eclipse.hyades.trace.ui.internal.piclient.XMLTraceDataProcessor;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.swt.widgets.Shell;


public class PDCoreUtil {
    //~ Methods ------------------------------------------------------------------------------------

    /**
     * attach to an agent
     *
     * @deprecated user should call attach(Sting,Vector,String,String)
     *  with portNumber specific instead.
     **/
    public static final Vector attach(String hostName, Vector agents, String selectedProject, String selectedMonitor) {
        return attach(hostName, agents, selectedProject, selectedMonitor, "10002", null);
    }

    public static final Vector attach(String hostName, Vector agents, String selectedProject, String selectedMonitor, String portNumber, String profileFile) {
        IPreferenceStore store = UIPlugin.getDefault().getPreferenceStore();

        try {
            UIPlugin.getActivePage().showView(PDPerspective.ID_PD_NAVIGATOR_VIEW);
        } catch (Exception exc) {
        }

        try {
            Node fnode = NodeFactory.createNode(hostName);

            hostName = fnode.getInetAddress().getHostName();
        } catch (UnknownHostException exc) {
        }

        //create the monitor
        IContainer container = createContainer(new Path(selectedProject));
        TRCMonitor monitor = createMonitor(container, selectedMonitor);

        //create the node   
        TRCNode node = createNode(monitor, hostName, portNumber);

        TRCProcessProxy process = null;
        Vector trcAgentList = new Vector();

        for (int idx = 0; idx < agents.size(); idx++) {
            Agent agent = (Agent) agents.elementAt(idx);
            Process pInfo = agent.getProcess();

            process = createProcess(node, pInfo);

            if (pInfo.getConsole().getDataProcessor() == null) {
                ConsoleDataProcessor consoleData = new ConsoleDataProcessor(process);

                consoleData.setConsole(pInfo.getConsole());
            }

            try {
                TRCAgentProxy agentProxy = createUniqueAgent(process, agent);

                doAttach(agent, agentProxy);
                
                activate(agentProxy);
                
                trcAgentList.add(agentProxy);

                ProfileEvent event = UIPlugin.getDefault().getProfileEvent();

                event.setSource(agentProxy);
                event.setType(ProfileEvent.ATTACH);
                UIPlugin.getDefault().notifyProfileEventListener(event);

                if ((idx == 0) && (profileFile != null) && !profileFile.equals("")) {
                    agentProxy.setProfileFile(profileFile);
                    agent.setProfileFile(agentProxy.getProfileFile());
                }
            } catch (Exception exc) {
                exc.printStackTrace();
            }
        }

        return trcAgentList;
    }

    /**
	 * @param agentProxy
	 */
	public static void activate(TRCAgentProxy agentProxy) {
		//deactivate previous agents
		for (Iterator iter = agentProxy.getProcessProxy().getAgentProxies().iterator(); iter.hasNext();) {
			TRCAgentProxy element = (TRCAgentProxy) iter.next();
			try {
				if(agentProxy.getName().equals(element.getName()) && agentProxy.getType().equals(element.getType()))
				{
					element.setActive(false);
					element.setAttached(false);
				}
			} catch (NullPointerException e) {
				// ignore null pointer exceptions
			}
		}
		agentProxy.setActive(true);
		agentProxy.setAttached(true);
		
	}

	public static final void detachAgent(TRCAgentProxy agent) throws InactiveAgentException, InactiveProcessException {
        TRCProcessProxy process = agent.getProcessProxy();
        String hostName = process.getNode().getName();
        Node node = PDCoreUtil.profileConnect(hostName, String.valueOf(process.getNode().getPort()));

        if (node != null) {
            Object instance = LoadersUtils.locateAgentInstance(agent);

            if ((instance != null) && instance instanceof Agent) {
                Agent a = (Agent) instance;
//				AgentListener al = (AgentListener)LoadersUtils.locateAgentListener(agent);
//				a.removeAgentListener(al);

                if (a.isAttached()) {
                    a.detach();
                }
            }
        }

        agent.setMonitored(false);
        agent.setAttached(false);

        LoadersUtils.deregisterDataProcessor(agent);
//		LoadersUtils.deregisterAgentInstance(agent);
//		LoadersUtils.deregisterAgentListener(agent);

        if (agent.isToProfileFile()) {
            XMLTraceDataProcessor processor = (XMLTraceDataProcessor) LoadersUtils.locateDataProcessor(agent);

            if (processor != null) {
                processor.endTrace(processor.getWriter());
                processor.setProfileFileName(null);

                //					  processor.setProfilingFile(null);
                processor.setWriter(null);
            }
        }
    }

    public static TRCAgentProxy createAgent(TRCProcessProxy process, Agent a) {
        TRCAgentProxy agentProxy = createAgent(process, a, "");

        //	setSessionStartTime(a.get, agentProxy);
        return agentProxy;
    }

	public static TRCAgentProxy getCorrespondingAgent(TRCProcessProxy processProxy,Agent ag,boolean activeProxy) {
		Iterator i = processProxy.getAgentProxies().iterator();

		while (i.hasNext()) {
			TRCAgentProxy agentProxy = (TRCAgentProxy) i.next();

			if (agentProxy.eIsProxy()) {
				continue;
			}

			if ((agentProxy != null) && agentProxy.getName().equals(ag.getName()) && (agentProxy.getRuntimeId() == null || agentProxy.getRuntimeId() == ag.getUUID()) && agentProxy.isActive()==activeProxy) {
				return agentProxy;
			}
		}

		return null;
	}
    /**
     * Insert the method's description here.
     * Creation date: (5/9/2001 3:29:43 PM)
     * @return com.ibm.etools.perftrace.TRCAgent
     * @param viewer com.ibm.etools.pd.core.PDProjectExplorer
     * @param process com.ibm.etools.perftrace.TRCProcessProxy
     * @param name java.lang.String
     */
    public static TRCAgentProxy createAgent(TRCProcessProxy processProxy, Agent agent, String sessionId) {
        TRCAgentProxy agentProxy = getCorrespondingAgent(processProxy,agent,false);
        if(agentProxy!=null)
        	return agentProxy;
//        String name = a.getName();

//        EList agents = process.getAgentProxies();
//        Iterator i = agents.iterator();
//
//        while (i.hasNext()) {
//            agent = (TRCAgentProxy) i.next();
//
//            if (agent.eIsProxy()) {
//                continue;
//            }
//
//            if (agent.getName().equals(name)) {
//                return agent;
//            }
//        }

        return createAgentNoCheck(processProxy, agent, sessionId);
    }

    /**
     * Insert the method's description here.
     * Creation date: (10/05/2000 10:03:14 AM)
     * @param path com.ibm.itp.common.IPath
     */
    public static final IContainer createContainer(IPath path) {
        IContainer container = null;
        IWorkspace workbench = UIPlugin.getPluginWorkbench();
        int segCount = path.segmentCount();

        for (int idx = 0; idx < segCount; idx++) {
            String seg = path.segment(idx);

            if (idx == 0) { //project

                IProject project = workbench.getRoot().getProject(path.uptoSegment(idx + 1).toString());

                if ((project == null) || !project.exists()) { //create the project

                    try {
                        project.create(null);
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
                }

                try {
                    project.open(null);
                } catch (Exception e) {
                }

                container = project;
            } else // (idx > 1)
             { //folder

                IFolder folder = workbench.getRoot().getFolder(path.uptoSegment(idx + 1));

                if ((folder == null) || !folder.exists()) { //create the folder

                    try {
                        folder.create(false, true, null);
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
                }

                container = folder;
            }
        }

        try {
            container.getProject().refreshLocal(IResource.DEPTH_INFINITE, null);
        } catch (CoreException exc) {
            exc.printStackTrace();
        }

        return container;
    }

    public static final TRCMonitor createMonitor(IContainer res, String name) {
        IWorkspace workbench = res.getWorkspace();

        TRCMonitor monitor = null;
        ArrayList monitors = PDContentProvider.getMonitors(res);

        for (int idx = 0; idx < monitors.size(); idx++) {
            Object mon = monitors.get(idx);

            if (mon instanceof TRCMonitor && ((TRCMonitor) mon).getName().equals(name)) {
                return (TRCMonitor) mon;
            }
        }

        String monitorName = TString.change(name, " ", "");
        String monitorFileName = monitorName + "." + TraceConstants.MONITOR_EXT;

        IPath filePath = res.getFullPath().append(monitorFileName);

        URI uri = URI.createURI("platform:/resource" + filePath.toString());
        Resource monDoc = Resource.Factory.Registry.INSTANCE.getFactory(uri).createResource(uri);
        monDoc.setModified(true);
        EList monExt = monDoc.getContents();

        UIPlugin.getDefault().getResourceSet().getResources().add(monDoc); // prevents reloading later

        HierarchyFactory factory = UIPlugin.getDefault().getPerftraceFactory();

        monitor = factory.createTRCMonitor();
        monitor.setName(name);

//        //*** adding support for multiple files
//        SaveUtil.addDocument(monDoc);

        monExt.add(monitor);

        PDContentProvider.addMonitor(res, monitor);

        return monitor;
    }

    public static final TRCNode createNode(TRCMonitor monitor, String name, String portNumber) {
        String ipAddress = "";

        try {
            Node fnode = NodeFactory.createNode(name);

            name = fnode.getInetAddress().getHostName();
            ipAddress = fnode.getInetAddress().getHostAddress();
        } catch (UnknownHostException exc) {
        }

        Object[] nodes = monitor.getNodes().toArray();

        for (int idx = 0; idx < nodes.length; idx++) {
            TRCNode node = (TRCNode) nodes[idx];

            if (node.eIsProxy()) {
                continue;
            }

            if (node.getName().equals(name)) {
                ((TRCNode) node).setPort(Integer.parseInt(portNumber));

                return (TRCNode) node;
            }
        }

        TRCNode node = null;

        String nodeName = TString.change(monitor.getName(), " ", "") + "_" + TString.change(name, " ", "");
        String nodeFileName = nodeName + "." + TraceConstants.NODE_EXT;

        String monPath = monitor.eResource().getURI().toString();
        IPath path = new Path(monPath);

        if (path.segmentCount() > 1) {
            monPath = path.removeLastSegments(1).toString();
        }

        IPath filePath = new Path(monPath).append(nodeFileName);

        URI uri = URI.createURI(filePath.toString());
        Resource nodeDoc = Resource.Factory.Registry.INSTANCE.getFactory(uri).createResource(uri);
        nodeDoc.setModified(true);
        EList nodeExt = nodeDoc.getContents();

        UIPlugin.getDefault().getResourceSet().getResources().add(nodeDoc); // prevents reloading later

        HierarchyFactory factory = UIPlugin.getDefault().getPerftraceFactory();

        node = factory.createTRCNode();
        node.setName(name);
        node.setIpAddress(ipAddress);
        node.setMonitor(monitor);
        node.setPort(Integer.parseInt(portNumber));
        nodeExt.add(node);

//        //*** adding support for multiple files
//        SaveUtil.addDocument(nodeDoc);

        EList mon_nodes = monitor.getNodes();

        try {
            mon_nodes.add(node);
        } catch (Exception e) {
            e.printStackTrace();
        }

        return node;
    }

    /**
     * Insert the method's description here.
     * Creation date: (5/9/2001 3:16:10 PM)
     * @return com.ibm.etools.perftrace.TRCProcessProxy
     * @param viewer com.ibm.etools.pd.core.PDProjectExplorer
     * @param node com.ibm.etools.perftrace.TRCNode
     * @param pID java.lang.String
     * @param pName java.lang.String
     */
    public static TRCProcessProxy createProcess(TRCNode node, Process pInfo) {
        try {
            TRCProcessProxy process = null;
            EList processes = node.getProcessProxies();
            Iterator i = processes.iterator();

            while (i.hasNext()) {
                process = (TRCProcessProxy) i.next();

                if (process.eIsProxy()) {
                    continue;
                }

                if (String.valueOf(process.getPid()).equals(pInfo.getProcessId())) {
                    if ((process.getRuntimeId() != null) && process.getRuntimeId().equals(pInfo.getUUID())) {
                        return process;
                    }

                    //check if the process is active; if not, this is
                    //a different process whith same id
                    Iterator agents = process.getAgentProxies().iterator();

                    while (agents.hasNext()) {
                        TRCAgentProxy agent = (TRCAgentProxy) agents.next();

                        if (agent.eIsProxy()) {
                            continue;
                        }

                        if (agent.isActive()) {
                            return process;
                        }
                    }
                }
            }

            TRCMonitor monitor = node.getMonitor();

            String rID = (new Date()).toString();

            rID = TString.change(rID, " ", "");
            rID = TString.change(rID, ":", "");

            String processName = monitor.getName() + "_" + node.getName() + "_" + pInfo.getProcessId() + "_" + rID;

            String fileName = processName + "." + TraceConstants.PROCESS_EXT;

            String monPath = node.eResource().getURI().toString();
            IPath path = new Path(monPath);

            if (path.segmentCount() > 1) {
                monPath = path.removeLastSegments(1).toString();
            }

            IPath filePath = new Path(monPath).append(fileName);

            URI uri = URI.createURI(filePath.toString());

            Resource pDoc = Resource.Factory.Registry.INSTANCE.getFactory(uri).createResource(uri);
            pDoc.setModified(true);
            EList pExt = pDoc.getContents();

            UIPlugin.getDefault().getResourceSet().getResources().add(pDoc); // prevents reloading later

//            // adding support for multiple files
//            SaveUtil.addDocument(pDoc);

            HierarchyFactory factory = UIPlugin.getDefault().getPerftraceFactory();

            //create process element
            process = factory.createTRCProcessProxy();
            process.setPid(Integer.parseInt(pInfo.getProcessId()));

            /* Ensure the runtime UUID is set on the process object */
            process.setRuntimeId(pInfo.getUUID());

            if (pInfo.getName() != null) {
                process.setName(pInfo.getName());
            } else {
                process.setName(pInfo.getExecutable());
            }

            process.setLocation(((ProcessImpl) pInfo).getlocation());
            process.setActive(true);
            process.setLaunchMode(0); //attach                       

            process.setNode(node);
            pExt.add(process);

            return process;
        } catch (InactiveProcessException exc) {
            exc.printStackTrace();
        }

        return null;
    }

    /**
     * Insert the method's description here.
     * Creation date: (12/5/2000 1:00:13 PM)
     */
    public static void detachAgents() {
//        Hashtable list = SaveUtil.getDocuments();

        for (Iterator e = HierarchyResourceSetImpl.getInstance().getResources().iterator(); e.hasNext();) {
            Resource res = ((Resource) e.next());
            EList ext = res.getContents();
            Iterator i = ext.iterator();

            while (i.hasNext()) {
                Object obj = i.next();

                if (obj instanceof TRCAgentProxy) {
                    TRCAgentProxy agent = (TRCAgentProxy) obj;
                    TRCProcessProxy process = agent.getProcessProxy();

                    if ((process == null) || !process.isActive() || !agent.isActive()) {
                        continue;
                    }

                    try {
                        String hostName = process.getNode().getName();
                        Node node = NodeFactory.createNode(hostName);

                        if (!node.isConnected()) {
                            continue;
                        }

                        Object instance = LoadersUtils.locateAgentInstance(agent);

                        if ((instance != null) && instance instanceof Agent) {
                            Agent a = (Agent) instance;

                            if ((a != null) && a.isAttached()) {
                                a.detach();
                            }
                        }
                    } catch (Exception exc) {
                        exc.printStackTrace();
                    }
                }
            }
        }
    }

    public static void doAttach(Agent agent, TRCAgentProxy agentProxy) throws InactiveAgentException, InactiveProcessException {
        PIProcessListener processListener = (PIProcessListener) LoadersUtils.locateAgentListener(agentProxy);

        if (processListener == null) {
            processListener = new PIAttachListener(agentProxy.getProcessProxy().getNode(), agentProxy.getProcessProxy());
            LoadersUtils.registerAgentListener(agentProxy, processListener);
        }

        //process agent active command for this listener 
        processListener.monitor(true);
        processListener.setLaunchMode(0); //attach
        agent.addAgentListener(processListener);

        agent.attach();
    }

    /**
     * Insert the method's description here.
     * Creation date: (5/2/2001 11:06:58 AM)
     * @param viewer com.ibm.etools.pd.core.PDProjectExplorer
     * @param agent com.ibm.etools.pd.core.launcher.TraceArguments
     */
    public static final boolean launchTrace(TraceArguments trace, ArrayList filters, Vector options, String selectedProject, String selectedMonitor) {
        if ((trace.getProcess() != null) && (trace.getAgent() == null)) {
            return launchTrace(trace.getProcess());
        }

        if ((trace.getProcess() != null) && (trace.getAgent() != null)) {
            return launchTrace(trace.getAgent());
        }

        try {
            UIPlugin.getActivePage().showView(PDPerspective.ID_PD_NAVIGATOR_VIEW);
        } catch (Exception exc) {
            exc.printStackTrace();
        }

        IPreferenceStore store = UIPlugin.getDefault().getPreferenceStore();

        try {
            String nodeName = trace.getHostName();
            Node lhost = profileConnect(nodeName, String.valueOf(trace.getPortNumber()));

            if (lhost == null) {
                return false;
            }

            nodeName = lhost.getName();

            //create the container
            IContainer container = createContainer(new Path(selectedProject));

            //create the monitor
            TRCMonitor monitor = createMonitor(container, selectedMonitor);

            //create the node
            TRCNode node = createNode(monitor, nodeName, String.valueOf(trace.getPortNumber()));

            PIProcessListener listener = new PIProcessListener(node);

            listener.monitor(true);

            Process lprocess = ProcessFactory.createProcess(lhost);

            //add process location
            lprocess.setLocation(trace.getLocation());
            lprocess.addProcessListener(listener);
            listener.setLaunchMode(1);
            listener.setAutoMonitoring(trace.isAutoMonitoring());

            boolean hasProfAgents = false;

            if (store.getBoolean(TraceConstants.LOG_OPTION_KEY)) {
                String agents = store.getString(TraceConstants.LOG_KEY);

                if ((agents != null) && !agents.equals("")) {
                    int idx = 0;

                    while (idx != -1) {
                        idx = agents.indexOf(",");

                        if (idx != -1) {
                            String agent = agents.substring(0, idx);
                            Agent a = AgentFactory.createAgent(lprocess, agent, TraceConstants.LOG_AGENT_TYPE);

                            a.addAgentListener(listener);

                            a.setAutoAttach(true);
                            agents = agents.substring(idx + 1);
                        } else {
                            Agent a = AgentFactory.createAgent(lprocess, agents, TraceConstants.LOG_AGENT_TYPE);

                            a.addAgentListener(listener);
                            a.setAutoAttach(true);
                        }
                    }
                }
            }

            if (store.getBoolean(TraceConstants.PROF_OPTION_KEY)) {
                hasProfAgents = true;

                Agent a = AgentFactory.createAgent(lprocess, TraceConstants.PROFILE_AGENT_NAME, TraceConstants.PROFILE_AGENT_TYPE);

                a.addAgentListener(listener);

                a.setAutoAttach(true);

                AgentConfiguration ac = a.getConfiguration();

                if (ac == null) {
                    a.setConfiguration(new AgentConfiguration());
                }

                // EC: make sure agent configuraiton is clear.	
                a.getConfiguration().clear();

                if (trace.isToProfileFile()) {
                    a.setProfileFile(trace.getProfileFile());
                }

                addDefaultProfileData(store, a, trace, filters);
                addDefaultOptionsData(store, a, trace, options);
            }

            String exec = trace.getClassName();

            if ((trace.getVMArguments() != null) && !trace.getVMArguments().equals("")) {
                exec = trace.getVMArguments() + " " + exec;
            }

            if ((trace.getParameters() != null) && !trace.getParameters().equals("")) {
                exec += (" " + trace.getParameters());
            }

            String param = "";

            if (!hasProfAgents) {
                param = exec;
            }

            if (hasProfAgents) {
                lprocess.setExecutable("java.exe");
                param = "-XrunpiAgent:server=controlled " + exec;
            } else {
                lprocess.setExecutable("java.exe");
                param = exec;
            }

            lprocess.setParameters(param);
            lprocess.addEnvironmentVariable(new Variable("CLASSPATH", TString.stripNewLineChar(trace.getClassPath())));
            lprocess.addEnvironmentVariable(new Variable("UICLASSPATH", trace.getClassPath()));
            listener.getProcessVariableList().add(new Variable("UICLASSPATH", trace.getClassPath()));

            setEnvironmentVariable(lprocess, trace.getEnvironmentVariable().toArray(), listener);

            try {
                lprocess.launch();
            } catch (Exception e) {
                String msg = UIPlugin.getResourceString("PROCESS_LAUNCH_ERROR_");

                msg = TString.change(msg, "%1", trace.getClassName());

                String text = UIPlugin.getResourceString("PROCESS_LAUNCH_ERRTEXT_INFO_");

                Status err = new Status(Status.WARNING, ResourcesPlugin.PI_RESOURCES, IResourceStatus.INTERNAL_ERROR, text, null);

                ErrorDialog.openError(UIPlugin.getActiveWorkbenchShell(), UIPlugin.getResourceString("TRACE_MSG"), msg, // no special message
                                      err);

                return true;
            }
        } catch (Exception exc) {
            exc.printStackTrace();
        }

        return true;
    }

    public static void setAgentConfiguration(TRCAgentProxy agent, Agent a) {
        AgentConfiguration aconfig = a.getConfiguration();

        aconfig.clear();

        //set profiling options
        if (agent.getType().equals(TraceConstants.PROFILE_AGENT_TYPE)) {
            Object[] configs = agent.getConfigurations().toArray();

            if (configs.length > 0) {
                TRCConfiguration config = (TRCConfiguration) configs[0];

                Vector v = new Vector();
                Object[] filters = config.getFilters().toArray();

                for (int idx = 0; idx < filters.length; idx++) {
                    TRCFilter filter = (TRCFilter) filters[idx];

                    if (filter.getActive().booleanValue()) {
                        AgentConfigurationEntry entry = new AgentConfigurationEntry();

                        /* RKD:  If the method portion of the filter is an empty String then this is a class filter */
                        if (filter.getOperation().equals("") || filter.getOperation().equals("\"\"")) {
                            entry.setName(filter.getPattern());
                            entry.setType("SETFILTER");
                            entry.setValue(filter.getMode());
                            entry.setEnabled(filter.getActive().booleanValue());
                        } else {
                            /* Method Filter */
                            entry.setName(filter.getPattern() + ' ' + filter.getOperation());
                            entry.setType("SETMETHODFILTER");
                            entry.setValue(filter.getMode());
                            entry.setEnabled(filter.getActive().booleanValue());
                        }

                        aconfig.addEntry(entry);
                    }
                }

                Object[] options = config.getOptions().toArray();

                for (int idx = 0; idx < options.length; idx++) {
                    TRCOption option = (TRCOption) options[idx];

                    AgentConfigurationEntry entry = new AgentConfigurationEntry();

                    entry.setType("SETOPTION");
                    entry.setName(option.getKey());
                    entry.setValue(option.getValue());

                    aconfig.addEntry(entry);
                }
            }
        }
    }

    /*
     * get the folder structure of a fully qualify file name
     */
    public static String getDirectoryName(String fName) {
        int i = fName.lastIndexOf(File.separatorChar);

        if (i >= 0) {
            fName = fName.substring(0, i);
        }

        return fName;
    }

    public static Enumeration getNodeConfiguration(String nodeName, int portNumber) {
        try {
            Node node = profileConnect(nodeName, String.valueOf(portNumber));

            if (node == null) {
                return null;
            }

            return node.listProcesses();
        } catch (NotConnectedException exc2) {
            exc2.printStackTrace();
        }

        return null;
    }

    /*
     * check the folders existence of a fully qualify file path
     */
    public static boolean isPathExists(String fdir) {
        File pf = new File(getDirectoryName(fdir));

        return pf.exists();
    }

    public static boolean isZipFile(InputStream readStream) throws IOException {
        byte[] magic = new byte[2];

        if (readStream.read(magic) > 1) {
            if ((magic[0] == 'P') && (magic[1] == 'K')) {
                return true;
            }
        }

        return false;
    }

    public static boolean isZipFile(String fileName) throws IOException {
        FileInputStream readStream = new FileInputStream(fileName);
        boolean ret = isZipFile(readStream);

        readStream.close();
        readStream = null;

        return ret;
    }

    public static boolean createDirectoryMessage(String fDir, Shell shell) {
        File directory = new File(PDCoreUtil.getDirectoryName(fDir));
        String msg = UIPlugin.getResourceString("STR_FOLDER_CREATION_MSG");

        msg = TString.change(msg, "%1", directory.getPath());

        if (!MessageDialog.openQuestion(shell, UIPlugin.getResourceString("TRACE_MSG"), msg)) {
            return false;
        }

        if (!directory.mkdirs()) {
            String errMsg = UIPlugin.getResourceString("STR_ERROR_FOLDER_CREATION");

            MessageDialog.openError(shell, UIPlugin.getResourceString("TRACE_MSG"), errMsg);

            return false;
        }

        return true;
    }

    /**
     * @param process
     * @param a
     * @return
     */
    public static TRCAgentProxy createUniqueAgent(TRCProcessProxy processProxy, Agent agent) {
        long timeStamp = (new Date()).getTime();
        double time = ((double) timeStamp) / 1000;
        TRCAgentProxy agentProxy = createAgentNoCheck(processProxy, agent, timeStamp + "");
		LoadersUtils.registerAgentInstance(agentProxy,agent);
        LoadersUtils.setSessionStartTime(time, agentProxy);

        return agentProxy;
    }

    public static Node profileConnect(String host, String port) {
        String text = "";

        if ((host == null) || (host.trim() == "") || (port == null) || (port.trim() == "")) {
            text = UIPlugin.getResourceString("INVALID_HOST");

            Status err = new Status(Status.WARNING, ResourcesPlugin.PI_RESOURCES, IResourceStatus.WARNING, text, null);

            ErrorDialog.openError(UIPlugin.getActiveWorkbenchShell(), UIPlugin.getResourceString("TRACE_MSG"), UIPlugin.getResourceString("CONNECTION_FAIL_TEXT"), err);

            return null;
        }

        IPreferenceStore store = UIPlugin.getDefault().getPreferenceStore();
        ConnectUtil util = new ConnectUtil(host, port, store.getString(HyadesConstants.SECURE_USERID), UIPlugin.getDefault());
        int result = util.connect();

        if (result == ConnectUtil.CONNECTION_SUCCESS) { //store the user id
            store.setValue(HyadesConstants.SECURE_USERID, util.getUserId());

            return util.getNode();
        }

        return null;
    }

    protected static TRCAgentProxy createAgentNoCheck(TRCProcessProxy process, Agent a, String sessionId) {
        TRCAgentProxy agent;
        String timestamp = DateFormat.getDateTimeInstance(DateFormat.FULL, DateFormat.FULL).format(new Date());
        String rID = timestamp;

        rID = TString.change(rID, " ", "");
        rID = TString.change(rID, ":", "");

        String agentName = new StringBuffer(process.getNode().getMonitor().getName()).append("_").append(process.getNode().getName()).append("_").append(process.getPid()).append("_").append(rID).append("_").append(sessionId).append("_").append(a.getName()).toString();

        agentName = TString.change(agentName, " ", "");

        String pPath = process.eResource().getURI().toString();
        IPath path = new Path(pPath);

        if (path.segmentCount() > 1) {
            pPath = path.removeLastSegments(1).toString();
        }

        String fileName = agentName + "." + TraceConstants.AGENT_EXT;
        IPath filePath = new Path(pPath).append(fileName);

        URI uri = URI.createURI(filePath.toString());

        Resource agDoc = Resource.Factory.Registry.INSTANCE.getFactory(uri).createResource(uri);
        EList agExt = agDoc.getContents();

        UIPlugin.getDefault().getResourceSet().getResources().add(agDoc); // prevents reloading later

        HierarchyFactory factory = UIPlugin.getDefault().getPerftraceFactory();

        agent = factory.createTRCAgentProxy();
        agent.setName(a.getName());

        if (a.getType() != null) {
            agent.setType(a.getType());
        }

        agent.setProcessProxy(process);

        TRCAgent iAgent = factory.createTRCAgent();

        iAgent.setAgentProxy(agent);
        agExt.add(iAgent);

		markModifiedResources(iAgent,true);

//        // adding support for multiple files
//        SaveUtil.addDocument(agDoc);

        //create configuration
        if (agent.getType().equals(TraceConstants.PROFILE_AGENT_TYPE)) {
            if (agent.getName().equals(TraceConstants.J2EE_AGENT_NAME)) {
                agent.setCollectionMode(TRCCollectionMode.HEAP_AND_EXECUTION_FULL_LITERAL);
            } else {
                TRCConfiguration config = factory.createTRCConfiguration();

                config.setName(UIPlugin.getDefault().getPluginPreferences().getString(TraceConstants.ACTIVE_FILTER_NAME));
                config.setActive(true);
                config.setAgentProxy(agent);
                agent.getConfigurations().add(config);

                AgentConfiguration aconfig = a.getConfiguration();
                boolean collectHeapInfo = false;
                boolean collectExecInfo = false;

                if (aconfig != null) {
                    for (int idx = 0; idx < aconfig.size(); idx++) {
                        AgentConfigurationEntry entry = aconfig.getEntryAt(idx);

                        if (entry.getType().equals("SETFILTER")) {
                            TRCFilter filter = factory.createTRCFilter();

                            filter.setMode(entry.getValue());
                            filter.setOperation("");
                            filter.setPattern(entry.getName());
                            filter.setType(entry.getValue());
                            filter.setActive(new Boolean(entry.isEnabled()));

                            filter.setConfiguration(config);
                        } else if (entry.getType().equals("SETMETHODFILTER")) {
                            String criteria = entry.getName();
                            int offset = criteria.indexOf(' ');

                            if (offset > 0) {
                                TRCFilter filter = factory.createTRCFilter();

                                filter.setMode(entry.getValue());
                                filter.setPattern(criteria.substring(0, offset));
                                filter.setOperation(criteria.substring(offset + 1, criteria.length()));
                                filter.setType(entry.getValue());
                                filter.setActive(new Boolean(entry.isEnabled()));
                                filter.setConfiguration(config);
                            }
                        } else if (entry.getType().equals("SETOPTION")) {
                            TRCOption opt = factory.createTRCOption();

                            opt.setKey(entry.getName());
                            opt.setValue(entry.getValue());
                            opt.setConfiguration(config);

                            if (entry.getName().equals(Options.OPTION_TRACE_MODE)) {
                                collectHeapInfo = entry.getValue().equals(Options.OPTION_VALUE_DEFAULT);
                            }

                            if (entry.getName().equals(Options.OPTION_STACK_INFORMATION)) {
                                collectExecInfo = !entry.getValue().equals(Options.OPTION_VALUE_NONE);
                            }
                        }
                    }
                }

                IPreferenceStore store = UIPlugin.getDefault().getPreferenceStore();

                if (collectExecInfo) {
                    if (collectHeapInfo) // heap + execution
                     {
                        if (store.getBoolean(TraceConstants.PROF_EXEC_STAT)) // compressed
                         {
                            if (store.getBoolean(TraceConstants.PROF_HEAP_INST)) //instance level
                             {
                                agent.setCollectionMode(TRCCollectionMode.HEAP_FULL_AND_EXECUTION_STATISTICS_ONLY_LITERAL);
                            } else {
                                agent.setCollectionMode(TRCCollectionMode.HEAP_AND_EXECUTION_STATISTICS_ONLY_LITERAL);
                            }
                        } else //detail
                         {
                            agent.setCollectionMode(TRCCollectionMode.HEAP_AND_EXECUTION_FULL_LITERAL);
                        }
                    } else // execution only
                     {
                        if (store.getBoolean(TraceConstants.PROF_EXEC_STAT)) // compressed
                         {
                            agent.setCollectionMode(TRCCollectionMode.EXECUTION_STATISTICS_ONLY_LITERAL);
                        } else {
                            agent.setCollectionMode(TRCCollectionMode.EXECUTION_FULL_LITERAL);
                        }
                    }
                } else // heap only
                 {
                    if (store.getBoolean(TraceConstants.PROF_HEAP_INST)) //instance level
                     {
                        agent.setCollectionMode(TRCCollectionMode.HEAP_FULL_LITERAL);
                    } else {
                        agent.setCollectionMode(TRCCollectionMode.HEAP_STATISTICS_ONLY_LITERAL);
                    }
                }
            }
        }

        if (agent.getAgent() != null) {
            agent.getAgent().setCollectionMode(agent.getCollectionMode());
        }

        iAgent.setName(agent.getName());
        iAgent.setRuntimeId(agent.getRuntimeId());
        iAgent.setType(agent.getType());
        iAgent.setCollectionMode(agent.getCollectionMode());

        return agent;
    }

    /**
     * Sets the modified flag on the Hierarchy resources up to the monitor.
     * 
	 */
	public static void markModifiedResources(EObject eObject, boolean modifiedState) {
		while(true)
		{
			if(eObject==null)
				break;
			if(eObject instanceof TRCAgent)
			{
				eObject.eResource().setModified(modifiedState);
				eObject = ((TRCAgent)eObject).getAgentProxy().getProcessProxy();
				continue;
			}
			if(eObject instanceof TRCAgentProxy)
			{
				eObject.eResource().setModified(modifiedState);
				eObject = ((TRCAgentProxy)eObject).getProcessProxy().getNode();
				continue;
			}
			if(eObject instanceof TRCProcessProxy)
			{
				eObject.eResource().setModified(modifiedState);
				eObject = ((TRCProcessProxy)eObject).getNode();
				continue;
			}
			if(eObject instanceof TRCNode)
			{
				eObject.eResource().setModified(modifiedState);
				eObject = ((TRCNode)eObject).getMonitor();
				continue;
			}
			if(eObject instanceof TRCMonitor)
			{
				eObject.eResource().setModified(modifiedState);
			}
			break;
		}
	}

	/**
     * Insert the method's description here.
     * Creation date: (5/4/2001 11:04:16 AM)
     * @param processListener com.ibm.etools.pd.core.piclient.PIProcessListener
     */
    private static void addDefaultOptionsData(IPreferenceStore store, Agent a, TraceArguments trace, Vector optionsList) {
        AgentConfiguration config = a.getConfiguration();

        for (int idx = 0; idx < optionsList.size(); idx++) {
            config.addEntry((AgentConfigurationEntry) optionsList.elementAt(idx));
        }
    }

    /**
     * Insert the method's description here.
     * Creation date: (5/4/2001 11:04:16 AM)
     * @param processListener com.ibm.etools.pd.core.piclient.PIProcessListener
     */
    private static void addDefaultProfileData(IPreferenceStore store, Agent a, TraceArguments trace, ArrayList filtersList) {
        AgentConfiguration config = a.getConfiguration();

        for (int idx = 0; idx < filtersList.size(); idx++) {
            FilterTableElement elem = (FilterTableElement) filtersList.get(idx);

            AgentConfigurationEntry entry = new AgentConfigurationEntry();

            /* RKD:  If there is no method filtercriteria then just send the class info */
            if (elem.getMethod().equals("")) {
                entry.setName(elem.getText());
                entry.setType("SETFILTER");
            } else {
                /* RKD:  The method filter criteria is specified by placing a space between the two strings */
                entry.setName(elem.getText() + ' ' + elem.getMethod());
                entry.setType("SETMETHODFILTER");
            }

            entry.setValue(elem.getVisibility());
            entry.setEnabled(true);

            config.addEntry(entry);
        }
    }

    /**
     * Insert the method's description here.
     * Creation date: (5/2/2001 11:06:58 AM)
     * @param viewer com.ibm.etools.pd.core.PDProjectExplorer
     * @param agent com.ibm.etools.perftrace.TRCAgent
     */
    private static final boolean launchTrace(TRCAgentProxy agent) {
        try {
            UIPlugin.getActivePage().showView(PDPerspective.ID_PD_NAVIGATOR_VIEW);
        } catch (Exception exc) {
        }

        TRCProcessProxy process = agent.getProcessProxy();
        TRCNode node = process.getNode();
        String type = agent.getType();

        try {
            String param = "";

            /* Create a process on the node */
            String nodeName = node.getName();
            Node lhost = profileConnect(nodeName, String.valueOf(node.getPort()));

            if (lhost == null) {
                return false;
            }

            nodeName = lhost.getName();

            Process lprocess = ProcessFactory.createProcess(lhost);

            lprocess.setLocation(process.getLocation());

            PIProcessListener listener = new PIProcessListener(node);

            listener.monitor(true);
            lprocess.addProcessListener(listener);

            //listener.setLaunchMode(process.getValueLaunchMode());
            listener.setLaunchMode(1);

            String exec = process.getName();

            if ((process.getVmArguments() != null) && !process.getVmArguments().equals("")) {
                exec = process.getVmArguments() + " " + exec;
            }

            if ((process.getParameters() != null) && !process.getParameters().equals("")) {
                exec += (" " + process.getParameters());
            }

            if (type.equals(TraceConstants.LOG_AGENT_TYPE)) {
                param = exec;
            } else {
                param = "-XrunpiAgent:server=controlled " + exec;
                lprocess.setExecutable("java.exe");
            }

            lprocess.setParameters(param);
            lprocess.addEnvironmentVariable(new Variable("CLASSPATH", TString.stripNewLineChar(process.getClasspath())));
            lprocess.addEnvironmentVariable(new Variable("UICLASSPATH", process.getClasspath()));
            listener.getProcessVariableList().add(new Variable("UICLASSPATH", process.getClasspath()));

            setEnvironmentVariable(lprocess, process.getEnvironmentVariables().toArray(), listener);

            setAgent(agent, lprocess, listener);

            try {
                lprocess.launch();
            } catch (Exception e) {
                String msg = UIPlugin.getResourceString("PROCESS_LAUNCH_ERROR_");

                msg = TString.change(msg, "%1", process.getName());

                String text = UIPlugin.getResourceString("PROCESS_LAUNCH_ERRTEXT_INFO_");

                Status err = new Status(Status.WARNING, ResourcesPlugin.PI_RESOURCES, IResourceStatus.INTERNAL_ERROR, text, null);

                ErrorDialog.openError(UIPlugin.getActiveWorkbenchShell(), UIPlugin.getResourceString("TRACE_MSG"), msg, // no special message
                                      err);

                return true;
            }

            //register the trace
            UIPlugin.getDefault().getTraceManager().registerTrace(agent, TraceConstants.LAUNCH_MODE);
        } catch (Exception exc) {
            exc.printStackTrace();
        }

        return true;
    }

    /**
     * Insert the method's description here.
     * Creation date: (5/2/2001 11:06:58 AM)
     * @param viewer com.ibm.etools.pd.core.PDProjectExplorer
     * @param agent com.ibm.etools.perftrace.TRCProcessProxy
     */
    private static final boolean launchTrace(TRCProcessProxy process) {
        try {
            UIPlugin.getActivePage().showView(PDPerspective.ID_PD_NAVIGATOR_VIEW);
        } catch (Exception exc) {
        }

        TRCNode node = process.getNode();
        Object[] agents = process.getAgentProxies().toArray();

        try {
            String param = "";
            String nodeName = node.getName();
            Node lhost = profileConnect(nodeName, String.valueOf(node.getPort()));

            if (lhost == null) {
                return false;
            }

            nodeName = lhost.getName();

            Process lprocess = ProcessFactory.createProcess(lhost);

            lprocess.setLocation(process.getLocation());

            PIProcessListener listener = new PIProcessListener(node);

            listener.monitor(true);

            listener.setLaunchMode(process.getLaunchMode());
            listener.setLaunchMode(1);
            lprocess.addProcessListener(listener);

            String exec = process.getName();

            if ((process.getVmArguments() != null) && !process.getVmArguments().equals("")) {
                exec = process.getVmArguments() + " " + exec;
            }

            if ((process.getParameters() != null) && !process.getParameters().equals("")) {
                exec += (" " + process.getParameters());
            }

            boolean hasProfAgents = false;

            for (int idx = 0; idx < agents.length; idx++) {
                TRCAgentProxy agent = (TRCAgentProxy) agents[idx];

                if (agent.eIsProxy()) {
                    continue;
                }

                if (agent.getType().equals(TraceConstants.PROFILE_AGENT_TYPE)) {
                    hasProfAgents = true;
                }

                setAgent(agent, lprocess, listener);
            }

            if (!hasProfAgents) {
                param = exec;
            }

            param = "-XrunpiAgent:server=controlled " + exec;
            lprocess.setExecutable("java.exe");

            lprocess.setParameters(param);
            lprocess.addEnvironmentVariable(new Variable("CLASSPATH", TString.stripNewLineChar(process.getClasspath())));
            lprocess.addEnvironmentVariable(new Variable("UICLASSPATH", process.getClasspath()));
            listener.getProcessVariableList().add(new Variable("UICLASSPATH", process.getClasspath()));

            setEnvironmentVariable(lprocess, process.getEnvironmentVariables().toArray(), listener);

            try {
                lprocess.launch();
            } catch (Exception e) {
                String msg = UIPlugin.getResourceString("PROCESS_LAUNCH_ERROR_");

                msg = TString.change(msg, "%1", process.getName());

                String text = UIPlugin.getResourceString("PROCESS_LAUNCH_ERRTEXT_INFO_");

                Status err = new Status(Status.WARNING, ResourcesPlugin.PI_RESOURCES, IResourceStatus.INTERNAL_ERROR, text, null);

                ErrorDialog.openError(UIPlugin.getActiveWorkbenchShell(), UIPlugin.getResourceString("TRACE_MSG"), msg, // no special message
                                      err);

                return true;
            }

            //register the trace
            UIPlugin.getDefault().getTraceManager().registerTrace(process, TraceConstants.LAUNCH_MODE);
        } catch (Exception exc) {
            exc.printStackTrace();
        }

        return true;
    }

    /**
     * Insert the method's description here.
     * Creation date: (5/2/2001 12:19:35 PM)
     * @param agent com.ibm.etools.perftrace.TRCAgent
     * @param processListener com.ibm.etools.pd.core.piclient.PIProcessListener
     */
    private static void setAgent(TRCAgentProxy agent, Process process, PIProcessListener listener) {
        Agent a = AgentFactory.createAgent(process, agent.getName(), agent.getType());

        a.addAgentListener(listener);

        setAgentConfiguration(agent, a);
    }

    /**
     *
     * @param process
     * @param envVariables
     * @param listener
     */
    private static void setEnvironmentVariable(Process process, Object[] envVariables, PIProcessListener listener) {
        if ((envVariables == null) || (process == null)) {
            return;
        }

        for (int i = 0; i < envVariables.length; i++) {
            EnvTableElement tableElement = (EnvTableElement) (envVariables[i]);

            try {
                process.addEnvironmentVariable(new Variable(tableElement.name(), tableElement.value()));
                listener.getProcessVariableList().add(new Variable(tableElement.name(), tableElement.value()));
            } catch (Exception e) {
                System.out.println("Error adding environment variable " + tableElement.name());
            }
        }
    }

	/**
	 * @param fAgentOld
	 * @param fAgent
	 */
	public static void copyAgentProperties(TRCAgentProxy fAgentOld, TRCAgentProxy fAgent) {
		fAgent.setCollectionMode(fAgentOld.getCollectionMode());
		fAgent.setProfileFile(fAgentOld.getProfileFile());
		fAgent.getConfigurations().clear();
		for (Iterator iter = fAgentOld.getConfigurations().iterator(); iter.hasNext();) {
			TRCConfiguration configuration = (TRCConfiguration) iter.next();
			fAgent.getConfigurations().add(EcoreUtil.copy(configuration));
			
		};
		TRCProcessProxy process = fAgent.getProcessProxy();
		TRCNode node = process.getNode();
		sendConfigurationToAgent(fAgent, node.getName(), process.getPid());			
				
	}
	private static void sendConfigurationToAgent(TRCAgentProxy agent, String host, int processId)
	{
		Node node = PDCoreUtil.profileConnect(host, String.valueOf(agent.getProcessProxy().getNode().getPort()));
		if(node == null)
			return;

	
		Process p = node.getProcess(String.valueOf(processId));

		if(p != null)
		{
			try {
				Agent a = p.getAgent(agent.getName());
				a.setProfileFile(agent.getProfileFile());
				PDCoreUtil.setAgentConfiguration(agent, a);
				if(a != null && a.isActive()) {
				  a.publishConfiguration();
			  
				  /* Apply the filters */
				  CustomCommand command = new CustomCommand();
				  command.setData("APPLYFILTERS");
				  a.invokeCustomCommand(command);			  
				}			
			}
			catch(InactiveAgentException aex)
			{
			}
		}
	}

}
