package org.eclipse.hyades.logging.parsers;

/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/


import java.text.ParsePosition;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.Locale;

import org.eclipse.hyades.logging.core.ILogger;
import org.eclipse.hyades.logging.events.IExtendedDataElement;

/**
 * AbstractErrorLogParser is the abstract superclass for the ApacheErrorLogParser.
 * <p>
 * For each error log record, this class parses the time stamp, message category,  
 * client IP address, and message.  This method also has a simplistic algorithm for 
 * finding a file name in the message field.  This parsed information and information 
 * about the local machine (machine where the parse is performed, but not 
 * necessarily the machine that produced the error log) is used to produce a
 * {@link org.eclipse.hyades.logging.events.ICommonBaseEvent} object for each record.
 */
public abstract class AbstractErrorLogParser extends Parser {

	//  Constants

	/**
	 * Constant for XML DateTime format.
	 */
	private final static String TARGET_FORMAT = "yyyy-MM-dd HH:mm:ss";

	/**
	 * Constant for time stamp format.
	 */
	private final static String TIME_STAMP_FORMAT = "EEE MMM d HH:mm:ss yyyy";

	/**
	 * Constant for String "category".
	 */
	private final static String CATEGORY = "category";

	/**
	 * Constant for String "CGI debugging output".
	 */
	private final static String CGI = "CGI_debugging_output";

	/**
	 * Constant code for "debug".
	 */
	private final static int DEBUG = 1;

	/**
	 * Constant code for "info".
	 */
	private final static int INFO = 2;

	/**
	 * Constant code for "notice".
	 */
	private final static int NOTICE = 3;

	/**
	 * Constant code for "warn".
	 */
	private final static int WARN = 4;

	/**
	 * Constant code for "error".
	 */
	private final static int ERROR = 5;

	/**
	 * Constant code for "crit".
	 */
	private final static int CRIT = 6;

	/**
	 * Constant code for "alert".
	 */
	private final static int ALERT = 7;

	/**
	 * Constant code for "emerg".
	 */
	private final static int EMERG = 8;

	/**
	 * Constant for String "debug".
	 */
	private final static String DEBUG_STRING = "debug";

	/**
	 * Constant for String "info".
	 */
	private final static String INFO_STRING = "info";

	/**
	 * Constant for String "notice".
	 */
	private final static String NOTICE_STRING = "notice";

	/**
	 * Constant for String "warn".
	 */
	private final static String WARN_STRING = "warn";

	/**
	 * Constant for String "error".
	 */
	private final static String ERROR_STRING = "error";

	/**
	 * Constant for String "crit".
	 */
	private final static String CRIT_STRING = "crit";

	/**
	 * Constant for String "alert".
	 */
	private final static String ALERT_STRING = "alert";

	/**
	 * Constant for String "emerg".
	 */
	private final static String EMERG_STRING = "emerg";

	// Variables to hold values parsed from error log records.  

	/**
	 * Parsed value of the message category field. 
	 */
	protected String category;

	/**
	 * Parsed value of the time stamp field converted to an XML DateTime  String. 
	 */
	protected StringBuffer creationTime;

	/**
	 * Parsed value of the client IP address field. 
	 */
	protected String hostID;

	/**
	 * Parsed value of the message field. 
	 */
	protected String body;

	// Variable to hold user-supplied value

	/**
	 * User-supplied value of the version of the web server product.  Produced in 
	 * the setUserInput method of the subclass.
	 */
	protected String productVersion;

	// Variable to hold product information

	/**
	 * Web server product name and version supplied by the subclass.   
	 */
	protected String sourceId;

	// variables associated with processing the parsed information

	/**
	 * The severity type associated with the message category.
	 */
	protected short Type;

	/**
	 * Variable to hold file name found in the message field. 
	 */
	protected String file;

	// Variables associated with counting messages with identical time stamp values

	/**
	 * XML DateTime String representation of time stamp in previous error log record. 
	 */
	private String currentCreationTime = "";

	/**
	 * Count of records with identical time stamp values. 
	 */
	private long duplicateCreationTimeCounter = 0;

	/**
	 * Used to indicate the current line that is being parsed.  This variable is used
	 * to display the line number that may have caused an error while parsing the file.
	 */
	private long lineNumber = 1;
	
	/**
	 * Stores the original content of the current line being parsed
	 */
	private String originalLine;

	/**
	 * Parses each error log record and produces a CommonBaseEvent object that is 
	 * written to the logger.
	 *
	 * @param argLogger  logger to which PD_Message objects are written  
	 * @exception LogParserException thrown if the parser is unable to parse the error log
	 */
	public void parse(ILogger argLogger) throws LogParserException {
		super.parse(argLogger);

		try {
			while (curLine != null) {
				reset();
                
                
				messages[arrayIndex].init();
				messages[arrayIndex].setSourceComponentId(eventFactory.createComponentIdentification());
				messages[arrayIndex].getSourceComponentId().init();
				messages[arrayIndex].setMsgDataElement(eventFactory.createMsgDataElement());
				messages[arrayIndex].getMsgDataElement().init();
                
				messages[arrayIndex].getSourceComponentId().setComponent(sourceId);
				messages[arrayIndex].getSourceComponentId().setLocation(localHostId);
				messages[arrayIndex].getSourceComponentId().setLocationType(localHostIdFormat);
				messages[arrayIndex].getSourceComponentId().setSubComponent(ParserConstants.UNKNOWN);
				messages[arrayIndex].getMsgDataElement().setMsgIdType(ParserConstants.NOT_APPLICABLE);
				messages[arrayIndex].getMsgDataElement().setMsgId(ParserConstants.NONE);

				if (parseDate(0, 0)) {
					parseRemainder();
					messages[arrayIndex].setSeverity(Type);
					if (category != null) {
						IExtendedDataElement categoryToken = createStringEDE(CATEGORY, category);
						messages[arrayIndex].addExtendedDataElement(categoryToken);
					}
					if (hostID != null) {
						IExtendedDataElement clientToken = createStringEDE(ParserConstants.CLIENT, hostID);
						messages[arrayIndex].addExtendedDataElement(clientToken);
					}
					if (file != null) {
						IExtendedDataElement fileToken = createStringEDE(ParserConstants.FILE, file);
						messages[arrayIndex].addExtendedDataElement(fileToken);
					}

					messages[arrayIndex].setMsg(body);
					messages[arrayIndex].getMsgDataElement().setMsgLocale(ParserConstants.LOCALE_EN_US);
				}
				// handle CGI debugging output
				else {
					body = curLine.trim();
					curLine = getNextLine();
					while (curLine != null) {
						if (!parseDate(0, 1)) {
							body += " " + curLine.trim();
							curLine = getNextLine();
						}
						else
							break;
					}
					if (body != null) {
						if (body.trim().length() == 0) {
							continue;
						}

						messages[arrayIndex].setMsg(body);
						IExtendedDataElement CGIToken = createStringEDE(CGI, body);
						messages[arrayIndex].addExtendedDataElement(CGIToken);
					}
					messages[arrayIndex].setSeverity(ParserConstants.SEVERITY3);
				} 
				messages[arrayIndex].setCreationTime(creationTime.toString());
				StringBuffer idString = new StringBuffer(localHostId);
				if (file_path != null) {
					idString.append(' ');
					idString.append(file_path);
				}
				if (localHostName != null) {
					idString.append(' ');
					idString.append(localHostName);
				}
				idString.append(' ');

				//Set the message counter (e.g., duplicate timestamp counter):
				if (currentCreationTime.equals(creationTime.toString().trim()))
					messages[arrayIndex].setSequenceNumber(++duplicateCreationTimeCounter);
				else {
					currentCreationTime = creationTime.toString().trim();
					duplicateCreationTimeCounter = 0;
				}

				//CBE           idString.append(creationTime.toString());
				//CBE           idString.append('_');
				//CBE           idString.append(messages[arrayIndex].getSequenceNumber());
				//CBE           messages[arrayIndex].setGlobalInstanceId(idString.toString());

				//if (arrayIndex == PDMessageArraySize - 1) {
				//    logger.write(messages);
				//    for (int i = 0; i < PDMessageArraySize; i++) {
				//        messages[i].init();
				//    }
				//    arrayIndex = 0;
				//}
				//else
				//    arrayIndex++;
				logger.write(messages[arrayIndex]);
			                
			}
			writePartialArray(); // for now just closes file
		}
		catch (Throwable throwable) {			
			ParserUtilities.exceptionHandler(throwable,lineNumber,originalLine,ParserUtilities.getResourceString("ERROR_LOG_PARSER_ERROR_"));
		}
	}

	/**
	 * Resets the parsed values of a log record before next record is parsed.
	 */
	protected void reset() {
		Type = 0;
		body = null;
		hostID = null;
		file = null;
		category = null;
		creationTime = null;
	}

	/**
	 * Parses a time stamp from an error log record, produces a Date object from the
	 * parsed information, and converts the Date object into a XML DateTime String.
	 *
	 * @param start  Position of time stamp in curLine (always set to 0) 
	 * @param mode   0 if time stamp is to be parsed; 1 if time stamp is only checked for validity
	 * @return       true if time stamp is valid; false otherwise 
	 */
	protected boolean parseDate(int start, int mode) {
		SimpleDateFormat formatter;
		String temp = "";
		curLine = curLine.trim();
		curLine = curLine + "\n";
		if (isChar(curLine, start + 0, '[') && isLet(curLine, start + 1, 3) && isChar(curLine, start + 4, ' ') && isLet(curLine, start + 5, 3) && isChar(curLine, start + 8, ' ') && isNum(curLine, start + 10, 1) && isChar(curLine, start + 11, ' ') && isNum(curLine, start + 12, 2) && isChar(curLine, start + 14, ':') && isNum(curLine, start + 15, 2) && isChar(curLine, start + 17, ':') && isNum(curLine, start + 18, 2) && isChar(curLine, start + 20, ' ') && isNum(curLine, start + 21, 4) && isChar(curLine, start + 25, ']')) {
			temp += curLine.substring(1, 25);
			formatter = new SimpleDateFormat(TIME_STAMP_FORMAT, Locale.US);
			ParsePosition parsePosition = new ParsePosition(0);
			Date artifactDate = formatter.parse(temp, parsePosition);
			if (artifactDate != null) {
				if (mode == 0) {
					formatter = new SimpleDateFormat(TARGET_FORMAT);
					creationTime = new StringBuffer(formatter.format(artifactDate));
					creationTime.replace(10, 11, "T");
					creationTime.append(ParserConstants.SIX_ZERO);
					Calendar localCalendar = Calendar.getInstance();
					localCalendar.setTime(artifactDate);

					// Time Zone (+/-hh:mm)

					char sign;
					int offset = (localCalendar.get(Calendar.ZONE_OFFSET) + localCalendar.get(Calendar.DST_OFFSET)) / 60000;
					String numberHours = null;
					String numberMinutes = null;
					if (offset >= 0) {
						sign = '+';
						numberHours = String.valueOf(offset / 60);
						numberMinutes = String.valueOf(offset % 60);
					}
					else {
						sign = '-';
						numberHours = String.valueOf(-1 * offset / 60);
						numberMinutes = String.valueOf(-1 * offset % 60);
					}
					creationTime.append(sign);

					if (numberHours.length() == 1)
						creationTime.append("0");
					creationTime.append(numberHours);
					creationTime.append(":");

					if (numberMinutes.length() == 1)
						creationTime.append("0");
					creationTime.append(numberMinutes);

					curLine = curLine.substring(26);
				}
				return true;
			}
		}
		else if (mode == 0) {
			
			/* create timestamp for CGI debugging output
			   if previous timestamp available, reuse */
			   
			if (!currentCreationTime.equals(""))
				creationTime = new StringBuffer(currentCreationTime);
			
			/* else set the time to zero */
			else {
				Date date = new Date();
				date.setTime(0);
				formatter = new SimpleDateFormat(TARGET_FORMAT);
				creationTime = new StringBuffer(formatter.format(date));
				creationTime.replace(10, 11, "T");
				creationTime.append(ParserConstants.SIX_ZERO);
				Calendar localCalendar = Calendar.getInstance();
				localCalendar.setTime(date);
				char sign;
				int offset = (localCalendar.get(Calendar.ZONE_OFFSET) + localCalendar.get(Calendar.DST_OFFSET)) / 60000;
				String numberHours = null;
				String numberMinutes = null;
				if (offset >= 0) {
					sign = '+';
					numberHours = String.valueOf(offset / 60);
					numberMinutes = String.valueOf(offset % 60);
				}
				else {
					sign = '-';
					numberHours = String.valueOf(-1 * offset / 60);
					numberMinutes = String.valueOf(-1 * offset % 60);
				}
				creationTime.append(sign);

				if (numberHours.length() == 1)
					creationTime.append("0");
				creationTime.append(numberHours);
				creationTime.append(":");

				if (numberMinutes.length() == 1)
					creationTime.append("0");
				creationTime.append(numberMinutes);
			}
		}
		return false;
	}

	/**
	 * Parses remainder of error log record following time stamp.
	 */
	protected void parseRemainder() {
		curLine = curLine.trim();
		curLine = curLine + "\n";
		parseType(0);
		parseBody();
	}

	/**
	 * Parses the message category and assigns a severity type.
	 *
	 * @param start  Position of message category in curLine (always set to 0) 
	 */
	protected void parseType(int start) {
		int typeLength = 0, typeCode = -1;
		boolean foundType = false;
		Type = ParserConstants.SEVERITY3;
		curLine = curLine.trim();
		curLine = curLine + "\n";
		if (isChar(curLine, start + 0, '[')) {
			if (isPara(curLine, start + 1, DEBUG_STRING) == 0) {
				foundType = true;
				typeLength = 5;
				typeCode = DEBUG;
				category = DEBUG_STRING;
			}
			else if (isPara(curLine, start + 1, INFO_STRING) == 0) {
				foundType = true;
				typeLength = 4;
				typeCode = INFO;
				category = INFO_STRING;
			}
			else if (isPara(curLine, start + 1, NOTICE_STRING) == 0) {
				foundType = true;
				typeLength = 6;
				typeCode = NOTICE;
				category = NOTICE_STRING;
			}
			else if (isPara(curLine, start + 1, WARN_STRING) == 0) {
				foundType = true;
				typeLength = 4;
				typeCode = WARN;
				category = WARN_STRING;
			}
			else if (isPara(curLine, start + 1, ERROR_STRING) == 0) {
				foundType = true;
				typeLength = 5;
				typeCode = ERROR;
				category = ERROR_STRING;
			}
			else if (isPara(curLine, start + 1, CRIT_STRING) == 0) {
				foundType = true;
				typeLength = 4;
				typeCode = CRIT;
				category = CRIT_STRING;
			}
			else if (isPara(curLine, start + 1, ALERT_STRING) == 0) {
				foundType = true;
				typeLength = 5;
				typeCode = ALERT;
				category = ALERT_STRING;
			}
			else if (isPara(curLine, start + 1, EMERG_STRING) == 0) {
				foundType = true;
				typeLength = 5;
				typeCode = EMERG;
				category = EMERG_STRING;
			}
		}
		if (foundType)
			if (isChar(curLine, start + typeLength + 1, ']')) {
				switch (typeCode) {
					case (EMERG) :
					case (ALERT) :
					case (CRIT) :
					case (ERROR) :
						Type = ParserConstants.SEVERITY1;
						break;
					case (WARN) :
						Type = ParserConstants.SEVERITY2;
						break;
					case (NOTICE) :
					case (INFO) :
					case (DEBUG) :
						Type = ParserConstants.SEVERITY3;
						break;
				}
				curLine = curLine.substring(start + typeLength + 2);
			}
	}

	/**
	 * Parses client IP address and message from an error log record.  This method
	 * also looks for a file name in the message field.
	 */
	protected void parseBody() {
		int startClient = curLine.indexOf("[client");
		int endClient = curLine.indexOf(']');
		if (startClient != -1 && endClient != -1 && endClient > startClient + 7) {
			hostID = (curLine.substring(curLine.indexOf("[client") + 7, curLine.indexOf(']'))).trim();
			curLine = curLine.substring(curLine.indexOf(']') + 1);
		}
		body = curLine.trim();
		if (body.indexOf(": ") != -1) {
			file = body.substring(body.indexOf(": ") + 2).trim();
		}
		curLine = getNextLine();
	}

	/**
	 * Looks for a search String h within another String line at a specified position.  
	 * 
	 * @param line   String in which to look for the search String
	 * @param pos    Position at which to look for the search String
	 * @param h      Search String 
	 * @return       0 if the search String is found at the specified position; nonzero value otherwise 
	 */
	protected int isPara(String line, int pos, String h) {
		try {
			return line.substring(pos, pos + h.length()).compareTo(h);
		}
		catch (Exception e) {
			return 1;
		}
	}
    
    
	/**
	 * A wrapper method used to return the next line in the file
	 * 
	 * @return		The next line in the log file
	 */
	private String getNextLine()
	{
		lineNumber++;
		originalLine = readLine();
		return originalLine;
		    	
	}

}