package org.eclipse.hyades.ui.sample.svg.generator;
/*******************************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/

import java.io.Serializable;
import java.util.Date;

import org.w3c.dom.CDATASection;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

/**
 * <code>SVGLineChart</code> generates a line chart graphic DOM using Scalable Vector Graphics (SVG).
 * 
 * @version 1.70.1.30
 */
public class SVGLineChart extends SVGXYChart implements IGraphicDocumentStyle, IDataInputProcessingExceptionCodes, Serializable {
	/**
	 * Sole constructor
	 */ 
	public SVGLineChart() {
		super();
		graphicType = IGraphicTypeConstants.LINE_CHART;						
	}


	/**
     * @see IGraphicDocumentGenerator#generateGraphicDocument(GraphicDocumentProperties)
     */       
	public Document generateGraphicDocument(GraphicDocumentProperties input) throws DataInputProcessingException {	
		// get the input
		GraphicAttributes graphicAttrs = createGraphicAttributes(input);	
		Document newDataDocument = graphicAttrs.getDataDocument();		
		int numberOfDatasets = 0;
		// make sure there is some data, or exit
		try {
			numberOfDatasets = dataRetriever.getNumberOfDatasets(newDataDocument);
		} catch (DataInputProcessingException e) {
			e.setWidgetType(graphicType);
			//System.out.println(e.getWidgetType() + " " + e.getElement() + " " + e.getMessage());
			throw e;		
		}		
		
		String rangeMaxValue = null;
		int gridXOffset = 60;
		int halfGridXOffset = gridXOffset/2;
		int gridYOffset = 50;
		
		// grid     
		try {
			xAxis = dataRetriever.getDataRange(newDataDocument, "S");
		} catch (DataInputProcessingException e) {
			e.setWidgetType(getGraphicType());
			//System.out.println(e.getWidgetType() + " " + e.getElement() + " " + e.getMessage());
			throw e;				
		}					

		try {
			yAxis = dataRetriever.getDataRange(newDataDocument, "W");
		} catch (DataInputProcessingException e) {
			e.setWidgetType(getGraphicType());
			//System.out.println(e.getWidgetType() + " " + e.getElement() + " " + e.getMessage());
			throw e;				
		}

		if ((xAxis == null) || (yAxis == null)) {
			throw new DataInputProcessingException("", NO_RANGE_FOUND, DATARANGE, getGraphicType());
		}
		if ((xAxis.getType() != DataRange.CONTINUUM) || (yAxis.getType() != DataRange.CONTINUUM)) {
			throw new DataInputProcessingException("", UNSUPPORTED_DATA_RANGE_TYPE, DATARANGE, getGraphicType());
		}

		xMarkers = xAxis.getSegmentMarkers();
		yMarkers = yAxis.getSegmentMarkers();

		if (yMarkers != null && yMarkers.length > 0) { 
			rangeMaxValue = (String)yMarkers[yMarkers.length - 1].getValueString();
		}		
		
		// The datasets
		dataSets = new DataSet[numberOfDatasets];
		for(int i=0; i < numberOfDatasets; i++) {
            dataSets[i] = dataRetriever.getDatasetWithPosition(newDataDocument, Integer.toString(i));
		}
		//get the Fly over text for the legend item.						
		legendFlyOverLabels = getLegendFlyOverLabels(dataSets);
		
		//get the max size of Ymarker Width.		
		double maxLengthOfYmarkerWidth = getMaxLengthOfYmarker(yMarkers, graphicAttrs);
	    double maxAllowableYMarkerWidth = graphicAttrs.getGraphicWidth() / 4;
	    if (maxLengthOfYmarkerWidth > maxAllowableYMarkerWidth) {
			maxLengthOfYmarkerWidth = maxAllowableYMarkerWidth;
		}
		
		//get the max size of Legent Lable Width.		
		double maxLengthOfLegendLableWidth = getMaxLengthOfLegendLabel(graphicAttrs, dataSets);
	    double maxAlloableLegendWidth = graphicAttrs.getGraphicWidth() / 3;
   		if (maxLengthOfLegendLableWidth > maxAlloableLegendWidth) {
			maxLengthOfLegendLableWidth = maxAlloableLegendWidth;
		}

		// Start calculations
        if (graphicAttrs.isLegendSuppressed()) {
			maxLengthOfLegendLableWidth = 0;
        }
        
		int xAxisLength = (int)(graphicAttrs.getGraphicWidth() - (maxLengthOfLegendLableWidth + maxLengthOfYmarkerWidth + gridXOffset));
   		int xLabelRowCount = getNumberOfAxisLabels(graphicAttrs, xAxisLength, xAxis);									
		int yAxisLength = (int)(graphicAttrs.getGraphicHeight() - (gridYOffset * 2.5) - (xLabelRowCount * 10));			
		Document generatedDocument = createSvgDocument(newDataDocument.getImplementation(), Short.toString(graphicAttrs.getGraphicWidth()), Short.toString(graphicAttrs.getGraphicHeight()));
		Element svgRoot = generatedDocument.getDocumentElement();
		registerEventHandler(svgRoot, "onload", "init(evt)");	
		addDefinitions(generatedDocument, graphicAttrs, numberOfDatasets);					
		addJavaScriptFunctions(generatedDocument, graphicAttrs, numberOfDatasets, gridXOffset, gridYOffset, xAxisLength, yAxisLength);
 		
 		// graphic outline 
 		if (!graphicAttrs.isOutlineSuppressed()) {
 			addOutline(generatedDocument, graphicAttrs);
 		}	 		
 		 		
		// timestamp
		if (!graphicAttrs.isTimestampSuppressed()) {
			try {
				String timestamp = dataRetriever.getTimestamp(newDataDocument);
				Date date = GraphicAttributes.parseISO8601Date(timestamp);
	        	addTimeStamp(generatedDocument, graphicAttrs, (date == null)?timestamp:graphicAttrs.formatDate(date));
			} catch (DataInputProcessingException e) {
				e.setWidgetType(graphicType);
				//System.out.println(e.getWidgetType() + " " + e.getElement() + " " + e.getMessage());
				throw e;
			}
		}
			
		// title bar
 		if (!graphicAttrs.isTitlebarSuppressed()) {	 		
	   		addTitleBar(generatedDocument, graphicAttrs);
 		}
 		// need to adjust graphic vertical position
 		else {
 			gridYOffset = super.resetGridOffsetForCenter(graphicAttrs.getGraphicHeight(), yAxisLength, 3.0);	
 		}
 		
		// preference icon
		if(!graphicAttrs.isUserPreferencesSuppressed() && graphicAttrs.getSvgFormatOnly())
		{
			addPreferenceIcon(generatedDocument, graphicAttrs);
		}
 
		// the actual x offset
		gridXOffset = (halfGridXOffset) + (int)maxLengthOfYmarkerWidth;
		
		// border, tick marks etc.
        addGrid(generatedDocument, svgRoot, xAxisLength, yAxisLength, gridXOffset, gridYOffset, xAxis, yAxis);
	
		// x and y axis labels
		addAxisLabels(generatedDocument, graphicAttrs, svgRoot, xAxisLength, yAxisLength, gridXOffset, gridYOffset, xAxis, yAxis);
        // axis titles
        addTitles(generatedDocument, graphicAttrs, svgRoot, xAxisLength, yAxisLength, gridXOffset, gridYOffset, xAxis, yAxis, xLabelRowCount, rangeMaxValue);

        // data lines
		Element gTextBoxElement = generatedDocument.createElement("g");
        addDataSets(generatedDocument, graphicAttrs, svgRoot, gTextBoxElement, xAxisLength, yAxisLength, gridXOffset, gridYOffset, xAxis, yAxis, dataSets); 

        // legend
        if (!graphicAttrs.isLegendSuppressed()) {      		
	  		try {
	       		String [] legendLabels = dataRetriever.getLegendLabels(newDataDocument);		       
				addLegend(generatedDocument, graphicAttrs, gTextBoxElement, legendLabels, legendFlyOverLabels, (int)maxLengthOfYmarkerWidth + xAxisLength + (halfGridXOffset), gridYOffset); 								

	  		} catch (DataInputProcessingException e) {
	  			e.setWidgetType(graphicType);
	  			//System.out.println(e.getWidgetType() + " " + e.getElement() + " " + e.getMessage());	  			
	  			throw e;
	  		} 		
    	}
		svgRoot.appendChild(gTextBoxElement);
    	return generatedDocument;
	}	
	
	/**
     * @see SVGDocumentGenerator#addLegendItemGraphic()
     */ 
	protected Element addLegendItemGraphic(Document generatedDocument, Element parent, int index, int fontSize, int x, int y) {
		final String[] legendShapeType = {"#legendcircleshape", "#legendsquareshape", "#legendhorizontalrectshape", "#legendverticalrectshape", "#legendtriangleshape", "#legenddiamondshape"};
		Element useElementShape, useElementLine, gColourElement;
		
		// add the standard shape to the legend
		super.addLegendItemGraphic(generatedDocument, parent, index, fontSize, x, y);
		
 		useElementShape = generatedDocument.createElement("use");
		useElementShape.setAttribute("xlink:href", legendShapeType[index % (legendShapeType.length)]);
		useElementShape.setAttribute("x", Integer.toString(x));
		// rectangles are placed using upper left corner
		useElementShape.setAttribute("y", Integer.toString(y + fontSize - 8));    
		
		// add the element to its color group for user preference changes
		gColourElement = generatedDocument.createElement("g");
		gColourElement.setAttribute("id", "shape" + index);		
		gColourElement.setAttribute("onclick","toggleVisibility('dataColour" + index + "');");
		gColourElement.setAttribute("class", "shape" + index);	
		gColourElement.appendChild(useElementShape);
		return gColourElement;
	}
	

	// specific to line chart 
	private void addDataSets(Document generatedDocument, GraphicAttributes attrs, Element parent, Element gTextBoxElement, int xAxisLength, int yAxisLength, int gridXOffset, int gridYOffset, DataRange xAxis, DataRange yAxis, DataSet[] dataSets)
		throws DataInputProcessingException {

		// add lines
		double xMin = xAxis.getMinValue();
		double xMax = xAxis.getMaxValue();
		double yMin = yAxis.getMinValue();
		double yMax = yAxis.getMaxValue();
		
		if ((xMin == xMax) || (yMin == yMax)) return;

		for (int i = 0; i < dataSets.length; i++) {
			DataSet dataset = dataSets[i];
			DataPoint[] dps = dataset.getDataPoints();
			double[] yPositions = new double[dps.length];
			double[] xPositions = new double[dps.length];
			
			for(int j=0; j < dps.length; j++) {
				xPositions[j] = xAxisLength * (dps[j].getValue1() - xMin) / (xMax - xMin);
				yPositions[j] = yAxisLength * (yMax - dps[j].getValue2()) / (yMax - yMin);
			}
			
		 	// add the line		
	  	 	addLine(generatedDocument, parent, dps, i, xPositions, yPositions, xAxisLength, yAxisLength, gridXOffset, gridYOffset);

	  	 	// add the texbox
			String stylesheetClass = "dataValues anchorAtMiddle";		
			Element gToggleElement = generatedDocument.createElement("g");
			gToggleElement.setAttribute("id","textbox" + i);
			gToggleElement.setAttribute("class", stylesheetClass);
			gToggleElement.setAttribute("transform", "translate(" + gridXOffset + "," + gridYOffset + ")");
			//gToggleElement.setAttribute("font-family", font);
			gToggleElement.setAttribute("visibility", "hidden");
			gToggleElement.setAttribute("onclick", "toggleVisibility('textbox" + i + "');");
			gTextBoxElement.appendChild(gToggleElement);	
			addTextBoxes(generatedDocument, attrs, gToggleElement, i, xPositions, yPositions, dps, xAxisLength, yAxisLength);
		}
	}
	
	// add style element with default style sheet rules for the line chart
	private void addDefaultInternalStylesheet(Document generatedDocument, GraphicAttributes attrs, Element parent, int sets) {
		StringBuffer styleRules = new StringBuffer(COMMON_STYLE);
		styleRules.append(XYCHART_STYLE);
				
		String [] palette = (String [])(attrs.getGraphicColorPalette()).get(attrs.getGraphicColorPaletteName());
		// palette has limited number of colours, so loop if more sets
		int paletteLength = palette.length;		
		for (int i = 0, j = 0; i < sets; i++) {
		  j = (i + 1) % paletteLength;
		  j = (j == 0) ? paletteLength - 1 : j-1;			
		  styleRules.append(" .dataset" + i + "{stroke-linecap:round;stroke-width:0.75pt; stroke:" + palette[j] + ";}"); 
		  if (!attrs.isLegendSuppressed()) {
			  styleRules.append(" .shapeline" + i + "{stroke-width:0.75pt;stroke:" + palette[j] + ";fill:" + palette[j] + ";}");
			  styleRules.append(" .shape" + i + "{ fill:" + palette[j] + ";}");
		  }
		}
		
		CDATASection newCDATASection = generatedDocument.createCDATASection(styleRules.toString());
		
		Element newElement = generatedDocument.createElement("style");
		newElement.setAttribute("id", "defaultStyleRules");
		newElement.setAttribute("type", "text/css");
		newElement.appendChild(newCDATASection);
		parent.appendChild(newElement);
	}	
	
	// add defs element with line chart definitions 
	protected void addDefinitions(Document generatedDocument, GraphicAttributes attrs, int numberOfDatasets) {		
		Element defsElement = super.addDefinitions(generatedDocument, attrs);
               
        // default internal style rules       
		addDefaultInternalStylesheet(generatedDocument, attrs, defsElement, numberOfDatasets);		
		
		if (!attrs.isLegendSuppressed()) {			
			// id: legendsquareshape
			Element gElement_legendsquare = generatedDocument.createElement("g");	
			defsElement.appendChild(gElement_legendsquare);
			gElement_legendsquare.setAttribute("id","legendsquareshape");
			
	    	Element legendsquareElement = generatedDocument.createElement("rect");
	    	gElement_legendsquare.appendChild(legendsquareElement);
			legendsquareElement.setAttribute("width", "8");
			legendsquareElement.setAttribute("height", "8");	
			legendsquareElement.setAttribute("transform", "translate(3,0)");
			gElement_legendsquare.appendChild(addLegendShapeLine(generatedDocument));
	
			// id: circleshape
			Element gElement_legendcircle = generatedDocument.createElement("g");
			defsElement.appendChild(gElement_legendcircle);
			gElement_legendcircle.setAttribute("id", "legendcircleshape");
			
			Element legendcircleElement = generatedDocument.createElement("circle");
			gElement_legendcircle.appendChild(legendcircleElement);
			legendcircleElement.setAttribute("r", "4");
			legendcircleElement.setAttribute("cx", "4");
			legendcircleElement.setAttribute("cy", "4");
			legendcircleElement.setAttribute("transform", "translate(3,0)");
			gElement_legendcircle.appendChild(addLegendShapeLine(generatedDocument));
			
	
			// id: diamondshape
			Element gElement_legenddiamond = generatedDocument.createElement("g");
			defsElement.appendChild(gElement_legenddiamond);
			gElement_legenddiamond.setAttribute("id", "legenddiamondshape");
			
			Element legenddiamondElement = generatedDocument.createElement("polygon");
			gElement_legenddiamond.appendChild(legenddiamondElement);
			legenddiamondElement.setAttribute("points", "0,4 4,0 8,4 4,8");
			legenddiamondElement.setAttribute("transform", "translate(3,0)");
			gElement_legenddiamond.appendChild(addLegendShapeLine(generatedDocument));
			
			
			// id: horizontalrectshape
			Element gElement_legendhorizontalrectshape = generatedDocument.createElement("g");
			defsElement.appendChild(gElement_legendhorizontalrectshape);
			gElement_legendhorizontalrectshape.setAttribute("id", "legendhorizontalrectshape");
			
			Element legendhorizontalrectElement = generatedDocument.createElement("rect");
			gElement_legendhorizontalrectshape.appendChild(legendhorizontalrectElement);
			legendhorizontalrectElement.setAttribute("width", "8");
			legendhorizontalrectElement.setAttribute("height", "4");
			legendhorizontalrectElement.setAttribute("transform", "translate(3,2)");
			gElement_legendhorizontalrectshape.appendChild(addLegendShapeLine(generatedDocument));
			
			// id: verticalrectshape
			Element gElement_legendverticalrectshape = generatedDocument.createElement("g");
			defsElement.appendChild(gElement_legendverticalrectshape);
			gElement_legendverticalrectshape.setAttribute("id", "legendverticalrectshape");
			
			Element legendverticalrectElement = generatedDocument.createElement("rect");
			gElement_legendverticalrectshape.appendChild(legendverticalrectElement);
			legendverticalrectElement.setAttribute("width", "4");
			legendverticalrectElement.setAttribute("height", "8");
			legendverticalrectElement.setAttribute("transform", "translate(5,0)");	
			gElement_legendverticalrectshape.appendChild(addLegendShapeLine(generatedDocument));
			
			
			// id: triangleshape 
			Element gElement_legendtriangleshape = generatedDocument.createElement("g");
			defsElement.appendChild(gElement_legendtriangleshape);
			gElement_legendtriangleshape.setAttribute("id", "legendtriangleshape");
			
			Element legendtriangleElement = generatedDocument.createElement("polygon");
			legendtriangleElement.setAttribute("points", "4 0, 0 8, 8 8");
			legendtriangleElement.setAttribute("transform", "translate(3,0)");
			gElement_legendtriangleshape.appendChild(legendtriangleElement);		
			gElement_legendtriangleshape.appendChild(addLegendShapeLine(generatedDocument));
			
		}			
			// id: squareshape
		Element gElement_square = generatedDocument.createElement("g");	
		defsElement.appendChild(gElement_square);
		gElement_square.setAttribute("id","squareshape");
			
	    Element squareElement = generatedDocument.createElement("rect");
	    gElement_square.appendChild(squareElement);
		squareElement.setAttribute("width", "5.6");
		squareElement.setAttribute("height", "5.6");
			
		// id: circleshape
		Element gElement_circle = generatedDocument.createElement("g");
		defsElement.appendChild(gElement_circle);
		gElement_circle.setAttribute("id", "circleshape");
			
		Element circleElement = generatedDocument.createElement("circle");
		gElement_circle.appendChild(circleElement);
		circleElement.setAttribute("r", "2.8");
		circleElement.setAttribute("cx", "2.8");
		circleElement.setAttribute("cy", "2.8");
		
		// id: holeshape
		Element gElement_hole = generatedDocument.createElement("g");
		defsElement.appendChild(gElement_hole);
		gElement_hole.setAttribute("id", "holeshape");
			
		Element holeElement = generatedDocument.createElement("circle");
		gElement_hole.appendChild(holeElement);
		holeElement.setAttribute("r", "2.8");
		holeElement.setAttribute("cx", "2.8");
		holeElement.setAttribute("cy", "2.8");
		holeElement.setAttribute("style", "fill:none;");
			
		// id: diamondshape
		Element gElement_diamond = generatedDocument.createElement("g");
		defsElement.appendChild(gElement_diamond);
		gElement_diamond.setAttribute("id", "diamondshape");
		
		Element diamondElement = generatedDocument.createElement("polygon");
		gElement_diamond.appendChild(diamondElement);
		diamondElement.setAttribute("points", "0,2.8 2.8,0 5.6,2.8 2.8,5.6");
			
		// id: horizontalrectshape
		Element gElement_horizontalrectshape = generatedDocument.createElement("g");
		defsElement.appendChild(gElement_horizontalrectshape);
		gElement_horizontalrectshape.setAttribute("id", "horizontalrectshape");
			
		Element horizontalrectElement = generatedDocument.createElement("rect");
		gElement_horizontalrectshape.appendChild(horizontalrectElement);
		horizontalrectElement.setAttribute("width", "5.6");
		horizontalrectElement.setAttribute("height", "2.8");
			
			
		// id: verticalrectshape
		Element gElement_verticalrectshape = generatedDocument.createElement("g");
		defsElement.appendChild(gElement_verticalrectshape);
		gElement_verticalrectshape.setAttribute("id", "verticalrectshape");
			
		Element verticalrectElement = generatedDocument.createElement("rect");
		verticalrectElement.setAttribute("width", "2.8");
		verticalrectElement.setAttribute("height", "5.6");		
		gElement_verticalrectshape.appendChild(verticalrectElement);
			
		// id: triangleshape 
		Element gElement_triangleshape = generatedDocument.createElement("g");
		defsElement.appendChild(gElement_triangleshape);
		gElement_triangleshape.setAttribute("id", "triangleshape");
			
		Element triangleElement = generatedDocument.createElement("polygon");
		triangleElement.setAttribute("points", "2.8 0, 0 5.6, 5.6 5.6");
		gElement_triangleshape.appendChild(triangleElement);		
				
	}	
	
	private Element addLegendShapeLine(Document generatedDocument) {
		Element lineElement = generatedDocument.createElement("polyline");
		lineElement.setAttribute("points", "0 4, 14 4");	
		return lineElement;
	}
	
	// Add a set of polyline elements for each dataset
	private void addLine(Document generatedDocument, Element parent, DataPoint[] dps, int linesetNumber, double[] xPositions, double[] yPositions, int xAxisLength, int yAxisLength, int gridXOffset, int gridYOffset) {

		final String[] shapeType = {"#circleshape", "#squareshape", "#horizontalrectshape", "#verticalrectshape", "#triangleshape", "#diamondshape"};
		final double[] shapeWidth = {5.6, 5.6, 5.6, 2.8, 5.6, 5.6};
		final double[] shapeHeight = {5.6, 5.6, 2.8, 5.6, 4.85, 5.6};
		final double BOX_HEIGHT = 18.0;

		int length = xPositions.length;
		double xPrevious = 0;
		double yPrevious = 0;
		double xNext;
		double yNext;
		
		// group for data colour changes and toggling exact values visibility
		Element gColourElement = generatedDocument.createElement("g");
		gColourElement.setAttribute("id", "dataColour" + linesetNumber);
		gColourElement.setAttribute("class", "dataset" + linesetNumber);
		gColourElement.setAttribute("visibility", "visible");
		gColourElement.setAttribute("onclick", "toggleVisibility('textbox" + linesetNumber + "');");
		
		parent.appendChild(gColourElement);
		
		// draw a polyline with data points
		int shapeNumber = linesetNumber % shapeType.length;
		for (int i = 0; i < length; i++) {
			xNext = xPositions[i];
			yNext = yPositions[i];
			
			if (i > 0) {
				Element newElement = generatedDocument.createElement("polyline");
				String points = xPrevious + " " + yPrevious + "," + xNext + " " + yNext;
				newElement.setAttribute("points", points);
				if (dps[i].getType() == DataPoint.HOLE || dps[i-1].getType() == DataPoint.HOLE)
				{
					newElement.setAttribute("style", "stroke-dasharray: 9, 5; stroke-width: 2;");
				}	
				newElement.setAttribute("transform", "translate(" + gridXOffset + "," + gridYOffset + ")");
				// add new polyline to group element	
				gColourElement.appendChild(newElement);
			}
			
			// Add shapes				
			double shapeX = xNext - shapeWidth[shapeNumber] / 2.0;
			if (shapeX <= 0) { // adjust shape x position if it crosses the left boundary
				shapeX = 0;	
			} else if (shapeX + shapeWidth[shapeNumber] > xAxisLength) { // adjust shape x position if it crosses the right boundary
				shapeX = xAxisLength - shapeWidth[shapeNumber];
			}
			
			double shapeY = yNext - shapeHeight[shapeNumber] / 2.0;
			if (shapeY + shapeHeight[shapeNumber] > yAxisLength) {  // adjust the shape y position if it crosses the x-axis
				shapeY = yAxisLength - shapeHeight[shapeNumber];	
			}
			
			Element shapeElement = generatedDocument.createElement("use");
			if (dps[i].getType() == DataPoint.HOLE)
			{
				shapeElement.setAttribute("xlink:href", "#holeshape");
			}else
			{
				shapeElement.setAttribute("xlink:href", shapeType[shapeNumber]);
			}			
			shapeElement.setAttribute("transform", "translate(" + (gridXOffset + shapeX) + "," + (gridYOffset + shapeY) + ")");
			//shapeElement.setAttribute("x", Double.toString(gridXOffset + shapeX));
			//shapeElement.setAttribute("y", Double.toString(gridYOffset + shapeY));
			shapeElement.setAttribute("visibility", "visible");
			gColourElement.appendChild(shapeElement);
						
			// add shapes on the data point
			xPrevious = xNext;
			yPrevious = yNext;	
		}		
	}	


	// add a textbox for each datapoint		
	protected void addTextBoxes(Document generatedDocument, GraphicAttributes attrs, Element parent, int id, 
	double[] xPositions, double[] yPositions, DataPoint[] dps,
	int xAxisLength, int yAxisLength) {

		int length = xPositions.length;
	
		for (int i = 0; i < length; i++) {
			String label = dps[i].getDisplayLabel(attrs);
			// determine text box position and dimensions	
			double textWidth = label.length();
			double boxWidth = 16 + (textWidth - 1) * 6.0;
			double boxHeight = 18;
			
			double xPos = xPositions[i];
			double yPos = yPositions[i];
			
			// Set the box x, y position 	
			double boxXPos = xPos;
			double boxYPos = yPos;
			
			// add the textbox					
			addEachTextBox(generatedDocument, attrs, parent, boxXPos, boxYPos, boxWidth, boxHeight, label, id, i);
		}	
	}		
	
	/**
	 * Initialize the line chart attributes
	 */
	protected GraphicAttributes createGraphicAttributes(GraphicDocumentProperties input) throws DataInputProcessingException {
		GraphicAttributes graphicAttrs = new GraphicAttributes();
			
		// set up the defaults
		graphicAttrs.setGraphicWidth(LINE_DEFAULT_WIDTH);
		graphicAttrs.setPreferencesPage(IUserPreferencesConstants.LINE_PREFS);
		graphicAttrs.setPreferencesPageWidth(IUserPreferencesConstants.PREFS_WIDTH);
		graphicAttrs.setPreferencesPageHeight(IUserPreferencesConstants.LINE_PREFS_HEIGHT);
		
		// get the input documents
		graphicAttrs.setConfigDocument(input.getConfigDocument());
		graphicAttrs.setDataDocument(input.getDataDocument());
		
		// (re)set any properties from the input documents		
		getConfigAttributes(graphicAttrs);
		
		// (re)set any properties from the input bean
		graphicAttrs.setAttributesFrom(input);		
				
		return graphicAttrs;
	}	

}

