/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.test.common.junit;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;

import junit.framework.AssertionFailedError;
import junit.framework.Test;
import junit.framework.TestCase;
import junit.framework.TestListener;
import junit.framework.TestResult;
import junit.framework.TestSuite;

import org.eclipse.hyades.test.common.event.ExecutionEvent;
import org.eclipse.hyades.test.common.event.InvocationEvent;
import org.eclipse.hyades.test.common.event.TypedEvent;
import org.eclipse.hyades.test.common.event.VerdictEvent;
import org.eclipse.hyades.test.common.util.BaseString;

/**
 * @author marcelop
 * @since 1.0.1
 */
public class HyadesTestRunner 
implements TestListener
{
	protected final static boolean SYSOUT = false;
	
	private Test rootTest;
	
	private Collection succeedTests;
	private Collection failureTests;
	private Collection errorTests;
	
	/**
	 * Constructor for HyadesTestRunner
	 */
	public HyadesTestRunner()
	{
		succeedTests = Collections.synchronizedCollection(new ArrayList());
		failureTests = Collections.synchronizedCollection(new ArrayList());
		errorTests = Collections.synchronizedCollection(new ArrayList());
	}
	
	/**
	 * Returns the root test, ie. the test that is passed to this
	 * runner.
	 * @return Test
	 */
	public Test getRoot()
	{
		return rootTest;
	}

	/**
	 * Sets the root test, ie. the test that is passed to this
	 * runner.
	 * @param root
	 */
	protected void setRoot(Test root)
	{
		rootTest = root;
	}
	
	/**
	 * Returns a collection with the hirarchy ids of the test cases that have 
	 * been succeed.
	 * @return Collection
	 */
	public Collection getSucceedTests()
	{
		return succeedTests;
	}

	/**
	 * Returns a collection with the hirarchy ids of the test cases that have 
	 * ended because of a failure.
	 * @return Collection
	 */
	public Collection getFailureTests()
	{
		return failureTests;
	}

	/**
	 * Returns a collection with the hirarchy ids of the test cases that have
	 * ended because of an error.
	 * @return Collection
	 */
	public Collection getErrorTests()
	{
		return errorTests;
	}

	/**
	 * Runs a suite extracted from a TestCase subclass.
	 * @param testClass
	 */
	public void run(Class testClass)
	{
		run(new HyadesTestSuite(testClass));
	}
	
	/**
	 * Runs a single test and collects its results. This method can be used to 
	 * start a test run from your program.
	 * @param suite
	 */
	public void run(Test test)
	{
		setRoot(test);

		TestResult result= createTestResult();
		result.addListener(this);

		runnerStarted();
		
		long startTime= getCurrentTime();
		try
		{
			test.run(result);
		}
		finally
		{
			long endTime= getCurrentTime();
			runnerExit((result == null || result.shouldStop()), (endTime-startTime));
		}		
	}

	/**
	 * Creates the TestResult to be used for the test run.
	 * @return TestResult
	 */
	protected TestResult createTestResult()
	{
		return new TestResult();
	}

	/**
	 * Returns the current timestamp.
	 * @return long
	 */
	protected long getCurrentTime()
	{
		return System.currentTimeMillis();
	}
	
	/**
	 * The runner has started. 
	 */
	protected void runnerStarted()
	{
		int count = HyadesTestUtil.countTests(new Test[]{getRoot()});
		if(SYSOUT)System.out.println("Run Started\n\tNumber of tests: " + count);
	}

	/**
	 * The runner is up to finish its activities. 
	 * @param runnerStopped
	 * @param the elapsed time
	 */
	protected void runnerExit(boolean runnerStopped, long elapsedTime)
	{
		if(SYSOUT)
		{
			if(runnerStopped)
				System.out.println("\nRun End");
			else
				System.out.println("\nRun Stop");
				
			System.out.println("\t" + HyadesTestUtil.getHierarchy(getRoot()) + "\n\t" + HyadesTestUtil.getHierarchyIds(getRoot()));
			System.out.println("\n\nResults:" 
				+ "\n\tsucceedCount: " + getSucceedTests().size()
				+ "\n\tfailureCount: " + getFailureTests().size()
				+ "\n\terrorCount: " + getErrorTests().size());
		}

		writeEvent(getDefaultVerdictEvent(getRoot()));

		TypedEvent typedEvent = new TypedEvent();
		typedEvent.setOwnerId(HyadesTestUtil.getHierarchyIds(getRoot()));
		typedEvent.setType(TypedEvent.STOP);
		typedEvent.setText(getLastEventText(runnerStopped, elapsedTime));
		writeEvent(typedEvent);
	}
	
	protected String getLastEventText(boolean runnerStopped, long elapsedTime)
	{
		return null;
	}
	
	/**
	 * @see junit.framework.TestListener#startTest(junit.framework.Test)
	 */
	public void startTest(Test test)
	{
		if(SYSOUT) System.out.println("\nStart\n\t" + HyadesTestUtil.getHierarchy(test) + "\n\t" + HyadesTestUtil.getHierarchyIds(test));
		
		if(test instanceof IHyadesTest)
		{
			IHyadesTest hyadesTest = (IHyadesTest)test;
			if(hyadesTest.getTestInvocationId() != null)
			{
				String hierarchyId = HyadesTestUtil.getHierarchyIds(hyadesTest.getParent());
				hierarchyId = HyadesTestUtil.appendToHierarchyId(hierarchyId, hyadesTest.getTestInvocationId(), hyadesTest.getIteration());
				
				InvocationEvent invocationEvent = new InvocationEvent();
				invocationEvent.setOwnerId(hierarchyId);
				invocationEvent.setStatus(InvocationEvent.STATUS_SUCCESSFUL);
				invocationEvent.setReason(InvocationEvent.REASON_NONE);
				invocationEvent.setInvokedId(HyadesTestUtil.getHierarchyIds(hyadesTest));
				writeEvent(invocationEvent);
			}
		}
		
		TypedEvent typedEvent = new TypedEvent();
		typedEvent.setOwnerId(HyadesTestUtil.getHierarchyIds(test));
		typedEvent.setType(TypedEvent.START);
		writeEvent(typedEvent);
	}

	/**
	 * @see junit.framework.TestListener#endTest(junit.framework.Test)
	 */
	public void endTest(Test test)
	{
		if(test == getRoot())
			return;

		if(SYSOUT)
		{
			System.out.println("\nEnd\n\t" + HyadesTestUtil.getHierarchy(test) + "\n\t" + HyadesTestUtil.getHierarchyIds(test));
			if((test instanceof HyadesTestSuite) && (((HyadesTestSuite)test).getArbiter() != null))
			{
				System.out.println("\tResults:" 
					+ "\n\t\tsucceedCount: " + filterTests(test, getSucceedTests()).size()
					+ "\n\t\tfailureCount: " + filterTests(test, getFailureTests()).size()
					+ "\n\t\terrorCount: " + filterTests(test, getErrorTests()).size());
			}
		}

		if((test instanceof TestCase) && (!getFailureTests().contains(test)) && (!getErrorTests().contains(test)))
			getSucceedTests().add(test);
			
		writeEvent(getDefaultVerdictEvent(test));

		TypedEvent typedEvent = new TypedEvent();
		typedEvent.setOwnerId(HyadesTestUtil.getHierarchyIds(test));
		typedEvent.setType(TypedEvent.STOP);
		writeEvent(typedEvent);
	}
	
	/**
	 * Returns the default verdict event, which is pass for test
	 * cases and the arbiter evaluation for test suites.
	 * @param test
	 * @return VerdictEvent
	 */
	protected VerdictEvent getDefaultVerdictEvent(Test test)
	{
		VerdictEvent verdictEvent = null;
		if(test instanceof TestSuite)
		{
			if(test instanceof HyadesTestSuite)
			{
				HyadesTestSuite hyadesTestSuite = (HyadesTestSuite)test;
				if(hyadesTestSuite.getArbiter() != null)
				{
					verdictEvent = hyadesTestSuite.getArbiter().analyse(hyadesTestSuite, filterTests(test, getSucceedTests()), filterTests(test, getErrorTests()), filterTests(test, getFailureTests()));
					if(verdictEvent != null)
					{
						if(verdictEvent.getOwnerId() == null)
							verdictEvent.setOwnerId(HyadesTestUtil.getHierarchyIds(test));
					}
				}
			}
		}
		else if(test instanceof TestCase)
		{
			if((!getFailureTests().contains(test)) && (!getErrorTests().contains(test)))
			{
				verdictEvent = new VerdictEvent();
				verdictEvent.setOwnerId(HyadesTestUtil.getHierarchyIds(test));
				verdictEvent.setVerdict(VerdictEvent.VERDICT_PASS);
			}
		}
		
		return verdictEvent;
	}

	/**
	 * @see junit.framework.TestListener#addError(junit.framework.Test, java.lang.Throwable)
	 */
	public void addError(Test test, Throwable throwable)
	{
		if(SYSOUT) System.out.println("\nError\n\t" + HyadesTestUtil.getHierarchy(test) + "\n\t" + HyadesTestUtil.getHierarchyIds(test));
		getErrorTests().add(test);

		VerdictEvent verdictEvent = new VerdictEvent();
		verdictEvent.setOwnerId(HyadesTestUtil.getHierarchyIds(test));
		verdictEvent.setVerdict(VerdictEvent.VERDICT_ERROR);
		verdictEvent.setReason(VerdictEvent.REASON_SEE_DESCRIPTION);
		verdictEvent.setText(BaseString.getStackTrace(throwable));
		writeEvent(verdictEvent);
	}

	/**
	 * @see junit.framework.TestListener#addFailure(junit.framework.Test, junit.framework.AssertionFailedError)
	 */
	public void addFailure(Test test, AssertionFailedError throwable)
	{
		if(SYSOUT) System.out.println("\nFailure\n\t" + HyadesTestUtil.getHierarchy(test) + "\n\t" + HyadesTestUtil.getHierarchyIds(test));
		getFailureTests().add(test);
		
		VerdictEvent verdictEvent = new VerdictEvent();
		verdictEvent.setOwnerId(HyadesTestUtil.getHierarchyIds(test));
		verdictEvent.setVerdict(VerdictEvent.VERDICT_FAIL);
		verdictEvent.setReason(VerdictEvent.REASON_SEE_DESCRIPTION);
		verdictEvent.setText(BaseString.getStackTrace(throwable));
		writeEvent(verdictEvent);
	}

	/**
	 * Writes an event.
	 * @param executionEvent
	 */	
	protected void writeEvent(ExecutionEvent executionEvent)
	{
		if(executionEvent != null)
			System.out.println(executionEvent);
	}
	
	protected Collection filterTests(Test parent, Collection tests)
	{
		Test[] testArray = (Test[])tests.toArray(new Test[tests.size()]);
		Collection filteredTests = new ArrayList(testArray.length);
		for (int i = 0, maxi = testArray.length; i < maxi; i++)
		{
			if((testArray[i] instanceof IHyadesTest) && HyadesTestUtil.isParentOf(parent, testArray[i]))
				filteredTests.add(testArray[i]);
		}
		
		return Collections.unmodifiableCollection(filteredTests);
	}
}
