/*******************************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui;

import org.eclipse.ui.ISharedImages;
import org.eclipse.ui.PlatformUI;

import org.eclipse.hyades.models.common.util.ICommonConstants;
import org.eclipse.hyades.ui.HyadesUI;
import org.eclipse.hyades.ui.HyadesUIPlugin;
import org.eclipse.hyades.ui.extension.IAssociationDescriptor;
import org.eclipse.hyades.ui.extension.IAssociationDescriptorFilter;
import org.eclipse.hyades.ui.extension.IAssociationMapping;
import org.eclipse.hyades.ui.extension.IAssociationMappingRegistry;

/**
 * Contains utility methods to work with the Test UI extensions.
 * 
 * @author marcelop
 * @since 0.0.1
 */
public class TestUIExtension
{
	private static class TestSuiteDescriptorFilter
	implements IAssociationDescriptorFilter
	{
		/**
		 * @see org.eclipse.hyades.ui.extension.IAssociationDescriptorFilter#isValid(org.eclipse.hyades.ui.extension.IAssociationMapping, org.eclipse.hyades.ui.extension.IAssociationDescriptor)
		 */
		public boolean isValid(IAssociationMapping associationMapping, IAssociationDescriptor associationDescriptor)
		{
			if(!ICommonConstants.TEST_SUITE_FILE_EXTENSION.equals(associationDescriptor.getExtension()))
				return false;

			String extensionPoint = associationMapping.getExtensionPoint();
			if(HyadesUIPlugin.EP_TYPE_DESCRIPTIONS.equals(extensionPoint))
			{
				if(associationDescriptor.applyToAllTypes())
					return false;
						
				if(associationDescriptor.types().length != 1)
					return false;
			}
			else if(HyadesUIPlugin.EP_EDITOR_EXTENSIONS.equals(extensionPoint))
			{
				if(associationDescriptor.getImageDescriptor() == null)
					associationDescriptor.setImageDescriptor(PlatformUI.getWorkbench().getSharedImages().getImageDescriptor(ISharedImages.IMG_OBJ_FILE));
			}
			else if(HyadesUIPlugin.EP_TYPE_VALIDATORS.equals(extensionPoint))
			{
				if(associationDescriptor.applyToAllTypes())
					return false;
			}
			else if(TestUIPlugin.EP_GENERATE_WIZARDS.equals(extensionPoint))
			{
				if(associationDescriptor.applyToAllTypes())
					return false;
			}
			else if(TestUIPlugin.EP_RUN_WIZARDS.equals(extensionPoint))
			{
				if(associationDescriptor.applyToAllTypes())
					return false;
			}
			
			return true;
		}
	}
	
	private static class TestCaseDescriptorFilter
	implements IAssociationDescriptorFilter
	{
		/**
		 * @see org.eclipse.hyades.ui.extension.IAssociationDescriptorFilter#isValid(org.eclipse.hyades.ui.extension.IAssociationMapping, org.eclipse.hyades.ui.extension.IAssociationDescriptor)
		 */
		public boolean isValid(IAssociationMapping associationMapping, IAssociationDescriptor associationDescriptor)
		{
			boolean isTestCaseWizard = TestUIPlugin.EP_TEST_CASE_WIZARD.equals(associationMapping.getExtensionPoint()); 
			boolean isTestCaseExtension = ICommonConstants.TEST_CASE_EXTENSION.equals(associationDescriptor.getExtension());
			
			if((!isTestCaseExtension) && (!isTestCaseWizard))
				return false;

			return true;
		}
	}	

	private static class DeploymentDescriptorFilter
	implements IAssociationDescriptorFilter
	{
		/**
		 * @see org.eclipse.hyades.ui.extension.IAssociationDescriptorFilter#isValid(org.eclipse.hyades.ui.extension.IAssociationMapping, org.eclipse.hyades.ui.extension.IAssociationDescriptor)
		 */
		public boolean isValid(IAssociationMapping associationMapping, IAssociationDescriptor associationDescriptor)
		{
			if(!ICommonConstants.DEPLOYMENT_FILE_EXTENSION.equals(associationDescriptor.getExtension()))
				return false;

			String extensionPoint = associationMapping.getExtensionPoint();
			if(HyadesUIPlugin.EP_EDITOR_EXTENSIONS.equals(extensionPoint))
			{
				if(associationDescriptor.getImageDescriptor() == null)
					associationDescriptor.setImageDescriptor(PlatformUI.getWorkbench().getSharedImages().getImageDescriptor(ISharedImages.IMG_OBJ_FILE));
			}
			
			return true;
		}
	}	

	private static class ExecutionDescriptorFilter
	implements IAssociationDescriptorFilter
	{
		/**
		 * @see org.eclipse.hyades.ui.extension.IAssociationDescriptorFilter#isValid(org.eclipse.hyades.ui.extension.IAssociationMapping, org.eclipse.hyades.ui.extension.IAssociationDescriptor)
		 */
		public boolean isValid(IAssociationMapping associationMapping, IAssociationDescriptor associationDescriptor)
		{
			if(!ICommonConstants.EXECUTION_FILE_EXTENSION.equals(associationDescriptor.getExtension()))
				return false;

			String extensionPoint = associationMapping.getExtensionPoint();
			if(HyadesUIPlugin.EP_EDITOR_EXTENSIONS.equals(extensionPoint))
			{
				if(associationDescriptor.getImageDescriptor() == null)
					associationDescriptor.setImageDescriptor(PlatformUI.getWorkbench().getSharedImages().getImageDescriptor(ISharedImages.IMG_OBJ_FILE));
			}
			
			return true;
		}
	}	
		
	/**
	 * The Hyades generic test suite type
	 */
	public final static String GENERIC_TEST_SUITE_TYPE = "org.eclipse.hyades.testSuite.generic";

	/**
	 * The Hyades generic test suite editor that is registered to the Eclipse
	 * framework. 
	 */
	public final static String GENERIC_TEST_SUITE_EDITOR_PART_ID = "org.eclipse.hyades.test.ui.editor.TestSuiteEditorPart";

	/**
	 * The Hyades generic test case type
	 */
	public final static String GENERIC_TEST_CASE_TYPE = "org.eclipse.hyades.testCase.generic";

	/**
	 * The Hyades deployment editor that is registered to the Eclipse
	 * framework. 
	 */
	public final static String DEPLOYMENT_EDITOR_PART_ID = "org.eclipse.hyades.test.ui.editor.DeploymentEditorPart";

	/**
	 * The Hyades execution result editor that is registered to the Eclipse
	 * framework. 
	 */
	public final static String EXECUTION_EDITOR_PART_ID = "org.eclipse.hyades.test.ui.editor.ExecutionEditorPart";


	/*
	 * Association mapping registries.
	 */
	private static IAssociationMappingRegistry testSuiteMappingRegistry;
	private static IAssociationMappingRegistry testCaseMappingRegistry;
	private static IAssociationMappingRegistry deploymentMappingRegistry;
	private static IAssociationMappingRegistry executionMappingRegistry;
	
	/**
	 * Disposes the resources used by this class.
	 */
	static void dispose()
	{
		if(testSuiteMappingRegistry != null)
			testSuiteMappingRegistry.dispose();

		if(testCaseMappingRegistry != null)
			testCaseMappingRegistry.dispose();
			
		if(deploymentMappingRegistry != null)
			deploymentMappingRegistry.dispose();	
	}
	
	/**
	 * Returns the Test Suite's Association mapping registry.
	 * 
	 * <p>When this method is called for the first time it initializes
	 * the registry.
	 * 
	 * @return IAssociationMappingRegistry
	 */
	public static IAssociationMappingRegistry getTestSuiteMappingRegistry()
	{
		if(testSuiteMappingRegistry == null)
		{
			testSuiteMappingRegistry = HyadesUI.createAssociationMappingRegistry();
			testSuiteMappingRegistry.initialize(ICommonConstants.TEST_SUITE_FILE_EXTENSION, new TestSuiteDescriptorFilter(), TestUIPlugin.getInstance().getPreferenceStore());
			
			testSuiteMappingRegistry.retrieveAssociationMapping(HyadesUIPlugin.EP_TYPE_DESCRIPTIONS, HyadesUIPlugin.getID());
			testSuiteMappingRegistry.retrieveAssociationMapping(HyadesUIPlugin.EP_EDITOR_EXTENSIONS, HyadesUIPlugin.getID());
			testSuiteMappingRegistry.retrieveAssociationMapping(HyadesUIPlugin.EP_TYPE_VALIDATORS, HyadesUIPlugin.getID());
			testSuiteMappingRegistry.retrieveAssociationMapping(TestUIPlugin.EP_GENERATE_WIZARDS, TestUIPlugin.getID());
			testSuiteMappingRegistry.retrieveAssociationMapping(TestUIPlugin.EP_RUN_WIZARDS, TestUIPlugin.getID());			
		}
		
		return testSuiteMappingRegistry;
	}
	
	/**
	 * Returns the Test Case's Association mapping registry.
	 * 
	 * <p>When this method is called for the first time it initializes
	 * the registry.
	 * 
	 * @return IAssociationMappingRegistry
	 */
	public static IAssociationMappingRegistry getTestCaseMappingRegistry()
	{
		if(testCaseMappingRegistry == null)
		{
			testCaseMappingRegistry = HyadesUI.createAssociationMappingRegistry();
			testCaseMappingRegistry.initialize(ICommonConstants.TEST_CASE_EXTENSION, new TestCaseDescriptorFilter(), TestUIPlugin.getInstance().getPreferenceStore());
			
			testCaseMappingRegistry.retrieveAssociationMapping(TestUIPlugin.EP_TEST_CASE_WIZARD, TestUIPlugin.getID());
			testCaseMappingRegistry.retrieveAssociationMapping(HyadesUIPlugin.EP_TYPE_DESCRIPTIONS, HyadesUIPlugin.getID());
			testCaseMappingRegistry.retrieveAssociationMapping(TestUIPlugin.EP_GENERATE_WIZARDS, TestUIPlugin.getID());
			testCaseMappingRegistry.retrieveAssociationMapping(TestUIPlugin.EP_RUN_WIZARDS, TestUIPlugin.getID());			
		}
		
		return testCaseMappingRegistry;
	}

	/**
	 * Returns the Deployment's Association mapping registry.
	 * 
	 * <p>When this method is called for the first time it initializes
	 * the registry.
	 * 
	 * @return IAssociationMappingRegistry
	 */
	public static IAssociationMappingRegistry getDeploymentMappingRegistry()
	{
		if(deploymentMappingRegistry == null)
		{
			deploymentMappingRegistry = HyadesUI.createAssociationMappingRegistry();
			deploymentMappingRegistry.initialize(ICommonConstants.DEPLOYMENT_FILE_EXTENSION, new DeploymentDescriptorFilter(), TestUIPlugin.getInstance().getPreferenceStore());
			
			deploymentMappingRegistry.retrieveAssociationMapping(HyadesUIPlugin.EP_EDITOR_EXTENSIONS, HyadesUIPlugin.getID());
		}
		
		return deploymentMappingRegistry;
	}

	/**
	 * Returns the Execution Result's Association mapping registry.
	 * 
	 * <p>When this method is called for the first time it initializes
	 * the registry.
	 * 
	 * @return IAssociationMappingRegistry
	 */
	public static IAssociationMappingRegistry getExecutionMappingRegistry()
	{
		if(executionMappingRegistry == null)
		{
			executionMappingRegistry = HyadesUI.createAssociationMappingRegistry();
			executionMappingRegistry.initialize(ICommonConstants.EXECUTION_FILE_EXTENSION, new ExecutionDescriptorFilter(), TestUIPlugin.getInstance().getPreferenceStore());
			
			executionMappingRegistry.retrieveAssociationMapping(HyadesUIPlugin.EP_EDITOR_EXTENSIONS, HyadesUIPlugin.getID());
		}
		
		return executionMappingRegistry;
	}

	/**
	 * Registers a test suite type to this Eclipse session cache.
	 * 
	 * <p>This is the mechanism available for a Hyades component - such as an
	 * editor - to register the type of test suite it is manipulating.  This is 
	 * necessary since there is no global registry of all the test suite types available.
	 * 
	 * @param type
	 */
	public synchronized static void registerTestSuiteType(String type)
	{
		if(type == null)
			return;
			
		IAssociationMapping mapping = getTestSuiteMappingRegistry().getAssociationMapping(HyadesUIPlugin.EP_TYPE_DESCRIPTIONS); 
		if(mapping.getAssociationDescriptors(type).length == 0)
		{
			IAssociationDescriptor descriptor = mapping.createAssociationDescriptor();
			descriptor.setExtension(ICommonConstants.TEST_SUITE_FILE_EXTENSION);
			descriptor.addType(type);
		}
	}
	
	/**
	 * Registers a test type to this Eclipse session cache.
	 * 
	 * <p>This is the mechanism available for a Hyades component - such as an
	 * editor - to register the type of test case it is manipulating.  This is 
	 * necessary since there is no global registry of all the test case types available.
	 * 
	 * @param type
	 */
	public synchronized static void registerTestCaseType(String type)
	{
		if(type == null)
			return;
			
		IAssociationMapping mapping = getTestCaseMappingRegistry().getAssociationMapping(HyadesUIPlugin.EP_TYPE_DESCRIPTIONS); 
		if(mapping.getAssociationDescriptors(type).length == 0)
		{
			IAssociationDescriptor descriptor = mapping.createAssociationDescriptor();
			descriptor.setExtension(ICommonConstants.TEST_CASE_EXTENSION);
			descriptor.addType(type);
		}
	}
}
