/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.execution.trace;

import java.io.BufferedWriter;
import java.io.FileWriter;
import java.io.IOException;
import java.io.PrintWriter;
import java.net.InetAddress;
import java.util.HashMap;

import org.eclipse.hyades.execution.trace.util.RecordAgentCreate;
import org.eclipse.hyades.execution.trace.util.RecordClassDef;
import org.eclipse.hyades.execution.trace.util.RecordMethodAccess;
import org.eclipse.hyades.execution.trace.util.RecordMethodDef;
import org.eclipse.hyades.execution.trace.util.RecordObjAlloc;
import org.eclipse.hyades.execution.trace.util.RecordParameterDef;
import org.eclipse.hyades.execution.trace.util.RecordThreadStart;
import org.eclipse.hyades.execution.trace.util.RecordTraceStart;
import org.eclipse.hyades.execution.trace.util.RecordVariable;
import org.eclipse.hyades.execution.trace.util.Utilities;
import org.eclipse.hyades.internal.execution.remote.AgentControllerListener;
import org.eclipse.hyades.internal.execution.remote.AgentNotRegisteredException;
import org.eclipse.hyades.internal.execution.remote.MonitorListener;
import org.eclipse.hyades.internal.execution.remote.RemoteClientListener;
import org.eclipse.hyades.internal.execution.remote.RemoteComponentSkeleton;


/**
 * This is the logger for distributed tracing.
 * 
 * @author Richard Duggan, Qiyan Li
 */
public class TraceLogger implements MonitorListener, AgentControllerListener, RemoteClientListener {

    /**
     * the singleton instance of the trace logger
     */
    private static TraceLogger _logger = null;

    /**
     * the peer timeout value
     */
    private static final long REQUEST_TIMEOUT = 10000;

    /**
     * the logging agent that will perform the actual logging
     */
    private RemoteComponentSkeleton _agent = null;

    /**
     * the name of the distributed trace agent
     */
    private String _name = "Distributed Trace Profiler";

    /**
     * the type of the distributed trace agent
     */
    private String _type = "Profiler";

    /**
     * the flag indicating whether the Agent Controller is active or not
     */
    private boolean _controllerAvailable = false;

    /**
     * the flag indicating whether the agent is currently being monitored
     */
    private boolean _isMonitored = false;

    /**
     * the stream to which the data goes
     */
    private PrintWriter _outWriter = null;

    /**
     * the filename correpsonding to the stream to which the data goes
     */
    private String _outputFileName = null;

    /**
     * the flag indicating whether the logger actively logging or idle
     */
    private boolean _loggingActive = false;

    /**
     * the record of the agent creation
     */
    private RecordAgentCreate _agentRecord = null;

    /**
     * the flag indiating whether this is a new trace
     */
    private boolean _isNewTrace = true;

    /**
     * the data structure holding the collection of method defitions
     */
    private HashMap methodCollection;

    /**
     * the data structure holding the collection of class defitions
     */
    private HashMap classCollection;

    /**
     * the data structure holding the collection of object definitions
     */
    private HashMap objectCollection;

    /**
     * the data structure holding the collection of parameter definitions
     */
    private HashMap parameterCollection;

    /**
     * the data structure holding the collection of thread definitions
     */
    private HashMap threadCollection;


    /**
     * Creates an instance of the logger.
     */
    private TraceLogger() {

        methodCollection = new HashMap();
        classCollection = new HashMap();
        objectCollection = new HashMap();
        parameterCollection = new HashMap();
        threadCollection = new HashMap();
        initialize();
    }

    /**
     * Get the singleton instance of this logger.
     */
    public static TraceLogger getInstance() {

        synchronized (TraceLogger.class) {
            if (_logger == null) {
                _logger = new TraceLogger();
            }
        }
        return _logger;
    }


    /**
     * Brings up the agent mechanism and sets any preferences that are stored in environment 
     * variable(s) (lowest priority) and/or the plugin's plugin.xml file that this logger 
     * belongs, if running within a plugin environment.
     */
    protected final void initialize() {

        /* Due to a race condition inside the RAC, this must be synchronized to give the initializer
           enough time to get initialize all fields properly so that agentControllerActive() does not
           get called too early. */
        synchronized (this) {
            if (_agent == null) {
                try {
                    _agent = new RemoteComponentSkeleton(_name, _type);
                    _agent.addAgentControllerListener(this);
                    _agent.addMonitorListener(this);
                    _agent.addRemoteClientListener(this);
                    _agent.initializeFast();
                    _controllerAvailable = true;
                } catch (Throwable e) {
                    _controllerAvailable = false;
                }
            }
        }
    }


    /**
     * This is called by the Logging Agent mechanism to indicate that the 
     * agent controller is present and running on the current node.
     */
    public final void agentControllerActive() {

        try {

            /* Due to a race condition inside the RAC, this synchronization will delay the creation
               of the record to ensure all fields are initialized. */
            synchronized (this) {
                _agentRecord = new RecordAgentCreate(_agent.getAgentUUID(), _agent.getJVMUUID(),
                    _agent.getNodeUUID(), _agent.getName(), Utilities.getCurrentTimeStamp());
            }
        } catch (Throwable e) {
            /* Exception gets thrown only if not registered.  It will be registered at this point,
               so we are OK. */
        }
        _controllerAvailable = true;
    }


    /**
     * This is called by the Logging Agent mechanism to indicate that the 
     * agent controller is not present and not running on the current node.
     */
    public final void agentControllerInactive() {

        _controllerAvailable = false;
        _isMonitored = false;
    }


    /**
     * Deregister this logging agent instance from the agent controller.
     */
    public final void finalize() {

        _agent.deregister();
    }


    /**
     * Invoked by the agent mechainism to indicate a remote client is collecting the data.
     * This method will flush any data held in the queue.
     */
    public final void monitorActive() {

        _isMonitored = true;
    }


    /**
     * Invoked by the logging agent mechanism to indicate a remote client is discontinuing
     * data collection.
     */
    public final void monitorInactive() {

        _isMonitored = false;

        /* Clear up the state from the previous trace. */
        classCollection.clear();
        methodCollection.clear();
        objectCollection.clear();
        parameterCollection.clear();
        threadCollection.clear();
        _isNewTrace = true;
    }


    /**
     * Invoked by the logging agent mechanism to indicate a remote client that was attached to
     * us has crashed.
     */
    public void clientInactive() {

        monitorInactive();
    }


    /**
     * Checks if the current logging level is set to log information.
     *
     * @return true if logging is turned on (i.e., Level.CONFIG - LEVEL.FINEST),
     *         otherwise false (i.e., Level.NONE).
     */
    public final boolean isLogging() {

        return _loggingActive;
    }


    /**
     * Checks if the current agent is being monitored.
     *
     * @return true if it is being monitored, otherwise false.
     */
    public final boolean isMonitored() {

        return _isMonitored;
    }

    /**
     * Writes a message to the file writer.  If I/O fails for some reason, the message will be
     * redirected to standard output.
     * 
     * @param message   the messge to be logged
     */
    private void outputFileWrite(String message) {

        try {
            _outWriter.println(message);
            _outWriter.flush();
        } catch (Exception e) {
            System.out.println("Error writing to file " + _outputFileName);
            System.out.println("REASON: " + e);
            System.out.println("NOTE: Logging output now set to standard out.");
            setFileName(null);
            System.out.println(message);
        }
    }


    /**
     * Logs a message to the logging agent, if available, and a local file, if specified.  Otherwise,
     * log the message to standard output.
     * 
     * @param message   the messge to be logged
     */
    public final boolean write(String message) {

        boolean result = false;
        try {
            if (_controllerAvailable && _isMonitored) {

                /* If this is a new trace, output agentCreate and the traceStart tags. */
                if (_isNewTrace) {
                    _agent.logMessageUTF8(_agentRecord.toString());
                    RecordTraceStart trcTag = new RecordTraceStart(_agent.getAgentUUID());
                    _agent.logMessageUTF8(trcTag.toString());

                    if (_outputFileName != null) {
                        outputFileWrite(_agentRecord.toString());
                        outputFileWrite(trcTag.toString());
                    }
                    _isNewTrace = false;
                }

                /* Output the message itself. */
                _agent.logMessageUTF8(message);
                result = true;
                if (_outputFileName != null) {
                    outputFileWrite(message);
                }
            } else {
                System.out.println(message);
            }

        } catch (Exception e) {
            System.out.println("Exception caught when logging messages to Trace Agent");
            e.printStackTrace();
        }
        return result;
    }


    /**
     * Retrieves the name of this logger.
     * 
     * @return a reference to the agent name
     */
    public final String getName() {
        return _name;
    }


    /**
     * Retrieves the agent UUID of this logger.
     * 
     * @return a reference to the agent UUID
     */
    public final String getAgentUUID() {
        return _agent.getAgentUUID();
    }


    /**
     * Retrieves the UUID of the agent JVM.
     * 
     * @return a reference to the UUID of the agent JVM
     */
    public final String getAgentJVMUUID() {

        String uuid = null;
        try {
            uuid = _agent.getJVMUUID();
        } catch (AgentNotRegisteredException e) {
        }
        return uuid;
    }


    /**
     * Retrieves the UUID of the agent node UUID.
     * 
     * @return a reference to the UUID of the agent node UUID
     */
    public final String getAgentNodeUUID() {

        String uuid = null;
        try {
            uuid = _agent.getNodeUUID();
        } catch (AgentNotRegisteredException e) {
        }
        return uuid;
    }


    /**
     * Returns the logging file name.
     *
     * @return a reference to the logging file name
     */
    public String getFileName() {

        return _outputFileName;
    }


    /**
     * This method sets the file name where messages are to be logged.  If stand-alone mode
     * is active, this overrides logging to standard out, by pumping all logs to the 
     * specified file. If standalone behavior is not active, then the file name is simply set 
     * for later use when standalone mode is enabled.  If <code>null</code> is passed as a parameter,
     * this stops logging to a file (and closes it), and 
     * redirects logging to Standard Output.
     * 
     * @param logFile   the new file name
     */
    public synchronized void setFileName(String logFile) {

        if ((logFile != null) && (logFile.length() != 0)) {
            try {
                _outWriter = new PrintWriter(new BufferedWriter(new FileWriter(logFile)), true);
                _outputFileName = logFile;
            } catch (IOException e) {
            }
        } else {
            closeFile();
            _outputFileName = null;
        }
    }


    /**
     * If the current logger has an associated file handle to an output file for logging, 
     * the existing file handle is closed.
     */
    private synchronized void closeFile() {

        if (_outputFileName != null) {
            _outWriter.close();
        }
        _outWriter = null;
        _outputFileName = null;
    }


    /**
     * Checks if the logger is actively logging or turned 'on'.
     *
     * @return true if the logger is actively logging or turned 'on'
     */
    public final boolean isActive() {
        return _loggingActive;
    }


    /**
     * This method sets the flag that turns the logger 'on' for active logging 
     * or 'off' for no logging.  If the logger has an associated file name as 
     * the output file for logging, a file handle is created when the logger is 
     * turned 'on' and the existing file handle is closed when the logger is 
     * turned 'off'.  All other logger preferences are maintained while the logger 
     * is turned 'off'.
     * 
     * @param active boolean flag that turns the logger 'on' or 'off'
     */
    public void setActive(boolean active) {
        _loggingActive = active;

        if (!_loggingActive)
            closeFile();
    }


    /** 
     * Call requestMonitorThroughPeer() to ask the RAC start monitor this agent.
     * 
     * @param addr      the IP address of the peer
     * @param agentUUID the agent UUID
     */
    public void requestMonitorThroughPeer(InetAddress addr, String agentUUID) {
        _agent.requestMonitorThroughPeer(addr, agentUUID, REQUEST_TIMEOUT);
    }


    /**
     * Returns whether the agent controller is available.
     */
    public boolean isRegisteredWithAgentController() {
        return _controllerAvailable;
    }


    /**
     * @see org.eclipse.hyades.internal.execution.remote.RemoteClientListener#clientActive()
     */
    public void clientActive() {
    }


    /**
     * Logs a particular event using the information provided in <code>self</code>
     * and <code>partner</code>, which may or may not be running on a remote entity.
     * 
     * @param eventType the type of event
     * @param self      the event to be logged
     * @param partner   the parent of the self event
     */
    public synchronized void logCorrelator(String eventType, TraceCorrelator self,
        TraceCorrelator partner) {

        /* Saves the thread definition if necessary. */
        Thread threadKey = Thread.currentThread();
        RecordThreadStart threadStart = (RecordThreadStart) threadCollection.get(threadKey);
        if (threadStart == null) {
            threadStart = new RecordThreadStart();
            threadStart.setTime(Utilities.getCurrentTimeStamp());
            threadCollection.put(threadKey, threadStart);
            write(threadStart.toString());
        }
        int threadIdRef = threadStart.getThreadId();

        /* Saves the class definition if necessary. */
        String className = self.getClazz().getName();
        RecordClassDef classDef = (RecordClassDef) classCollection.get(className);
        if (classDef == null) {
            classDef = new RecordClassDef(className);
            classDef.setTime(Utilities.getCurrentTimeStamp());
            classDef.setThreadIdRef(threadIdRef);
            classCollection.put(className, classDef);
            write(classDef.toString());
        }

        /* Saves the object definition if necessary. */
        int objRef = self.getObject();
        RecordObjAlloc objAlloc = (RecordObjAlloc) objectCollection.get(new Integer(objRef));
        if (objAlloc == null) {
            objAlloc = new RecordObjAlloc();
            objAlloc.setTime(Utilities.getCurrentTimeStamp());
            objAlloc.setClassIdRef(classDef.getClassId());
            objAlloc.setThreadIdRef(threadIdRef);
            objectCollection.put(new Integer(objRef), objAlloc);
            write(objAlloc.toString());
        }

        /* Saves the method definition if necessary. */
        String methodName = self.getInvokedMethod().toString();
        String methodSignature = Utilities.getInvokedMethodSignature(self.getParmClasses(),
            self.getReturnClass() == null ? null : Utilities.getJniNotation(self.getReturnClass()));
        String methodKey = className + "#" + methodName + methodSignature;
        RecordMethodDef methodDef = (RecordMethodDef) methodCollection.get(methodKey);
        if (methodDef == null) {
            methodDef = new RecordMethodDef(methodName, methodSignature, classDef.getClassId());
            methodCollection.put(methodKey, methodDef);
            write(methodDef.toString());
        }

        /* If any of the three input parameters is null, or if the parameter lists are of different lengths,
           the parameter list is assumed to be empty. */
        String[] parameterNames = self.getParmNames();
        Class[] parameterClasses = self.getParmClasses();
        Object[] parameterValues = self.getParmValues();
        int listLength = (parameterNames == null || parameterClasses == null || parameterValues == null ||
            parameterNames.length != parameterClasses.length ||
            parameterClasses.length != parameterValues.length ? 0 : parameterNames.length);

        /* Saves the parameter definitions if necessary. */
        int[] parameterIds = new int[listLength];
        for (int i = 0; i < listLength; i++) {
            String parameterKey = Integer.toString(methodDef.getMethodId()) + "@" + Integer.toString(i);
            RecordParameterDef parameterDef = (RecordParameterDef) parameterCollection.get(parameterKey);
            if (parameterDef == null) {
                parameterDef = new RecordParameterDef(parameterClasses[i], methodDef.getMethodId(), i);
                parameterDef.setName(parameterNames[i]);
                parameterCollection.put(parameterKey, parameterDef);
                write(parameterDef.toString());
            }
            parameterIds[i] = parameterDef.getParameterId();
        }

        /* Saves the event itself. */
        RecordMethodAccess methodAccess =
            new RecordMethodAccess(eventType, methodDef.getMethodId(), objAlloc.getObjId());
        methodAccess.setTime(Utilities.getCurrentTimeStamp());
        methodAccess.setSequenceCounter(self.getOperationCounter());
        methodAccess.setTicket(Long.toString(self.getApplicationCounter()));
        methodAccess.setParameterList(parameterIds, parameterClasses, parameterValues);
        methodAccess.setThreadIdRef(threadIdRef);

        /* Set the location information for the local correlator. */
        self.setAgentIdRef(_logger.getAgentUUID());
        self.setProcessIdRef(_logger.getAgentJVMUUID());
        self.setNodeIdRef(_logger.getAgentNodeUUID());
        self.setThreadIdRef(threadIdRef);

        /* Saves the partner information if the invocation is remote. */
        if (partner != null && partner.getAgentIdRef() != null
            && !self.getAgentIdRef().equals(partner.getAgentIdRef())
            && (eventType == RecordMethodAccess.METHOD_RECEIVE ||
            eventType == RecordMethodAccess.METHOD_RETURN)) {
            methodAccess.setRemoteContext(partner.getNodeIdRef(), partner.getProcessIdRef(),
                partner.getAgentIdRef(), partner.getThreadIdRef(), partner.getApplicationCounter(),
                partner.getOperationCounter());
        }


        /* Saves the return value if the event is an exit event, and the return type is not null. */
        if (eventType == RecordMethodAccess.METHOD_EXIT && self.getReturnClass() != null
            && self.getReturnClass() != void.class) {
            methodAccess.setReturnValue(
                new RecordVariable(RecordVariable.RETURN_VALUE, 0, self.getReturnClass(), self.getReturnValue()));
        }
        write(methodAccess.toString());
    }
}
