/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v0.5
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v05.html
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.models.hierarchy.util;

import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.emf.ecore.xmi.XMIResource;
import org.eclipse.emf.ecore.xmi.XMLLoad;
import org.eclipse.emf.ecore.xmi.XMLSave;
import org.eclipse.emf.ecore.xmi.impl.XMIResourceImpl;
import org.eclipse.hyades.loaders.util.AgentsContext;
import org.eclipse.hyades.loaders.util.HierarchyContext;
import org.eclipse.hyades.loaders.util.LoadersUtils;
import org.eclipse.hyades.loaders.util.LookupServiceExtensions;
import org.eclipse.hyades.loaders.util.ResourceExtensions;
import org.eclipse.hyades.loaders.util.XMLResourceLoader;
import org.eclipse.hyades.models.hierarchy.TRCAgent;
import org.eclipse.hyades.models.hierarchy.UnresolvedCorrelation;
import org.eclipse.hyades.models.hierarchy.plugin.HierarchyPlugin;


public class HierarchyXMIResourceImpl extends XMIResourceImpl implements XMIResource {
    //~ Instance fields ----------------------------------------------------------------------------

    protected HierarchyContext context = null;
    protected boolean register = true;
    private ArrayList intAndStrList = new ArrayList(1000);

    //~ Constructors -------------------------------------------------------------------------------

    public HierarchyXMIResourceImpl() {
        super();

        initHierarchyResource();
    }

    public HierarchyXMIResourceImpl(URI uri) {
        super(uri);

        initHierarchyResource();
    }

    //~ Methods ------------------------------------------------------------------------------------

    public EObject getEObject(String uriFragment) {
        if (!isPath(uriFragment)) {
            if (idToEObjectMap != null) {
                EObject eObject = (EObject) idToEObjectMap.get(uriFragment);

                return eObject;
            } else {
                return null;
            }
        }

        return super.getEObject(uriFragment);
    }

    public String getIndexStr(int index) {
        // index String slot size
        //   0     "0"    0    1
        //   1     "1"    1    2
        //   2     "2"    2    3 	
        int size = intAndStrList.size();

        if (index >= size) {
            intAndStrList.ensureCapacity(index + 1);

            for (int i = size; i < index; i++) {
                intAndStrList.add(i, null);
            }

            String indexStr = Integer.toString(index);

            intAndStrList.add(index, indexStr);

            return indexStr;
        }

        if (intAndStrList.get(index) == null) {
            String indexStr = Integer.toString(index);

            intAndStrList.set(index, indexStr);

            return indexStr;
        }

        return (String) intAndStrList.get(index);
    }

    public String getURIFragment(EObject eObject) {
        String id = EcoreUtil.getID(eObject);

        if (id != null) {
            return id;
        } else {
            StringBuffer result = new StringBuffer();

            for (EObject container = eObject.eContainer(); container != null;
                     container = eObject.eContainer()) {
                EStructuralFeature eStructuralFeature = eObject.eContainmentFeature();

                if (eStructuralFeature.isMany()) {
                    org.eclipse.emf.common.util.EList eList = (org.eclipse.emf.common.util.EList) container.eGet(eStructuralFeature, false);
                    int index = eList.indexOf(eObject);

                    if (index < 10000) {
                        result.insert(0, getIndexStr(index));
                    } else {
                        result.insert(0, index);
                    }

                    result.insert(0, '.');
                    result.insert(0, eStructuralFeature.getName());
                    result.insert(0, '@');
                } else {
                    result.insert(0, eStructuralFeature.getName());
                    result.insert(0, '@');
                }

                result.insert(0, '/');
                eObject = container;
            }

            result.insert(0, getURIFragmentRootSegment(eObject));
            result.insert(0, '/');

            return result.toString();
        }
    }

    /**
     * @see org.eclipse.emf.ecore.resource.Resource.Internal#attached(org.eclipse.emf.ecore.EObject)
     */
    public void attached(EObject value) {
        if (register) {
            LookupServiceExtensions.getInstance().register(getContext(value), value);
        }
    }

    /**
     * @see org.eclipse.emf.ecore.resource.Resource.Internal#detached(org.eclipse.emf.ecore.EObject)
     */
    public void detached(EObject value) {
        LookupServiceExtensions.getInstance().deregister(getContext(value), value);
    }

    public void doLoad(InputStream inputStream, Map options) throws IOException {
        register = false;
        super.doLoad(inputStream, options);

        ContainmentTraverser containmentTraverser = new ContainmentTraverser(getContents());

        EObjectVisitor objectVisitor = new EObjectVisitor() {
            public boolean afterChildren(EObject element) {
                return true;
            }

            public boolean beforeChildren(EObject element) {
                LookupServiceExtensions.getInstance().register(getContext(element), element);

                return true;
            }
        };

        containmentTraverser.registerVisitors(objectVisitor);

        containmentTraverser.traverse();

        register = true;
    }

    /**
     * This method unloads the resource and cleanup the memory.
     *
     *
     * */
    public void doShallowDelete() {
        for (Iterator iter = EMFUtil.delete(this).iterator(); iter.hasNext();) {
            Resource res = (Resource) iter.next();

            if (res != null) {
                res.setModified(true);
            }
        }

        if ((context != null) && (context.getAgentProxy() != null)) {
            context.getAgentProxy().setAgent(null);
        }

        unloadLookupContext();
//        SaveUtil.removeDocument(this);
    }

    public void doShallowUnload() {
        EMFUtil.unload(this);
        unloadLookupContext();
//        SaveUtil.removeDocument(this);
    }

    /**
     * @return
     */
    protected HierarchyContext getContext(EObject object) {
        //        context = (HierarchyContext) LookupServiceExtensions.getInstance().locate(null, HierarchyContext.class, getURI().toString());
        if ((context == null) && object instanceof TRCAgent) {
            context = (HierarchyContext) LookupServiceExtensions.getInstance().locate(null, HierarchyContext.class, getURI().toString());

            if (context == null) {
                context = new HierarchyContext();
            }

            TRCAgent agent = (TRCAgent) object;

            AgentsContext agentsContext = (AgentsContext) LookupServiceExtensions.getInstance().locate(null, AgentsContext.class, LoadersUtils.getLookUpKey(agent.getRuntimeId()));

            if (agentsContext == null) {
                agentsContext = new AgentsContext(agent.getRuntimeId());
                LookupServiceExtensions.getInstance().register(null, agentsContext);
            }

            agentsContext.registerAgent(agent);

            context.setAgent(agent);
            context.setContextURI(getURI().toString());
            LookupServiceExtensions.getInstance().register(null, context);

            return null;
        }
        else if (object instanceof UnresolvedCorrelation) 
        {
        	return null; //this should be added to the global context
        }

        return context;
    }

    protected XMLLoad createXMLLoad() {
        XMLResourceLoader resourceLoader = null;

        try {
            resourceLoader = (XMLResourceLoader) ResourceExtensions.getInstance().get(getURI().fileExtension());

            if (resourceLoader != null) {
                resourceLoader = (XMLResourceLoader) resourceLoader.getClass().newInstance();
                resourceLoader.setXMLHelper(createXMLHelper());
            }
        } catch (Exception e) {
            HierarchyPlugin.log(e.getLocalizedMessage());
        }

        if (resourceLoader == null) {
            return super.createXMLLoad();
        }

        return resourceLoader;
    }

    /**
     * @see org.eclipse.emf.ecore.xmi.impl.XMLResourceImpl#createXMLSave()
     */
    protected XMLSave createXMLSave() {
        return new HierarchyXMISaveImpl(createXMLHelper());
    }

    /* (non-Javadoc)
     * @see org.eclipse.emf.ecore.resource.impl.ResourceImpl#doUnload()
     */
    protected void doUnload() {
        doShallowUnload();
    }

    protected void unloadLookupContext() {
        if (context != null) {
            if (context.getAgent() != null) {
                //                LookupServiceExtensions.getInstance().deregister(null, TRCAgentImpl.class, context.getAgent().getRuntimeId());
                TRCAgent agent = context.getAgent();
                AgentsContext agentsContext = (AgentsContext) LookupServiceExtensions.getInstance().locate(null, AgentsContext.class, LoadersUtils.getLookUpKey(agent.getRuntimeId()));

                if (agentsContext != null) {
                    agentsContext.deregisterAgent(agent);
                }

                deregisterContents(getContents());
            }

            LookupServiceExtensions.getInstance().deregister(context);
            LookupServiceExtensions.getInstance().deregister(null, HierarchyContext.class, getURI().toString());
            context = null;
        }
    }

    private void deregisterContents(List l) {
        ContainmentTraverser containmentTraverser = new ContainmentTraverser(l);

        EObjectVisitor objectVisitor = new EObjectVisitor() {
            public boolean afterChildren(EObject element) {
                LookupServiceExtensions.getInstance().deregister(context, element);

                return true;
            }

            public boolean beforeChildren(EObject element) {
                return true;
            }
        };

        containmentTraverser.registerVisitors(objectVisitor);

        containmentTraverser.traverse();
    }

    private void initHierarchyResource() {
        for (int i = 0; i < 1000; i++) {
            intAndStrList.add(i, null);
        }
    }
}
