package org.eclipse.hyades.logging.core;

/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

/** 
 * Utility class for assisting in generating XML data.
 *  
 * 
 * @author    Paul Slauenwhite
 * @version   9 July 2003
 */

public class XmlUtility {

    /**
     * Normalizes the parameter string according to the XML specification for attribute-value 
     * normalization (<a href="http://www.w3.org/TR/REC-xml">http://www.w3.org/TR/REC-xml</a>)
     * and valid characters (<a href="http://www.w3.org/TR/REC-xml#charsets">http://www.w3.org/TR/REC-xml#charsets</a>).
     * <p>
     * Valid characters, according to the XML specification, in all Unicode characters, excluding the surrogate blocks, 
     * 0xFFFE, and 0xFFFF.
     * 
     * @param string The string to be normalized.
     * @return The normalized string.
     */
    public static String normalize(String string) {

        //Return 'null' if the string is null:
        if (string == null)
            return "null";

        //Return an empty string if the string is empty:
        if (string.length() == 0)
            return "";

        StringBuffer normalizedString = new StringBuffer();
        char character;

        //Check if any characters require normalization or replacement of non-valid characters:
        for (int counter = 0; counter < string.length(); counter++) {

            character = string.charAt(counter);

            //0x003C:
            if (character == '<')
                normalizedString.append("&lt;");

            //0x003E:
            else if (character == '>')
                normalizedString.append("&gt;");

            //0x0026:
            else if (character == '&')
                normalizedString.append("&amp;");

            //0x0022:
            else if (character == '"')
                normalizedString.append("&quot;");

            //0x0027:
            else if (character == '\'')
                normalizedString.append("&apos;");

            //0x0009:
            else if (character == '\t')
                normalizedString.append("&#x9;");

            //0x000A:
            else if (character == '\n')
                normalizedString.append("&#xA;");

            //0x000D:
            else if (character == '\r')
                normalizedString.append("&#xD;");

            /*
            //0x0020:
            			else if (character == ' ')
            				normalizedString.append("&#x20;");
            */

            //Valid character range:
            else if (((((int) (character)) >= 0x0020) && (((int) (character)) <= 0xD7FF)) || ((((int) (character)) >= 0xE000) && (((int) (character)) <= 0xFFFD)) || ((((int) (character)) >= 0x10000) && (((int) (character)) <= 0x10FFFF)))
                normalizedString.append(character);

            else
                normalizedString.append('?');
        }

        return (normalizedString.toString());
    }

    /**
     * Denormalizes the parameter string.
     * 
     * @param string the String to be denormalized.
     * @return The denormalized String.
     */
    public static String denormalize(String string) {

        if (string == null)
            return "null";

        StringBuffer denormalizedString = new StringBuffer();
        char character = 0;
        int semiColonIndex = -1;
        String name = null;

        //Locate and denormalize all entity references:
        for (int counter = 0; counter < string.length(); counter++) {

            character = string.charAt(counter);

            //Check if this character is the start of a possible entity reference (e.g. ampersand in &<name>;) and find a possible end to the possible entity reference (e.g. semi-solon in &<name>;):
            if ((character == '&') && ((semiColonIndex = string.indexOf(';', (counter + 1))) != -1)) {

                name = string.substring((counter + 1), semiColonIndex).trim();

                if (name.equals("lt"))
                    denormalizedString.append('<');

                else if (name.equals("gt"))
                    denormalizedString.append('>');

                else if (name.equals("amp"))
                    denormalizedString.append('&');

                else if (name.equals("quot"))
                    denormalizedString.append('"');

                else if (name.equals("apos"))
                    denormalizedString.append('\'');

                else if (name.equals("#x9"))
                    denormalizedString.append('\t');

                else if (name.equals("#xA"))
                    denormalizedString.append('\n');

                else if (name.equals("#xD"))
                    denormalizedString.append('\r');

                /*
                else if (name.equals("#x20"))
                	  denormalizedString.append(' ');
                */

                //Not a supported entity reference:
                else {
                    denormalizedString.append('&');
                    denormalizedString.append(name);
                    denormalizedString.append(';');
                }

                counter = semiColonIndex;
            }
            else
                denormalizedString.append(character);
        }

        return (denormalizedString.toString());
    }
}
