/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.sdb.internal.wizard;

import java.io.*;
import java.lang.reflect.*;
import java.net.*;
import java.util.*;

import org.eclipse.core.resources.*;
import org.eclipse.core.runtime.*;
import org.eclipse.hyades.analysis.engine.AnalysisEngineConstants;
import org.eclipse.hyades.analysis.engine.AnalysisEnginePlugin;
import org.eclipse.hyades.models.internal.sdb.loader.*;
import org.eclipse.hyades.sdb.internal.*;
import org.eclipse.hyades.sdb.internal.help.*;
import org.eclipse.hyades.sdb.internal.util.*;
import org.eclipse.jface.dialogs.*;
import org.eclipse.jface.operation.*;
import org.eclipse.jface.preference.*;
import org.eclipse.jface.viewers.*;
import org.eclipse.swt.*;
import org.eclipse.swt.events.*;
import org.eclipse.swt.layout.*;
import org.eclipse.swt.widgets.*;
import org.eclipse.ui.*;
import org.eclipse.ui.dialogs.*;
import org.eclipse.ui.help.*;

/**
 *  Page 1 of the base resource import-from-zip Wizard.
 *
 *  Note that importing from .jar is identical to importing from .zip, so
 *  all references to .zip in this class are equally applicable to .jar
 *  references.
 */
public class ImportSymptomDBPage2
	extends WizardDataTransferPage
	implements Listener, ModifyListener, SelectionListener, IRunnableWithProgress {

	// widgets
	private Button _localHost;
	private Button _remoteHost;
	private Label _localLabel;
	private Text _localFileText;
	private Button _sourceBrowse;
	private Button _browseLocation;
	private Label _urlLabel;
	private Combo _urlText;
	private Text _location;
	private Text _targetFileName;
	private Button _overwriteCheckbox;
	private Button _createFolderCheckbox;
	private Button _loadDBCheckbox;
	private Button _openEditorCheckbox;

	private String fError;
	private IStructuredSelection fSelection;
	private final String _dbname = "symptomdb.trcdbxmi";
	private String _xmiFile;
	private String _xmlFile;
	private String _urlFile;
	private boolean _isLocalHost;
	private ArrayList _urlList = new ArrayList();

	class ColumnData {
		private String _name;
		private String _url;

		public ColumnData(String name, String url) {
			_name = name;
			_url = url;
		}

		public String name() {
			return _name;
		}
		public String url() {
			return _url;
		}

		public void name(String name) {
			_name = name;
		}

		public void url(String url) {
			_url = url;
		}
	}
	/**
	 *  Creates an instance of this class
	 * @param aWorkbench IWorkbench
	 * @param selection IStructuredSelection
	 */
	public ImportSymptomDBPage2(
		IWorkbench aWorkbench,
		IStructuredSelection selection) {
		super("xmlTraceFileImportPage1");
		setTitle(SDbPlugin.getResourceString("IMPORT_SYMPTOM_DB_PTITLE"));
		setDescription(SDbPlugin.getResourceString("IMPORT_SYMPTOM_DB_PDESC"));
		setImageDescriptor(
			SDbPluginImages.getImageDescriptor(SDbPluginImages.IMG_IMPORT_DB_WIZ_BAN));

		fSelection = selection;

	}
	protected boolean allowNewContainerName() {
		return true;
	}
	/**
	 * Creates a new button with the given id.
	 * <p>
	 * The <code>Dialog</code> implementation of this framework method
	 * creates a standard push button, registers for selection events
	 * including button presses and registers
	 * default buttons with its shell.
	 * The button id is stored as the buttons client data.
	 * Note that the parent's layout is assumed to be a GridLayout and 
	 * the number of columns in this layout is incremented.
	 * Subclasses may override.
	 * </p>
	 *
	 * @param parent the parent composite
	 * @param id the id of the button (see
	 *  <code>IDialogConstants.*_ID</code> constants 
	 *  for standard dialog button ids)
	 * @param label the label from the button
	 * @param defaultButton <code>true</code> if the button is to be the
	 *   default button, and <code>false</code> otherwise
	 */
	protected Button createButton(
		Composite parent,
		int id,
		String label,
		boolean defaultButton) {
		// increment the number of columns in the button bar
		 ((GridLayout) parent.getLayout()).numColumns++;

		Button button = new Button(parent, SWT.PUSH);

		GridData data = new GridData(GridData.HORIZONTAL_ALIGN_FILL);
		data.heightHint = convertVerticalDLUsToPixels(IDialogConstants.BUTTON_HEIGHT);
		data.widthHint =
			convertHorizontalDLUsToPixels(IDialogConstants.BUTTON_WIDTH + 2);
		button.setLayoutData(data);

		button.setData(new Integer(id));
		button.setText(label);

		if (defaultButton) {
			Shell shell = parent.getShell();
			if (shell != null) {
				shell.setDefaultButton(button);
			}
			button.setFocus();
		}
		button.setFont(parent.getFont());
		return button;
	}
	/** (non-Javadoc)
	 * Method declared on IDialogPage.
	 */
	public void createControl(Composite parent) {
		Composite composite = new Composite(parent, SWT.NULL);
		composite.setLayout(new GridLayout());
		GridData data = GridUtil.createFill();
		data.widthHint = 300;
		composite.setLayoutData(data);
		composite.setLayoutData(data);

		createSourceGroup(composite);
		createSpacer(composite);

		createBoldLabel(
			composite,
			SDbPlugin.getResourceString("STR_IMPORT_SYMPTOMDB_PSOURCEQ"));

		Composite result = new Composite(composite, SWT.NULL);
		GridLayout layout = new GridLayout();
		layout.numColumns = 3;
		layout.verticalSpacing = 2;
		result.setLayout(layout);
		data = GridUtil.createHorizontalFill();
		data.widthHint = 300;
		result.setLayoutData(data);

		Label locationLabel = new Label(result, SWT.NULL);
		locationLabel.setText(SDbPlugin.getResourceString("STR_SYMPTOM_DB_LOCATION"));

		_location = new Text(result, SWT.BORDER);
		_location.setLayoutData(GridUtil.createHorizontalFill());

		_browseLocation = new Button(result, SWT.NULL);
		_browseLocation.setText(SDbPlugin.getResourceString("STR_IMPORT_XMLLOG_BROWSE"));

		locationLabel = new Label(result, SWT.NULL);
		locationLabel.setText(
			SDbPlugin.getResourceString("STR_SYMPTOM_DB_FILE_LOCATION"));

		_targetFileName = new Text(result, SWT.BORDER);
		_targetFileName.setLayoutData(GridUtil.createHorizontalFill());

		createSpacer(composite);
		createPlainLabel(
			composite,
			SDbPlugin.getResourceString("ImportDBWizardPage.options"));

		createOptionsGroup(composite);

		restoreWidgetValues();
		updateWidgetEnablements();
		setPageComplete(determinePageCompletion());

		setControl(composite);

		_location.addModifyListener(this);
		_localFileText.addModifyListener(this);
		_targetFileName.addModifyListener(this);
		_browseLocation.addSelectionListener(this);

		initData();

		WorkbenchHelp.setHelp(_location, ContextIds.IMPORT_SYMDB_LOCATION);
		WorkbenchHelp.setHelp(_targetFileName, ContextIds.IMPORT_SYMDB_FILE);
	}
	/**
	 *	Create the import options specification widgets.
	 *
	 *	@param parent org.eclipse.swt.widgets.Composite
	 */
	protected void createOptionsGroup(Composite parent) {
		Composite optionsGroup = new Composite(parent, SWT.NONE);
		GridLayout layout = new GridLayout();
		layout.marginHeight = 0;
		optionsGroup.setLayout(layout);

		_overwriteCheckbox = new Button(optionsGroup, SWT.CHECK | SWT.LEFT);
		_overwriteCheckbox.setText(
			SDbPlugin.getResourceString("ImportDBWizardPage.override.text"));

		_createFolderCheckbox = new Button(optionsGroup, SWT.CHECK | SWT.LEFT);
		_createFolderCheckbox.setText(
			SDbPlugin.getResourceString("ImportDBWizardPage.create.folder.str"));

		_createFolderCheckbox.setSelection(true);
		
		_loadDBCheckbox = new Button(optionsGroup, SWT.CHECK | SWT.LEFT);
		_loadDBCheckbox.setText(
			SDbPlugin.getResourceString("ImportDBWizardPage.options.load"));

		_loadDBCheckbox.setSelection(true);
		
		_openEditorCheckbox = new Button(optionsGroup, SWT.CHECK | SWT.LEFT);
		_openEditorCheckbox.setText(
			SDbPlugin.getResourceString("ImportDBWizardPage.options.openEditor"));

		_openEditorCheckbox.setSelection(true);

		WorkbenchHelp.setHelp(
			_overwriteCheckbox,
			ContextIds.IMPORT_SYMDB_OVERRIDE_RESOURCE);
		WorkbenchHelp.setHelp(
			_createFolderCheckbox,
			ContextIds.IMPORT_SYMDB_CREATE_FOLDER);
	}
	private String createModel() {
		_isLocalHost = _localHost.getSelection();
		_xmlFile = _localFileText.getText().trim();
		_xmiFile =
			_location.getText().trim() + File.separator + _targetFileName.getText().trim();

		_urlFile = "";
		int i = _urlText.getSelectionIndex();
		if (i >= 0 && i < _urlList.size())
			_urlFile = ((ColumnData) _urlList.get(i)).url();

		try {
			getWizard().getContainer().run(true, true, this);
		} catch (InterruptedException e) {
			return "";
		} catch (InvocationTargetException e) {
			return "";
		}

		return fError;
	}
	/**
	 *  Create the import source widgets
	 */
	protected void createSourceGroup(Composite parent) {

		Composite sourceGrp = new Composite(parent, SWT.NONE);
		GridLayout layout = new GridLayout();
		layout.numColumns = 3;
		layout.verticalSpacing = 2;
		layout.horizontalSpacing = 2;
		GridData data = GridUtil.createHorizontalFill();
		data.widthHint = 300;
		sourceGrp.setLayout(layout);
		sourceGrp.setLayoutData(data);

		_remoteHost = new Button(sourceGrp, SWT.RADIO);
		data = GridUtil.createHorizontalFill();
		data.horizontalSpan = 3;
		_remoteHost.setLayoutData(data);
		_remoteHost.setText(SDbPlugin.getResourceString("STR_IMPORT_REMOTEHOST"));
		_remoteHost.setSelection(true);

		_urlLabel = new Label(sourceGrp, SWT.NONE);
		data = new GridData();
		data.horizontalIndent = 20;
		_urlLabel.setLayoutData(data);
		_urlLabel.setText(SDbPlugin.getResourceString("STR_IMPORT_URL"));

		_urlText = new Combo(sourceGrp, SWT.BORDER | SWT.READ_ONLY);
		data = GridUtil.createHorizontalFill();
		data.horizontalSpan = 2;
		_urlText.setLayoutData(data);

		_urlText.addModifyListener(this);

		Label label = new Label(sourceGrp, SWT.NULL);
		data = new GridData();
		data.horizontalSpan = 3;
		label.setLayoutData(data);

		_localHost = new Button(sourceGrp, SWT.RADIO);
		data = GridUtil.createHorizontalFill();
		data.horizontalSpan = 3;
		_localHost.setLayoutData(data);
		_localHost.setText(SDbPlugin.getResourceString("STR_IMPORT_LOCALHOST"));

		Composite localComp = new Composite(sourceGrp, SWT.NONE);
		layout = new GridLayout();
		layout.numColumns = 3;
		layout.verticalSpacing = 2;
		layout.horizontalSpacing = 2;
		data = GridUtil.createHorizontalFill();
		data.horizontalSpan = 3;
		data.widthHint = 300;
		localComp.setLayout(layout);
		localComp.setLayoutData(data);

		_localLabel = new Label(localComp, SWT.NONE);
		data = new GridData();
		data.horizontalIndent = 15;
		_localLabel.setLayoutData(data);
		_localLabel.setText(
			SDbPlugin.getResourceString("STR_IMPORT_SYMPTOM_DB_SOURCE"));

		// source name entry field
		_localFileText = new Text(localComp, SWT.BORDER);
		data = GridUtil.createHorizontalFill();
		//data.grabExcessHorizontalSpace = true;
		_localFileText.setLayoutData(data);

		// source browse button
		_sourceBrowse = new Button(localComp, SWT.PUSH);
		_sourceBrowse.setText(SDbPlugin.getResourceString("STR_IMPORT_LOG_FILE_BROWSE1"));

		_sourceBrowse.addListener(SWT.Selection, this);
		_localHost.addSelectionListener(this);

		enableSourceGroup();
		_localFileText.setFocus();
		
		IPreferenceStore store = SDbPlugin.getDefault().getPreferenceStore();
		String SymDBFileName = store.getString(SdUIConstants.SYM_DB_FILE_NAME);
		if (SymDBFileName != null && !SymDBFileName.equals("")) {
			_localFileText.setText(SymDBFileName);
		}
		
		WorkbenchHelp.setHelp(_localHost, ContextIds.IMPORT_SYMDB_HOST_LOCAL);
		WorkbenchHelp.setHelp(_localFileText, ContextIds.IMPORT_SYMDB_HOST_LOCAL);
		WorkbenchHelp.setHelp(_remoteHost, ContextIds.IMPORT_SYMDB_HOST_REMOTE);
		WorkbenchHelp.setHelp(_urlText, ContextIds.IMPORT_SYMDB_HOST_REMOTE);
	}
	/**
	 * Returns whether this page is complete. This determination is made based upon
	 * the current contents of this page's controls.  Subclasses wishing to include
	 * their controls in this determination should override the hook methods 
	 * <code>validateSourceGroup</code> and/or <code>validateOptionsGroup</code>.
	 *
	 * @return <code>true</code> if this page is complete, and <code>false</code> if
	 *   incomplete
	 * @see #validateSourceGroup
	 * @see #validateOptionsGroup
	 */
	protected boolean determinePageCompletion() {
		boolean complete =
			validateSourceGroup() && validateDestinationGroup() && validateOptionsGroup();

		// Avoid draw flicker by not clearing the error
		// message unless the input is valid.
		if (complete)
			setErrorMessage(null);

		return complete;
	}
	/**
	 * The Finish button was pressed.  Try to do the required work now and answer
	 * a boolean indicating success.  If <code>false</code> is returned then the
	 * wizard will not close.
	 *
	 * @return boolean
	 */
	public boolean finish() {
		
		if (_localFileText.getText().trim().length() > 0) {
			IPreferenceStore store = SDbPlugin.getDefault().getPreferenceStore();
			store.setValue(SdUIConstants.SYM_DB_FILE_NAME, _localFileText.getText().trim());
		}
		
		IPath targetFileName = new Path(_targetFileName.getText().trim());
		String ext = targetFileName.getFileExtension();
		if (ext == null || !ext.equals("trcdbxmi")) {
			String errMsg = SDbPlugin.getResourceString("STR_FINISH_IMPORT_ERROR_");
			String err = SDbPlugin.getResourceString("STR_DB_EXT_REASON_");

			Status err1 =
				new Status(
					Status.INFO,
					ResourcesPlugin.PI_RESOURCES,
					IResourceStatus.INTERNAL_ERROR,
					err,
					null);

			ErrorDialog.openError(
				getShell(),
				SDbPlugin.getResourceString("SYMPTOMDB_MESSAGE"),
				errMsg,
				err1);

			_targetFileName.setFocus();
			_targetFileName.selectAll();

			return false;
		}

		fError = "";
		IWorkspace workbench = ResourcesPlugin.getWorkspace();
		IContainer container = null;
		IPath path = new Path(_location.getText().trim());
		if (_createFolderCheckbox.getSelection()) {
			container = createContainer(path);
		} else {
			int idx = path.segmentCount();
			if (idx == 1) {
				//project
				container = workbench.getRoot().getProject(path.toString());
			} else // (idx > 1)
				{ //folder

				container = workbench.getRoot().getFolder(path);
			}

		}

		if (container == null || !container.exists()) {
			//return error
			String errMsg = SDbPlugin.getResourceString("STR_FINISH_IMPORT_ERROR_");
			String err = SDbPlugin.getResourceString("STR_PATH_NOT_FOUND_REASON_");

			Status err1 =
				new Status(
					Status.INFO,
					ResourcesPlugin.PI_RESOURCES,
					IResourceStatus.INTERNAL_ERROR,
					err,
					null);

			ErrorDialog.openError(
				getShell(),
				SDbPlugin.getResourceString("SYMPTOMDB_MESSAGE"),
				errMsg,
				err1);

			return false;

		}

		if (!_overwriteCheckbox.getSelection() && container.exists(targetFileName)) {
			String msg = SDbPlugin.getResourceString("STR_FILE_EXISTS");
			msg =
				TString.change(
					msg,
					"%1",
					_location.getText().trim()
						+ java.io.File.separator
						+ _targetFileName.getText());

			if (!MessageDialog
				.openQuestion(getShell(), SDbPlugin.getResourceString("SYMPTOMDB_MESSAGE"), msg))
				return false;

		}

		fError = createModel();
		if (fError != null && !fError.equals("")) {
			String errMsg = SDbPlugin.getResourceString("STR_LOAD_XML_ERROR_");
			Status err1 =
				new Status(
					Status.WARNING,
					ResourcesPlugin.PI_RESOURCES,
					IResourceStatus.INTERNAL_ERROR,
					fError,
					null);

			ErrorDialog.openError(
				getShell(),
				SDbPlugin.getResourceString("SYMPTOMDB_MESSAGE"),
				errMsg,
				err1);

			return false;
		}

		/* save imported sdb in preference store*/
		if(fError==null || (fError!=null && fError.length()==0)){

			savePreference(_loadDBCheckbox.getSelection());
		}


		/* open the editor */
		if (_openEditorCheckbox.getSelection()) {
			try
			{
				IFile file = (IFile)container.findMember(targetFileName);
				SDbPlugin.getDefault().getWorkbench().getActiveWorkbenchWindow().getActivePage().openEditor(file);
			}
			catch (Exception e)
			{
				e.printStackTrace();
			}
		}

		return true;
	}
	/**
	 *  Handle all events and enablements for widgets in this dialog
	 *
	 * @param event Event
	 */
	public void handleEvent(Event event) {
		if (event.widget == _sourceBrowse)
			handle_sourceBrowsePressed();

	}
	/**
	 *  Open an appropriate source browser so that the user can specify a source
	 *  to import from
	 */
	protected void handle_sourceBrowsePressed() {

		String currentSource = this._localFileText.getText();

		FileDialog dlg = new FileDialog(_localFileText.getShell());
		dlg.setFilterPath(currentSource);
		dlg.setFilterExtensions(new String[] { "*.xml", "*.*" });
		dlg.open();

		String fileName = dlg.getFileName();
		if (fileName != null && !fileName.equals("")) {
			_localFileText.setText(dlg.getFilterPath() + File.separator + fileName);
		}
	}
	/**
	 * Sent when the text is modified.
	 *
	 * @param e an event containing information about the modify
	 */
	public void modifyText(ModifyEvent e) {
		setPageComplete(determinePageCompletion());
	}
	public void run(IProgressMonitor mon) {

		String destination = "";
		if (_xmiFile.startsWith("\\") || _xmiFile.startsWith("/")) {
			_xmiFile = _xmiFile.substring(1);
		}
		
		_xmiFile = new Path(_xmiFile).toString();
		destination = "platform:/resource/" + _xmiFile;
			
		if (_isLocalHost) {
			try {
				SDBLoader l = new SDBLoader();
				l.loadDatabase(_xmlFile, destination, mon);
				fError = l.getErrorMessage();
			} catch (Exception exc) {
				//fError = exc.toString();

				if(fError==null || (fError!=null && fError.length()==0))
					fError = SDbPlugin.getResourceString("STR_EXC_REASON_");
			}
		} else {
			try {
				URL url = new URL(_urlFile);

				InputStream is = url.openStream();
				if (is != null) {
					SDBLoader l = new SDBLoader();
					l.loadDatabase(is, _urlFile, destination, mon);
					fError = l.getErrorMessage();
				}
			} catch (MalformedURLException exc) {
				//fError = exc.toString();
				fError = SDbPlugin.getResourceString("STR_MALFORMEDURLEXC_REASON_");
			} catch (IOException exc1) {
				//fError = exc1.toString();
				fError = SDbPlugin.getResourceString("STR_IOEXC_REASON_");				

			} catch (Exception e) {
				//fError = e.toString();
				fError = SDbPlugin.getResourceString("STR_EXC_REASON_");
			}
		}

	}
	/**
	 * Check if widgets are enabled or disabled by a change in the dialog.
	 * @param event Event
	 */
	protected void updateWidgetEnablements() {

		boolean pageComplete = determinePageCompletion();
		setPageComplete(pageComplete);
		if (pageComplete)
			setMessage(null);
		super.updateWidgetEnablements();
	}
	/**
	 * Returns whether this page's options group's controls currently all contain
	 * valid values.
	 * <p>
	 * The <code>WizardDataTransferPage</code> implementation of this method returns
	 * <code>true</code>. Subclasses may reimplement this hook method.
	 * </p>
	 *
	 * @return <code>true</code> indicating validity of all controls in the options
	 *   group
	 */
	protected boolean validateOptionsGroup() {
		return (
			!_location.getText().trim().equals("")
				&& !_targetFileName.getText().trim().equals(""));
	}
	/**
	 * Returns whether this page's source specification controls currently all
	 * contain valid values.
	 * <p>
	 * The <code>WizardDataTransferPage</code> implementation of this method returns
	 * <code>true</code>. Subclasses may reimplement this hook method.
	 * </p>
	 *
	 * @return <code>true</code> indicating validity of all controls in the 
	 *   source specification group
	 */
	protected boolean validateSourceGroup() {
		return (
			(_localHost.getSelection() && !_localFileText.getText().trim().equals(""))
				|| (!_localHost.getSelection() && !_urlText.getText().trim().equals("")));
	}
	/**
	 * Insert the method's description here.
	 * Creation date: (8/7/2001 12:06:04 PM)
	 */
	private void showResourceDialog() {
		IPath path = new Path(_location.getText().trim());
		IResource res = ResourcesPlugin.getWorkspace().getRoot().findMember(path);
		IContainer cont = null;

		if (res != null && res.exists() && res instanceof IContainer)
			cont = (IContainer) res;

		ContainerSelectionDialog dialog =
			new ContainerSelectionDialog(
				getShell(),
				cont,
				true,
				SDbPlugin.getResourceString("STR_RES_DLG_MSG"));

		if (dialog.open() == ContainerSelectionDialog.OK) {
			Object[] result = dialog.getResult();
			if (result != null && result.length == 1 && result[0] != null) {
				_location.setText(((IPath) result[0]).toOSString());
			}
		}
	}
	/**
	 * Sent when default selection occurs in the control.
	 * <p>
	 * For example, on some platforms default selection occurs
	 * in a List when the user double-clicks an item or types
	 * return in a Text.
	 * </p>
	 *
	 * @param e an event containing information about the default selection
	 */
	public void widgetDefaultSelected(SelectionEvent e) {
	}
	/**
	 * Sent when selection occurs in the control.
	 * <p>
	 * For example, on some platforms selection occurs in
	 * a List when the user selects an item or items.
	 * </p>
	 *
	 * @param e an event containing information about the selection
	 */
	public void widgetSelected(SelectionEvent e) {
		if (e.widget == _browseLocation)
			showResourceDialog();
		else if (e.widget == _localHost) {
			enableSourceGroup();
			setPageComplete(determinePageCompletion());
		}
	}
	/**
	 * Insert the method's description here.
	 * Creation date: (8/7/2001 8:18:18 PM)
	 */
	private void initData() {
		if (fSelection != null && !fSelection.isEmpty()) {
			Object elem = fSelection.getFirstElement();
			if (elem == null)
				return;

			if (elem instanceof IProject || elem instanceof IFolder) {
				_location.setText(((IResource) elem).getFullPath().toOSString());
			} else if (elem instanceof IFile) {
				_location.setText(
					((IResource) elem).getFullPath().removeLastSegments(1).toOSString());
			}
		}

        if(_location.getText().equals(""))
           _location.setText(SdUIConstants.LOG_ANALYZER_PRJ);
           
		_localHost.setSelection(false);
		_remoteHost.setSelection(true);
		enableSourceGroup();

		//urls = name\tvalue\nname\tvalue
		String urls =
			SDbPlugin.getDefault().getPreferenceStore().getString(
				SdUIConstants.SYMPTOM_DB_URL);
		if (urls != null && !urls.equals("")) {
			int idx = urls.indexOf("\n");
			while (idx != -1) {
				String value = urls.substring(0, idx);
				int i = value.indexOf("\t");
				if (i != -1) {
					ColumnData data = new ColumnData(value.substring(0, i), value.substring(i + 1));
					_urlList.add(data);
				}

				urls = urls.substring(idx + 1);
				idx = urls.indexOf("\n");
			}

			int i = urls.indexOf("\t");
			if (i != -1) {
				ColumnData data = new ColumnData(urls.substring(0, i), urls.substring(i + 1));
				_urlList.add(data);
			}
		}

		for (int i = 0; i < _urlList.size(); i++)
			_urlText.add(((ColumnData) _urlList.get(i)).name() + " [ " + ((ColumnData) _urlList.get(i)).url() + " ] ");

		if (_urlText.getItemCount() > 0)
			_urlText.setText(_urlText.getItem(0));

		_targetFileName.setText(_dbname);

	}
	/**
	 * Insert the method's description here.
	 * Creation date: (10/05/2000 10:03:14 AM)
	 * @param path com.ibm.itp.common.IPath
	 */
	private IContainer createContainer(IPath path) {
		IContainer container = null;
		IWorkspace workbench = ResourcesPlugin.getWorkspace();
		int segCount = path.segmentCount();

		for (int idx = 0; idx < segCount; idx++) {
			String seg = path.segment(idx);

			if (idx == 0) {
				//project
				IProject project =
					workbench.getRoot().getProject(path.uptoSegment(idx + 1).toString());
				if (project == null || !project.exists()) {
					//create the project

					try {
						project.create(null);
						project.open(null);
					} catch (Exception e) {
						e.printStackTrace();
					}
				}
				container = project;
			} else // (idx > 1)
				{ //folder

				IFolder folder = workbench.getRoot().getFolder(path.uptoSegment(idx + 1));
				if (folder == null || !folder.exists()) {
					//create the folder

					try {
						folder.create(false, true, null);
					} catch (Exception e) {
						e.printStackTrace();
					}
				}
				container = folder;
			}

		}

		try {

			container.getProject().refreshLocal(IResource.DEPTH_INFINITE, null);
		} catch (CoreException exc) {
			
			exc.printStackTrace();
		}

		return container;
	}
	private void enableSourceGroup() {

		_localFileText.setEnabled(_localHost.getSelection());
		_localLabel.setEnabled(_localHost.getSelection());
		_sourceBrowse.setEnabled(_localHost.getSelection());

		_urlText.setEnabled(!_localHost.getSelection());
		_urlLabel.setEnabled(!_localHost.getSelection());

	}
	
	private void savePreference(boolean check){
		IPreferenceStore store = AnalysisEnginePlugin.getPlugin().getPreferenceStore();
		String prefPath = store.getString(AnalysisEngineConstants.SYMPTOM_DB_PATH);
		int i = prefPath.indexOf(_xmiFile);
		StringBuffer sdbPath = new StringBuffer();
		
		if(i>-1){
			sdbPath.append(prefPath);
			sdbPath.replace(i+_xmiFile.length()+1, i+_xmiFile.length()+2,check ? "1" : "0");
		}
		else{
			if(prefPath.length()> 0){
				sdbPath.append(prefPath);
				sdbPath.append(";");
			
			}
			sdbPath.append("/");
			sdbPath.append(_xmiFile);
			sdbPath.append(",");
			sdbPath.append(check ? 1 : 0);

		}

		store.setValue(AnalysisEngineConstants.SYMPTOM_DB_PATH, sdbPath.toString());
	
	}
}
