package org.eclipse.hyades.logging.events;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.ObjectStreamField;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import org.eclipse.hyades.logging.events.exceptions.InvalidExtendedDataElementTypeException;

/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

/**
 * This is the class that embodies the Extended Data Element as described in 
 * "Canonical Situation Data Format: The Common Base Event V1.0.1 
 * ACAB.BO0301.0.1"
 * 
 * 
 * @author Richard K. Duggan
 * @author Paul E. Slauenwhite
 * @author Jason Cornpropst
 * @version 1.0.1
 * @since 1.0
 * @see IExtendedDataElement 
 */
public class ExtendedDataElementImpl implements IExtendedDataElement, Serializable, Cloneable {

    //define the serialiVersionUID for serialization
    //if this object is ever changed that breaks backward compatability
    //this value MUST be changed Every effort should be made not to
    //break backward compatability
    static final long serialVersionUID = 6633427043459375461L;

    // define the fields that will be serialized. 
    // if any new variables are added to the class
    // this table should be updated along with the readObject and writeObject methods
    private static final ObjectStreamField[] serialPersistentFields = { new ObjectStreamField("name", String.class), new ObjectStreamField("type", int.class), new ObjectStreamField("value", Object.class), new ObjectStreamField("children", IExtendedDataElement[].class)};

    /**
     * Fields.
     */
    private String name = null;

    private int type = -1;

    /**
     * Holder for the data of this event property.
     */
    private Object value = null;

    /**
     * The associations elements with the other artifacts.
     */
    private ArrayList children = new ArrayList();

    /**
     * @deprecated Not required for CBE 1.0.1.
     */
    private String id = null;

    //~ Constructors .....................................................................
    /**
     * 
     */
    protected ExtendedDataElementImpl() {
        super();
    }

    //~ Methods ..........................................................................

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#addChildDataElement(IExtendedDataElement)
     */
    public void addChildDataElement(IExtendedDataElement element) {

        if (children == null) {
            children = new ArrayList();
        }

        this.children.add(element);
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#getChildDataElement(java.lang.String)
     */
    public IExtendedDataElement getChildDataElement(String name) {

        if (children != null) {

            for (int counter = 0; counter < children.size(); counter++) {

                if (((IExtendedDataElement) (children.get(counter))).getName().equals(name)) {
                    return ((IExtendedDataElement) (children.get(counter)));
                }
            }
        }

        return null;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#getChildDataElements()
     */
    public List getChildDataElements() {
        return children;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#removeChildDataElement(String)
     */
    public void removeChildDataElement(String name) {

        if (children != null) {

            for (int counter = 0; counter < children.size(); counter++) {

                if (((IExtendedDataElement) (children.get(counter))).getName().equals(name)) {
                    children.remove(counter--);
                }
            }
        }
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#clearChildDataElements()
     */
    public void clearChildDataElements() {
        children = null;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#getValues()
     */
    public List getValues() {

        if (value == null) {
            return null;
        }

        if (!(value instanceof String[])) {
            throw new InvalidExtendedDataElementTypeException("getValues was called for an illegal event property type");
        }

        return (Arrays.asList((String[]) (value)));
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#getValuesAsStringArray()
     */
    public String[] getValuesAsStringArray() {

        if (value == null) {
            return null;
        }

        if (!(value instanceof String[])) {
            throw new InvalidExtendedDataElementTypeException("getValues was called for an illegal event property type");
        }

        return ((String[]) (value));
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#getHexValues()
     */
    public char[] getHexValues() {
        return getHexBinarys();
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#getHexValue()
     */
    public byte[] getHexValue() {
        return getHexBinary();
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#getId()
     * @deprecated Not required for CBE 1.0.1.
     */
    public String getId() {
        return id;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#getName()
     */
    public String getName() {
        return name;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#getTypeValue()
     */
    public int getTypeValue() {
        return type;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#getType()
     */
    public String getType() {

        switch (type) {
            case TYPE_BYTE_VALUE :
                return TYPE_BYTE;

            case TYPE_SHORT_VALUE :
                return TYPE_SHORT;

            case TYPE_INT_VALUE :
                return TYPE_INT;

            case TYPE_LONG_VALUE :
                return TYPE_LONG;

            case TYPE_BOOLEAN_VALUE :
                return TYPE_BOOLEAN;

            case TYPE_FLOAT_VALUE :
                return TYPE_FLOAT;

            case TYPE_DOUBLE_VALUE :
                return TYPE_DOUBLE;

            case TYPE_STRING_VALUE :
                return TYPE_STRING;

            case TYPE_DATE_TIME_VALUE :
                return TYPE_DATE_TIME;

            case TYPE_BYTE_ARRAY_VALUE :
                return TYPE_BYTE_ARRAY;

            case TYPE_SHORT_ARRAY_VALUE :
                return TYPE_SHORT_ARRAY;

            case TYPE_INT_ARRAY_VALUE :
                return TYPE_INT_ARRAY;

            case TYPE_LONG_ARRAY_VALUE :
                return TYPE_LONG_ARRAY;

            case TYPE_BOOLEAN_ARRAY_VALUE :
                return TYPE_BOOLEAN_ARRAY;

            case TYPE_FLOAT_ARRAY_VALUE :
                return TYPE_FLOAT_ARRAY;

            case TYPE_DOUBLE_ARRAY_VALUE :
                return TYPE_DOUBLE_ARRAY;

            case TYPE_STRING_ARRAY_VALUE :
                return TYPE_STRING_ARRAY;

            case TYPE_DATE_TIME_ARRAY_VALUE :
                return TYPE_DATE_TIME_ARRAY;

            case TYPE_HEX_BINARY_VALUE :
                return TYPE_HEX_BINARY;
        }

        return null;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#setId(String)
     * @deprecated Not required for CBE 1.0.1.
     */
    public void setId(String id) {
        this.id = id;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#setName(String)
     */
    public void setName(String name) {
        this.name = name;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#setType(byte)
     */
    public void setType(int type) {

        if ((type >= 0) && (type <= TYPE_HEX_BINARY_VALUE)) {
            this.type = type;
        }
        else {
            throw new InvalidExtendedDataElementTypeException("setType was called with an invalid datatype value of  " + getType());
        }
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#setType(String)
     */
    public void setType(String type) {

        if (type.equals(TYPE_BYTE)) {
            this.type = TYPE_BYTE_VALUE;
        }
        else if (type.equals(TYPE_BOOLEAN)) {
            this.type = TYPE_BOOLEAN_VALUE;
        }
        else if (type.equals(TYPE_SHORT)) {
            this.type = TYPE_SHORT_VALUE;
        }
        else if (type.equals(TYPE_INT)) {
            this.type = TYPE_INT_VALUE;
        }
        else if (type.equals(TYPE_LONG)) {
            this.type = TYPE_LONG_VALUE;
        }
        else if (type.equals(TYPE_FLOAT)) {
            this.type = TYPE_FLOAT_VALUE;
        }
        else if (type.equals(TYPE_DOUBLE)) {
            this.type = TYPE_DOUBLE_VALUE;
        }
        else if (type.equals(TYPE_DATE_TIME)) {
            this.type = TYPE_DATE_TIME_VALUE;
        }
        else if (type.equals(TYPE_STRING)) {
            this.type = TYPE_STRING_VALUE;
        }
        else if (type.equals(TYPE_BYTE_ARRAY)) {
            this.type = TYPE_BYTE_ARRAY_VALUE;
        }
        else if (type.equals(TYPE_BOOLEAN_ARRAY)) {
            this.type = TYPE_BOOLEAN_ARRAY_VALUE;
        }
        else if (type.equals(TYPE_SHORT_ARRAY)) {
            this.type = TYPE_SHORT_ARRAY_VALUE;
        }
        else if (type.equals(TYPE_INT_ARRAY)) {
            this.type = TYPE_INT_ARRAY_VALUE;
        }
        else if (type.equals(TYPE_LONG_ARRAY)) {
            this.type = TYPE_LONG_ARRAY_VALUE;
        }
        else if (type.equals(TYPE_FLOAT_ARRAY)) {
            this.type = TYPE_FLOAT_ARRAY_VALUE;
        }
        else if (type.equals(TYPE_DOUBLE_ARRAY)) {
            this.type = TYPE_DOUBLE_ARRAY_VALUE;
        }
        else if (type.equals(TYPE_DATE_TIME_ARRAY)) {
            this.type = TYPE_DATE_TIME_ARRAY_VALUE;
        }
        else if (type.equals(TYPE_STRING_ARRAY)) {
            this.type = TYPE_STRING_ARRAY_VALUE;
        }
        else if (type.equals(TYPE_HEX_BINARY)) {
            this.type = TYPE_HEX_BINARY_VALUE;
        }
        else {
            throw new InvalidExtendedDataElementTypeException("setType was called with an invalid datatype value of  " + getType());
        }
    }

    /**
     * Sets the value.
     * @param value The value to set
     */
    public void setValues(String[] values) {
        value = values;
    }

    public void setHexValues(char[] hexValues) {

        type = TYPE_HEX_BINARY_VALUE;

        value = hexValues;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#setHexValue(byte[])
     */
    public void setHexValue(byte[] hexValue) {

        char[] hexValues = null;

        if (hexValue != null) {

            hexValues = new char[hexValue.length];

            for (int counter = 0; counter < hexValue.length; counter++) {

                //Adjust the signed 8-bit byte (e.g. -128 to 127) to a unsigned 8-bit character (e.g. 0 to 255):
                hexValues[counter] = ((char) ((hexValue[counter] + 128)));
            }
        }

        setHexValues(hexValues);
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#getString()
     */
    public String getString() {

        if (type != TYPE_STRING_VALUE) {
            throw new InvalidExtendedDataElementTypeException("getString was called for an event property of type " + getType());
        }

        if (value == null) {
            return null;
        }

        if (!(value instanceof String)) {
            throw new InvalidExtendedDataElementTypeException("getString was called for an illegal event property type");
        }

        return ((String) (value));
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#getDateTimeAsLong()
     */
    public long getDateTimeAsLong() {

        if (type != TYPE_DATE_TIME_VALUE) {
            throw new InvalidExtendedDataElementTypeException("getDateTimeAsLong was called for an event property of type " + getType());
        }

        if (value == null) {
            return 0;
        }

        if (!(value instanceof String)) {
            throw new InvalidExtendedDataElementTypeException("getDateTimeAsLong was called for an illegal event property type");
        }

        return (CbeFormatter.convertXmlSchemaDateTimeToDate(((String) (value))));
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#getDateTime()
     */
    public String getDateTime() {

        if (type != TYPE_DATE_TIME_VALUE) {
            throw new InvalidExtendedDataElementTypeException("getDateTime was called for an event property of type " + getType());
        }

        if (value == null) {
            return null;
        }

        if (!(value instanceof String)) {
            throw new InvalidExtendedDataElementTypeException("getDateTime was called for an illegal event property type");
        }

        return ((String) (value));
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#getBoolean()
     */
    public boolean getBoolean() {

        if (type != TYPE_BOOLEAN_VALUE) {
            throw new InvalidExtendedDataElementTypeException("getBoolean was called for an event property of type " + getType());
        }

        if (value == null) {
            return true;
        }

        if (!(value instanceof Boolean)) {
            throw new InvalidExtendedDataElementTypeException("getBoolean was called for an illegal event property type");
        }

        return (((Boolean) (value)).booleanValue());
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#getByte()
     */
    public byte getByte() {

        if (type != TYPE_BYTE_VALUE) {
            throw new InvalidExtendedDataElementTypeException("getByte was called for an event property of type " + getType());
        }

        if (value == null) {
            return 0;
        }

        if (!(value instanceof Byte)) {
            throw new InvalidExtendedDataElementTypeException("getByte was called for an illegal event property type");
        }

        return (((Byte) (value)).byteValue());
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#getShort()
     */
    public short getShort() {

        if (type != TYPE_SHORT_VALUE) {
            throw new InvalidExtendedDataElementTypeException("getShort was called for an event property of type " + getType());
        }

        if (value == null) {
            return 0;
        }

        if (!(value instanceof Short)) {
            throw new InvalidExtendedDataElementTypeException("getShort was called for an illegal event property type");
        }

        return (((Short) (value)).shortValue());
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#getInt()
     */
    public int getInt() {

        if (type != TYPE_INT_VALUE) {
            throw new InvalidExtendedDataElementTypeException("getInt was called for an event property of type " + getType());
        }

        if (value == null) {
            return 0;
        }

        if (!(value instanceof Integer)) {
            throw new InvalidExtendedDataElementTypeException("getInt was called for an illegal event property type");
        }

        return (((Integer) (value)).intValue());
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#getLong()    
     */
    public long getLong() {

        if (type != TYPE_LONG_VALUE) {
            throw new InvalidExtendedDataElementTypeException("getLong was called for an event property of type " + getType());
        }

        if (value == null) {
            return 0;
        }

        if (!(value instanceof Long)) {
            throw new InvalidExtendedDataElementTypeException("getLong was called for an illegal event property type");
        }

        return (((Long) (value)).longValue());
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#getFloat()
     */
    public float getFloat() {

        if (type != TYPE_FLOAT_VALUE) {
            throw new InvalidExtendedDataElementTypeException("getFloat was called for an event property of type " + getType());
        }

        if (value == null) {
            return 0;
        }

        if (!(value instanceof Float)) {
            throw new InvalidExtendedDataElementTypeException("getFloat was called for an illegal event property type");
        }

        return (((Float) (value)).floatValue());
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#getDouble()
     */
    public double getDouble() {

        if (type != TYPE_DOUBLE_VALUE) {
            throw new InvalidExtendedDataElementTypeException("getDouble was called for an event property of type " + getType());
        }

        if (value == null) {
            return 0;
        }

        if (!(value instanceof Double)) {
            throw new InvalidExtendedDataElementTypeException("getDouble was called for an illegal event property type");
        }

        return (((Double) (value)).doubleValue());
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#getStringArray()
     */
    public String[] getStringArray() {

        if (type != TYPE_STRING_ARRAY_VALUE) {
            throw new InvalidExtendedDataElementTypeException("getStringArray was called for an event property of type " + getType());
        }

        if (value == null) {
            return null;
        }

        if (!(value instanceof String[])) {
            throw new InvalidExtendedDataElementTypeException("getStringArray was called for an illegal event property type");
        }

        return ((String[]) (value));
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#getDateTimeArrayAsLong()
     */
    public long[] getDateTimeArrayAsLong() {

        if (type != TYPE_DATE_TIME_ARRAY_VALUE) {
            throw new InvalidExtendedDataElementTypeException("getDateTimeArrayAsLong was called for an event property of type " + getType());
        }

        if (value == null) {
            return null;
        }

        if (!(value instanceof String[])) {
            throw new InvalidExtendedDataElementTypeException("getDateTimeArrayAsLong was called for an illegal event property type");
        }

        String[] dateTimeArray = ((String[]) (value));
        long[] dateTimeAsLongArray = null;

        if (dateTimeArray != null) {

            dateTimeAsLongArray = new long[dateTimeArray.length];

            for (int counter = 0; counter < dateTimeArray.length; counter++) {
                dateTimeAsLongArray[counter] = (CbeFormatter.convertXmlSchemaDateTimeToDate(dateTimeArray[counter]));
            }
        }

        return dateTimeAsLongArray;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#getDateTimeArray()
     */
    public String[] getDateTimeArray() {

        if (type != TYPE_DATE_TIME_ARRAY_VALUE) {
            throw new InvalidExtendedDataElementTypeException("getDateTimeArray was called for an event property of type " + getType());
        }

        if (value == null) {
            return null;
        }

        if (!(value instanceof String[])) {
            throw new InvalidExtendedDataElementTypeException("getDateTimeArray was called for an illegal event property type");
        }

        return ((String[]) (value));
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#getBooleanArray()
     */
    public boolean[] getBooleanArray() {

        if (type != TYPE_BOOLEAN_ARRAY_VALUE) {
            throw new InvalidExtendedDataElementTypeException("getBooleanArray was called for an event property of type " + getType());
        }

        if (value == null) {
            return null;
        }

        if (!(value instanceof boolean[])) {
            throw new InvalidExtendedDataElementTypeException("getBooleanArray was called for an illegal event property type");
        }

        return ((boolean[]) (value));
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#getByteArray()
     */
    public byte[] getByteArray() {

        if (type != TYPE_BYTE_ARRAY_VALUE) {
            throw new InvalidExtendedDataElementTypeException("getByteArray was called for an event property of type " + getType());
        }

        if (value == null) {
            return null;
        }

        if (!(value instanceof byte[])) {
            throw new InvalidExtendedDataElementTypeException("getByteArray was called for an illegal event property type");
        }

        return ((byte[]) (value));
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#getShortArray()
     */
    public short[] getShortArray() {

        if (type != TYPE_SHORT_ARRAY_VALUE) {
            throw new InvalidExtendedDataElementTypeException("getShortArray was called for an event property of type " + getType());
        }

        if (value == null) {
            return null;
        }

        if (!(value instanceof short[])) {
            throw new InvalidExtendedDataElementTypeException("getShortArray was called for an illegal event property type");
        }

        return ((short[]) (value));
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#getIntArray()
     */
    public int[] getIntArray() {

        if (type != TYPE_INT_ARRAY_VALUE) {
            throw new InvalidExtendedDataElementTypeException("getIntArray was called for an event property of type " + getType());
        }

        if (value == null) {
            return null;
        }

        if (!(value instanceof int[])) {
            throw new InvalidExtendedDataElementTypeException("getIntArray was called for an illegal event property type");
        }

        return ((int[]) (value));
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#getLongArray()
     */
    public long[] getLongArray() {

        if (type != TYPE_LONG_ARRAY_VALUE) {
            throw new InvalidExtendedDataElementTypeException("getLongArray was called for an event property of type " + getType());
        }

        if (value == null) {
            return null;
        }

        if (!(value instanceof long[])) {
            throw new InvalidExtendedDataElementTypeException("getLongArray was called for an illegal event property type");
        }

        return ((long[]) (value));
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#getFloatArray()
     */
    public float[] getFloatArray() {

        if (type != TYPE_FLOAT_ARRAY_VALUE) {
            throw new InvalidExtendedDataElementTypeException("getFloatArray was called for an event property of type " + getType());
        }

        if (value == null) {
            return null;
        }

        if (!(value instanceof float[])) {
            throw new InvalidExtendedDataElementTypeException("getFloatArray was called for an illegal event property type");
        }

        return ((float[]) (value));
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#getDoubleArray()
     */
    public double[] getDoubleArray() {

        if (type != TYPE_DOUBLE_ARRAY_VALUE) {
            throw new InvalidExtendedDataElementTypeException("getDoubleArray was called for an event property of type " + getType());
        }

        if (value == null) {
            return null;
        }

        if (!(value instanceof double[])) {
            throw new InvalidExtendedDataElementTypeException("getDoubleArray was called for an illegal event property type");
        }

        return ((double[]) (value));
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#getHexBinarys()
     */
    public char[] getHexBinarys() {

        if (type != TYPE_HEX_BINARY_VALUE) {
            throw new InvalidExtendedDataElementTypeException("getHexBinarys was called for an event property of type " + getType());
        }

        if (value == null) {
            return null;
        }

        if (!(value instanceof char[])) {
            throw new InvalidExtendedDataElementTypeException("getHexBinarys was called for an illegal event property type");
        }

        return ((char[]) (value));
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#getHexBinary()
     */
    public byte[] getHexBinary() {

        if (type != TYPE_HEX_BINARY_VALUE) {
            throw new InvalidExtendedDataElementTypeException("getHexBinary was called for an event property of type " + getType());
        }

        if (value == null) {
            return null;
        }

        if (!(value instanceof char[])) {
            throw new InvalidExtendedDataElementTypeException("getHexBinary was called for an illegal event property type");
        }

        byte[] hexValuesAsBytes = null;

        char[] hexValues = ((char[]) (value));

        if (hexValues != null) {

            hexValuesAsBytes = new byte[hexValues.length];

            for (int counter = 0; counter < hexValues.length; counter++) {

                //Check if higher order characters (e.g. 256 - 65535), if so, set to the largest signed 8-bit byte (e.g. 127):
                if (hexValues[counter] > 255)
                    hexValuesAsBytes[counter] = 127;

                //Adjust the unsigned 8-bit character (e.g. 0 to 255) to a signed 8-bit byte (e.g. -128 to 127):
                else
                    hexValuesAsBytes[counter] = ((byte) ((hexValues[counter] - 128)));
            }
        }

        return hexValuesAsBytes;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#setString()
     */
    public void setString(String value) {

        type = TYPE_STRING_VALUE;

        this.value = value;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#setDateTime()
     */
    public void setDateTime(long value) {

        type = TYPE_DATE_TIME_VALUE;

        this.value = CbeFormatter.convertDateToXmlSchemaDateTime(value);
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#setDateTime()
     */
    public void setDateTime(String value) {

        type = TYPE_DATE_TIME_VALUE;

        this.value = value;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#setBoolean()
     */
    public void setBoolean(boolean value) {

        type = TYPE_BOOLEAN_VALUE;

        this.value = new Boolean(value);
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#setByte()
     */
    public void setByte(byte value) {

        type = TYPE_BYTE_VALUE;

        this.value = new Byte(value);
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#setShort()
     */
    public void setShort(short value) {

        type = TYPE_SHORT_VALUE;

        this.value = new Short(value);
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#setInt()
     */
    public void setInt(int value) {

        type = TYPE_INT_VALUE;

        this.value = new Integer(value);
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#setLong()
     */
    public void setLong(long value) {

        type = TYPE_LONG_VALUE;

        this.value = new Long(value);
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#setFloat()
     */
    public void setFloat(float value) {

        type = TYPE_FLOAT_VALUE;

        this.value = new Float(value);
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#setDouble()
     */
    public void setDouble(double value) {

        type = TYPE_DOUBLE_VALUE;

        this.value = new Double(value);
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#setStringArray()
     */
    public void setStringArray(String[] values) {

        type = TYPE_STRING_ARRAY_VALUE;

        this.value = values;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#setDateTimeArray()
     */
    public void setDateTimeArray(long[] values) {

        type = TYPE_DATE_TIME_ARRAY_VALUE;

        if (values != null) {

            String[] dateTimeArray = new String[values.length];

            for (int counter = 0; counter < values.length; counter++) {
                dateTimeArray[counter] = (CbeFormatter.convertDateToXmlSchemaDateTime(values[counter]));
            }

            this.value = dateTimeArray;
        }
        else {
            values = null;
        }
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#setDateTimeArray()
     */
    public void setDateTimeArray(String[] values) {

        type = TYPE_DATE_TIME_ARRAY_VALUE;

        this.value = values;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#setBooleanArray()
     */
    public void setBooleanArray(boolean[] values) {

        type = TYPE_BOOLEAN_ARRAY_VALUE;

        this.value = values;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#setByteArray()
     */
    public void setByteArray(byte[] values) {

        type = TYPE_BYTE_ARRAY_VALUE;

        this.value = values;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#setShortArray()
     */
    public void setShortArray(short[] values) {

        type = TYPE_SHORT_ARRAY_VALUE;

        this.value = values;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#setIntArray()
     */
    public void setIntArray(int[] values) {

        type = TYPE_INT_ARRAY_VALUE;

        this.value = values;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#setLongArray()
     */
    public void setLongArray(long[] values) {

        type = TYPE_LONG_ARRAY_VALUE;

        this.value = values;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#setFloatArray()
     */
    public void setFloatArray(float[] values) {

        type = TYPE_FLOAT_ARRAY_VALUE;

        this.value = values;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#setDoubleArray()
     */
    public void setDoubleArray(double[] values) {

        type = TYPE_DOUBLE_ARRAY_VALUE;

        this.value = values;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#setHexBinarys()
     */
    public void setHexBinarys(char[] values) {

        type = TYPE_HEX_BINARY_VALUE;

        this.value = values;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#setHexBinary(byte[])
     */
    public void setHexBinary(byte[] values) {

        type = TYPE_HEX_BINARY_VALUE;

        if (values != null) {

            char[] hexValues = new char[values.length];

            for (int counter = 0; counter < values.length; counter++) {
                hexValues[counter] = ((char) ((values[counter] + 128)));
            }

            this.value = hexValues;
        }
        else {
            values = null;
        }
    }

    /**
     * @deprecated Use the init() as of 1.0.1.
     */
    public void clearValues() {
        init();
    }

    /**
     * Resets the object's properties to their initial (e.g. null) state.
     * 
     * @since 1.0.1
     */
    public void init() {

        type = -1;

        name = null;
        id = null;
        value = null;

        children = null;
    }

    /**
     * Determines is two different objects are equal.
     */
    public boolean equals(Object obj) {

        //It's the same object reference, so it HAS to be equal
        if (this == obj) {
            return true;
        }
        else if (obj instanceof ExtendedDataElementImpl) {

            ExtendedDataElementImpl extendedDataElement = ((ExtendedDataElementImpl) (obj));

            //Does the object represent the same data type?
            if (((getName() != null) && (getName().equals(extendedDataElement.getName())) || (getName() == null) && (extendedDataElement.getName() == null)) && (getTypeValue() == extendedDataElement.getTypeValue())) {

                if (((children != null) && (children.equals(extendedDataElement.children))) || ((children == null) && (extendedDataElement.children == null))) {

                    if ((this.value != null) && (extendedDataElement.value != null)) {

                        if ((this.value instanceof Boolean) && (extendedDataElement.value instanceof Boolean)) {
                            return ((Boolean) (this.value)).equals(((Boolean) (extendedDataElement.value)));
                        }
                        else if ((this.value instanceof boolean[]) && (extendedDataElement.value instanceof boolean[])) {

                            if (((boolean[]) (this.value)).length == ((boolean[]) (extendedDataElement.value)).length) {

                                for (int counter = 0; counter < ((boolean[]) (this.value)).length; counter++) {

                                    if (((boolean[]) (this.value))[counter] != ((boolean[]) (extendedDataElement.value))[counter]) {
                                        return false;
                                    }
                                }

                                return true;
                            }
                        }
                        else if ((this.value instanceof Byte) && (extendedDataElement.value instanceof Byte)) {
                            return ((Byte) (this.value)).equals(((Byte) (extendedDataElement.value)));
                        }
                        else if ((this.value instanceof byte[]) && (extendedDataElement.value instanceof byte[])) {

                            if (((byte[]) (this.value)).length == ((byte[]) (extendedDataElement.value)).length) {

                                for (int counter = 0; counter < ((byte[]) (this.value)).length; counter++) {

                                    if (((byte[]) (this.value))[counter] != ((byte[]) (extendedDataElement.value))[counter]) {
                                        return false;
                                    }
                                }

                                return true;
                            }
                        }
                        else if ((this.value instanceof String) && (extendedDataElement.value instanceof String)) {
                            return ((String) (this.value)).equals(((String) (extendedDataElement.value)));
                        }
                        else if ((this.value instanceof String[]) && (extendedDataElement.value instanceof String[])) {

                            if (((String[]) (this.value)).length == ((String[]) (extendedDataElement.value)).length) {

                                for (int counter = 0; counter < ((String[]) (this.value)).length; counter++) {

                                    if (!((String[]) (this.value))[counter].equals(((String[]) (extendedDataElement.value))[counter])) {
                                        return false;
                                    }
                                }

                                return true;
                            }
                        }
                        else if ((this.value instanceof Double) && (extendedDataElement.value instanceof Double)) {
                            return ((Double) (this.value)).equals(((Double) (extendedDataElement.value)));
                        }
                        else if ((this.value instanceof double[]) && (extendedDataElement.value instanceof double[])) {

                            if (((double[]) (this.value)).length == ((double[]) (extendedDataElement.value)).length) {

                                for (int counter = 0; counter < ((double[]) (this.value)).length; counter++) {

                                    if (((double[]) (this.value))[counter] != ((double[]) (extendedDataElement.value))[counter]) {
                                        return false;
                                    }
                                }

                                return true;
                            }
                        }
                        else if ((this.value instanceof Float) && (extendedDataElement.value instanceof Float)) {
                            return ((Float) (this.value)).equals(((Float) (extendedDataElement.value)));
                        }
                        else if ((this.value instanceof float[]) && (extendedDataElement.value instanceof float[])) {

                            if (((float[]) (this.value)).length == ((float[]) (extendedDataElement.value)).length) {

                                for (int counter = 0; counter < ((float[]) (this.value)).length; counter++) {

                                    if (((float[]) (this.value))[counter] != ((float[]) (extendedDataElement.value))[counter]) {
                                        return false;
                                    }
                                }

                                return true;
                            }
                        }
                        else if ((this.value instanceof char[]) && (extendedDataElement.value instanceof char[])) {

                            if (((char[]) (this.value)).length == ((char[]) (extendedDataElement.value)).length) {

                                for (int counter = 0; counter < ((char[]) (this.value)).length; counter++) {

                                    if (((char[]) (this.value))[counter] != ((char[]) (extendedDataElement.value))[counter]) {
                                        return false;
                                    }
                                }

                                return true;
                            }
                        }
                        else if ((this.value instanceof Integer) && (extendedDataElement.value instanceof Integer)) {
                            return ((Integer) (this.value)).equals(((Integer) (extendedDataElement.value)));
                        }
                        else if ((this.value instanceof int[]) && (extendedDataElement.value instanceof int[])) {

                            if (((int[]) (this.value)).length == ((int[]) (extendedDataElement.value)).length) {

                                for (int counter = 0; counter < ((int[]) (this.value)).length; counter++) {

                                    if (((int[]) (this.value))[counter] != ((int[]) (extendedDataElement.value))[counter]) {
                                        return false;
                                    }
                                }

                                return true;
                            }
                        }
                        else if ((this.value instanceof Long) && (extendedDataElement.value instanceof Long)) {
                            return ((Long) (this.value)).equals(((Long) (extendedDataElement.value)));
                        }
                        else if ((this.value instanceof long[]) && (extendedDataElement.value instanceof long[])) {

                            if (((long[]) (this.value)).length == ((long[]) (extendedDataElement.value)).length) {

                                for (int counter = 0; counter < ((long[]) (this.value)).length; counter++) {

                                    if (((long[]) (this.value))[counter] != ((long[]) (extendedDataElement.value))[counter]) {
                                        return false;
                                    }
                                }

                                return true;
                            }
                        }
                        else if ((this.value instanceof Short) && (extendedDataElement.value instanceof Short)) {
                            return ((Short) (this.value)).equals(((Short) (extendedDataElement.value)));
                        }
                        else if ((this.value instanceof short[]) && (extendedDataElement.value instanceof short[])) {

                            if (((short[]) (this.value)).length == ((short[]) (extendedDataElement.value)).length) {

                                for (int counter = 0; counter < ((short[]) (this.value)).length; counter++) {

                                    if (((short[]) (this.value))[counter] != ((short[]) (extendedDataElement.value))[counter]) {
                                        return false;
                                    }
                                }

                                return true;
                            }
                        }
                    }
                    else if ((this.value == null) && (extendedDataElement.value == null)) {
                        return true;
                    }
                }
            }
        }

        return false;
    }

    private void writeObject(ObjectOutputStream out) throws IOException {

        // set the values of the Serializable fields
        ObjectOutputStream.PutField fields = out.putFields();
        fields.put("name", name);
        fields.put("type", type);
        fields.put("value", value);

        if ((children != null) && (!children.isEmpty())) {
            fields.put("children", ((IExtendedDataElement[]) (children.toArray(new IExtendedDataElement[children.size()]))));
        }

        // save them
        out.writeFields();
    }

    private void readObject(ObjectInputStream in) throws IOException, ClassNotFoundException {

        // prepare to read the alternate persistent fields
        ObjectInputStream.GetField fields = null;
        fields = in.readFields();

        // read the alternate persistent fields
        name = (String) fields.get("name", name);
        type = (int) fields.get("type", type);
        value = (Object) fields.get("value", value);

        Object tempArray[] = null;
        tempArray = (Object[]) fields.get("children", null);
        if (tempArray != null) {
            int arraySize = tempArray.length;
            for (int i = 0; i < arraySize; i++) {
                addChildDataElement((IExtendedDataElement) tempArray[i]);
            }
        }
    }

    /**
     * return a cloned (copy by value) object. This is a deep copy version, in which all the
     * objects within this object will be copied by value.
     *  
     * @return Object
     */
    public Object clone() throws CloneNotSupportedException {

        ExtendedDataElementImpl copy = ((ExtendedDataElementImpl) (super.clone()));

        Object tempElement = null;

        //Clone the children array list:
        if (this.children != null) {

            copy.children = ((ArrayList) (this.children.clone()));
            for (int ctr = 0; ctr < this.children.size(); ctr++) {
                tempElement = children.get(ctr);
                /* if the Elements in children arrayList are some other implementations,
                 * it will only be shallow cloned.
                 */
                if (tempElement instanceof ExtendedDataElementImpl) {
                    copy.children.set(ctr, ((IExtendedDataElement) ((ExtendedDataElementImpl) (tempElement)).clone()));
                }
            }
        }

        if (this.value != null) {

            if (this.value instanceof Boolean) {
                copy.value = new Boolean(((Boolean) (this.value)).booleanValue());
            }
            else if (this.value instanceof boolean[]) {
                copy.value = ((boolean[]) (this.value)).clone();
            }
            else if (this.value instanceof Byte) {
                copy.value = new Byte(((Byte) (this.value)).byteValue());
            }
            else if (this.value instanceof byte[]) {
                copy.value = ((byte[]) (this.value)).clone();
            }
            else if (this.value instanceof String) {
                copy.value = new String(((String) (this.value)));
            }
            else if (this.value instanceof String[]) {
                copy.value = ((String[]) (this.value)).clone();
            }
            else if (this.value instanceof Double) {
                copy.value = new Double(((Double) (this.value)).doubleValue());
            }
            else if (this.value instanceof double[]) {
                copy.value = ((double[]) (this.value)).clone();
            }
            else if (this.value instanceof Float) {
                copy.value = new Float(((Float) (this.value)).floatValue());
            }
            else if (this.value instanceof float[]) {
                copy.value = ((float[]) (this.value)).clone();
            }
            else if (this.value instanceof char[]) {
                copy.value = ((char[]) (this.value)).clone();
            }
            else if (this.value instanceof Integer) {
                copy.value = new Integer(((Integer) (this.value)).intValue());
            }
            else if (this.value instanceof int[]) {
                copy.value = ((int[]) (this.value)).clone();
            }
            else if (this.value instanceof Long) {
                copy.value = new Long(((Long) (this.value)).longValue());
            }
            else if (this.value instanceof long[]) {
                copy.value = ((long[]) (this.value)).clone();
            }
            else if (this.value instanceof Short) {
                copy.value = new Short(((Short) (this.value)).shortValue());
            }
            else if (this.value instanceof short[]) {
                copy.value = ((short[]) (this.value)).clone();
            }
        }

        return copy;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#addChild(java.lang.String, int, java.lang.String)
     */
    public IExtendedDataElement addChild(String name, int type, String value) {

        IExtendedDataElement extendedDataElement = new ExtendedDataElementImpl();
        extendedDataElement.setName(name);
        extendedDataElement.setType(type);
        extendedDataElement.setValues(new String[] { value });

        addChildDataElement(extendedDataElement);

        return extendedDataElement;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#addChild(java.lang.String, int, java.lang.String[])
     */
    public IExtendedDataElement addChild(String name, int type, String[] values) {

        IExtendedDataElement extendedDataElement = new ExtendedDataElementImpl();
        extendedDataElement.setName(name);
        extendedDataElement.setType(type);
        extendedDataElement.setValues(values);

        addChildDataElement(extendedDataElement);

        return extendedDataElement;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#addChild(java.lang.String, java.lang.String)
     */
    public IExtendedDataElement addChild(String name, String value) {

        IExtendedDataElement extendedDataElement = new ExtendedDataElementImpl();
        extendedDataElement.setName(name);
        extendedDataElement.setType(IExtendedDataElement.TYPE_STRING_VALUE);
        extendedDataElement.setValues(new String[] { value });

        addChildDataElement(extendedDataElement);

        return extendedDataElement;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#addChild(java.lang.String, java.lang.String[])
     */
    public IExtendedDataElement addChild(String name, String[] values) {

        IExtendedDataElement extendedDataElement = new ExtendedDataElementImpl();
        extendedDataElement.setName(name);
        extendedDataElement.setType(IExtendedDataElement.TYPE_STRING_VALUE);
        extendedDataElement.setValues(values);

        addChildDataElement(extendedDataElement);

        return extendedDataElement;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#addChild(java.lang.String)
     */
    public IExtendedDataElement addChild(String name) {

        IExtendedDataElement extendedDataElement = new ExtendedDataElementImpl();
        extendedDataElement.setName(name);

        addChildDataElement(extendedDataElement);

        return extendedDataElement;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#addChildWithBooleanArrayValue(java.lang.String, boolean[])
     */
    public IExtendedDataElement addChildWithBooleanArrayValue(String name, boolean[] values) {

        IExtendedDataElement extendedDataElement = new ExtendedDataElementImpl();
        extendedDataElement.setName(name);
        extendedDataElement.setType(IExtendedDataElement.TYPE_BOOLEAN_ARRAY_VALUE);

        String[] stringValues = new String[values.length];

        for (int counter = 0; counter < stringValues.length; counter++) {
            stringValues[counter] = String.valueOf(values[counter]);
        }

        extendedDataElement.setValues(stringValues);

        addChildDataElement(extendedDataElement);

        return extendedDataElement;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#addChildWithBooleanValue(java.lang.String, boolean)
     */
    public IExtendedDataElement addChildWithBooleanValue(String name, boolean value) {

        IExtendedDataElement extendedDataElement = new ExtendedDataElementImpl();
        extendedDataElement.setName(name);
        extendedDataElement.setType(IExtendedDataElement.TYPE_BOOLEAN_VALUE);
        extendedDataElement.setValues(new String[] { String.valueOf(value)});

        addChildDataElement(extendedDataElement);

        return extendedDataElement;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#addChildWithByteArrayValue(java.lang.String, byte[])
     */
    public IExtendedDataElement addChildWithByteArrayValue(String name, byte[] values) {

        IExtendedDataElement extendedDataElement = new ExtendedDataElementImpl();
        extendedDataElement.setName(name);
        extendedDataElement.setType(IExtendedDataElement.TYPE_BYTE_ARRAY_VALUE);

        String[] stringValues = new String[values.length];

        for (int counter = 0; counter < stringValues.length; counter++) {
            stringValues[counter] = String.valueOf(values[counter]);
        }

        extendedDataElement.setValues(stringValues);

        addChildDataElement(extendedDataElement);

        return extendedDataElement;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#addChildWithByteValue(java.lang.String, byte)
     */
    public IExtendedDataElement addChildWithByteValue(String name, byte value) {

        IExtendedDataElement extendedDataElement = new ExtendedDataElementImpl();
        extendedDataElement.setName(name);
        extendedDataElement.setType(IExtendedDataElement.TYPE_BYTE_VALUE);
        extendedDataElement.setValues(new String[] { String.valueOf(value)});

        addChildDataElement(extendedDataElement);

        return extendedDataElement;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#addChildWithDateArrayValue(java.lang.String, java.lang.String[])
     */
    public IExtendedDataElement addChildWithDateArrayValue(String name, String[] values) {

        IExtendedDataElement extendedDataElement = new ExtendedDataElementImpl();
        extendedDataElement.setName(name);
        extendedDataElement.setType(IExtendedDataElement.TYPE_DATE_TIME_ARRAY_VALUE);

        String[] stringValues = new String[values.length];

        for (int counter = 0; counter < stringValues.length; counter++) {
            stringValues[counter] = String.valueOf(values[counter]);
        }

        extendedDataElement.setValues(stringValues);

        addChildDataElement(extendedDataElement);

        return extendedDataElement;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#addChildWithDateAsLongValue(java.lang.String, long)
     */
    public IExtendedDataElement addChildWithDateAsLongValue(String name, long value) {

        IExtendedDataElement extendedDataElement = new ExtendedDataElementImpl();
        extendedDataElement.setName(name);
        extendedDataElement.setType(IExtendedDataElement.TYPE_DATE_TIME_VALUE);
        extendedDataElement.setValues(new String[] { CbeFormatter.convertDateToXmlSchemaDateTime(value)});

        addChildDataElement(extendedDataElement);

        return extendedDataElement;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#addChildWithDatesAsLongValue(java.lang.String, long[])
     */
    public IExtendedDataElement addChildWithDatesAsLongValue(String name, long[] values) {

        IExtendedDataElement extendedDataElement = new ExtendedDataElementImpl();
        extendedDataElement.setName(name);
        extendedDataElement.setType(IExtendedDataElement.TYPE_DATE_TIME_ARRAY_VALUE);

        String[] stringValues = new String[values.length];

        for (int counter = 0; counter < stringValues.length; counter++) {
            stringValues[counter] = CbeFormatter.convertDateToXmlSchemaDateTime(values[counter]);
        }

        extendedDataElement.setValues(stringValues);

        addChildDataElement(extendedDataElement);

        return extendedDataElement;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#addChildWithDateValue(java.lang.String, java.lang.String)
     */
    public IExtendedDataElement addChildWithDateValue(String name, String value) {

        IExtendedDataElement extendedDataElement = new ExtendedDataElementImpl();
        extendedDataElement.setName(name);
        extendedDataElement.setType(IExtendedDataElement.TYPE_DATE_TIME_VALUE);
        extendedDataElement.setValues(new String[] { String.valueOf(value)});

        addChildDataElement(extendedDataElement);

        return extendedDataElement;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#addChildWithDoubleArrayValue(java.lang.String, double[])
     */
    public IExtendedDataElement addChildWithDoubleArrayValue(String name, double[] values) {

        IExtendedDataElement extendedDataElement = new ExtendedDataElementImpl();
        extendedDataElement.setName(name);
        extendedDataElement.setType(IExtendedDataElement.TYPE_DOUBLE_ARRAY_VALUE);

        String[] stringValues = new String[values.length];

        for (int counter = 0; counter < stringValues.length; counter++) {
            stringValues[counter] = String.valueOf(values[counter]);
        }

        extendedDataElement.setValues(stringValues);

        addChildDataElement(extendedDataElement);

        return extendedDataElement;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#addChildWithDoubleValue(java.lang.String, double)
     */
    public IExtendedDataElement addChildWithDoubleValue(String name, double value) {

        IExtendedDataElement extendedDataElement = new ExtendedDataElementImpl();
        extendedDataElement.setName(name);
        extendedDataElement.setType(IExtendedDataElement.TYPE_DOUBLE_VALUE);
        extendedDataElement.setValues(new String[] { String.valueOf(value)});

        addChildDataElement(extendedDataElement);

        return extendedDataElement;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#addChildWithFloatArrayValue(java.lang.String, float[])
     */
    public IExtendedDataElement addChildWithFloatArrayValue(String name, float[] values) {

        IExtendedDataElement extendedDataElement = new ExtendedDataElementImpl();
        extendedDataElement.setName(name);
        extendedDataElement.setType(IExtendedDataElement.TYPE_FLOAT_ARRAY_VALUE);

        String[] stringValues = new String[values.length];

        for (int counter = 0; counter < stringValues.length; counter++) {
            stringValues[counter] = String.valueOf(values[counter]);
        }

        extendedDataElement.setValues(stringValues);

        addChildDataElement(extendedDataElement);

        return extendedDataElement;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#addChildWithFloatValue(java.lang.String, float)
     */
    public IExtendedDataElement addChildWithFloatValue(String name, float value) {

        IExtendedDataElement extendedDataElement = new ExtendedDataElementImpl();
        extendedDataElement.setName(name);
        extendedDataElement.setType(IExtendedDataElement.TYPE_FLOAT_VALUE);
        extendedDataElement.setValues(new String[] { String.valueOf(value)});

        addChildDataElement(extendedDataElement);

        return extendedDataElement;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#addChildWithHexValue(java.lang.String, java.lang.String)
     */
    public IExtendedDataElement addChildWithHexValue(String name, String value) {

        IExtendedDataElement extendedDataElement = new ExtendedDataElementImpl();
        extendedDataElement.setName(name);
        extendedDataElement.setType(IExtendedDataElement.TYPE_HEX_BINARY_VALUE);

        char[] hexValues = new char[value.length()];

        for (int counter = 0; counter < hexValues.length; counter++) {
            hexValues[counter] = value.charAt(counter);
        }

        extendedDataElement.setHexValues(hexValues);

        addChildDataElement(extendedDataElement);

        return extendedDataElement;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#addChildWithHexValueAsByteArray(java.lang.String, byte[])
     */
    public IExtendedDataElement addChildWithHexValueAsByteArray(String name, byte[] value) {

        IExtendedDataElement extendedDataElement = new ExtendedDataElementImpl();
        extendedDataElement.setName(name);
        extendedDataElement.setType(IExtendedDataElement.TYPE_HEX_BINARY_VALUE);

        char[] hexValues = new char[value.length];

        for (int counter = 0; counter < hexValues.length; counter++) {
            hexValues[counter] = ((char) ((value[counter] + 128)));
        }

        extendedDataElement.setHexValues(hexValues);

        addChildDataElement(extendedDataElement);

        return extendedDataElement;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#addChildWithIntArrayValue(java.lang.String, int[])
     */
    public IExtendedDataElement addChildWithIntArrayValue(String name, int[] values) {

        IExtendedDataElement extendedDataElement = new ExtendedDataElementImpl();
        extendedDataElement.setName(name);
        extendedDataElement.setType(IExtendedDataElement.TYPE_INT_ARRAY_VALUE);

        String[] stringValues = new String[values.length];

        for (int counter = 0; counter < stringValues.length; counter++) {
            stringValues[counter] = String.valueOf(values[counter]);
        }

        extendedDataElement.setValues(stringValues);

        addChildDataElement(extendedDataElement);

        return extendedDataElement;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#addChildWithIntValue(java.lang.String, int)
     */
    public IExtendedDataElement addChildWithIntValue(String name, int value) {

        IExtendedDataElement extendedDataElement = new ExtendedDataElementImpl();
        extendedDataElement.setName(name);
        extendedDataElement.setType(IExtendedDataElement.TYPE_INT_VALUE);
        extendedDataElement.setValues(new String[] { String.valueOf(value)});

        addChildDataElement(extendedDataElement);

        return extendedDataElement;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#addChildWithLongArrayValue(java.lang.String, long[])
     */
    public IExtendedDataElement addChildWithLongArrayValue(String name, long[] values) {

        IExtendedDataElement extendedDataElement = new ExtendedDataElementImpl();
        extendedDataElement.setName(name);
        extendedDataElement.setType(IExtendedDataElement.TYPE_LONG_ARRAY_VALUE);

        String[] stringValues = new String[values.length];

        for (int counter = 0; counter < stringValues.length; counter++) {
            stringValues[counter] = String.valueOf(values[counter]);
        }

        extendedDataElement.setValues(stringValues);

        addChildDataElement(extendedDataElement);

        return extendedDataElement;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#addChildWithLongValue(java.lang.String, long)
     */
    public IExtendedDataElement addChildWithLongValue(String name, long value) {

        IExtendedDataElement extendedDataElement = new ExtendedDataElementImpl();
        extendedDataElement.setName(name);
        extendedDataElement.setType(IExtendedDataElement.TYPE_LONG_VALUE);
        extendedDataElement.setValues(new String[] { String.valueOf(value)});

        addChildDataElement(extendedDataElement);

        return extendedDataElement;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#addChildWithShortArrayValue(java.lang.String, short[])
     */
    public IExtendedDataElement addChildWithShortArrayValue(String name, short[] values) {

        IExtendedDataElement extendedDataElement = new ExtendedDataElementImpl();
        extendedDataElement.setName(name);
        extendedDataElement.setType(IExtendedDataElement.TYPE_SHORT_ARRAY_VALUE);

        String[] stringValues = new String[values.length];

        for (int counter = 0; counter < stringValues.length; counter++) {
            stringValues[counter] = String.valueOf(values[counter]);
        }

        extendedDataElement.setValues(stringValues);

        addChildDataElement(extendedDataElement);

        return extendedDataElement;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#addChildWithShortValue(java.lang.String, short)
     */
    public IExtendedDataElement addChildWithShortValue(String name, short value) {

        IExtendedDataElement extendedDataElement = new ExtendedDataElementImpl();
        extendedDataElement.setName(name);
        extendedDataElement.setType(IExtendedDataElement.TYPE_SHORT_VALUE);
        extendedDataElement.setValues(new String[] { String.valueOf(value)});

        addChildDataElement(extendedDataElement);

        return extendedDataElement;
    }
}
