package org.eclipse.hyades.logging.events;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.ObjectStreamField;
import java.io.Serializable;

/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

/**
 * The StartSituation deals with the start up process for a component. 
 * Messages that indicate that a component has begun the startup process, 
 * that it has finished the startup process, or that it has aborted the 
 * startup process all fall into this category. Existing messages include 
 * words like starting, started, initializing, and initialized, for example:
 * 
 * <code>
 * DIA3206I The TCP/IP protocol support was started successfully.
 * DIA3000I "%1S" protocol support was successfully started.
 * DIA3001E "%1S" protocol support was not successfully started.
 * WSVR0037I: Starting EJB jar: {0}
 * </code>
 * 
 * 
 * @author Richard K. Duggan
 * @author Paul E. Slauenwhite
 * @version 1.0.1
 * @since 1.0.1
 * @see IStartSituation
 * @see ISituationType
 */
public class StartSituationImpl extends SituationTypeImpl implements IStartSituation, Serializable, Cloneable {

    //Define the serialiVersionUID for serialization:
    //NOTE: If this object is ever changed which breaks backward compatibility,
    //this value MUST be changed.  Every effort should be made not to break 
    //backward compatibility.
    static final long serialVersionUID = -3888096669772662967L;

    //Define the fields that will be serialized: 
    //NOTE: If any new variables are added to the class this table should be 
    //updated along with the readObject and writeObject methods.
    private static final ObjectStreamField[] serialPersistentFields = { new ObjectStreamField("successDisposition", String.class), new ObjectStreamField("situationQualifier", String.class)};

    //Class instance variable(s):
    protected String successDisposition = null;
    protected String situationQualifier = null;

    /**
     * No-argument protected constructor for StartSituationImpl.
     */
    protected StartSituationImpl() {
		super();
    }

    /**
     * Returns the successDisposition.
     * 
     * This property specifies the success disposition of an operation of a 
     * situation that caused the situation to be reported.  The successDisposition 
     * is a string with the following set of values:
     * 
     * SUCCESSFUL
     * UNSUCCESSFUL
     * 
     * This is a REQUIRED property and once it is set it MUST NOT change. 
     * The string length for this property MUST NOT exceed 64 characters.
     * 
     * @return String The returned successDisposition.
     * @since 1.0.1
     */
    public String getSuccessDisposition() {
        return successDisposition;
    }

    /**
     * Returns the situationQualifier.
     * 
     * This property specifies the situation qualifiers that are representation 
     * of the parameters necessary to describe the situation.
     * The situationQualifier is a string with the following set of values:
     * 
     * START INITIATED
     * RESTART INITIATED
     * START COMPLETED
     * 
     * This is a REQUIRED property and once it is set it MUST NOT change. 
     * The string length for this property MUST NOT exceed 64 characters.
     * 
     * @return String The returned situationQualifier.
     * @since 1.0.1
     */
    public String getSituationQualifier() {
        return situationQualifier;
    }

    /**
     * Sets the successDisposition.
     * 
     * This property specifies the success disposition of an operation of a 
     * situation that caused the situation to be reported.  The successDisposition 
     * is a string with the following set of values:
     * 
     * SUCCESSFUL
     * UNSUCCESSFUL
     * 
     * This is a REQUIRED property and once it is set it MUST NOT change. 
     * The string length for this property MUST NOT exceed 64 characters.
     * 
     * @param successDisposition The successDisposition to be set.
     * @since 1.0.1
     */
    public void setSuccessDisposition(String successDisposition) {
        this.successDisposition = successDisposition;
    }

    /**
     * Sets the situationQualifier.
     * 
     * This property specifies the situation qualifiers that are representation 
     * of the parameters necessary to describe the situation.
     * The situationQualifier is a string with the following set of values:
     * 
     * START INITIATED
     * RESTART INITIATED
     * START COMPLETED
     * 
     * This is a REQUIRED property and once it is set it MUST NOT change. 
     * The string length for this property MUST NOT exceed 64 characters.
     * 
     * @param situationQualifier The situationQualifier to be set.
     * @since 1.0.1
     */
    public void setSituationQualifier(String situationQualifier) {
        this.situationQualifier = situationQualifier;
    }

    /**
     * Overrides the <code>java.lang.Object</code>'s equals() API to 
     * determine if the parameter <code>object</code> is equal to this object.
     * 
     * @param object The <code>java.lang.Object</code> to be compared to this object.
     * @return true if this object is the same as the parameter <code>object</code>, false otherwise.
     * @see java.lang.Object#equals(java.lang.Object)
     */
    public boolean equals(Object object) {

        //Check if the parameter object's super class is equal to the this object's super class:
        if (super.equals(object)) {

            StartSituationImpl startSituation = ((StartSituationImpl) (object));

            //Compare the value(s) of all object properties:
            String thisObjectsProperty = getSituationQualifier();
            String parameterObjectsProperty = startSituation.getSituationQualifier();

            if (((thisObjectsProperty != null) && (thisObjectsProperty.equals(parameterObjectsProperty))) || ((thisObjectsProperty == null) && (parameterObjectsProperty == null))) {

                thisObjectsProperty = getSuccessDisposition();
                parameterObjectsProperty = startSituation.getSuccessDisposition();

                if (((thisObjectsProperty != null) && (thisObjectsProperty.equals(parameterObjectsProperty))) || ((thisObjectsProperty == null) && (parameterObjectsProperty == null)))
                    return true;
            }
        }

        return false;
    }

    private void writeObject(ObjectOutputStream out) throws IOException {

        //Set the values of the serializable field(s):
        ObjectOutputStream.PutField fields = out.putFields();

        fields.put("situationQualifier", situationQualifier);
        fields.put("successDisposition", successDisposition);

        //Write the values of the serializable field(s):
        out.writeFields();
    }

    private void readObject(ObjectInputStream in) throws IOException, ClassNotFoundException {

        //Prepare and read the alternate persistent field(s):
        ObjectInputStream.GetField fields = null;

        fields = in.readFields();

        //Retrieve the alternate persistent field(s):
		situationQualifier = ((String) (fields.get("situationQualifier", situationQualifier)));
		successDisposition = ((String) (fields.get("successDisposition", successDisposition)));
    }

    /**
     * Return a clone (copy by value) of this object. This is a deep copy version, 
     * in which all the objects within this object will be copied by value.
     *  
     * @return Object The clone of this object.
     */
    public Object clone() throws CloneNotSupportedException {
        return (super.clone());
    }

    /**
     * Resets the object's properties to their initial (e.g. null) state.
     * 
     * @see org.eclipse.hyades.logging.events.ISituationType#init()
     */
    public void init() {

        super.init();

        situationQualifier = null;
        successDisposition = null;
    }
}