/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.logging.adapter.sensors;
import java.io.BufferedReader;
import java.io.EOFException;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.RandomAccessFile;

import org.eclipse.hyades.logging.adapter.impl.Sensor;
import org.eclipse.hyades.logging.adapter.util.Messages;
import org.eclipse.hyades.logging.adapter.internal.util.AdapterSensor;
import org.eclipse.hyades.logging.adapter.AdapterInvalidConfig;
import org.w3c.dom.Element;
import org.w3c.dom.Node; 
import org.w3c.dom.NodeList;
/**
 * Sample Implementation of an OS File Sensor.
 * 
 * 
 * @version 0.1
 * @since 0.1
 * @see org.eclipse.hyades.logging.adapter.impl.Sensor
 * @see org.eclipse.hyades.logging.adapter.ISensor
 */
public class SingleOSFileSensor extends Sensor
{
	/**
	 * The relative or absolute directory containing the log file.
	 */
	private String directory = null;
	/**
	 * The name including extension of the log file.  
	 * 
	 * Note: This value may contain a regular expression.
	 */
	private String fileName = null;
	/**
	 * The full (e.g. absolute paths) command string of the converter application.  
	 */
	private String converterCommand = null;
	private String converterShell = null;
	private String rawLogFileName = null;
	private File inputFile = null;
	private FileReader rawLogIn = null;
	private BufferedPeriodicReader input = null;
	// allocate the max blocking size 
	private String[] lastLine = new String[maximumBlocking];
	/**
	 * Current thread lock for spawning child threads. 
	 */
	private Object currentThreadLock = new Object();
	/**
	 * Counter for tracking completed child threads. 
	 */
	private int childThreadsDone = 0;
	/**
	 * No-arguement constructor to create a SingleOSFileSensor.
	 */
	public SingleOSFileSensor()
	{
		super();
	}
	/**
	 * Stops the sensor from reading the log file.  All object 
	 * references are cleaned, file handles are closed and temporary 
	 * files deleted.
	 * 
	 * @see org.eclipse.hyades.logging.adapter.IComponent#stop()
	 */
	public void stop()
	{
		super.stop();
		clean();
	}
	/**
	 * Update the configuration based on the configuration Element.
	 * 
	 * @see org.eclipse.hyades.logging.adapter.IComponent#update()
	 */
	public void update() throws AdapterInvalidConfig
	{
		// first get the basic configuration set
		super.update();
		// maximumBlocking is set by the sensor config
		lastLine = new String[maximumBlocking];
		String directory = null;
		String fileName = null;
		String expiration = null;
		Element sensorInstance = null;
		Element element = getConfiguration();
		// This sensor is configured with a sub element of the config
		NodeList sensors = element.getChildNodes();
		for (int i = 0; i < sensors.getLength(); i++)
			if (sensors.item(i).getNodeType() == Node.ELEMENT_NODE)
				sensorInstance = (Element) sensors.item(i);
		if (sensorInstance.hasAttribute(Messages.getString("HyadesGAdirectoryAttributeName")))
			directory = sensorInstance.getAttribute(Messages.getString("HyadesGAdirectoryAttributeName"));
		if (sensorInstance.hasAttribute(Messages.getString("HyadesGAfileNameAttributeName")))
			fileName = sensorInstance.getAttribute(Messages.getString("HyadesGAfileNameAttributeName"));
		if (sensorInstance.hasAttribute(Messages.getString("HyadesGAexpirationAttributeName")))
			expiration = sensorInstance.getAttribute(Messages.getString("HyadesGAexpirationAttributeName"));
		setDirectory(directory);
		setFileName(fileName);
		
		String converterCmdAttribute =
			sensorInstance.getAttribute(Messages.getString("HyadesGAconverterCmdAttributeName"));
		if (converterCmdAttribute.length() > 0)
		{
			setConverterCommand(converterCmdAttribute);
		}

		if (sensorInstance.hasAttribute(Messages.getString("HyadesGAconverterShellAttributeName"))) {
			converterShell = sensorInstance.getAttribute(Messages.getString("HyadesGAconverterShellAttributeName"));
			if (converterShell.length() == 0)
				converterShell = null;
		}

	}
	/**
	 * 	simulates a getNext
	 * @see com.ibm.acad.general.sensor.ISensor#testGetNext()
	 */
	public Object[] testGetNext()
	{
		return testGetNextLine();
	}
	/**
	 * implements the testGetNext behaviour by returning a test string
	 * @return String[]
	 */
	public String[] testGetNextLine()
	{
		String[] lineArray = new String[2];
		lineArray[0] = "test string";
		return lineArray;
	}
	/**
	 * 	returns last read line
	* @see com.ibm.acad.general.sensor.ISensor#getNext()
	 */
	public Object[] getNext()
	{
		if (rawLogFileName == null)
		{
			/* If the directory does not end with the proper file
			 * separator then we need to add one.  RKD:  We probably
			 * need to make sure we convert incorrect path separators
			 * to the proper platform.
			 */
			String pathSeparator=System.getProperty("file.separator");
			if(getDirectory().endsWith(pathSeparator)) {
				rawLogFileName = getDirectory() + getFileName();
			}
			else {
				rawLogFileName = getDirectory() + pathSeparator + getFileName();
			}
			
			if(rawLogFileName != null)
			{
				
				try
				{
					input = new BufferedPeriodicReader(rawLogFileName,converterCommand, converterShell);
				}
				catch (Exception e)
				{
					log(e.toString(), AdapterSensor.CRITICAL_LEVEL);
				}    			
			}
		}
		if (input != null)
		{
			
			
			for (int i = 0; i < maximumBlocking; i++)
			{
				try
				{
					lastLine[i] = input.readLine();
				}
				catch (Exception e)
				{
					log(e.toString(), AdapterSensor.WARN_LEVEL);
				}
				if (lastLine[i] == null)
				{
					i = maximumBlocking;
				}
			}
		}
		if (lastLine[0] != null)
			return lastLine;
		else
			return null;
	}
	/**
	 * do all cleaning here
	 */
	public void clean()
	{
		try
		{
			if (input != null)
			{
				input.close();
			}
		}
		catch (Exception e)
		{
			log(e.toString(), AdapterSensor.HARMLESS_LEVEL);
		}
	}
	/**
	 * Returns the directory.
	 * @return String
	 */
	final public String getDirectory()
	{
		return directory;
	}
	/**
	 * Returns the fileName.
	 * @return String
	 */
	final public String getFileName()
	{
		return fileName;
	}
	/**
	 * Returns the converter command.
	 * @return String
	 */
	final public String getConverterCommand()
	{
		return converterCommand;
	}
	/**
	 * Sets the directory.
	 * @param directory The directory to set
	 */
	final public void setDirectory(String directory)
	{
		this.directory = directory;
	}
	/**
	 * Sets the fileName.
	 * @param fileName The fileName to set
	 */
	final public void setFileName(String fileName)
	{
		this.fileName = fileName;
	}
	/**
	 * Sets the converter command.
	 * @param converterCmd The converter command to set
	 */
	final public void setConverterCommand(String converterCmd)
	{
		this.converterCommand = converterCmd;
	}
	/**
	 * BufferedPeriodicReader reads a file line by line and populates the returned
	 * string array. It holds a pointer offset to the last read byte in order to resume.
	 */
	private class BufferedPeriodicReader
	{
		private String filename = null;
		private String converter = null;
		private String shell = null;
		private RandomAccessFile fileReader = null;
		long lastBookmark = 0;
		public BufferedPeriodicReader(String newFilename)
		{
			this(newFilename,null,null);
		}
		
		public BufferedPeriodicReader(String newFilename, String converter)
		{
			this(newFilename,converter,null);
		}

		public BufferedPeriodicReader(String newFilename, String converter, String shell)
		{
			
			this.filename = newFilename;
			this.converter = converter;
			this.shell = shell;
			
			prepareFile();
		}	
		/**
		 * prepare file for the reading
		 */
		public boolean prepareFile()
		{		
			//If the path to the converter is set, convert the file first:
			if (converter != null)
			{
				try
				{
					//Execute the converter command as a runtime process and monitor any stdout and stderr output:
					Process converterProcess;
					
					//PROBLEM: Because some native platforms only provide limited buffer size for standard input streams, failure to read the input stream of the process may cause the process to block, and even deadlock.
					//SOLUTION: Process input streams must be read on seperate threads.
					synchronized (currentThreadLock)
					{
						//Reset the counter for tracking completed child threads:
						childThreadsDone = 0;
						
						// Execute the converter command:
							
						String platform = System.getProperty("os.name", null);
					
						// Use the converter command alone on Windows
						if (platform.startsWith("Windows")) {
							converterProcess = Runtime.getRuntime().exec(converter);
						}
						// Run the converter in a shell on all other platforms
						else if (platform.startsWith("AIX")) {
							// For AIX we need to specify the command as an array of strings
							
							// If no shell is specified, use sh
							if (shell == null) {
								shell = "sh";
							}
							
							String [] cmdarr = {shell, converter};
							//Execute the converter command string array
							converterProcess = Runtime.getRuntime().exec(cmdarr);
						}
						else {
							// For all other non-Windows platforms we can run the command as one string
							
							// If no shell is specified, use sh
							if (shell == null) {
								shell = "sh";
							}
							// Add the shell to the converter command
							String converterCmd = shell + " " + converter;
							
							//Execute the converter command string
							converterProcess = Runtime.getRuntime().exec(converterCmd);
						}
						
						//Monitor any stdout and stderr output from the execution of the converter command:
						new ConverterOutputReader(converterProcess.getErrorStream()).start();
						new ConverterOutputReader(converterProcess.getInputStream()).start();

						//ASSUMPTION: The process is only terminated after all its streams (i.e. stdout and stderr) have terminated.
						//Wait for all of the stdout and stderr output to be read before continuing:
						try
						{
							//Wait no more than 5 minutes (300000 ms):
							currentThreadLock.wait(300000);
						}
						catch (InterruptedException e)
						{
						}
					}

					//Capture the exit value (by convention, the value 0 indicates normal termination) from the terminated converter process:
					int exitValue = -1;
					
					try
					{
						exitValue = converterProcess.waitFor();
					}
					catch (InterruptedException i)
					{
						//Destroy the converter process if it has not yet exited:
						converterProcess.destroy();
					}

					//Throw an exception is the process has not exited or exits abnormally (by convention, the value 0 indicates normal termination):
					if (exitValue != 0) {
						throw new Exception();
					}
				}
				catch (Exception e)
				{
					log(e.toString(), AdapterSensor.CRITICAL_LEVEL);
					return false;
				}
			}

			boolean fileIsUsable = false;
			try
			{
				fileReader = new RandomAccessFile(filename, "r");
			}
			catch (Exception e)
			{
				fileReader = null;
			}
			try
			{
				if (fileReader != null && fileReader.length() > lastBookmark)
					fileIsUsable = true;
			}
			catch (IOException e)
			{
				log(e.toString(), AdapterSensor.WARN_LEVEL);
			}
			return fileIsUsable;
		}
		/**
		 * close file
		 */
		public void close()
		{
			if (fileReader != null)
				try
				{
					fileReader.close();
				}
				catch (Exception e)
				{
				}
			fileReader = null;
		}
		/**
		 * 
		 * @return
		 * @throws IOException
		 */
		public String readLine() throws IOException
		{
			if (fileReader == null)
			{
				prepareFile();
				fileReader.seek(lastBookmark);
			}
			if (fileReader != null)
			{
				String last = null;
				fileReader.seek(lastBookmark);
				if (fileReader != null)
				{
					try
					{
						last = fileReader.readLine();
					}
					catch (EOFException e)
					{
						last = null;
					}
					catch (NullPointerException e)
					{
						last = null;
					}
					if (last != null)
					{
						lastBookmark = fileReader.getFilePointer();
						return last;
					}
					else
					{
						close();
					}
				}
			}
			
			return null;
		}
	}
	//PROBLEM: Because some native platforms only provide limited buffer size for standard input streams, failure to read the input stream of the process may cause the process to block, and even deadlock.
	//SOLUTION: Process input streams must be read on seperate threads.
	//Class used to read input streams from the converter process on seperate threads.
	class ConverterOutputReader extends Thread
	{
		private BufferedReader reader;
		
		public ConverterOutputReader(InputStream inputStream)
		{
			reader = new BufferedReader(new InputStreamReader(inputStream));
		}
		public void run()
		{
			String line;
			try
			{
				//Read the input stream until the stream is closed (i.e. null termination):
				while (true){
					line = reader.readLine();
					if(line == null){
						break;
					}
				}
			}
			catch (IOException e)
			{
			}
			
			//If both input streams (i.e. stdout and stderr) are closed, notify the waiting parent thread:
			synchronized (currentThreadLock)
			{
				if (++childThreadsDone == 2)
					currentThreadLock.notify();
			}
		}
	}
}
