/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.probekit;

import java.util.*;

import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.resource.*;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.eclipse.hyades.models.probekit.internal.*;

/**
 * This is the main Probekit compiler class. You feed it one or more resources
 * of type org.eclipse.hyades.models.probekit.Probekit and then extract
 * the artifacts necessary to use the probes: a string representing
 * the Java source for the class containing the probe fragments,
 * and a string representing the engine script that will instrument
 * class files based on those probes.
 *
 * An inner class ProbekitException is defined and can be thrown.
 * This wraps all exceptions that can occur inside the compiler,
 * including internal errors like NullPointerException.
 * The exception text string tells what really happened (in English only).
 * 
 * Use the Compiler class this way:
 *	1. Create an instance
 *		Use default constructor or convenience constructors
 *	2. Populate the instance
 *		a. addProbeKit()
 *		b. addFile()
 *		c. addResource()
 *	3. Optionally call verify() to catch certain probe errors in advance.
 *	   (It will run automatically in any case, but maybe you want to preflight
 *	   your probes.)
 *	4. Use getGeneratedSource() and getEngineScript() (in either order) to
 *	   get the Java source and engine script text, respectively.
 *
 * Usually that's the end, but you can add more resources if you want
 * and regenerate the sources.
 * 
 * @author apratt
 */

public class Compiler {
	//------------------------------- constructors ------------------------------
	public Compiler() {
		// State initializes to "dirty"
	}

	public Compiler(Probekit probekit) {
		probekits.add(probekit);
		// State initializes to "dirty"
	}

	/**
	 * Create a Compiler and add probekits found at the top level
	 * of the named file to it. See addFile.
	 * @param file the name of the file to load.
	 * @throws ProbekitException
	 */
	public Compiler(String file) throws ProbekitException {
		addFile(file);
		// State initializes to "dirty"
	}

	//------------------------------- parameter setters ------------------------------
	public void setClassPrefix(String prefix) {
		this.classPrefix = prefix;
		setDirtyState();
	}

	//------------------------------- exception class ------------------------------
	/**
	 * The exception class for probekit exceptions.
	 * All thrown exceptions are of this type, and are created with a string value which
	 * describes the problem. The string is in English. 
	 */
	public static class ProbekitException extends Exception {
		/**
		 * String constructor. Exceptions are always created with a string telling what's wrong.
		 * @param string A message (in English) telling what's wrong.
		 */
		public ProbekitException(String string) {
			super(string);
		}
	}

	//------------------------------- resource adders ------------------------------
	/**
	 * Add a probekit to the list of probekits this compiler will compile.
	 * @param probekit the Probekit to add
	 */
	public void addProbekit(Probekit probekit) {
		probekits.add(probekit);
	}

	/**
	 * Load the named file as a resource and add probekit objects found
	 * at the top level to this Compiler.
	 * 
	 * @param file the name of the file to load.
	 * @throws ProbekitException
	 */
	public void addFile(String file) throws ProbekitException {
		URI fileURI = URI.createFileURI(file);
		ResourceSet resourceSet = new ResourceSetImpl();
		Resource res = resourceSet.getResource(fileURI, true);
		if (res == null) {
			throw new ProbekitException("Failed to read resource from file " + file);
		}
		addResource(res);
	}
	
	/**
	 * Add probekits that appear at the top level of a resource 
	 * to the probekits this compiler will compile.
	 * 
	 * @param res the Resource to scan for probekits.
	 */
	public void addResource(Resource res) {
		Iterator iter = res.getContents().iterator();

		while (iter.hasNext()) {
			Object obj = iter.next();
			if (!(obj instanceof Probekit)) {
				// ignore things in the resource that aren't probekits
			}
			else {
				probekits.add(obj);
			}
		}		
		setDirtyState();
	}

	//------------------------------- verify ------------------------------
	/**
	 * Tests a probe set for correctness and consistency. Throws an exception if the probe set fails
	 * any tests. The tests tend to be for things that aren't already forbidden by the modeling.
	 * @throws ProbekitException if there is any inconsistency in the set. The exception string
	 * describes one or more problems encountered, separated by the host-specific newline separator.
	 * It's possible that not all problems will be reported - for instance, only one internal
	 * integrity failure for a given probe might get reported, even if it has more than one problem.
	 *
	 * This verification step permits the core implementation to have less internal
	 * consistency checking logic.
	 */
	public void verify() throws ProbekitException {
		// verify each probe's structure
		String errorString = new String();

		// Verify the class name prefix set by setClassPrefix
		if (!isValidJavaIdentifier(getOuterClassName())) {
			errorString += 
				"Class prefix \"" + classPrefix + "\"" +
				"results in an invalid Java identifier " +
				"\"" + getOuterClassName() + "\"" + 
				lineSeparator;
		}

		try {
			int count = 1;
			for (Iterator probekitIter = probekits.iterator(); probekitIter.hasNext(); ) {
				Probekit pk = (Probekit)probekitIter.next();
				if (pk.getProbe() == null) {
					throw new ProbekitException("Probe set includes a Probekit object with a null probe list");
				}
				for (Iterator iter = pk.getProbe().iterator(); iter.hasNext(); ) {
					Probe probe = (Probe)iter.next();
					try {
						if (probe == null) {
							throw new ProbekitException("Probe list includes a null probe reference");
						}
						verifyOneProbe(probe);
					}
					catch (ProbekitException e) {
						errorString += "Probe #" + count + ": " + e.getMessage() + lineSeparator;
					}
					++count;
				}
			}
		}
		catch (Throwable t) {
			throw new ProbekitException("Verification error(s): " + lineSeparator + 
				errorString + lineSeparator + 
				"Unexpected exception while verifying probe set: " + t);
		}
		
		if (errorString.length() != 0) {
			throw new ProbekitException("Verification error(s): " + lineSeparator + errorString);
		}
	}

	private void verifyOneProbe(Probe probe) throws ProbekitException {
		// These verification tests are implemented:
		// has a non-null fragment list (but the list may be empty)
		// no duplicate fragment types
		// no unrecognized fragment type names
		// no invalid combination of fragments (callsite vs. method)
		// no duplicate data items in a fragment
		// no unrecognized data items in a fragment
		// data items in all fragments are appropriate for that fragment
		// data item names are valid Java identifiers
		// all fragments should have some "code" (even an empty string - not null)
		// If there is a FragmentAtClassScope, its code should not be null
		// well formed imports (if they exist, string is not null)
		// well formed fragment at class scope (if it exists, string is not null)
		// well formed targets - no nulls, and only valid rule strings (include and exclude)
		// No controlKey - it's not implemented yet.
		// No invocationObject - it's mostly implemented but not tested.

		// TODO: These verification tests are not implemented:
		// no duplicate languages for names, descriptions, control names
		// only one of each type that can have "lang" is allowed NOT to have a "lang"
		// targets/filters that provably exclude everything
		// targets/filters with some "include" expressions and no "excludes" - clearly user error
		// if there's a controlKey, it should be a valid identifier
		// only one controlKey
		// invocation object name (if present) is a valid Java identifier
		// no two data items have the same name, or the name of the invocationObject

		// Note about missing strings: a "missing" string might be null,
		// or it might be a string with no characters in it.
		// In the current (December 2003) editor, if you never fill anything 
		// in you get null, but if you fill in something and then delete it all 
		// you get an empty string. All verification of strings must check for both.

		// verify fragment types and combinations
		boolean hasEntry = false;
		boolean hasExit = false;
		boolean hasCatch = false;
		boolean hasBeforeCall = false;
		boolean hasAfterCall = false;

		// verify imports
		if (probe.getImport() != null) {
			for (Iterator importIter = probe.getImport().iterator() ;
				 importIter.hasNext() ; ) 
			{
			 	Import imp = (Import)importIter.next();
			 	String impText = imp.getText();
			 	if (impText == null || impText.equals("")) throw new ProbekitException("An import has no text");
			 	// Hard to flag errors in imports, since we want to permit e.g. comments in them.
			 	// TODO: restrict imports to just package/class name patterns?
			}
		}

		// verify fragment at class scope
		if (probe.getFragmentAtClassScope() != null) {
			if (probe.getFragmentAtClassScope().getValue() == null) {
				throw new ProbekitException("The fragment at class scope has a null value");
			}
		}

		// Having no fragments is allowed - a probe with
		// nothing but a FragmentAtClassScope, for instance, is valid
		// (but useless, since that class will never be called).
		// This tests for a null list, not an empty one. A null list would cause
		// trouble (or require extra checks) in other code, so test for it here.
		if (probe.getFragment() == null) {
			throw new ProbekitException("The fragment list is null");
		}

		for (Iterator iter = probe.getFragment().iterator(); iter.hasNext(); ) {
			boolean isEntry = false;
			boolean isExit = false;
			boolean isCatch = false;
			boolean isBeforeCall = false;
			boolean isAfterCall = false;
			Fragment f = (Fragment)iter.next();
			FragmentEnum fenum = f.getType();
			if (fenum == null) throw new ProbekitException("A fragment type is null");
			String ftype = fenum.getName();
			if (ftype == null) throw new ProbekitException("A fragment type string is null");
			if (ftype.equals("entry")) {
				isEntry = true;
				if (hasEntry) throw new ProbekitException("only one entry fragment is allowed");
				hasEntry = true;
			}
			else if (ftype.equals("exit")) {
				isExit = true;
				if (hasExit) throw new ProbekitException("only one exit fragment is allowed");
				hasExit = true;
			}
			else if (ftype.equals("catch")) {
				isCatch = true;
				if (hasCatch) throw new ProbekitException("only one catch fragment is allowed");
				hasCatch = true;
			}
			else if (ftype.equals("beforeCall")) {
				isBeforeCall = true;
				if (hasBeforeCall) throw new ProbekitException("only one beforeCall fragment is allowed");
				hasBeforeCall = true;
			}
			else if (ftype.equals("afterCall")) {
				isAfterCall = true;
				if (hasAfterCall) throw new ProbekitException("only one afterCall fragment is allowed");
				hasAfterCall = true;
			}
			else {
				throw new ProbekitException("Unrecognized fragment type string " + ftype);
			}

			if ((hasEntry || hasExit || hasCatch) && (hasBeforeCall || hasAfterCall)) {
				throw new ProbekitException("Invalid fragment type combination: callsite and method types can not appear together");
			}

			// Verify data item types
			boolean hasClassName = false;
			boolean hasMethodName  = false;
			boolean hasMethodSig = false;
			boolean hasThisObject = false;
			boolean hasArgs = false;
			boolean hasReturnedObject = false;
			boolean hasExceptionObject = false;
			boolean hasIsFinally = false;

			if (f.getData() != null) {
				for (Iterator dataIter = f.getData().iterator(); dataIter.hasNext(); ) {
					Data data = (Data)dataIter.next();
					DataEnum denum = data.getType();
					if (denum == null) throw new ProbekitException("a data item type enum is null");
					String dtype = denum.getName();
					if (dtype == null) throw new ProbekitException("a data item has null type string");
					if (dtype.equals("className")) {
						if (hasClassName) throw new ProbekitException("Duplicate data item: className");
						hasClassName = true;
					}
					else if (dtype.equals("methodName")) {
						if (hasMethodName) throw new ProbekitException("Duplicate data item: methodName");
						hasMethodName = true;
					}
					else if (dtype.equals("methodSig")) {
						if (hasMethodSig) throw new ProbekitException("Duplicate data item: methodSig");
						hasMethodSig = true;
					}
					else if (dtype.equals("thisObject")) {
						if (hasThisObject) throw new ProbekitException("Duplicate data item: thisObject");
						hasThisObject = true;
					}
					else if (dtype.equals("args")) {
						if (hasArgs) throw new ProbekitException("Duplicate data item: args");
						hasArgs = true;
					}
					else if (dtype.equals("exceptionObject")) {
						if (hasExceptionObject) throw new ProbekitException("Duplicate data item: exceptionObject");
						hasExceptionObject = true;
						if (isEntry || isBeforeCall) throw new ProbekitException("Invalid data item: exceptionObject in entry/beforecall fragment");
					}
					else if (dtype.equals("returnedObject")) {
						if (hasReturnedObject) throw new ProbekitException("Duplicate data item: returnedObject");
						hasReturnedObject = true;
						if (isEntry || isBeforeCall || isCatch) throw new ProbekitException("Invalid data item: returnedObject in entry/beforecall/catch fragment");
					}
					else if (dtype.equals("isFinally")) {
						if (hasIsFinally ) throw new ProbekitException("Duplicate data item: isFinally");
						hasIsFinally = true;
						if (!isCatch) throw new ProbekitException("Invalid data item: isFinally in non-catch fragment");
					}
					else {
						throw new ProbekitException("Invalid data item type string: " + dtype);
					}
					if (!isValidJavaIdentifier(data.getName())) {
						throw new ProbekitException("Invalid Java identifier for data item: " + data.getName());
					}
				}
			}

			if (f.getCode() == null || 
				f.getCode().getValue() == null || 
				f.getCode().getValue().equals(""))
			{
				throw new ProbekitException("Invalid fragment: has no code");
			}
		}

		// Verify targets
		for (Iterator targetIter = probe.getTarget().iterator(); targetIter.hasNext() ; ) {
			Target targ = (Target)targetIter.next();
			String targType = targ.getType();
			if (targType == null || targType.equals("")) {
				throw new ProbekitException("Invalid target/filter: missing type string");
			}
			if (!(targType.equals("include") || targType.equals("exclude"))) {
				throw new ProbekitException("Unknown target type (not include or exclude): " + targ.getType());
			}
			// Note: null or empty wildcard pattern strings are allowed
			if (hasWhitespace(targ.getClassName()) ||
				hasWhitespace(targ.getMethod()) ||
				hasWhitespace(targ.getPackage()) ||
				hasWhitespace(targ.getSignature()))
			{
					throw new ProbekitException("A target wildcard pattern contains a space, tab, or other whitespace character");
			}
		}

		// TODO: remove this when controlKey is implemented and tested
		if (probe.getControlKey() != null) {
			throw new ProbekitException("Probe uses controlKey, which is not yet implemented");
		}

		// TODO: remove this when invocationObject is fully implemented and tested
		if (probe.getInvocationObject() != null) {
			throw new ProbekitException("Probe uses invocationObject, which is not yet implemented");
		}
	}

	/**
	 * Checks to see if a given string is a valid Java identifier.
	 * @param id the string to test
	 * @return true if it is valid, false if not
	 */
	boolean isValidJavaIdentifier(String id) {
		if (id.length() == 0) return false;
		if (!Character.isJavaIdentifierStart(id.charAt(0))) return false;
		for (int i = 1; i < id.length(); i++) {
			if (!Character.isJavaIdentifierPart(id.charAt(i))) {
				return false;
			}
		}
		return true;
	}

	/**
	 * Checks to see if the string has any whitespace characters in it.
	 * Returns true if so. Uses java.lang.Character.isWhitespace().
	 * Accepts null as a string to test, and returns false.
	 * @param s the string to test
	 * @return true if there are any whitespace characters in the string
	 */
	boolean hasWhitespace(String s) {
		if (s == null) return false;
		for (int i = 0; i < s.length(); i++) {
			if (java.lang.Character.isWhitespace(s.charAt(i))) {
				return true;
			}
		}
		return false;
	}
	
	//------------------------------- main functionality ------------------------------

	/**
	 * Get the generated source. If the source hasn't been generated yet, it will be.
	 * @return the generated source as a string.
	 */
	public String getGeneratedSource() throws ProbekitException {
		if (compilerState == STATE_DIRTY) {
			// Generate now. Throws exceptions if there are errors.
			generateSource();
		}
		// if we get here then generating went off without a hitch
		return generatedSource.toString();	
	}

	/**
	 * Get the script that describes this probe set to the BCI engine.
	 * 
	 * @return the string form of the description data
	 * @throws ProbekitException
	 */
	public String getEngineScript() throws ProbekitException {
		if (compilerState == STATE_DIRTY) {
			generateSource();
		}

		StringBuffer outBuffer = new StringBuffer();
		
		for (Iterator iter = probeDetailsMap.entrySet().iterator(); iter.hasNext() ; ) {
			Map.Entry entry = (Map.Entry)iter.next();
			Probe probe = (Probe)entry.getKey();
			ProbeDetails details = (ProbeDetails)entry.getValue();
			
			outBuffer.append("REM " + details.commentName + lineSeparator);
			outBuffer.append("PROBE" + lineSeparator);
			
			for (Iterator targetIter = probe.getTarget().iterator(); targetIter.hasNext(); ) {
				// Note: a "missing" wildcard might be null or it might be the empty string.
				// See verify() for an explanation.
				Target t = (Target)targetIter.next();
				String pkg = t.getPackage();
				if (pkg == null || pkg.equals("")) pkg = "*";
				String cls = t.getClassName();
				if (cls == null || cls.equals("")) cls = "*";
				String mth = t.getMethod();
				if (mth == null || mth.equals("")) mth = "*";
				String sig = t.getSignature();
				if (sig == null || sig.equals("")) sig = "*";
				outBuffer.append("RULE " + 
								 pkg + " " + 
								 cls + " " +
								 mth + " " +
								 sig + " " +
								 t.getType() + lineSeparator);
			}
			
			for (Iterator fragIter = probe.getFragment().iterator(); fragIter.hasNext(); ) {
				Fragment f = (Fragment)fragIter.next();
				outBuffer.append("REF ");

				// emit the type string corresopnding to the type
				String fragType = f.getType().getName();
				int fragTypeNumber = FragTypeNameToNumber(fragType);

				if (fragType.equals("entry")) {
					outBuffer.append("ONENTRY ");
				}
				else if (fragType.equals("exit")) {
					outBuffer.append("ONEXIT ");
				}
				else if (fragType.equals("catch")) {
					outBuffer.append("ONCATCH ");
				}
				else if (fragType.equals("beforeCall")) {
					outBuffer.append("BEFORECALL ");
				}
				else if (fragType.equals("afterCall")) {
					outBuffer.append("AFTERCALL ");
				}
				else {
					throw new ProbekitException("Unsupported fragment type; not entry, exit, or catch");
				}

				// Emit the internal-form class name for this fragment				
				// It's an inner class of getOuterClassName
				outBuffer.append(getOuterClassName() + "$" + details.uniqueClassName + " ");

				// Note: it's a bad internal error if any of these fragmentDetails arrays 
				// isn't appropriately populated (i.e. causes NullPointerException)

				// Emit the method name for this fragment
				outBuffer.append(details.fragmentDetails[fragTypeNumber].functionName + " ");

				// Emit the signature for this fragment
				outBuffer.append(details.fragmentDetails[fragTypeNumber].signature + " ");
				
				// Emit the list of arguments for this fragment
				outBuffer.append(details.fragmentDetails[fragTypeNumber].argumentList + lineSeparator);			
			}
		}
		return outBuffer.toString();
	}

	/**
	 * The main source code (and metadata) generation method of the Compiler. 
	 * The generated source is a compilable Java source file
	 * containing one class, whose name is PROBE_PACKAGE_NAME.classPrefix_probe.
	 * This function discards all previously-generated state of the Compiler, if any,
	 * and generates new metadata ("probe details") in parallel with the new source code.
	 * 
	 * The "probe details" object generated here includes the metadata that describes the
	 * probe set to the agent.
	 */
		
	private void generateSource() throws ProbekitException {
		// Verify the integrity of the probe and throw a descriptive error if it's bad
		verify();

		try {
			// Reset all accumulators and generated stuff,
			// and set the state to "dirty" now. This way
			// if we don't exit this function normally
			// it won't falsely be seen as "clean."
			controlKeysSet = new HashSet();
			generatedSource = new StringBuffer();
			probeDetailsMap = new HashMap();	// See TODO at declaration re IdentityHashMap
			nextSerialNumber = 0;
			compilerState = STATE_DIRTY;
	
			emitln("// generated source from ProbeKit compiler");
	
			// Loop through all probes and initialize the details map
			emitln("// List of probes in this probe set:");
			for (Iterator probekitIter = probekits.iterator(); probekitIter.hasNext(); ) {
				Probekit pk = (Probekit)probekitIter.next();
				for (Iterator iter = pk.getProbe().iterator(); iter.hasNext(); ) {
					Probe probe = (Probe)iter.next();
					ProbeDetails details = new ProbeDetails(probe);
					probeDetailsMap.put(probe, details);
					
					// Name this probe as being in this probe set
					emitln("//   " + details.commentName);
				}
			}
	
			// Note: we would emit "package foo;" here if the probe class went into
			// a package, but it does not. This might seem like bad form, but it
			// means we get to put the generated class files at the top level of
			// the jar file instead of putting them under subdirectories.

			emitln("// \"imports\" specifications for probes (if any):");
			emitImports();
	
			// Emit the outer class declaration
			emitln("class " + getOuterClassName() + " {");
			
			// Emit the field members of the outer class: the probe disable flags.
			// First, the global flag. TODO: make the global disable flag real, not a placeholder
			emitln("  public static boolean placeHolderForGlobalEnableFlag = true;");
			
			// Now, the per-probe control keys
			emitln("  // Probe control keys appear below (if any):");
			emitControlKeyDeclarations();
	
			// Generate one inner class per probe in the list of probekits.
			for (Iterator probekitIter = probekits.iterator(); probekitIter.hasNext(); ) {
				Probekit pk = (Probekit)probekitIter.next();
				for (Iterator iter = pk.getProbe().iterator(); iter.hasNext(); ) {
					Probe probe = (Probe)iter.next();
					emitOneProbeInnerClass(probe);
				}
			}
			emitln("}");	// closing brace for the outer class
			
			// Mark the state as "clean," ready to return results
			compilerState = STATE_CLEAN;
		}
		catch (RuntimeException e) {
			// Convert the runtime exception into a ProbekitException
			// so the caller sees our internal errors as Probekit errors.
			throw new ProbekitException(e.toString());
		}
	}

	//------------------------------- support and helper functions -------------------

	/**
	 * The probekits that have been added to this Compiler
	 */
	private List probekits = new LinkedList();

	/**
	 * The state of the compiler object. It's "dirty" until you
	 * generate code from a set of probekits, and then it's "clean"
	 * until you add more probekits.
	 */
	private int compilerState = STATE_DIRTY;
	private static final int STATE_DIRTY = 0;
	private static final int STATE_CLEAN = 1;
	
	/**
	 * The generated source, produced by the generateSource method.
	 * Methods that return this string will regenerate it if the
	 * compiler state is STATE_DIRTY;
	 */
	private StringBuffer generatedSource;

	/**
	 * Set the state to "dirty" and release any 
	 * previously generated source string.
	 */
	private void setDirtyState() {
		generatedSource = null;
		compilerState = STATE_DIRTY;
	}

	/**
	 * A private shortcut that emits its argument into the
	 * generatedSource buffer.
	 */
	private void emit(String s) {
		generatedSource.append(s);
	}
	
	/**
	 * A private shortcut that emits its argument and a newline into
	 * the generatedSource buffer. Uses the stored lineSeparator
	 * string rather than constantly calling System.getProperty.
	 */
	private void emitln(String s) {
		generatedSource.append(s);
		generatedSource.append(lineSeparator);
	}

	/**
	 * A convenience variable so we don't have to call
	 * System.getProperty all the time.
	 */
	static private final String lineSeparator = System.getProperty("line.separator");

	/**
	 * The prefix to use on generated class names.
	 * If the caller doesn't set one with setClassPrefix, it's empty.
	 */
	private String classPrefix = "";
	
	/**
	 * Returns the outer class name that encloses all fragment classes
	 * @return the outer class name
	 */
	private String getOuterClassName() { 
		return classPrefix + "_probe";
	}

	// ------------------------------------------------------------------------
	// DataItemDesc system, describing data items.

	/**
	 * Holds a single data item description. This structure maps
	 * data item type strings that appear in the probekit file (like
	 * "className" and "isFinally") to the Java source code data
	 * types of the corresponding parameters and variables (String
	 * and boolean), and also the internal signature strings
	 * (Ljava/lang/String and Z).
	 */
	static private class DataItemDesc {
		public DataItemDesc(String n, String t, String s) {
			typeName = n;
			typeType = t;
			typeSignature = s;
		}
		public String typeName;
		public String typeType;
		public String typeSignature;
	};

	/**
	 * A static array of descriptors for each type of data item, in
	 * the canonical order. 
	 *
	 * @see DataItemDesc
	 * @see "the Compiler class static constructor"
	 * @see "the BCI engine's canonical order, which this must match"
	 */	
	static private DataItemDesc[] dataItemDescs;
	
	/**
	 * Static constructor for the Compiler class: populates the dataItemDescs array.
	 *
	 * IMPORTANT: This is the "canonical order" for arguments. It is the order
	 * they'll appear in the Java function signature, so it also has to be the
	 * order used by the BCI engine when it passes arguments.
	 *
	 * Implementation details of the BCI engine require that returnedObject and
	 * exceptionObject be the first and second arguments, respectively.
	 * See comments in the BCI engine at CProbeRef::PushArguments about this,
	 * and the ordering of the enum bitmap in class CProbeRef
	 */
	{
		// IMPORTANT: this is the "canonical order" and must match the order
		// that the BCI engine will use when emitting code to pass these arguments.
		dataItemDescs = new DataItemDesc[8];
		dataItemDescs[0] = new DataItemDesc("returnedObject", "Object", "Ljava/lang/Object;");
		dataItemDescs[1] = new DataItemDesc("exceptionObject", "Throwable", "Ljava/lang/Throwable;"); 
		dataItemDescs[2] = new DataItemDesc("className", "String", "Ljava/lang/String;");
		dataItemDescs[3] = new DataItemDesc("methodName", "String", "Ljava/lang/String;");
		dataItemDescs[4] = new DataItemDesc("methodSig", "String", "Ljava/lang/String;");
		dataItemDescs[5] = new DataItemDesc("thisObject", "Object", "Ljava/lang/Object;");
		dataItemDescs[6] = new DataItemDesc("args", "Object[]", "[Ljava/lang/Object;");
		dataItemDescs[7] = new DataItemDesc("isFinally", "boolean", "Z");
	}

	//------------------- Constant strings for names of things -----------------
	/**
	 * The prefix for the names of generated probe inner classes. The actual classes will
	 * have a trailing number to make their names unique. 
	 */
	static final String PROBE_INNER_CLASS_NAME_PREFIX = "Probe_";

	/**
	 * Type signature string for the parameter type and return type of the invocationObject.
	 * TODO: the invocationObject system is implmented but not tested.
	 */
	static final String INVOCATION_OBJECT_TYPE_SIGNATURE = "Ljava/lang/Object;";

	/**
	 * Holds the computed details about a probe, like its unique name,
	 * the signatures of the fragment functions, and convenience flags and attributes.
	 * The information about a probe is developed over time - it's not complete
	 * until generateSource finishes.
	 * 
	 * The details class contains an array to hold details about each fragment.
	 */
	public static class ProbeDetails {
		public ProbeDetails(Probe probe) {
			// Initialize the "comment name" of the probe.
			// This is the first name string regardless of language.
			// It will appear as the probe's name in comments in generated code,
			// and in exception error messages.
			if (probe.getName() == null || probe.getName().size() == 0) {
				commentName = "unnamed_probe";
			}
			else {
				commentName = ((Name)(probe.getName().get(0))).getText();
			}
		}
		
		public String commentName = "";
		public String uniqueClassName = "";
		public boolean hasInvocationObject = false;

		static class FragmentDetails {
			public String functionName;
			public String signature;
			public String argumentList;
		}
		public FragmentDetails[] fragmentDetails = new FragmentDetails[FRAGMENT_TYPE_COUNT];
	}

	/**
	 * This constant must equal the number of fragment types. The details arrays are
	 * allocated to this size (see class ProbeDetails) and the FragTypeNameToNumber
	 * function had better return a value 0 <= x < FRAGMENT_TYPE_COUNT.
	 */	
	static final int FRAGMENT_TYPE_COUNT = 5;
	
	/**
	 * Convert a fragment type name to a number. This number is used as
	 * the index into the fragmentDetails array in the ProbeDetails object.
	 * It's internal only - there isn't any restriction on the ordering.
	 * I am not using DataEnum.getValue() because I don't want to couple those
	 * enum values to this code in any way.
	 */
	private static int FragTypeNameToNumber(String name) throws ProbekitException {
		if (name.equals("entry")) {
			return 0;
		}
		else if (name.equals("exit")) {
			return 1;
		}
		else if (name.equals("catch")) {
			return 2;
		}
		else if (name.equals("beforeCall")) {
			return 3;
		}
		else if (name.equals("afterCall")) {
			return 4;
		}
		else {
			throw new ProbekitException("Bad fragment type name: " + name);
		}
	}
	
	/**
	 * This map collects the ProbeDetails of each generated probe.
	 * The key is the Probe, the value is the ProbeDetails.
	 *
	 * TODO: make this IdentityHashMap when I can use 1.4-only features.
	 */
	private Map probeDetailsMap;
	
	/**
	 * The set of ControlKey names used by the probes in this probe set.
	 * This set is part of the output of the compiler, since the GUI
	 * and the BCI engine will want to know all their names. 
	 */
	private Set controlKeysSet;

	//----------------------- source generator helper methods ------------------------
	
	/**
	 * Goes through each probe in the probekit and emits its import statement. 
	 */
	private void emitImports() throws ProbekitException {
		for (Iterator probekitIter = probekits.iterator(); probekitIter.hasNext(); ) {
			Probekit probekit = (Probekit)probekitIter.next();
			for (Iterator probeIter = probekit.getProbe().iterator() ; probeIter.hasNext() ; ) {
				Probe probe = (Probe)probeIter.next();
				if (probe.getImport() != null) {
					Object temp_o = probeDetailsMap.get(probe);
					if (temp_o == null) {
						throw new ProbekitException("Internal error: failed to find details for probe");
					}
					ProbeDetails details = (ProbeDetails)temp_o;
					for (Iterator importIter = probe.getImport().iterator(); importIter.hasNext() ; ) {
						Import imp = (Import)importIter.next();
						emitln("import " + imp.getText() + "; // from " + details.commentName);
					}
				}
			}
		}
	}

	/**
	 * A descriptor for a controlKey, the enabling flag for a probe.
	 * Holds the key name and its initial value (true or false).
	 * Overrides "equals" to compare just the key name.
	 */
	class ControlKeyDescriptor {
		String key;
		String initialValue;
		ControlKeyDescriptor(String k, String iv) {
			key = k;
			initialValue = iv;
		}
		
		/**
		 * Override for Object.equals: compare just the key name.
		 * The descriptor goes into a map that should have just one entry per key.
		 */
		public boolean equals(Object other) {
			if (other instanceof ControlKeyDescriptor) {
				return key.equals(((ControlKeyDescriptor)other).key);
			}
			else return false; 
		}

	}
	
	/**
	 * Goes through each probe in the probekit and collects its control key.
	 * Emits static variable declarations for the unique keys.
	 * Emits "true" as the default value for each key.
	 * Adds each key to the controlKeys collection, so it can be
	 * passed back out as part of the metadata for this probe set.
	 * TODO: emit false or a default taken from the probe resource.
	 */
	private void emitControlKeyDeclarations() throws ProbekitException {		
		for (Iterator probekitIter = probekits.iterator(); probekitIter.hasNext(); ) {
			Probekit probekit = (Probekit)probekitIter.next();
			for (Iterator probeIter = probekit.getProbe().iterator() ; probeIter.hasNext() ; ) {
				Probe probe = (Probe)probeIter.next();
				Object temp_o = probeDetailsMap.get(probe);
				if (temp_o == null) {
					throw new ProbekitException("Internal error: failed to find details for probe");
				}
				ProbeDetails details = (ProbeDetails)temp_o;
				if (probe.getControlKey() != null) {
					String key = probe.getControlKey().getName();
					controlKeysSet.add(key);
					emitln("  // key " + key + " is used by " + details.commentName);
				}
			}
		}

		// TODO: control keys default to TRUE now, should be FALSE for real
		for (Iterator i = controlKeysSet.iterator(); i.hasNext() ; ) {
			String key = (String)i.next();			
			emitln("  public static boolean " + key + " = true;");
		}
	}

	/**
	 * A counter used by getNextProbeClassName to keep the names unique 
	 */
	private int nextSerialNumber;

	/**
	 * Using nextSerialNumber, generate the next unique class name for a probe.
	 * Called from emitOneProbeInnerClass.
	 * @return the class name, unique within this probe set
	 */
	private String getNextProbeClassName() {
		String s = PROBE_INNER_CLASS_NAME_PREFIX + nextSerialNumber;
		nextSerialNumber++;
		return s;
	}

	/**
	 * Emit the probe inner class for one probe. This inner class has member functions
	 * for each fragment the probe defines. A unique name is generated for the
	 * inner class, and this name, the fragment method signatures, and other
	 * metadata are saved as part of the "details" for this probe.
	 * 
	 * @param probe the probe to generate the inner class for.
	 * @throws ProbekitException
	 */
	private void emitOneProbeInnerClass(Probe probe) throws ProbekitException {
		// Get the details for this probe
		Object temp_o = probeDetailsMap.get(probe);
		if (temp_o == null) {
			throw new ProbekitException("Internal error: failed to find details for probe");
		}
		ProbeDetails details = (ProbeDetails)temp_o;
		
		emitln("  // Class for probe " + details.commentName);

		// Compute the inner class name for this probe and remember it in details		
		details.uniqueClassName = getNextProbeClassName();
		
		// Emit the inner class declaration for this probe's class
		emitln("  public static class " + 	details.uniqueClassName + " {");

		// Emit the "fragment at class scope"

		FragmentAtClassScope facs = probe.getFragmentAtClassScope(); 
		if (facs != null) {
			emitln("    // Fragment at class scope");
			emitln(facs.getValue());
		}
	
		// Emit the fragments. Each fragment is a function with its own name (entry, catch, exit).
		// No error checking is done here for malformed probes, like two "entry" fragments,
		// or an entry fragment with "isFinally" as one of the arguments.

		for (Iterator fragIter = probe.getFragment().iterator(); fragIter.hasNext(); ) {
			Fragment frag = (Fragment)fragIter.next();
			String fragType = frag.getType().getName();
			int fragTypeNumber = FragTypeNameToNumber(fragType);
			
			String funcName = "_" + fragType;	// prepend underscore so "catch" isn't a keyword
			details.fragmentDetails[fragTypeNumber] = new ProbeDetails.FragmentDetails();
			details.fragmentDetails[fragTypeNumber].functionName = funcName;

			// Compute the return type of the probe: void for most, but Object
			// for entry probes that ask to have an invocationObject.
			String returnType;
			String returnSignature;
			if (fragType.equals("entry") && details.hasInvocationObject) {
				returnType = "Object";
				returnSignature = INVOCATION_OBJECT_TYPE_SIGNATURE;
			}
			else {
				returnType = "void";
				returnSignature = "V";
			}

			// Emit the function declaration
			emitln("    public static " + returnType + " " + funcName + " (");

			// Crank through the desired data items and emit them as parameter declarations.
			// Emit them in the canonical order. This means going through
			// the dataItemDescs array in its order, and looking for
			// a data item with each corresponding name in the fragment.
			// This way the user can specify them in any order.
			// See comments near "class DataItemDesc" and the static
			// initializer that populates it for important details of ordering. 
			//
			// In parallel, build the internal form of the fragment function signature.
		
			boolean first = true;
			details.fragmentDetails[fragTypeNumber].signature = "(";
			details.fragmentDetails[fragTypeNumber].argumentList = new String();
			for (int j = 0; j < dataItemDescs.length; j++) {
				// Look through the fragment's data items for one with a matching string
				for (Iterator dataIter = frag.getData().iterator(); dataIter.hasNext(); ) {
					Data data = (Data)dataIter.next();
					if (data.getType().getName().equals(dataItemDescs[j].typeName)) {
						String paramName = data.getName();
						if (!first) emitln(","); 
						emit("      " + dataItemDescs[j].typeType + " /*" + dataItemDescs[j].typeName + "*/ " + paramName);
						details.fragmentDetails[fragTypeNumber].signature += dataItemDescs[j].typeSignature;
						if (!first) {
							details.fragmentDetails[fragTypeNumber].argumentList += ",";
						}
						details.fragmentDetails[fragTypeNumber].argumentList += dataItemDescs[j].typeName;
						first = false;
					}
				}
			}

			// Emit the invocation object parameter if any, for "catch" or "exit"
			if (details.hasInvocationObject && 
				(fragType.equals("catch") || fragType.equals("exit"))) 
			{
				if (!first) emitln(",");
//				emit("      Object " + probe.getInvocationObject().getName()); // TODO: use this instead of the placeholder
				emit("      Object placeholder_for_invocation_object_name");
				details.fragmentDetails[fragTypeNumber].signature += INVOCATION_OBJECT_TYPE_SIGNATURE;
			}

			// Close paren for the signature, open brace for the body
			emitln("      ) {");
			
			// Append the return type to the internal signature
			details.fragmentDetails[fragTypeNumber].signature += ")" + returnSignature;
			emitln("      // Internal signature for this method: " + details.fragmentDetails[fragTypeNumber].signature);

			// Emit global control flag test logic
			emitln("      if (!placeHolderForGlobalEnableFlag)");
			if (details.hasInvocationObject) {
				emitln("        return null;");
			}
			else {
				emitln("        return;");
			}
								
			// Emit the control key test logic if this probe has a custom key.
			if (probe.getControlKey() != null) {
				emitln("      if (!" + probe.getControlKey().getName() + ") return;");					
			}

			// Emit the local variable declaration for the invocation object (ENTRY fragments only)
			if (details.hasInvocationObject && fragType.equals("entry")) {
//				emitln("      Object " + frag.getInvocationObject().getName());	TODO: use this not placeholder
				emitln("      Object placeholder_for_invocation_object_name;");
			}
			
			// Emit the user-written source code for this fragment.
			emitln("//------------------ begin user-written fragment code ----------------");
			emitln(frag.getCode().getValue());
			emitln("//------------------- end user-written fragment code -----------------");

			if (details.hasInvocationObject && fragType.equals("entry")) {
//				emitln("      return " + frag.getInvocationObject().getName());	TODO: use this not placeholder
				emitln("      return placeholder_for_invocation_object_name;");
			}
					
			// Close brace for this function
			emitln("    }");
		}
		// Close brace for this class
		emitln("  }");
	}
}

