/*******************************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.ui;

import java.net.URL;
import java.util.MissingResourceException;
import java.util.ResourceBundle;

import org.eclipse.core.runtime.*;
import org.eclipse.hyades.ui.adapter.HyadesAdapterFactory;
import org.eclipse.hyades.ui.internal.logicalfolder.LogicalFolder;
import org.eclipse.hyades.ui.internal.util.ResourceBundleManager;
import org.eclipse.hyades.ui.util.ILogger;
import org.eclipse.ui.plugin.AbstractUIPlugin;

/**
 * Hyades UI Plugin class.  This class should not be used outside the context of
 * Eclipse.
 * 
 * @author marcelop
 * @author popescu
 * @since 0.0.1
 */
public class HyadesUIPlugin 
extends AbstractUIPlugin
{
	/**
	 * Extension point id used to register the editor extensions.
	 */
	public static final String EP_EDITOR_EXTENSIONS = "editorExtensions";

	/**
	 * Extension point id used to register the analyzer extensions.
	 */
	public static final String EP_ANALYZER_EXTENSIONS = "analyzerExtensions";
	
	/**
	 * Extension point id used to register the report extensions.
	 */
	public static final String EP_REPORT_EXTENSIONS = "reportExtensions";
	
	/**
	 * Extension point id used to register the type descriptions.
	 */
	public static final String EP_TYPE_DESCRIPTIONS = "typeDescriptions";
	
	/**
	 * Extension point id used to register the type validators.
	 */
	public static final String EP_TYPE_VALIDATORS = "typeValidators";
	

	private static HyadesUIPlugin instance;
	private ILogger logger;
	private ResourceBundleManager resourceBundleManager;
	
	/**
	 * Constructor for HyadesUIPlugin
	 * @param descriptor
	 */
	public HyadesUIPlugin(IPluginDescriptor descriptor)
	{
		super(descriptor);
		instance = this;
	}
	
	/**
	 * Returns the instance of this class created by the eclipse framework.
	 * @return HyadesUIPlugin
	 */
	public static HyadesUIPlugin getInstance()
	{
		return instance;
	}
	
	/**
	 * @see org.eclipse.core.runtime.Plugin#startup()
	 */
	public void startup()
	throws CoreException
	{
		resourceBundleManager = new ResourceBundleManager(); 
		resourceBundleManager.add(getDescriptor().getResourceBundle());

		Platform.getAdapterManager().registerAdapters(HyadesAdapterFactory.INSTANCE, LogicalFolder.class);

		super.startup();		

		try
		{
			HyadesUIImages.INSTANCE.initialize(new URL(getDescriptor().getInstallURL(), "icons/full/"), getImageRegistry());
		}
		catch(Exception e)
		{
				logError(e);
		}		
	}

	/**
	 * @see org.eclipse.core.runtime.Plugin#shutdown()
	 */
	public void shutdown()
	throws CoreException
	{
		resourceBundleManager.dispose();
		instance = null;
		savePluginPreferences();
		logger = null;
		Platform.getAdapterManager().unregisterAdapters(HyadesAdapterFactory.INSTANCE);
		HyadesAdapterFactory.INSTANCE.dispose();
				
		super.shutdown();
	}
	
	/**
	 * Returns the an instance of {@link ILogger} that used the
	 * <code>log</code> methods of this class.
	 * @return ILogger
	 */
	public static ILogger getLogger()
	{
		if(instance.logger == null)
		{
			instance.logger = new ILogger()
			{
				public void logError(Throwable throwable)
				{
					logError(throwable);
				}

				public void logError(String text)
				{
					logError(text);
				}

				public void logInfo(String text)
				{
					logInfo(text);
				}
			};
		}
		return instance.logger;
	}

	/**
	 * Returns this plugin's id.
	 * @return String
	 */
	public static String getID()
	{
		return instance.getDescriptor().getUniqueIdentifier();
	}
	
	/**
	 * Logs an error described by a throwable.
	 * 
	 * <p>This method should be used whenever a class in this plugin
	 * has to log an error since it adheres to the global logging
	 * strategy.
	 * 
	 * @param throwable
	 */
	public static void logError(Throwable throwable)
	{
		Status status = new Status(1, getID(), 0, throwable.toString(), throwable);
		getInstance().getLog().log(status);
	}
	
	/**
	 * Logs an error described by a text.
	 * 
	 * <p>This method should be whenever a class in this plugin
	 * has to log an error since it adheres to the global logging
	 * strategy.
	 * 
	 * @param text
	 */
	public static void logError(String text)
	{
		logError(new Throwable(text));
	}
	
	/**
	 * Logs an information described by a text.
	 * 
	 * <p>This method should be whenever a class in this plugin
	 * has to log an information since it adheres to the global logging
	 * strategy.
	 * 
	 * @param text
	 */
	public static void logInfo(String text)
	{
		Status status = new Status(3, getID(), 0, text, new Throwable(text));
		getInstance().getLog().log(status);
	}
	
	/**
	 * Returns the resource bundle used by this plugin.
	 * 
	 * <p>IMPORTANT: Don't use this method to retrieve values from the
	 * resource bundle.  For this purpose use the static <code>getString()</code> 
	 * defined in this class.
	 * 
	 * <p>This method is provided so this resource bundle can
	 * be used as the parent of another resource bundle.
	 * 
	 * @return ResourceBundle
	 */
	public ResourceBundle getResourceBundle()
	{
		return getDescriptor().getResourceBundle();
	}

	/**
	 * Returns the "plugin.properties" file's value associate to a given key.
	 * @param key
	 * @return String
	 * @throws java.util.MissingResourceException if the key is not in the file
	 * @throws NullPointerException if key is null
	 */
	public static String getString(String key)
	throws NullPointerException, MissingResourceException
	{
		return instance.resourceBundleManager.getString(key);
	}
	
	/**
	 * Returns the string value associate to a given key.  The key is passed to
	 * each resource bundle in the order they are retrieve by the 
	 * {@link #iterator()} method.
	 * 
	 * <p>The <code>arg</code> string defined replaces the %1 
	 * variable defined in the file's values.
	 * 
	 * <p>Example: If the value associated to the key <code>"a"</code> is 
	 * <code>"%0 %1 %2 %3 %4"</code> and arg is <code>"v1"</code>,
	 * the return of this method is <code>"%0 v1  %2 %3 %4"</code>.
	 * 
	 * @param key
	 * @param arg
	 * @return String
	 * @throws java.util.MissingResourceException if the key is not in the file
	 * @throws NullPointerException if key is null
	 */
	public static String getString(String key, String arg)
	throws NullPointerException, MissingResourceException
	{
		return instance.resourceBundleManager.getString(key, arg);
	}	

	/**
	 * Returns the "plugin.properties" file's value associate to a given key.
	 * 
	 * <p>The strings defined in <code>args</code> replaces the %n (where n>=1) 
	 * variables defined in the file's values.
	 * 
	 * <p>Example: If the value associated to the key <code>"a"</code> is 
	 * <code>"%0 %1 %2 %3 %4"</code> and args is <code>{"v1", null, "v3"}</code>,
	 * the return of this method is <code>"%0 v1  v3 %4"</code>.
	 * 
	 * @param key
	 * @param args
	 * @return String
	 * @throws java.util.MissingResourceException if the key is not in the file
	 * @throws NullPointerException if key is null
	 */
	public static String getString(String key, String[]args)
	throws NullPointerException, MissingResourceException
	{
		return instance.resourceBundleManager.getString(key, args);
	}
}
