/*******************************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.internal.model;

import java.io.IOException;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.ResourcesPlugin;

import org.eclipse.emf.common.notify.Adapter;
import org.eclipse.emf.common.notify.Notifier;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.emf.ecore.xmi.XMLResource;

import org.eclipse.hyades.test.ui.TestUIPlugin;
import org.eclipse.hyades.ui.internal.util.FileUtil;

/**
 * Contains generic utility methods to work with EMF objects.
 * 
 * @author marcelop
 * @since 0.2.0
 */
public class EMFUtil
{
	public static final Map RESOURCE_OPTIONS = new HashMap();
	{
		RESOURCE_OPTIONS.put(XMLResource.OPTION_DECLARE_XML, Boolean.TRUE);
		RESOURCE_OPTIONS.put(XMLResource.OPTION_SKIP_ESCAPE, Boolean.FALSE);
	}
	
	/**
	 * Saves a resource.
	 * @param resource
	 * @param progressMonitor
	 * @throws IllegalArgumentException if the resource is null
	 * @throws Exception if an exception happens during the saving attempt.
	 */
	public static void save(Resource resource)
	throws Exception
	{
		if(resource == null)
			throw new IllegalArgumentException(TestUIPlugin.getString("_ERROR_ARG_NULL", "resource"));
		
		boolean isModified = resource.isModified();
		try
		{
			resource.save(RESOURCE_OPTIONS);
		}
		catch(Exception e)
		{
			resource.setModified(isModified);
			throw e;
		}
	}
	
	/**
	 * Returns an EObject from the resource set that has the same URI fragment or
	 * id of the specifie eObject.  If the eObject is not in the resource set or can't
	 * be loaded, this method returns <code>null</code>.
	 * 
	 * @param resourceSet
	 * @param eObject
	 * @param load
	 * @return EObject
	 */
	public static EObject getEObject(ResourceSet resourceSet, EObject eObject, boolean load)
	{
		if((resourceSet == null) || (eObject == null))
			return null;
		
		Resource resource = eObject.eResource();
		if(resource == null)
			return null;
					
		if(resourceSet.equals(resource.getResourceSet()))
			return eObject;
			
		String uriFragment = resource.getURIFragment(eObject);
		resource = resourceSet.getResource(resource.getURI(), load);
		if(resource == null)
			return null;
					
		eObject = resource.getEObject(uriFragment);
		if(eObject == null)
			return null;
				
		return eObject;			
	}
	
	/**
	 * Returns the EObjects in the specified file from the Resource Set without 
	 * loading it.
	 * 
	 * <p>If the resource set is <code>null</code> or if the file is not loaded
	 * into it, this method returns an empty array.
	 * 
	 * @param resourceSet 
	 * @param file
	 * @return EObject[]
	 */
	public static EObject[] getEObjects(ResourceSet resourceSet, IFile file)
	{
		if(file == null)
			return new EObject[0];

		return getEObjects(resourceSet, file.getFullPath().toString());
	}

	/**
	 * Returns the EObjects in the specified file paht from the Resource Set 
	 * without loading it.
	 * 
	 * <p>If the resource set is <code>null</code> or if the file is not loaded
	 * into it, this method returns an empty array.
	 * 
	 * @param resourceSet 
	 * @param file
	 * @return EObject[]
	 */
	public static EObject[] getEObjects(ResourceSet resourceSet, String fullPath)
	{
		if((resourceSet == null) && (fullPath == null))
			return new EObject[0];

		Resource resource = resourceSet.getResource(URI.createPlatformResourceURI(fullPath), false);
		if(resource == null)
			return new EObject[0];
			
		return (EObject[])resource.getContents().toArray(new EObject[resource.getContents().size()]);
	}

	/**
	 * Loads the specified file returning all the root EObjects.  If the
	 * resource set argument is <code>null</code> then this method tries
	 * to find a registered resource factory.
	 * 
	 * <p>If the load attempt fails this method adds an entry to the
	 * log and returns an empty array.
	 * 
	 * @param resourceSet 
	 * @param file
	 * @return EObject[]
	 */
	public static EObject[] load(ResourceSet resourceSet, IFile file)
	{
		if(file == null)
			return new EObject[0];

		return load(resourceSet, file.getFullPath().toString());
	}
	
	/**
	 * Loads the specified file returning all the root EObjects.  If the
	 * resource set argument is <code>null</code> then this method tries
	 * to find a registered resource factory.
	 * 
	 * <p>If the load attempt fails this method adds an entry to the
	 * log and returns an empty array.
	 * 
	 * @param resourceSet 
	 * @param fullPath
	 * @return EObject[]
	 */
	public static EObject[] load(ResourceSet resourceSet, String fullPath)
	{
		if(fullPath == null)
			return new EObject[0];
			
		URI uri = URI.createPlatformResourceURI(fullPath);
		Resource resource = null;
		try
		{
			if(resourceSet != null)
			{
				resource = resourceSet.getResource(uri, true);
			}
			else
			{
				Resource.Factory factory = getResourceFactory(FileUtil.getFileExtension(fullPath));
				if(factory != null)
				{
					resource = factory.createResource(uri);
					resource.load(RESOURCE_OPTIONS);
				}
			}
		}
		catch(Exception e)
		{
			TestUIPlugin.logError(e);
		}
		
		if(resource == null)
			return new EObject[0];
			
		return (EObject[])resource.getContents().toArray(new EObject[resource.getContents().size()]);			
	}
	
	/**
	 * Returns the resource factory associated with a given file extension.
	 * If the file extension is <code>null</code> or if there isn't a specifc
	 * factory for the file extension, the this method returns the generic
	 * factory.
	 * @param fileExtension
	 * @return the Resource.Factory or <code>null</code> if none. 
	 */
	public static Resource.Factory getResourceFactory(String fileExtension)
	{
		if(fileExtension != null)
		{
			Resource.Factory factory = (Resource.Factory)Resource.Factory.Registry.INSTANCE.getExtensionToFactoryMap().get(fileExtension);
			if(factory != null)
				return factory;
		}
			
		return (Resource.Factory)Resource.Factory.Registry.INSTANCE.getExtensionToFactoryMap().get(Resource.Factory.Registry.DEFAULT_EXTENSION); 
	}
	
	/**
	 * Returns the workspace file that is storing a given eObject or
	 * <code>null</code> if the object is not persisted. 
	 * @param eObject
	 * @return IFile
	 */
	public static IFile getWorkspaceFile(EObject eObject)
	{
		if(eObject == null)
			return null;
		return getWorkspaceFile(eObject.eResource());
	}

	/**
	 * Returns the workspace file that corresponds to a given resource or
	 * <code>null</code> if the resource is not available in the 
	 * workspace.
	 * @param resource
	 * @return IFile
	 */
	public static IFile getWorkspaceFile(Resource resource)
	{
		String filePath = getWorkspaceFilePath(resource);
		if(filePath == null)
			return null;
			
		IResource workspaceResource = ResourcesPlugin.getWorkspace().getRoot().findMember(filePath);
		if((workspaceResource == null) || (workspaceResource.getType() != IResource.FILE))
			return null;
			
		return (IFile)workspaceResource;
	}
	
	public static String getWorkspaceFilePath(Resource resource)
	{
		if(resource == null)
			return null;
			
		String uriToString = resource.getURI().toString();
		if(!uriToString.startsWith("platform:/resource"))
			return null;
			
		return uriToString.substring("platform:/resource".length());
	}
	
	/**
	 * Returns a resource without loading it.  If the resource is <code>null</code>,
	 * a new resource is created if a resource factory is available for the file
	 * extension.
	 * @param resourceSet
	 * @param file
	 * @return Resource
	 */
	public static Resource getResource(ResourceSet resourceSet, IFile file)
	{
		URI uri = URI.createPlatformResourceURI(file.getFullPath().toString());
		if(resourceSet != null)
			return resourceSet.getResource(uri, false);

		Resource.Factory factory = getResourceFactory(file.getFileExtension());
		if(factory != null)
			return factory.createResource(uri);
			
		return null;
	}
	
	/**
	 * Reloads a given eObject returning a <b>new</b> instance from the
	 * same resource.
	 * 
	 * <p>This method unloads and reloads the entire eObject's resource
	 * and tries to find the object in the new resource content.
	 * 
	 * <p>Important: after reloading the resource all the objects
	 * in that resource are reinstantiated.  This has a big impact if an object
	 * is being cached somewhere.
	 * 
	 * @param eObject
	 * @return the reloaded eObject
	 */
	public static EObject reload(EObject eObject)
	{
		if(eObject == null)
			return null;
			
		Resource resource = eObject.eResource();
		if(resource == null)
			return eObject;
			
		String uriFragment = resource.getURIFragment(eObject);
		if(resource.isLoaded())
			resource.unload();
			
		try
		{
			resource.load(RESOURCE_OPTIONS);
		}
		catch (IOException e)
		{
			TestUIPlugin.logError(e);
			return null;
		}
		
		return resource.getEObject(uriFragment);
	}
	
	
	/**
	 * Removes the specified eObject from its container and breaks
	 * all the references to this object within the context of its resource
	 * set. 
	 * @param eObject
	 * @return the changed resources including the eObject's resource.
	 */
	public static Resource[] remove(EObject eObject)
	{
		if(eObject == null)
			return new Resource[0];
			
		if(eObject.eResource() == null)
		{
			EcoreUtil.remove(eObject);
			return new Resource[0];
		}	
		
		Resource resource = eObject.eResource();
		Set changedResources = org.eclipse.hyades.models.hierarchy.util.EMFUtil.delete(eObject, resource.getResourceSet());
		changedResources.add(resource);
		
		if(eObject.eContainer() != null)
			EcoreUtil.remove(eObject);
			
		return (Resource[])changedResources.toArray(new Resource[changedResources.size()]);
	}
	
	/**
	 * Removes the specified resource from its resource set and breaks
	 * all the references to objects inside this resource within the context 
	 * of the resource set. 
	 * @param resource
	 * @return the changed resources excluding the specified resource. 
	 */
	public static Resource[] remove(Resource resource)
	{
		if(resource == null)
			return new Resource[0];
			
		Set changedResources = org.eclipse.hyades.models.hierarchy.util.EMFUtil.delete(resource);
		return (Resource[])changedResources.toArray(new Resource[changedResources.size()]);
	}	

	/**
	 * Adds the specified adapter to the notifiers in the given iterator.
	 * @param objectIterator
	 * @param adapter
	 */
	public static void addAdapter(Iterator objectIterator, Adapter adapter)
	{
		while(objectIterator.hasNext())
		{
			Object object = objectIterator.next();
			if(object instanceof Notifier)
				((Notifier)object).eAdapters().add(adapter);
		}
	}
	
	/**
	 * Removes the specified adapter from the notifiers in the given 
	 * iterator.
	 * @param objectIterator
	 * @param adapter
	 */
	public static void removeAdapter(Iterator objectIterator, Adapter adapter)
	{
		while(objectIterator.hasNext())
		{
			Object object = objectIterator.next();
			if(object instanceof Notifier)
				((Notifier)object).eAdapters().remove(adapter);
		}
	}
	
	/**
	 * Removes all adapters from the notifiers in the given 
	 * iterator.
	 * @param objectIterator
	 */
	public static void removeAdapters(Iterator objectIterator)
	{
		while(objectIterator.hasNext())
		{
			Object object = objectIterator.next();
			if(object instanceof Notifier)
				((Notifier)object).eAdapters().clear();
		}
	}	
}
