/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.sdb.internal.editors;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.Locale;

import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.jface.action.*;
import org.eclipse.jface.viewers.*;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.*;
import org.eclipse.ui.help.WorkbenchHelp;

import org.eclipse.hyades.models.internal.sdb.*;
import org.eclipse.hyades.models.internal.sdb.impl.*;
import org.eclipse.hyades.sdb.internal.SDbPlugin;
import org.eclipse.hyades.sdb.internal.help.ContextIds;
import org.eclipse.hyades.sdb.internal.util.*;

public class SymptomDBEditorDetailsPage extends EditorPage implements SelectionListener{

private TreeViewer treeViewer;
private DetailsContent details;
private Menu     popup;
private FindAction findAction;
private AddAction      addAction;
private AddSymptomAction addSymptomAction;
private DeleteAction   deleteAction;
private Separator separator;
private Button btnAdd;
private Button btnDel;
private Button btnSearch;
private Object root;
private EObject parentOfCurrentSel = null;
private SDBFactoryImpl objFactory = null;

private SymptomDBEditor fEditor;
private EObject currentSelection = null;
private boolean wasCurrentSelValidated = false;
private static SymptomDBSearchDialog dialog;

public SymptomDBEditorDetailsPage(RASWidgetFactory factory){
	super(factory);		
}

public void createContent(Composite parent){
	
	
    createHeader(parent);

	Composite composite =  this.factory.createComposite(parent);
	GridLayout detailLayout = new GridLayout();
	detailLayout.numColumns = 2;

	composite.setLayout(detailLayout);
	composite.setLayoutData(GridUtil.createFill());
    		
    createDetail(composite);
	
	Composite sepFooter = this.factory.createCompositeSeparator(composite);
	GridData sepFooterGrid = GridUtil.createHorizontalFill();
	sepFooterGrid.heightHint = 3;
	sepFooterGrid.horizontalSpan = 2;
	sepFooter.setLayoutData(sepFooterGrid);
	
    createFooter(composite);
    factory.paintChildControlsFor(composite);
        
	treeViewer.setContentProvider(new DBContentProvider());
	treeViewer.setLabelProvider(new DBLabelProvider());
	treeViewer.getTree().addSelectionListener(this);
	
	objFactory = SDbPlugin.getDefault().getSDBFactory();
	
    makeActions();
    
	MenuManager menuMgr = new MenuManager("#PopupMenu");
 	menuMgr.setRemoveAllWhenShown(true);
	menuMgr.addMenuListener(new IMenuListener() {
	  public void menuAboutToShow(IMenuManager manager) {
		   SymptomDBEditorDetailsPage.this.fillContextMenu(manager);
	  }
	});
 	Menu menu = menuMgr.createContextMenu(treeViewer.getTree());
	treeViewer.getTree().setMenu(menu);	    
	    
}

public void init(SymptomDBEditor editor){
	fEditor = editor;
	if(getTreeRoot()!=null)	
		details.init(fEditor);

}
private void createHeader(Composite content){

	Composite sepHeader = this.factory.createCompositeSeparator(content);

}

private void createDetail(Composite composite){
			    
    Tree tree = this.factory.createTree(composite, RASWidgetFactory.BORDER_STYLE | SWT.H_SCROLL | SWT.V_SCROLL | SWT.MULTI);

    treeViewer = new TreeViewer(tree);
	GridData dummy = GridUtil.createFill();
	dummy.widthHint = 200;
    treeViewer.getTree().setLayoutData(dummy);

    details = new DetailsContent(this.factory, composite, SWT.NONE);
    details.createContent();
			    
}

private void createFooter(Composite footer){
	
	Composite buttons = this.factory.createComposite(footer); 
	GridLayout buttonsLayout = new GridLayout();
	GridData data = GridUtil.createHorizontalFill();	
	
	buttonsLayout.numColumns = 2;
	buttons.setLayout(buttonsLayout);
	buttons.setLayoutData(data);	
		
    createButtons(buttons);      
}

private void createButtons(Composite buttons) {

	Composite update = this.factory.createComposite(buttons);
	GridLayout updateLayout = new GridLayout();
	GridData data = GridUtil.createHorizontalFill();

	updateLayout.numColumns = 4;
	update.setLayout(updateLayout);
	update.setLayoutData(data);

	btnAdd = this.factory.createButton(update, SDbPlugin.getResourceString("STR_ADD_LABEL"), SWT.PUSH);
	GridData addGrid = new GridData(GridData.END);
	//Need wider button for German translation of "Add", i.e. "Hinzufugen".
	if (Locale.getDefault().toString().equals("en_US"))
		addGrid.widthHint = 50;
	else
		addGrid.widthHint = 80;
	btnAdd.setLayoutData(addGrid);

	btnDel = this.factory.createButton(update, SDbPlugin.getResourceString("STR_DELETE_LABEL"), SWT.PUSH);
	GridData delGrid = new GridData(GridData.END);
	//Make the Delete button the same size as the Add button.
	if (Locale.getDefault().toString().equals("en_US"))
		delGrid.widthHint = 50;
	else
		delGrid.widthHint = 80;
	btnDel.setLayoutData(delGrid);

	Label label = new Label(update, SWT.NONE);
	GridData labelGrid = new GridData(GridData.END);
	labelGrid.widthHint = 50;
	label.setLayoutData(labelGrid);

	btnSearch = this.factory.createButton(update, SDbPlugin.getResourceString("STR_SEARCH_LABEL"), SWT.PUSH);
	GridData searchGrid = new GridData(GridData.END);
	//Need wider button for French translation of "Find...", i.e. "Rechercher...".
	if (Locale.getDefault().toString().equals("en_US"))
		searchGrid.widthHint = 50;
	else
		searchGrid.widthHint = 80;
	btnSearch.setLayoutData(searchGrid);

	btnAdd.addSelectionListener(this);
	btnDel.addSelectionListener(this);
	btnSearch.addSelectionListener(this);
}

public void fillContextMenu(IMenuManager menu) {

   IStructuredSelection selection = (IStructuredSelection) treeViewer.getSelection();
   updateActions(selection);

   Object object = selection.getFirstElement();
      
   if(object instanceof SDBSymptom){
   		menu.add(addSymptomAction);
   		menu.add(separator);
   		WorkbenchHelp.setHelp(addSymptomAction, ContextIds.SYMDB_EDITOR_POPUP_ADD_SYMPTOM);
   }
   
   if (object instanceof SDBRuntime || object instanceof SDBSymptom || object instanceof SDBSolution ) {
    	menu.add(addAction);
   }
   
   if (object instanceof SDBSymptom || object instanceof SDBSolution || object instanceof SDBDirective) {
      	menu.add(deleteAction);
      	WorkbenchHelp.setHelp(deleteAction, ContextIds.SYMDB_EDITOR_POPUP_DELETE);
   }
    
}

protected void updateActions(IStructuredSelection selection) {
       addAction.selectionChanged(selection);
       addSymptomAction.selectionChanged(selection);
       deleteAction.selectionChanged(selection);
}


public void updateButtons(Object data) {
    if(data instanceof SDBRuntime){
	    btnAdd.setEnabled(true);
		btnDel.setEnabled(false);
    }
    else if(data instanceof SDBSymptom || data instanceof SDBSolution){
	    btnAdd.setEnabled(true);
		btnDel.setEnabled(true);
    }
    else if(data instanceof SDBDirective){
	    btnAdd.setEnabled(false);
		btnDel.setEnabled(true);
    }
    else{
	    btnAdd.setEnabled(false);
		btnDel.setEnabled(false);
    }
    
}   

private void makeActions() {

	addSymptomAction = new AddSymptomAction();
   	
   	separator = new Separator();   	
	addAction = new AddAction();
   	deleteAction = new DeleteAction();
   	if(findAction==null)
	   	findAction = new FindAction(this);
}

public Object getTreeRoot(){
	return ((Object[])root)[0];
}

public void setTreeRoot(Object newRoot){
	root = newRoot;
	if(((Object[])root)[0]==null){
		btnAdd.setEnabled(false);
		btnDel.setEnabled(false);
		btnSearch.setEnabled(false);
	}
}

public SymptomDBSearchDialog getDialog(){
	return dialog;
}
public void setDialog(SymptomDBSearchDialog dialog){
	SymptomDBEditorDetailsPage.dialog = dialog;
}

public TreeViewer getViewer(){
	return treeViewer;
}

public Action getFindAction(){
	return findAction;
}

public void widgetDefaultSelected(SelectionEvent event) {
}

public void widgetSelected(SelectionEvent event) {

	statusErrorMessage("");

	if(event.item instanceof TreeItem) {
	  
       EObject current = (EObject)event.item.getData();
       if(current==null)
       	return;
       
		if(fEditor.isModelDirty())	
       		updateCurrentSelection(currentSelection);
    		
       details.showDetailsFor(current);

	   if(current!=currentSelection){
	       
	       if(current instanceof SDBSolution||current instanceof SDBDirective)
	       	parentOfCurrentSel = (EObject)((TreeItem)event.item).getParentItem().getData();
	       else if(current instanceof SDBSymptom)
	        parentOfCurrentSel = (EObject)getTreeRoot();
	       else
	        parentOfCurrentSel =null;
       		currentSelection = (EObject)current;
	   }
       
       updateButtons(current);       

	  return;
	}

	if(event.widget == btnAdd) {   
		TreeItem[] sel = treeViewer.getTree().getSelection();
		if(sel.length > 0){	
			add(new TreeSelection(sel[0]));
		}
		else{
			statusErrorMessage(SDbPlugin.getResourceString("STR_NO_SEL_ADD_MSG"));
		}
	}
	if (event.widget == btnDel) {
	  
	  TreeItem[] selection = treeViewer.getTree().getSelection();
	  if(selection.length > 0){
			Object[] sel = new Object[selection.length];
			for(int i=0;i<selection.length;i++){
				sel[i] = ((TreeItem)selection[i]).getData();
			}
			deleteAction.updateSelection(new StructuredSelection(sel));
			deleteAction.run();
	  }
	  else{
	  	statusErrorMessage(SDbPlugin.getResourceString("STR_NO_SEL_DEL_MSG"));
	  }
	  return;
	}
	
	if(event.widget == btnSearch) {
		findAction.run();
	}
	
}

private void statusErrorMessage(String message) {

	IStatusLineManager manager= fEditor.getStatusLineManager();
	if (manager == null)				
		return;
			
	manager.setMessage("");
	manager.setErrorMessage(message); 
	
}


public EObject getCurrentSelection(){
	return currentSelection;		
}

public EObject getParentOfCurrentSel(){
	return parentOfCurrentSel;
}

public boolean updateCurrentSelection(EObject currentSelection){

	if(currentSelection!=null){

		details.updateCurrentSelection(currentSelection);		
		fEditor.updateModelDirtyStatus(false);
		treeViewer.refresh();			
		return true;		

	}
	return false;
}

private void add(TreeSelection parentList){
	

	if(!fEditor.validateState(null, null, null))
		return;

	EObject parent = (EObject)parentList.getObject();

	if(parent instanceof SDBRuntime){

		SDBSymptom newSymptom  = objFactory.createSDBSymptom();
		int i = ((SDBRuntime)parent).getSymptoms().size();
		newSymptom.setDescription(SDbPlugin.getResourceString("STR_NEW_SYMPTOM_LABEL") + ++i);
		SDBRuntime db = (SDBRuntime)getTreeRoot();
		db.getSymptoms().add(newSymptom);

		treeViewer.add(db, newSymptom);
		treeViewer.setExpandedState(db, true);		
		parentList.add(newSymptom);

	}
	
	if(parent instanceof SDBSymptom){

		SDBSolution newSolution = objFactory.createSDBSolution();
		int i = ((SDBRuntime)getTreeRoot()).getSolutions().size();
		newSolution.setDescription(SDbPlugin.getResourceString("STR_NEW_SOLUTION_INIT") + ++i);
		SDBRuntime db = (SDBRuntime)getTreeRoot();
		db.getSolutions().add(newSolution);

		((SDBSymptom)parent).getSolutions().add(newSolution);
		newSolution.getSymptoms().add((SDBSymptom)parent);

		treeViewer.add(parent, newSolution);
		treeViewer.setExpandedState(parent, true);

		parentList.add(newSolution);

	}

	if(parent instanceof SDBSolution){

		SDBDirective newDirective = objFactory.createSDBDirective();
		int i = ((SDBRuntime)getTreeRoot()).getDirectives().size();
		newDirective.setDescription(SDbPlugin.getResourceString("STR_NEW_DIRECTIVE_INIT")+ ++i);
		newDirective.setDirectiveString("");
		SDBRuntime db = (SDBRuntime)getTreeRoot();
		db.getDirectives().add(newDirective);

		((SDBSolution)parent).getDirectives().add(newDirective);
		newDirective.getSolutions().add((SDBSolution)parent);

		treeViewer.add(parent, newDirective);
		treeViewer.setExpandedState(parent, true);

		parentList.add(newDirective);
		
	}

	setSelection(parentList.getSelection());
	fEditor.updateModelDirtyStatus(true);
	fEditor.updateUIDirtyStatus(true);
	
	details.setFocusFor(parentList.getObject());
}

private void addSymptom(){

		if(!fEditor.validateState(null, null, null))
			return;
		SDBSymptom newSymptom  = objFactory.createSDBSymptom();
		SDBRuntime db = (SDBRuntime)getTreeRoot();
		int i = db.getSymptoms().size();
		newSymptom.setDescription(SDbPlugin.getResourceString("STR_NEW_SYMPTOM_LABEL") + ++i);

		db.getSymptoms().add(newSymptom);

		treeViewer.add(db, newSymptom);

		fEditor.updateUIDirtyStatus(true);

		setSelection(new Object[]{getTreeRoot(), newSymptom, null, null});
		details.setFocusFor(newSymptom);
}

public void remove(ArrayList selections){
	
	if(!fEditor.validateState(null, null, null))
		return;
	
	ArrayList parents = new ArrayList();
	ArrayList current = null;
	int i=0;
	
	for(i=0; i < selections.size()-1;i++){
	  current = remove((TreeSelection)selections.get(i), false);
	  parents.addAll(current);
	}
	
	current = remove((TreeSelection)selections.get(i), true);
	parents.addAll(current);

	for(i=0;i<parents.size();i++){
		treeViewer.refresh(parents.get(i));
	}
}


private ArrayList remove(TreeSelection sel, boolean newSel){
	
	ArrayList parentObj = new ArrayList();
	Object[] selection = sel.getSelection();

	EObject object = sel.getObject();
	int index = -1;
	if(object instanceof SDBSymptom){
		
		EList sol = ((SDBSymptom)object).getSolutions();
		
		ArrayList lst = new ArrayList();
		lst.addAll(sol);
		
		clearSolutions((EObject)object, lst);
		
		SDBRuntime db = (SDBRuntime)getTreeRoot();
		index = db.getSymptoms().indexOf((SDBSymptom)object);
		db.getSymptoms().remove((SDBSymptom)object);
		EList symptoms = db.getSymptoms();

		if(!symptoms.isEmpty())
			if(index < symptoms.size())
				selection[1] = (SDBSymptom)symptoms.get(index);
			else
				selection[1] = (SDBSymptom)symptoms.get(index -1);
		else
			selection[1] = null;

		parentObj.add(db);
	}
	else if (object instanceof SDBSolution){

		SDBSolution current = (SDBSolution)object;
		SDBSymptom parent = (SDBSymptom)sel.getSelection()[1];
		
		index = parent.getSolutions().indexOf((SDBSolution)current);

		parentObj.addAll(current.getSymptoms());
		parent.getSolutions().remove(current);
		current.getSymptoms().remove(parent);

		if(current.getSymptoms().isEmpty()){
			EList dir = current.getDirectives();
				
			ArrayList lst = new ArrayList();
			lst.addAll(dir);
				
			clearDirectives(current, lst);
			
			((SDBRuntime)getTreeRoot()).getSolutions().remove(current);
			details.updateRegistry(current, false);
		}
		
		EList solutions = parent.getSolutions();
				
		if(!solutions.isEmpty())
			if(index < solutions.size())
				selection[2] = (SDBSolution)solutions.get(index);
			else
				selection[2] = (SDBSolution)solutions.get(index -1);
		else
			selection[2] = null;

	}
	else if (object instanceof SDBDirective){

		SDBDirective current = (SDBDirective)object;
		SDBSolution parent = (SDBSolution)sel.getSelection()[2];
		
		index = parent.getDirectives().indexOf((SDBDirective)current);
		for(int i=0;i<current.getSolutions().size();i++){
			parentObj.addAll(((SDBSolution)current.getSolutions().get(i)).getSymptoms());
		}

		parent.getDirectives().remove(current);
		current.getSolutions().remove(parent);

		if(current.getSolutions().isEmpty()){
			((SDBRuntime)getTreeRoot()).getDirectives().remove(current);
			details.updateRegistry(current, false);
		}
		
		EList directives = parent.getDirectives();
				
		if(!directives.isEmpty())
			if(index < directives.size())
				selection[3] = (SDBDirective)directives.get(index);
			else
				selection[3] = (SDBDirective)directives.get(index -1);
		else
			selection[3] = null;

	}
	
	fEditor.updateUIDirtyStatus(true);
	fEditor.updateModelDirtyStatus(false);
	
	if(newSel)	
		setSelection(selection);

	return parentObj;	
}

public void setSelection(Object[] sel){
		
	if(sel[0]==null){

		return;
	}
						
	TreeItem rootItem = treeViewer.getTree().getItems()[0];
	if(sel[1]==null){
		notifyWidget(rootItem);
		return;
	}
		
	treeViewer.setExpandedState(root, true);
				
	//it might be possible that the tree item we are looking for hasn't been initialized, although this shouldn't happen
	// because we are forcing a refresh before looking for an item on a tree level
	
	// symptom
	TreeItem symptomItem = null;
	TreeItem[] items = rootItem.getItems();
	for(int i=0;i<items.length;i++){
		Object data = items[i].getData();
		if(data!=null && data.equals(sel[1])){
			symptomItem = items[i];
			break;
		}
	}
			
	if(sel[2]==null){
		notifyWidget(symptomItem);
		return;
	}
	
	symptomItem.setExpanded(true);
	treeViewer.refresh(sel[1]);
	
	//solution
	TreeItem solItem = null;
	items = symptomItem.getItems();
	for(int i=0;i<items.length;i++){
		Object data = items[i].getData();
		if(data!=null && data.equals(sel[2])){
			solItem = items[i];				
			break;
		}
	}
	        
	if(sel[3]==null){
		notifyWidget(solItem);
		return;
	}
			
	solItem.setExpanded(true);
	treeViewer.refresh(sel[1]);
		
	//directive
	TreeItem directiveItem = null;
	items = solItem.getItems();
	for(int i=0;i<items.length;i++){
		Object data = items[i].getData();
		if(data!=null && data.equals(sel[3])){			
			directiveItem = items[i];
			break;
		}
	}
	notifyWidget(directiveItem);

}

private void notifyWidget(TreeItem item){

	Tree tree = treeViewer.getTree();
	tree.setSelection(new TreeItem[]{item});
	tree.showSelection();
	Event event = new Event();
	event.widget = tree;
	event.item = item;
	widgetSelected(new SelectionEvent(event));

}

private void clearSolutions(EObject parent, ArrayList lst){
	
	for(int i=0;i<lst.size();i++){
		SDBSolution item = (SDBSolution)lst.get(i);

		EList dir = item.getDirectives();
		
		ArrayList dirLst = new ArrayList();
		dirLst.addAll(dir);
					
		item.getSymptoms().remove(parent);
		if(item.getSymptoms().isEmpty()) {
			clearDirectives(item, dirLst);
			((SDBRuntime)getTreeRoot()).getSolutions().remove(item);
			details.updateRegistry(item, false);				
		}
	}
		
	((SDBSymptom)parent).getSolutions().clear();
	((SDBSymptom)parent).getPatterns().clear();
	
}

private void clearDirectives(EObject parent, ArrayList lst){

	for(int i=0;i<lst.size();i++){
		SDBDirective item = (SDBDirective)lst.get(i);
		item.getSolutions().remove(parent);
		if(item.getSolutions().isEmpty()) {
			((SDBRuntime)getTreeRoot()).getDirectives().remove(item);
			details.updateRegistry(item, false);				
		}
	}
		
	((SDBSolution)parent).getDirectives().clear();

}

class AddAction extends org.eclipse.ui.actions.SelectionListenerAction {
	TreeSelection selection;

	public AddAction() {
		super(SDbPlugin.getResourceString("STR_ADD_LABEL"));
	}
	
	public void run() {
		add(selection);	
	}
	
    public boolean updateSelection(IStructuredSelection s) {    	
       
       TreeItem[] sel = treeViewer.getTree().getSelection();

	   if (!super.updateSelection(s))
		   return false;
		   
	   if (sel.length==0 || sel.length > 1)
	       return false;
		
	   TreeItem item = sel[0];
	   selection = new TreeSelection(item);
	   Object current = item.getData();
       if (current!=null) {
       	  
       	  if (current instanceof SDBRuntime) {
       	  	  setText(SDbPlugin.getResourceString("STR_ADD_SYMPTOM_LABEL"));
       	  	  WorkbenchHelp.setHelp(addAction, ContextIds.SYMDB_EDITOR_POPUP_ADD_SYMPTOM);
       	  }
       	  else if(current instanceof SDBSymptom) {
       	  	  setText(SDbPlugin.getResourceString("STR_ADD_SOLUTION_LABEL"));
       	  	  WorkbenchHelp.setHelp(addAction, ContextIds.SYMDB_EDITOR_POPUP_ADD_SOLUTION);     	  	  
       	  }
       	  else if(current instanceof SDBSolution){
       	  	  setText(SDbPlugin.getResourceString("STR_ADD_DIRECTIVE_LABEL"));
       	  	  WorkbenchHelp.setHelp(addAction, ContextIds.SYMDB_EDITOR_POPUP_ADD_DIRECTIVE);
       	  }	
       	  else if (current instanceof SDBDirective){
         	  return false;
       	  }
       }
       
       return true;        

    }
}

class AddSymptomAction extends org.eclipse.ui.actions.SelectionListenerAction {
	Object selection;

	public AddSymptomAction() {
		super(SDbPlugin.getResourceString("STR_ADD_SYMPTOM_LABEL"));
	}
	
	public void run() {
		addSymptom();	
	}
	
    public boolean updateSelection(IStructuredSelection s) {    	
       

	   if (!super.updateSelection(s))
		   return false;
		   
	   if (s.isEmpty() || s.size() > 1)
	       return false;
		
       Iterator i = s.iterator();
       if (i.hasNext()) {
       	  Object current = i.next();
       	  
       	  if (current instanceof SDBSymptom) {
       	  	  selection = (SDBSymptom)current;

       	  }
       	  else {
         	  return false;
       	  }
       }
       
       return true;        

    }
}

class DeleteAction extends org.eclipse.ui.actions.SelectionListenerAction {
	ArrayList selection;
	 
	public DeleteAction() {
		super(SDbPlugin.getResourceString("STR_DELETE_LABEL"));
		selection = new ArrayList();
	}
	
	public void run() {
		remove(selection);
	}
	
    public boolean updateSelection(IStructuredSelection s) {
       selection.clear();
	   TreeItem[] sel = treeViewer.getTree().getSelection();
	   if (!super.updateSelection(s))
		   return false;
		   
	   if (sel.length < 1)
	       return false;
		
	   for(int i=0;i<sel.length;i++){
		  Object current = sel[i].getData();
       	  if (current instanceof SDBSymptom || current instanceof SDBSolution || current instanceof SDBDirective)
       	  	  selection.add(new TreeSelection(sel[i]));
       	  else
       	   return false;
       	  
       }
       
       return true;        
    }
}

public class FindAction extends Action {

	
	private SymptomDBEditorDetailsPage page;
	/**
	 * Constructor for FindAction.
	 * 
	 */
	public FindAction(SymptomDBEditorDetailsPage page) {
		super();
		this.page = page;
	}

   	public void run(){

		SymptomDBSearchDialog dialog = page.getDialog();   		
		if (dialog != null && fEditor.getSite().getShell() != dialog.getParentShell()) {
			dialog.close();
		}
	
	    if(dialog==null){
			dialog = new SymptomDBSearchDialog(fEditor.getSite().getShell(), (SDBRuntime)getTreeRoot(), page);
			page.setDialog(dialog);
		}
		else{	
			page.getDialog().setTarget(page);
		}
		
		dialog.open();

   	}

}

public class TreeSelection {

	private TreeItem selectedItem;
	private Object[] selection;
	
	public TreeSelection(TreeItem sel){
		selectedItem = sel;
		selection = getSelectedObjectsFromTreeItem();
	}
	
	public Object[] getSelection(){
		return selection;
	}
	
	public void setSelection(Object[] newSel){
		for(int i=0;i<newSel.length; i++){
			selection[i] = newSel[i];
		}
	}
	
	public EObject getObject(){
		EObject object = null;
		
		if(selection[3]!=null){
			return (EObject)selection[3];
		}
			
		for(int i=1;i<selection.length;i++){
			if(selection[i]==null){
				object = (EObject)selection[i-1];
				break;
			}
		}
		
		return object;
	}
	
	private Object[] getSelectedObjectsFromTreeItem(){

	   Object current = selectedItem.getData();
	   Object[] selection = new Object[4];
       if (current!=null) {
       	  
       	  if (current instanceof SDBRuntime) {
       	  	  selection[0] = current;
       	  }
       	  else if(current instanceof SDBSymptom) {
       	  	  selection[0] = selectedItem.getParentItem().getData();
       	  	  selection[1] = current;
       	  }
       	  else if(current instanceof SDBSolution){
       	      selection[0] = selectedItem.getParentItem().getParentItem().getData();
       	      selection[1] = selectedItem.getParentItem().getData();
       	      selection[2] = current;
       	      
       	  }	
       	  else if (current instanceof SDBDirective){
       	      selection[3] = current;
       	      TreeItem parent = selectedItem.getParentItem();
       	      selection[2] = parent.getData();
			  selection[1] = parent.getParentItem().getData();	
       	      selection[0] = parent.getParentItem().getParentItem().getData();

       	  }
       }
       return selection;
	}
	
	public void add(EObject object){
       	  if (object instanceof SDBRuntime) {
       	  	  selection[0] = object;
       	  }
       	  else if(object instanceof SDBSymptom) {
       	  	  selection[1] = object;
       	  }
       	  else if(object instanceof SDBSolution){
       	      selection[2] = object;       	      
       	  }	
       	  else if (object instanceof SDBDirective){
       	      selection[3] = object;
       	  }		
	}
}
}
