/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v0.5
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v05.html
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.models.internal.sdb.util;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtension;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.IPluginDescriptor;
import org.eclipse.core.runtime.IPluginRegistry;

abstract class RegistryReader {
    //~ Static fields/initializers -----------------------------------------------------------------

    protected static final String TAG_DESCRIPTION = "description";

    //~ Instance fields ----------------------------------------------------------------------------

    protected IPluginRegistry pluginRegistry;
    String extensionPointID;
    String pluginID;

    //~ Constructors -------------------------------------------------------------------------------

    public RegistryReader(IPluginRegistry pluginRegistry, String pluginID, String extensionPointID) {
        super();
        this.pluginRegistry = pluginRegistry;
        this.pluginID = pluginID;
        this.extensionPointID = extensionPointID;
    }

    //~ Methods ------------------------------------------------------------------------------------

    /**
     * Reads from the plugin registry and parses it.
     */
    public void readRegistry() {
        IExtensionPoint point = pluginRegistry.getExtensionPoint(pluginID, extensionPointID);

        if (point != null) {
            IConfigurationElement[] elements = point.getConfigurationElements();

            for (int i = 0; i < elements.length; i++) {
                internalReadElement(elements[i]);
            }
        }
    }

    /**
     * Logs the error in the desktop log using the provided
     * text and the information in the configuration element.
     */
    protected void logError(IConfigurationElement element, String text) {
        IExtension extension = element.getDeclaringExtension();
        IPluginDescriptor descriptor = extension.getDeclaringPluginDescriptor();

        System.err.println("Plugin " + descriptor.getUniqueIdentifier() + ", extension " + extension.getExtensionPointUniqueIdentifier());
        System.err.println(text);
    }

    /**
     * Logs a very common registry error when a required attribute is missing.
     */
    protected void logMissingAttribute(IConfigurationElement element, String attributeName) {
        logError(element, "The required attribute '" + attributeName + "' not defined");
    }

    /**
     * Implement this method to read element attributes.
     * If this element has subelements, the reader will recursively cycle through them
     * and will call this method, so don't do it here.
     */
    abstract boolean readElement(IConfigurationElement element);

    private void internalReadElement(IConfigurationElement element) {
        boolean recognized = this.readElement(element);

        if (recognized) {
            IConfigurationElement[] children = element.getChildren();

            for (int i = 0; i < children.length; ++i) {
                internalReadElement(children[i]);
            }
        } else {
            logError(element, "Error processing extension: " + element);
        }
    }

    //~ Inner Classes ------------------------------------------------------------------------------

    static class PluginClassDescriptor {
        protected IConfigurationElement element;
        protected String attributeName;

        public PluginClassDescriptor(IConfigurationElement element, String attributeName) {
            this.element = element;
            this.attributeName = attributeName;
        }

        public Object createInstance() {
            try {
                return element.createExecutableExtension(attributeName);
            } catch (CoreException e) {
                throw new RuntimeException(e.getLocalizedMessage());
            }
        }
    }
}
