/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * Scapa Technologies Limited - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.statistical.ui.editor.internal;

/**
 * StatCon modules allows third party data tracing and system control tools to plug in to a generic interface.
 * When a StatCon editor is opened, all StatCon module implementations are loaded and once copy of each is instantiated.
 * One module instance per StatCon instance.  The module must implement this interface so that StatCon can deal with
 * it as a generic module.  The module will be loaded with the load() call and may then use any of the StatCon interface
 * to set up its extensions as it likes.  When StatCon shuts down the module will be unloaded with the unload() call.  
 * NOTE: Each module should implement this interface and should also have an empty constructor so that StatCon can
 * instantiate a new copy of the module with no arguments.
 * NOTE: modules may want to use the XMLConfigUtil file to generate and parse StatCon files.
 */
public interface StatConModule {

	/**
	 * This is the first point of call for every StatCon module.  This method 
	 * is called so that the StatCon module can initialise itself and set up its
	 * drop listeners and any top level module nodes it wants to add.
	 * A module may throw an Exception but if it does so it must clean up any
	 * modifications made to the StatCon instance first.
	 * @param statcon the StatConInterface representing this StatCon instance
	 * @throws Exception
	 */
	public void load(StatConInterface statcon) throws Exception;

	/**
	 * When a StatCon file is loaded or a configuration is applied to a statcon instance
	 * each module gets a chance to intercept any module-specific xml it might have added.
	 * This allows a StatCon module to persist any configuration made within StatCon.
	 * Module configuration data is stored just under the root node.  Modules
	 * should not use the tag names "<graph>" "<agent>" "<descriptor>".  Best practice here is
	 * to prefix module root XML nodes with "<module-...>".
	 * @param statcon the StatCon interface associated with this configuration
	 * @param xml the XML file containing the configuration data
	 * @throws Exception
	 */
	public void applyConfig(StatConInterface statcon, String xml) throws Exception;

	/**
	 * When StatCon builds a configuration file to be saved into a .statcon resource, it asks
	 * each module to generate any XML configuration data that it could load with an applyConfig() call
	 * when the file is next opened.  This allows a module to save configuration data and reload it.
	 * The module may generate one or more root nodes.  Best practice is to prefix each root XML node with 
	 * "<module-...>" so that it does not clash with any XML elements generated by StatCon.  e.g. a Perfmon
	 * StatCon module could generate "<module-perfmon>" as its root element.  This element (or elements) would
	 * then be appended under the root node of the StatCon XML file.
	 * @param statcon the StatCon interface associated with the generate config request
	 * @return an XML string to be appended under the root node of the StatCon configuration file
	 */
	public String generateConfig(StatConInterface statcon);
	
	/**
	 * This is the last point of call for every StatCon module. This will typically be called on shutdown of
	 * a StatCon instance (editor) but it may be called at other points.  Upon calling this a module is expected to
	 * remove all references to itself from StatCon and free all resources.
	 * @param statcon the StatCon interface associated with the unload request
	 * @throws Exception
	 */
	public void unload(StatConInterface statcon) throws Exception;
}