/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.execution.testgen.http;

import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.ListIterator;
import java.util.StringTokenizer;

import org.eclipse.hyades.internal.execution.testgen.TestgenException;
import org.eclipse.hyades.models.common.facades.behavioral.ILoop;
import org.eclipse.hyades.test.http.internal.util.RequestHelper;
// mdd for Hyades-O 1.2
import java.util.LinkedList;
// need HttpRequest for PAGE_END, etc info
import org.eclipse.hyades.test.http.runner.HttpRequest;


/**
 * @author mdunn
 *
 * To change the template for this generated type comment go to
 * Window&gt;Preferences&gt;Java&gt;Code Generation&gt;Code and Comments
 */
public class TestGenHttp 
{
	private static final String	PACKET_A_CONNECTIONNAME
														= "connectionNumber";
	private static final String	PACKET_S_DATA
														= "data";
	private static final String	PACKET_S_DATA_A_ENCODING
														= "encoding";
	private static final String	PACKET_S_DATA_A_ENCODING_V_NONE
														= "NONE";
	private static final String	PACKET_S_DATA_A_ENCODING_V_ASCIIFY
														= "ASCIIFY";
	private static final String	PACKET_S_DATA_A_ENCODING_V_BASE64
														= "BASE64";
	private static final String	PACKET_S_DATA_A_LENGTH
														= "length";
	private static final String	PACKET_S_DATA_A_TYPE
														= "type";
	private static final String	PACKET_S_DATA_A_TYPE_V_HTTPCONTENT
														= "HTTPCONTENT";
	private static final String	PACKET_S_DATA_A_TYPE_V_HTTPMESSAGE
														= "HTTPMESSAGE";
	private static final String	PACKET_A_FROM
														= "from";
	private static final String	PACKET_A_FROM_V_CLIENT
														= "CLIENT";
	private static final String	PACKET_A_FROM_V_SERVER
														= "SERVER";
	private static final String	PACKET_A_TICKET
														= "ticket";
	private static final String	PACKET_A_TIMESTAMP
														= "timestamp";
	private static final String	PACKET_A_TYPE
														= "type";
	private static final String	PACKET_A_TYPE_V_HTTP
														= "HTTP";
	private static final String	PACKET_A_TYPE_V_HTTPS
														= "HTTPS";

	private static final String	CRNL	= "\r\n";
	
	private static final char	SLASH	= '/';
	private static final char	COLON	= ':';
	private static final String BLANK   = " ";
	
	private static boolean isFirstPage = true;
	
	private class ParsedHttpMessage
	{
		String			method;
		String			protocol;
		String			host;
		int				port;
		String			path;
		String			version;
		boolean 		bSSL;
		int 			enumPage;
		int 			nPageNum;
		int 			nConnID;
		boolean 		newPage;
		String 			timeStampString;
		ArrayList		headerNames = new ArrayList();
		ArrayList		headerValues = new ArrayList();
		StringBuffer	body = new StringBuffer();
	}
	// New Object to handle 
	private class ConnectionNode
	{
		int connectionNumber;
		LinkedList childrenNodes;
	}
	LinkedList connectionsList;
	
	// Object to hold Response information
	private class ResponseInfo {
		String timeStampString;
		boolean newPage;
	}
	
	// private HashMap				lastRecvHashMap = new HashMap();
	private ParsedHttpMessage	currentHttpRequest;
	private ParsedHttpMessage 	previousHttpRequest;
	TRCContext context = null;
	LinkedList newPageList; 

	/**
	 * @param context
	 */
	public TestGenHttp (TRCContext context)
	{
		super();
		
		this.context = context;
		isFirstPage = true;
		connectionsList = new LinkedList();
		newPageList = new LinkedList();
	}
	
	/**
	 * Parse Request Message for Message information
	 * @param dataNode
	 * @throws TestgenException
	 */
	private void httpRequestMessage (TRCNode dataNode)
		throws TestgenException
	{
		String content = dataNode.getElement().getContent();
		if (content == null)
			throw new TestgenException(TestgenHttpPlugin.
					getResourceString("E_DATA_NO_CONTENT"), null);

		String encoding = TRCNodeHandler.getAttributeValue(PACKET_S_DATA_A_ENCODING, dataNode);
		if (encoding == null)
			throw new TestgenException(TestgenHttpPlugin.
					getResourceString("E_DATA_NO_ENCODING_ATT"), null);
		// TODO: dhinson: needs to handle non-NONE encodings eventually			
		// TODO: dhinson: change to equals() when DuWayne fixes value
		if (encoding.equalsIgnoreCase(PACKET_S_DATA_A_ENCODING_V_NONE))
			; // do nothing (no translation required)
		else if (encoding.equals(PACKET_S_DATA_A_ENCODING_V_ASCIIFY))
			return; // silently ignore
		else if (encoding.equals(PACKET_S_DATA_A_ENCODING_V_BASE64))
			return; // silently ignore
		else
			return; // silently ignore
			
		StringTokenizer bodyTokenizer = new StringTokenizer(content, CRNL);
		for (int headerCount = 0; bodyTokenizer.hasMoreTokens();
			++headerCount) {
			String header = bodyTokenizer.nextToken();
			if (headerCount == 0) {
				StringTokenizer headerTokenizer = new StringTokenizer(header);
				
				currentHttpRequest.method = headerTokenizer.nextToken();
				
				String urn = headerTokenizer.nextToken();
				if (urn.length() > 6 && urn.substring(0, 7).equals("http://")) {
					try {
						URL url = new URL(urn);
						currentHttpRequest.path = url.getPath();
					}
					catch (MalformedURLException e) {
					}
				} else
					currentHttpRequest.path = urn;
				
				String version = headerTokenizer.nextToken();
				currentHttpRequest.version =
					version.substring(version.indexOf(SLASH) + 1,
						version.length());				
			}
			else {
				int colonIndex = header.indexOf(COLON);
				String name = header.substring(0, colonIndex);
				String value = header.substring(colonIndex + 1, header.length());
				currentHttpRequest.headerNames.add(name);
				currentHttpRequest.headerValues.add(value);
				
				// fetch host and port from Host: header
				if (name.trim().equals("Host")) {
					try {
						URL url = new URL("http://" + value.trim());
						currentHttpRequest.host = url.getHost();
						currentHttpRequest.port = url.getPort();
						if (currentHttpRequest.port == -1) {
							// Check if SSL - if so, use 443 for port
							if (currentHttpRequest.bSSL)
								currentHttpRequest.port = 443;
							else
								currentHttpRequest.port = 80;
						}
					}
					catch (MalformedURLException e) {
						e.printStackTrace();
					}
				}
			}
		}
	}
	
	/**

	 * 
	 * @param dataNode
	 * @throws TestgenException
	 */
	private void httpRequestContent (TRCNode dataNode)
		throws TestgenException
	{
		String content = dataNode.getElement().getContent();
		if (content == null)
			throw new TestgenException(TestgenHttpPlugin.
					getResourceString("E_DATA_NO_CONTENT"), null);

		String encoding = TRCNodeHandler.getAttributeValue(PACKET_S_DATA_A_ENCODING, dataNode);
		if (encoding == null)
			throw new TestgenException(TestgenHttpPlugin.
					getResourceString("E_DATA_NO_ENCODING_ATT"), null);
		// TODO: dhinson: needs to handle non-NONE encodings eventually			
		// TODO: dhinson: change to equals() when DuWayne fixes value
		if (encoding.equalsIgnoreCase(PACKET_S_DATA_A_ENCODING_V_NONE))
			currentHttpRequest.body.append(content);
		else if (encoding.equals(PACKET_S_DATA_A_ENCODING_V_ASCIIFY))
			; // silently ignore
		else if (encoding.equals(PACKET_S_DATA_A_ENCODING_V_BASE64))
			; // silently ignore
		else
			; // silently ignore
	}
	
	/**
	 * This method is called when the TRCRecordingInfo node handler
	 * has determined that a STOP message has been found
	 * 
	 * This method will call reorderNodeList() to reorder the list of
	 * nodes, then call writeNewPageList() to parse the node list and 
	 * write the information into the behavior model.
	 * 
	 * @throws TestgenException
	 */
	protected void handle_http_stop() throws TestgenException {
		// remove_unwanted_responses();
		// Call new routine to group the nodes
		reorderNodeList();
		writeNewPageList();
		//determinePageEnum(); 
		//writePageList();
		context.setPageNumber(0);	
		
	}
	/**
	 * This method will parse a http response and return whether it
	 * is a new page or not (Content-Type: equals text/html )
	 *  
	 * @param node
	 * @return true if Content-Type header equals text/html
	 * 
	 * @throws TestgenException
	 */
	private boolean checkResponseForNewPage(TRCNode node) 
		throws TestgenException {
		boolean retcode = false;

				
		List dataNodeList = TRCNodeHandler.getSubNodeList(PACKET_S_DATA, node);
		if (dataNodeList.isEmpty())
			throw new TestgenException(TestgenHttpPlugin.
				getResourceString("E_PACKET_NO_DATA"), null);

		// populate currentHttpRequest
		ListIterator itr = dataNodeList.listIterator();
		while (itr.hasNext()) {
			TRCNode dataNode = (TRCNode)itr.next();

			String type = TRCNodeHandler.getAttributeValue(PACKET_S_DATA_A_TYPE, dataNode);
			if (type == null)
				throw new TestgenException(TestgenHttpPlugin.
					getResourceString("E_DATA_NO_TYPE_ATT"), null);

			if (type.equals(PACKET_S_DATA_A_TYPE_V_HTTPMESSAGE)){
				retcode = httpRequestMessageContentType(dataNode);
				break;
			}
			else
				// ignore any others
				continue;
		}
		return retcode;	
	}
	/**
	 * Parse http Request to see if Content-Type: is 
	 * text/html.  Return true to indicate new page.
	 * 
	 * @param dataNode
	 * @return
	 * @throws TestgenException
	 */
	private boolean httpRequestMessageContentType(TRCNode dataNode) 
		throws TestgenException {
		boolean isNewPage = false;
		
		String content = dataNode.getElement().getContent();
		if (content == null)
			throw new TestgenException(TestgenHttpPlugin.
					getResourceString("E_DATA_NO_CONTENT"), null);
		
		StringTokenizer bodyTokenizer = new StringTokenizer(content, CRNL);
		for (int headerCount = 0; bodyTokenizer.hasMoreTokens();
			++headerCount) {
			String header = bodyTokenizer.nextToken();
			if (headerCount == 0) {
				// Do nothing 		
			}
			else {
				int colonIndex = header.indexOf(COLON);
				String name = header.substring(0, colonIndex);
				String value = header.substring(colonIndex + 1, header.length());
				
				// fetch Content-Type
				if (name.trim().equalsIgnoreCase("Content-Type")) {
					String tempstr = value.trim().toLowerCase();
					if (tempstr.trim().startsWith("text/html"))
						isNewPage = true;
					else
						isNewPage = false;
					break;
				}
			}
		}	
		if (!isNewPage) {
			// now check for Redirection - status code of 301 or 302
			isNewPage = checkForRedirect(dataNode);
		}
		return isNewPage;
	}
	/**
	 * Loops through the original list of TRCNode nodes, removing responses 
	 * that contain html or jpeg information (that don't have headers)
	 * 
	 * @throws TestgenException
	 */
	private void remove_unwanted_responses() throws TestgenException {
		LinkedList pageList = null;
		boolean writethisNode = true;
		int totalNodes = 0;
		int i = 0;
		TRCNode node = null;
		LinkedList newPageList = new LinkedList();
		
		pageList = context.getPageList();
		totalNodes = pageList.size();
		for (i = 0; i < totalNodes; i++) {
			node = (TRCNode)pageList.get(i);
			writethisNode = shouldIWriteNode(node);
			if (writethisNode)
				newPageList.add(node);
		}
		int newTotalNodes = newPageList.size();
		if (newTotalNodes > 0){
			pageList.clear();
			context.setPageList(newPageList);
		}
	}
	/**
	 * Determine whether a  node has information we need and should be 
	 * included in the new list of nodes.
	 * 
	 * @param node
	 * @return false if this node is from server and is content only
	 * @throws TestgenException
	 */
	boolean shouldIWriteNode(TRCNode node) throws TestgenException {
		boolean writeNode = true;
		boolean fromClient = false;
		boolean fromServer = false;
		boolean isContent = false;
		boolean isMessage = false;
				
		// populate currentHttpRequest
		
		String from = TRCNodeHandler.getAttributeValue(PACKET_A_FROM, node);
		if (from.equals(PACKET_A_FROM_V_CLIENT)) {
			fromClient = true;
			return true;
		}
		else if (from.equals(PACKET_A_FROM_V_SERVER)) {
			fromServer = true;
		}
		
		List dataNodeList = TRCNodeHandler.getSubNodeList(PACKET_S_DATA, node);
		if (dataNodeList.isEmpty())
			throw new TestgenException(TestgenHttpPlugin.
				getResourceString("E_PACKET_NO_DATA"), null);
		
		ListIterator itr = dataNodeList.listIterator();
		while (itr.hasNext()) {
			TRCNode dataNode = (TRCNode)itr.next();

			String type = TRCNodeHandler.getAttributeValue(PACKET_S_DATA_A_TYPE, dataNode);
			if (type == null)
				throw new TestgenException(TestgenHttpPlugin.
					getResourceString("E_DATA_NO_TYPE_ATT"), null);
			if (type.equals(PACKET_S_DATA_A_TYPE_V_HTTPMESSAGE)) {
				isMessage = true;
				return true;
			}
			if (type.equals(PACKET_S_DATA_A_TYPE_V_HTTPCONTENT)) {
				isContent = true;
				break;
			}
			else
				// ignore any others
				continue;
		}
		if (fromServer && isContent)
			writeNode = false;
		
		return writeNode;
	}
	
	/**
	 * @param node
	 * @return true if this node is a response (from server)
	 */
	boolean checkIsResponse(TRCNode node) {
		boolean isResponse = false;
		
		String from = TRCNodeHandler.getAttributeValue(PACKET_A_FROM, node);
		if (from.equals(PACKET_A_FROM_V_CLIENT)){
			isResponse = false;
		}
		else if (from.equals(PACKET_A_FROM_V_SERVER)) {
			isResponse = true;
		}
		return isResponse;	
	}
	
	/**
	 * @param node
	 * @return true if this node is from the CLIENT 
	 */
	boolean checkIsRequest(TRCNode node) {
		boolean isRequest = false;
	
		String from = TRCNodeHandler.getAttributeValue(PACKET_A_FROM, node);
		if (from.equals(PACKET_A_FROM_V_CLIENT)){
			isRequest = true;
		}
		else if (from.equals(PACKET_A_FROM_V_SERVER)) {
			isRequest = false;
		}
		return isRequest;	
	}
	/**
	 * Check response for status code of 301 or 302 which indicates redirection
	 * 
	 * @param node
	 * @return
	 * @throws TestgenException
	 */
	boolean checkForRedirect(TRCNode node) throws TestgenException {
		
		boolean isRedirect = false;
		String content = node.getElement().getContent();
		if (content == null) {
			throw new TestgenException(TestgenHttpPlugin.
					getResourceString("E_DATA_NO_CONTENT"), null);
		}
		StringTokenizer bodyTokenizer = new StringTokenizer(content, CRNL);
		String statusLine = bodyTokenizer.nextToken();
		// at this point have HTTP/1.1 301 Object Temporarily Moved
		StringTokenizer status = new StringTokenizer(statusLine, BLANK);
		String thisStatus = status.nextToken();  // This is HTTP/1.1
		String actualStatus = status.nextToken();// This is 200 or 302 or 301, etc...
		Integer intStat = new Integer(actualStatus);
		int actualVal = intStat.intValue();
		if ((actualVal == 302) || (actualVal == 301))
			isRedirect = true;
		
		return isRedirect;
	}
	/**
	 * @param node
	 * @return
	 * @throws TestgenException
	 */
	private boolean parseHttpRequest (TRCNode node)
		throws TestgenException
	{
		boolean retcode = false;
		int timestamp = 0;
		String timestampString = TRCNodeHandler.getAttributeValue(PACKET_A_TIMESTAMP, node);
		if (timestampString != null)
			timestamp = Integer.parseInt(timestampString);
				
		List dataNodeList = TRCNodeHandler.getSubNodeList(PACKET_S_DATA, node);
		if (dataNodeList.isEmpty())
			throw new TestgenException(TestgenHttpPlugin.
				getResourceString("E_PACKET_NO_DATA"), null);

		// reset currentHttpRequest
		currentHttpRequest = new ParsedHttpMessage();

		String typeValue = TRCNodeHandler.getAttributeValue(PACKET_A_TYPE, node);
		if (typeValue.equals(PACKET_A_TYPE_V_HTTPS))
			currentHttpRequest.bSSL = true;
		else 
			currentHttpRequest.bSSL = false;

		// populate currentHttpRequest
		ListIterator itr = dataNodeList.listIterator();
		while (itr.hasNext()) {
			TRCNode dataNode = (TRCNode)itr.next();

			String type = TRCNodeHandler.getAttributeValue(PACKET_S_DATA_A_TYPE, dataNode);
			if (type == null)
				throw new TestgenException(TestgenHttpPlugin.
					getResourceString("E_DATA_NO_TYPE_ATT"), null);

			if (type.equals(PACKET_S_DATA_A_TYPE_V_HTTPMESSAGE))
				httpRequestMessage(dataNode);
			else if (type.equals(PACKET_S_DATA_A_TYPE_V_HTTPCONTENT))
				httpRequestContent(dataNode);
			else
				// ignore any others
				continue;
		}
		
		int pageNumber = context.getPageNumber();
		currentHttpRequest.nPageNum = pageNumber;
		String connectionNumber = TRCNodeHandler.getAttributeValue(PACKET_A_CONNECTIONNAME, node);
		Integer thisConn = new Integer(connectionNumber);
		currentHttpRequest.nConnID = thisConn.intValue();
		ILoop thisLoop = null;
		currentHttpRequest.timeStampString = timestampString;

		return retcode;
	}
	/**
	 * @param thisMessage
	 * @return
	 */
	private boolean createPage(ParsedHttpMessage thisMessage){
		boolean retcode = false;
		int myenum = 0;
		
		ILoop thisLoop = null;
		int timestamp = 0;
		
		int pageNumber = context.getPageNumber();
		
		String timestampString = thisMessage.timeStampString;
		if (timestampString != null)
			timestamp = Integer.parseInt(timestampString);
		if ((thisMessage.enumPage == HttpRequest.PAGE_ONLY) || (thisMessage.enumPage == HttpRequest.PAGE_START)
			|| (pageNumber == 0))
		{
			pageNumber += 1;
			context.setPageNumber(pageNumber);
			// thisLoop = context.createNewPage(pageNumber);
			String newPageName = thisMessage.host + thisMessage.path;
			thisLoop = context.createNewPage(newPageName);
		}
		String testCaseName = thisMessage.method + " " +
			thisMessage.host;
		context.newTestCaseContext(testCaseName);
		thisMessage.nPageNum = pageNumber;
		// fetch and populate RequestHelper...
		RequestHelper requestHelper = context.getRequestHelper();
		
		// ... headers
		requestHelper.setAttribute(RequestHelper.ATT_METHOD,
			thisMessage.method);
		requestHelper.setAttribute(RequestHelper.ATT_HOST,
			thisMessage.host);
		requestHelper.setAttribute(RequestHelper.ATT_PORT,
			Integer.toString(thisMessage.port));
		requestHelper.setAttribute(RequestHelper.ATT_ABS_PATH,
			thisMessage.path);
		requestHelper.setAttribute(RequestHelper.ATT_VERSION,
			thisMessage.version);
		for (int i = 0; i < thisMessage.headerNames.size(); ++i)
			requestHelper.setHeader(((String)thisMessage.headerNames.
				get(i)).trim(), ((String)thisMessage.headerValues.
				get(i)).trim());

		//  New attributes from RequestHelper mdd
		// ATT_PAGE_NUM = 8;
		// ATT_PAGE_ORDER = 9;
		// ATT_CONN_ID = 10;
		// ATT_CONN_ORDER = 11;
		// ATT_IS_SSL = 12;

		requestHelper.setAttribute(RequestHelper.ATT_PAGE_NUM,
			Integer.toString(thisMessage.nPageNum));
		requestHelper.setAttribute(RequestHelper.ATT_PAGE_ORDER,
			Integer.toString(thisMessage.enumPage));
		requestHelper.setAttribute(RequestHelper.ATT_CONN_ID,
			Integer.toString(thisMessage.nConnID));
		if (thisMessage.bSSL)
			requestHelper.setAttribute(RequestHelper.ATT_IS_SSL, "true");
		else
			requestHelper.setAttribute(RequestHelper.ATT_IS_SSL, "false");
			
		// ... body
		requestHelper.setAttribute(RequestHelper.ATT_BODY,
		thisMessage.body.toString());

		// ... calculate and set request's think time
		HashMap	myLastRecvHashMap = context.getLastRecvHashMap();
		
		int thinkTime = 0;
		Integer lastRecvInteger = (Integer)myLastRecvHashMap.get(TRCContext.GLOBAL_LASTRECV);
		if (lastRecvInteger != null) {
			int lastRecv = lastRecvInteger.intValue();
			// thinkTime = (lastRecv == -1) ? timestamp : timestamp - lastRecv;
			thinkTime = (lastRecv == -1) ? 0 : timestamp - lastRecv;
			if (thinkTime < 0)
				thinkTime = 0;
		}
		requestHelper.setAttribute(RequestHelper.ATT_THINK_TIME,
			Integer.toString(thinkTime));
		
		return retcode;
	}
	
	/**
	 * @param node
	 * @return
	 * @throws TestgenException
	 */
	private ResponseInfo parseHttpResponse(TRCNode node) throws TestgenException {
		ResponseInfo thisResponse = new ResponseInfo();
		HashMap	myLastRecvHashMap = context.getLastRecvHashMap();
			
		int timestamp = Integer.parseInt(TRCNodeHandler.getAttributeValue(PACKET_A_TIMESTAMP,
			node));
		myLastRecvHashMap.put(TRCContext.GLOBAL_LASTRECV, new Integer(timestamp));
		context.setLastRecvHashMap(myLastRecvHashMap);
		thisResponse.timeStampString = TRCNodeHandler.getAttributeValue(PACKET_A_TIMESTAMP,node);
		thisResponse.newPage = checkResponseForNewPage(node);
		
		return thisResponse;
	}

	/**
	 * @throws TestgenException
	 * This routine will reorder the node list in connections then use the new
	 * this list to process.
	 */
	private void reorderNodeList() throws TestgenException {
		int totalNodes = -1;
		int i = 0;
		remove_unwanted_responses();
		LinkedList originalNodeList = context.getPageList();
		totalNodes = originalNodeList.size();
		TRCNode node; 
		int connectionIndex = -1;
		int connectionNumber = -1;
		
		for (i = 0; i < totalNodes; i++){
			// check connection number first
			node = (TRCNode)originalNodeList.get(i);
			String strConnectionNumber = TRCNodeHandler.getAttributeValue(PACKET_A_CONNECTIONNAME, node);	
			Integer thisConn = new Integer(strConnectionNumber);
			connectionNumber = thisConn.intValue();
			
			connectionIndex = findConnection(connectionNumber);
			if (connectionIndex == -1) {
				addConnectionNode(connectionNumber,node); 
			}
			else {
				addConnectionChild(connectionIndex, node);
			}
		} 
		// Now replace old List with New List
		createNewPageList();
		int newTotalNodes = newPageList.size();
		
		
		
	}
	/**
	 * This routine will return the index for this connection, if it exists
	 * 
	 * @param connectionNumber 
	 * 		the connection number to find
	 * @return 
	 * 		The index of the found connection number or 
	 * 		-1 if not found
	 */
	private int findConnection(int connectionNumber) {
		int foundIndex = -1;
		int numberOfConnections = -1;
		int i = 0;
		ConnectionNode thisConnection;
		
		numberOfConnections = connectionsList.size();
		if (numberOfConnections == 0) {
			foundIndex = -1;	
		}
		else {
			for (i = 0; i < numberOfConnections; i++) {
				thisConnection = (ConnectionNode)connectionsList.get(i);
				if (thisConnection.connectionNumber == connectionNumber) {
					foundIndex = i;
					break;
				}	
			}
		}
		return foundIndex;
	}
	
	/**
	 * This method creates a new Connection node, and adds newNode as
	 * the first child of that node
	 * 
	 * @param connectionNumber
	 * @param newNode
	 * @return
	 */
	private boolean addConnectionNode(int connectionNumber, TRCNode newNode){
		boolean retcode = true;
			
		// Create a new ConnectionNode
		ConnectionNode newConnectionNode = new ConnectionNode();
		// Add the connection number and the new child
		newConnectionNode.connectionNumber = connectionNumber;
		newConnectionNode.childrenNodes = new LinkedList();
		newConnectionNode.childrenNodes.add(newNode);
		// Add this connectionNode to the ConnectionsList 
		connectionsList.add(newConnectionNode);
		
		
		return retcode;
	}
	/**
	 * This method adds newNode as a child of the Connection node found
	 * at connectionIndex
	 * 
	 * @param connectionIndex
	 * @param newNode
	 * @return
	 */
	public boolean addConnectionChild(int connectionIndex, TRCNode newNode){
		boolean retcode = true;
		
		ConnectionNode thisConnection = (ConnectionNode)connectionsList.get(connectionIndex);
		thisConnection.childrenNodes.add(newNode);
		return retcode;	
	}
	
	/**
	 * This method goes through each Connection node, and adds each child
	 * found for that Connection node to the reordered list of TRCNode nodes
	 * 
	 * This newPageList will be processed to create the testsuite.
	 * 
	 */
	private void createNewPageList() {
		int totalConnections = -1;
		int i = 0;
		int j = 0; 
		ConnectionNode thisConnection = null;
		TRCNode thisNode = null;
		int totalChildren = -1;
		
		totalConnections = connectionsList.size();
		for (i = 0; i < totalConnections; i++){
			thisConnection = (ConnectionNode)connectionsList.get(i);
			totalChildren = thisConnection.childrenNodes.size();
			for (j = 0; j < totalChildren; j++){
				thisNode = (TRCNode)thisConnection.childrenNodes.get(j);
				newPageList.add(thisNode);
			}
		}
	}
	
	/**
	 * This method goes through the newly reordered TRCNode node list, 
	 *	and creates Behavior Model entries for each. One of the elements of 
	 *	the behavior model is page enumeration (begin, middle, end or only).
	 *	Since you don't know the enumeration of a particular request/response
	 *	pair until you have looked at the subsequent pair, we must store the 
	 *	request and compare the current request to the previous response to 
	 *	determine enumeration.
	 *
	 *		Determine if node is a HttpRequest or HttpResponse
	 *		If HttpRequest
	 *			parse request and populate currentHttpRequest
	 *		If HttpResponse
	 *			parse request and determine if it is a New Page 
	 *
	 *		If have both the request and response
	 *			Determine the Page Enumeration for Previous Response
	 *			Write Previous Response
	 *
	 * @throws TestgenException
	 */
	public void writeNewPageList()  throws TestgenException {
		int totalNodes = -1;
		int i = 0;
		boolean isARequest = false;
		TRCNode currentNode = null;
		boolean haveRequest = false;
		boolean haveResponse = false;
		boolean isNewPage = false;
		boolean err = false;
		ResponseInfo responseInfo = null;

		// Process the entire node list
		totalNodes = newPageList.size();
		for (i = 0; i < totalNodes; i++) {
			currentNode = (TRCNode)newPageList.get(i);
			isARequest = checkIsRequest(currentNode);
			if (isARequest) {
				haveRequest = true;
				err = parseHttpRequest(currentNode);
			}
			else {
				haveResponse = true;
				responseInfo = parseHttpResponse(currentNode);
				currentHttpRequest.timeStampString = responseInfo.timeStampString;
				currentHttpRequest.newPage = responseInfo.newPage;
			}
			
			if (haveRequest && haveResponse) {
				haveRequest = false;
				haveResponse = false;
				if (totalNodes == 2) {
					currentHttpRequest.newPage = true;
					currentHttpRequest.enumPage = HttpRequest.PAGE_ONLY;
					createPage(currentHttpRequest);
					break;
				}
				if ((previousHttpRequest == null) || (i == 1)) {
					// This is the First Page
					currentHttpRequest.newPage = true;
					currentHttpRequest.enumPage = HttpRequest.PAGE_START;
					previousHttpRequest = currentHttpRequest;
				}
				else {
					if (previousHttpRequest.newPage) {
						// Previous Request IS a new page
						if (currentHttpRequest.newPage) {
							previousHttpRequest.enumPage = HttpRequest.PAGE_ONLY;
						}
						else {
							// Current Request NOT a new page
							previousHttpRequest.enumPage = HttpRequest.PAGE_START;
						}
					}
					else {
						// Previous Message NOT a new page
						if (currentHttpRequest.newPage) {
							previousHttpRequest.enumPage = HttpRequest.PAGE_END;
						}
						else {
							previousHttpRequest.enumPage = HttpRequest.PAGE_MID;
						}
					}
					createPage(previousHttpRequest);
					previousHttpRequest = currentHttpRequest;
					previousHttpRequest.enumPage = HttpRequest.PAGE_END;
					if (i == (totalNodes - 1)) {
						if (previousHttpRequest.newPage)
							previousHttpRequest.enumPage = HttpRequest.PAGE_ONLY;
						createPage(previousHttpRequest);
					}
					
				}
			}
		}
	}
}
