/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.sd.ui.internal.model;

import java.util.*;

import org.eclipse.hyades.sd.ui.internal.loader.*;

/**
 * A Graph will always be the holder of all top level objects. If being used as 
 * the root of a view, such as a sequence diagram, the association topNodes 
 * represent the objects across the diagram from left to right in order.
 * Typically the top level nodes will be instances of NodeContainer.
 */
public class Graph
{
  /**
  * Graph provides a set of static type values that indicate the type of 
  * graph that has been populated. For example PROCESSoverCLASS is a grpah that
  * consists of Process objects as TopNodes with Class objects as internal nodes.
  * <ul/>
  * <li>UNKNOWN indicates the graph is untyped.
  * <li>NODEoverPROCESS indicates the graph shows the interaction between processes in nodes.
  * <li>PROCESSoverTHREAD indicates the graph shows the interaction between threads in processes.
  * <li>PROCESSoverOBJECT indicates the graph shows the interaction between objects in processes.
  * <li>PROCESSoverCLASS indicates the graph shows the interaction between classes in processes.
  * <li>OBJECToverMETHOD indicates the graph shows the interaction between methods in objects.
  * <li>CLASSoverMETHOD indicates the graph shows the interaction between methods in classes.
  * 
  * <li>LOGoverRECORD indicates the graph shows the interaction between records in logs.
  * <li>THREADoverRECORD indicates the graph shows the interaction between records in threads.
  * <eul/>
  * In all cases the connections are based on method calls. The Y in XoverY determines the
  * type of object that the topNodes contains. 
  */
  public static final int UNKNOWN = 0;
  public static final int NODEoverAGENT = 1;
  
  public static final int NODEoverPROCESS = 2;
  public static final int AGENToverPROCESS = 3;

  public static final int NODEoverTHREAD = 4;
  public static final int AGENToverTHREAD = 5;
  public static final int PROCESSoverTHREAD = 6;
  
  public static final int NODEoverOBJECT = 7;
  public static final int AGENToverOBJECT = 8;
  public static final int PROCESSoverOBJECT = 9;
  public static final int THREADoverOBJECT = 10;
  
  public static final int NODEoverCLASS = 11;
  public static final int AGENToverCLASS = 12;
  public static final int PROCESSoverCLASS = 13;
  public static final int THREADoverCLASS = 14;
  
  public static final int NODEoverMETHOD = 15;
  public static final int OBJECToverMETHOD = 16;
  public static final int CLASSoverMETHOD = 17;
  public static final int THREADoverMETHOD = 18;
  public static final int PROCESSoverMETHOD = 19;
  public static final int AGENToverMETHOD = 20;

  public static final int LOGoverRECORD = 21;
  public static final int THREADoverRECORD = 22;

  /**
   * name represents the logical name of a graph. The value will be set by the graph 
   * creator, but can be changed for convience by the graph consumer.
   */
  private String name;

  /**
   * A graph type in an enumeration value. The type is meant to reflect the type of 
   * content it contains.
   * The values are the XoverY static member of this class. 
   */
  private int type = 0;

  /**
   * The userArea Hashtable is an open structure for users of this object to store
   * anything they want. Graph provides no management of this object.
   */
  private Hashtable userArea = null;
  /**
   * The lastReadNode GraphNode is to be used by the readers of a Graph to indicate
   * in a sequence of nodes what has been read. Graph provides no management of this reference.
   */
  private GraphNode lastReadNode = null;
  /**
   * The topNodes array of GrapNode is used to hold the GraphNodes at the top of each
   * column in the viewer. The reader of a Graph should not alter this array.
   */
  private GraphNode topNodes[] = new GraphNode[3];
  /**
   * nextNewGraphNode is used internally to keep track of the number of nodes created.
   */
  private int nextNewGraphNode = 0;
  /**
   * The modelLoader is used internally to hold a reference to the model loader instance used
   * to populate this Graph.
   */
  private IModelLoader modelLoader = null;

  private ContentFactory contentFactory = null;
  private boolean filtered = false; 
  private double startTime = 0;
  private double endTime = 0;
  private double weightUnit = 0;
  private double weightRange = 255;
  private int currentMaxIncrement = 0; //the size of incremenet list
  private Increment incrementList[] = new Increment[5000];

  private Graph() {}
  
  protected Graph(int inputType) {
    this();
    setType(inputType);
  }
  
  public void load() {
	if (getModelLoader() != null)
	  getModelLoader().load();
  }
  
  public static GraphFactory getGraphFactory() {
	return GraphFactory.getGraphFactory();
  }
  
  public GraphNode createGraphNode() {
	return contentFactory.createGraphNode();
  }

  public NodeConnection createNodeConnection() {
	return contentFactory.createNodeConnection();
  }
 
  public NodeContainer createNodeContainer() {
	return contentFactory.createNodeContainer();
  }

  public int getTopNodeCount() {
    return nextNewGraphNode;
  }

  public void addTopNode(GraphNode node)
  {
    try {
      topNodes[nextNewGraphNode] = node;
    }
    catch (ArrayIndexOutOfBoundsException e) {
      GraphNode[] tempArray = new GraphNode[2 * topNodes.length];
      System.arraycopy(topNodes, 0, tempArray, 0, topNodes.length);
      topNodes = tempArray;
      topNodes[nextNewGraphNode] = node;
    }
    node.setIndexInContainer(nextNewGraphNode);
    nextNewGraphNode++;
  }

  public double getStartTime() {
    return startTime;
  }
  protected void setStartTime(double time) {
    if ((startTime == 0) || (startTime > time))
    	startTime = time;
  }
  
  public double getEndTime() {
    return endTime;
  }
  protected void setEndTime(double time) {
    if (time > endTime)
         endTime = time;
  }

  protected double getWeightUnit(double timeSpan) {
    if (weightUnit == 0)
      weightUnit = timeSpan / weightRange;
    return weightUnit;
  }
  
  public Increment[] getIncrementWeights()
  {
    Increment[] iList = getIncrementList();
    //    double currentWeightUnit = getWeightUnit(getTimeRange());
    double currentWeightUnit = getWeightUnit(getLargestIncrementSpread(iList));
    if (iList[0] != null) {
      iList[0].setWeight(currentWeightUnit, iList[0]);
    }
    for (int i = 1; i < currentMaxIncrement; i++) {
      iList[i].setWeight(currentWeightUnit, iList[i - 1]);
    }
    return iList;
  }
  
  private double getLargestIncrementSpread(Increment[] iList)
  {
    double spread = 0;
    double currentSpread = 0;

    for (int i = 1; i < currentMaxIncrement; i++)
    {
      currentSpread = iList[i].getTime() - iList[i - 1].getTime();
      if (currentSpread > spread)
        spread = currentSpread;
    }
    return spread;
  }

  public Increment addIncrementAppendDup(double newTime)
  {
  	//first create a new Incremenet with newTime and append to the end of the increment list
	Increment increment = createAndAppendIncrement(newTime);
    
    int target = increment.getValue();
    currentMaxIncrement++;
    
    // now recover from the assumption walking backwards since all these increments are ordered
    // and need to be bumped out one assuming an insertion
    if (currentMaxIncrement > 1)
    {
      // find the insertion point which is after the last Increment with the same or smaller time	
      for (int i = currentMaxIncrement - 2; i >= 0; i--)
      {
        if (newTime < incrementList[i].getTime()) {
          target = i; 	// a candidate position
        }
        else {
          moveIncrementAndAdjust(increment, target);
          i = -1; 		// get outahere
        }
      }
    }
    return increment;
  }
  
  /**
   * create an increment with time given and append to the list last position
   * @param newTime
   * @return Increment
   */
  private Increment createAndAppendIncrement(double newTime)
  {
    Increment increment = new Increment();
    increment.setValue(currentMaxIncrement);
    increment.setTime(newTime);
    setStartTime(newTime);
    setEndTime(newTime);
    // boldly assume the new increment is also the newest time
    try
    {
      incrementList[currentMaxIncrement] = increment;
    }
    catch (ArrayIndexOutOfBoundsException e)
    {
      Increment[] tempArray = new Increment[incrementList.length + 5000];
      System.arraycopy(incrementList, 0, tempArray, 0, incrementList.length);
      incrementList = tempArray;
      incrementList[currentMaxIncrement] = increment;
    }
    return increment;
  }
  
  public void moveIncrementAndAdjust(Increment increment, int target)
  {
    // if increment is already in the correct position smile
    if (increment.getValue() == target)
      return;

    // if increment is the last in array just shift the array right and insert it
    if (increment.getValue() == currentMaxIncrement - 1) {
      shiftIncrementsRight(target, currentMaxIncrement - 2);
    }
    // if increment is between the insertion point and the end only move what
    // is in between (shift right) and then insert
    else if (increment.getValue() > target) {
      shiftIncrementsRight(target, increment.getValue() - 1);
    }
    // if the increment comes from before the insertion point only move what is 
    // in between (shift left) and then insert
    else if (increment.getValue() < target) {
      shiftIncrementsLeft(increment.getValue() + 1, target);
    }

	incrementList[target] = increment;
	increment.setValue(target);
  }
  
  private void shiftIncrementsRight(int left, int right)
  {
    // parms are source array, start pnt, target array, target pnt, length
    System.arraycopy(
      incrementList,
      left,
      incrementList,
      left + 1,
      right - left + 1);
    for (int i = left + 1; i < right + 2; i++)
      incrementList[i].setValue(i);
  }
  
  private void shiftIncrementsLeft(int left, int right)
  {
	// parms are source array, start pnt, target array, target pnt, length
    System.arraycopy(
      incrementList,
      left,
      incrementList,
      left - 1,
      right - left + 1);
    for (int i = left - 1; i < right; i++)
      incrementList[i].setValue(i);
  }
  
  public void setMaxIncrement(int i) {
	currentMaxIncrement = i;
  }
  public int getMaxIncrement() {
	return currentMaxIncrement;
  }

  public Increment[] getIncrementList() {
    return incrementList;
  }
  public void setIncrementList(Increment[] list) {
    incrementList = list;
  }
  
  public void setFiltered(boolean setting) {
	filtered = setting;
  }
  public boolean getFiltered() {
	return filtered;
  }

  public void setModelLoader(IModelLoader loader) {
	modelLoader = loader;
	return;
  }
  public IModelLoader getModelLoader() {
	return modelLoader;
  }

  protected void setContentFactory(ContentFactory factory) {
	contentFactory = factory;
  }
  public ContentFactory getContentFactory() {
	return contentFactory;
  }
  
  public void setName(String inputName) {
	name = inputName;
  }
  public String getName() {
	return name;
  }

  protected void setType(int inputType) {
	type = inputType;
  }
  public int getType() {
	return type;
  }

  public void setLastReadNode(GraphNode node) {
	lastReadNode = node;
  }
  public GraphNode getLastReadNode() {
	return lastReadNode;
  }

  public GraphNode[] getTopNodes() {
	return topNodes;
  }
  public void setTopNodes(GraphNode[] nodes) {
	topNodes = nodes;
  }

  public Hashtable getUserArea() {
	return userArea;
  }
  public void setUserArea(Hashtable area) {
	userArea = area;
  }

}
