/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.sd.ui.internal.util;

import org.eclipse.draw2d.Graphics;
import org.eclipse.draw2d.geometry.Rectangle;
import org.eclipse.hyades.sd.ui.SDPlugin;
import org.eclipse.hyades.sd.ui.internal.model.ColumnFigure;
import org.eclipse.hyades.sd.ui.internal.model.ColumnNode;
import org.eclipse.hyades.sd.ui.internal.model.GraphNode;
import org.eclipse.hyades.sd.ui.internal.model.NodeLink;
import org.eclipse.hyades.sd.ui.internal.views.SDViewerUI;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.DisposeEvent;
import org.eclipse.swt.events.DisposeListener;
import org.eclipse.swt.events.MouseAdapter;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.events.MouseMoveListener;
import org.eclipse.swt.events.PaintEvent;
import org.eclipse.swt.events.PaintListener;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.ImageData;
import org.eclipse.swt.graphics.PaletteData;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.graphics.RGB;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Canvas;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;

/**
 * CONTEXT_ID ovbn0000 for Over View button in Sequence Diagram.
 * CONTEXT_ID ovar0000 for OverView area in Sequence Diagram.
 */

public class DrawAreaOverview extends MouseAdapter implements SelectionListener, PaintListener, MouseMoveListener {

    Composite graphComposite;
    Button overviewButton;
    Canvas overviewCanvas = null;
    SelectorFigure selectorFigure= null;
    SDViewerUI sdViewerUI;
    GC gc;
    boolean expanded = false;
    Point parentSize;
    int overviewResizeRatio=1;
    int canvasSizeX, canvasSizeY;
    boolean mouseDown = false;
    boolean firstTime = true;
    int mouseDownX=0, mouseDownY=0;
    int mouseOffsetX=0, mouseOffsetY=0;
    Point overviewButtonPreferredSize;
    int headerSize;
    boolean drawLifeLines = true;
    int[] columnPositions = null;
	int boxSpace;
	int boxHeight;

	/**
	 * Constructor for DrawAreaOverview
	 */
	public DrawAreaOverview(Composite graphComposite, Point pS){
		this.graphComposite = graphComposite;
		parentSize = pS;
		sdViewerUI = (SDViewerUI)graphComposite.getParent();
		overviewButton = new Button(graphComposite,SWT.PUSH);
		overviewButton.setText(SDPlugin.getString("STR_OVERVIEW"));
		overviewButton.setCapture(false);
		overviewButtonPreferredSize = overviewButton.computeSize(SWT.DEFAULT, SWT.DEFAULT, true);
		overviewButton.setSize(overviewButtonPreferredSize.x, overviewButtonPreferredSize.y);
		headerSize = overviewButtonPreferredSize.y;
		setButtonLocation();
		overviewButton.addSelectionListener(this);
		overviewButton.moveAbove(null);
		boxSpace = LogCDrawUtils.getInitBoxSpace();
		boxHeight = LogCDrawUtils.getInitBoxSpace();
		
		org.eclipse.ui.help.WorkbenchHelp.setHelp(
			overviewButton,
			SDPlugin.getPluginId() + ".ovbn0000");
		org.eclipse.ui.help.WorkbenchHelp.setHelp(
			graphComposite,
			SDPlugin.getPluginId() + ".ovar0000");	
			
	}
	
	/**
	 * reset button location when overview is closed
	 */
	private void setButtonLocation(){
		
	   if(System.getProperty("os.name").startsWith("Windows"))
	      overviewButton.setLocation(parentSize.x-overviewButtonPreferredSize.x-13,parentSize.y-overviewButtonPreferredSize.y);
	   else
	      overviewButton.setLocation(parentSize.x-overviewButtonPreferredSize.x-24,parentSize.y-overviewButtonPreferredSize.y-9);
	}
	
	//PaintListener ->begin
	public void paintControl(PaintEvent e) {

		if(expanded)
		   drawOverview(e);
	}
	//PaintListener ->end
	
	//SelectionListener ->begin
	public void widgetSelected(SelectionEvent e){
	   if(parentSize.x==0||parentSize.y==0)
	      return;
	   if(expanded){
	   	  closeUI();
	   }else{
	   	  try{
	   	      setCanvasSize(sdViewerUI.getEndOfTime()*LogCDrawUtils.getTimeUnit()+LogCDrawUtils.getInitBoxSpace());
	   	  }catch(java.lang.ArithmeticException ae){
	   	   	  return;
	   	  }
	   	  openUI();
	   }
	   expanded = !expanded;
	}
	public void widgetDefaultSelected(SelectionEvent e){}
	//SelectionListener ->end
	
	public void close(){
		if(expanded){
		   closeUI();
		   expanded = !expanded;
		}
	}
	
	private void openUI(){
		if(overviewCanvas == null){
	   	     overviewCanvas = new Canvas(graphComposite,SWT.BORDER);
	   	     overviewCanvas.setBackground(LogCDrawUtils.getBackgroundColor());
	         overviewCanvas.addPaintListener(this);
	         overviewCanvas.addMouseMoveListener(this);
	         overviewCanvas.addMouseListener(this); 
	         selectorFigure = new SelectorFigure();
	         gc = new GC(overviewCanvas);	  
	         
	         overviewCanvas.addDisposeListener(new DisposeListener() {
					public void widgetDisposed(DisposeEvent e) {
					if (selectorFigure!=null) selectorFigure.dispose();
					selectorFigure=null;
					if (gc!=null) gc.dispose();
					gc=null;
				}
			});   
		}
	   	  overviewCanvas.setSize(canvasSizeX,canvasSizeY);
		  overviewCanvas.setLocation(parentSize.x-overviewCanvas.getSize().x,parentSize.y-overviewCanvas.getSize().y);
		  int overviewRatio = LogCDrawUtils.getOverviewRatio();
		  Point canvasLocation = overviewCanvas.getLocation();
		  selectorFigure.moveFigure(graphComposite.getHorizontalBar().getSelection()/overviewRatio, graphComposite.getVerticalBar().getSelection()*LogCDrawUtils.getTimeUnit()/overviewRatio);
		  overviewButton.setLocation(overviewButton.getLocation().x,canvasLocation.y-overviewButton.getSize().y);
		  overviewCanvas.moveAbove(null);
	}
	
	private void closeUI(){
		overviewCanvas.setVisible(false);
		if (overviewCanvas!=null) overviewCanvas.dispose();
	   	overviewCanvas=null;
		if (selectorFigure!=null) selectorFigure.dispose();
		selectorFigure=null;
		if (gc!=null) gc.dispose();
		gc=null;
	   	sdViewerUI.redraw();
	   	setButtonLocation();
	}
	
	public void adjustParentSize(int x, int y){
		parentSize.x = x;
		parentSize.y = y;
		setButtonLocation();
	}
	
	/**
	 * calculate overview size, and its scale ratio
	 * @param y
	 */
	public void setCanvasSize(int y){
		float realY = y;
		float realX=0;
		GraphNode [] columnFigures = sdViewerUI.getColumnFigures();
		ColumnFigure columnFigure;
		for(int i=0;i<sdViewerUI.getGraph().getTopNodeCount(); i++){
			columnFigure = (ColumnFigure)columnFigures[i];
			realX+=columnFigure.getWidth();
		}
       float availableX = parentSize.x;
		float availableY = parentSize.y-headerSize-overviewButtonPreferredSize.y;
		float resizeRatioFloat = Math.max(realX/availableX , realY/availableY);
		overviewResizeRatio = (new Float(resizeRatioFloat)).intValue();
		overviewResizeRatio++;
		if(overviewResizeRatio<1)
		   overviewResizeRatio = 1;
		LogCDrawUtils.setOverviewRatio(overviewResizeRatio);
	   	canvasSizeY=(new Float(realY/overviewResizeRatio)).intValue();
	   	canvasSizeX=(new Float(realX/overviewResizeRatio)).intValue();
		if(canvasSizeX/sdViewerUI.getGraph().getTopNodeCount()<3)
		   drawLifeLines = false;
		else
		   drawLifeLines = true;
	}
	
	private void drawOverview(PaintEvent e){
		
		gc.fillRectangle(0,0,overviewCanvas.getSize().x,overviewCanvas.getSize().y);
		int columnCount = sdViewerUI.getGraph().getTopNodeCount();
		//if(columnPositions==null || sdEditorUI.getGraph().getTopNodeCount()!=columnPositions.length)
		columnPositions = new int[columnCount];
		GraphNode [] columnFigures = sdViewerUI.getColumnFigures();
		ColumnFigure columnFigure;
		ColumnNode node;
		NodeLink nodeConnection;
		int position;
		
		//draw life lines of each columnFigure
		for(int i=0;i<columnCount; i++){
		   columnFigure = (ColumnFigure)columnFigures[i];
		   columnPositions[i]=columnFigure.drawSelf(gc,new Point(canvasSizeX/columnCount,canvasSizeY),drawLifeLines,boxHeight,overviewResizeRatio);
		}
		
		int timeUnit = LogCDrawUtils.getTimeUnit();
		Color originalColor = gc.getForeground();
		//gc.setForeground(LogCDrawUtils.getConnectionColor());
		gc.setForeground(LogCDrawUtils.getLogConnectionColor());
		
		//draw node connections
		for(int i=0;i<columnCount; i++){
			columnFigure = (ColumnFigure)columnFigures[i];
		    for (int j = 0; j < columnFigure.getInternalNodeCount(); j++){
			   node = (ColumnNode) columnFigure.getInternalNodes()[j];
			   for (int k = 0; k < node.getSourceConnections().length; k++) {
			      if(node.getSourceConnections()[k]!=null){
			         nodeConnection = (NodeLink) node.getSourceConnections()[k];
			         position = (boxSpace + timeUnit* nodeConnection.getStartIncrementValue())/overviewResizeRatio;
			         gc.drawLine(columnPositions[nodeConnection.getSource().getContainer().getIndexInContainer()],
			                     position,
			                     columnPositions[nodeConnection.getTarget().getContainer().getIndexInContainer()],
			                     position);
			      }
		       }
		    }
		}
		gc.setForeground(originalColor);
		originalColor.dispose();
		
		selectorFigure.paintFigure(new org.eclipse.draw2d.SWTGraphics(gc));
		if(firstTime){
		   mouseDownX=selectorFigure.getTargetBounds().width/2;
		   mouseDownY=selectorFigure.getTargetBounds().height/2;
		   firstTime=false;
		}
	}
	
	public void redraw(){
	   //if(overviewCanvas!=null)
	      //drawOverview();	
	}
	
	//MouseAdapter ->begin
	public void mouseDoubleClick(MouseEvent e){
			if(!selectorFigure.positionContained(e.x,e.y)){
			   mouseDown=true;
			   moveShadowDC(e.x, e.y);
			   mouseDown=false;
			}
		}
	
	public void mouseDown(MouseEvent e){
	   if(selectorFigure.positionContained(e.x,e.y)){
	      mouseDown = true;
	      mouseDownX = e.x;
	      mouseDownY = e.y;
	      mouseOffsetX = e.x-selectorFigure.getTargetBounds().x;
	      mouseOffsetY = e.y-selectorFigure.getTargetBounds().y; 
	   }
	}
	
	public void mouseUp(MouseEvent e){
	      mouseDown = false;
	}
	//	MouseAdapter ->end
	
	//MouseMoveListener ->begin
	public void mouseMove(MouseEvent e){
		moveShadow(e.x,e.y);
	}
	//MouseMoveListener ->end
	
	private void moveShadowDC(int x, int y){
		
		if(mouseDown){
	   	   Rectangle tBounds = selectorFigure.getTargetBounds();
	   	   tBounds.x=x-tBounds.width/2; 	//mouse position is center of the new target shadow
	   	   if(tBounds.x<0){
	   	      tBounds.x=0;
	   	   }else if(tBounds.getRight().x>canvasSizeX){
		      tBounds.x=canvasSizeX-tBounds.width;
	   	   }
	   	   mouseDownX=tBounds.x+tBounds.width/2; //set MouseDownX to be center of new target shadow
	   	   
	   	   tBounds.y=y-tBounds.height/2;
	   	   if(tBounds.y<0){
	   	      tBounds.y=0;
	   	   }else if(tBounds.getBottom().y>canvasSizeY){
		      tBounds.y=canvasSizeY-tBounds.height;
	   	   }
		   mouseDownY=tBounds.y+tBounds.height/2; //set MouseDownY to be center of new target shadow
	   	   
	   	   overviewCanvas.redraw();
	   	   selectorFigure.paintFigure(new org.eclipse.draw2d.SWTGraphics(gc));
		   sdViewerUI.setScrollbars(tBounds.getLeft().x*overviewResizeRatio, (tBounds.getTop().y*overviewResizeRatio+LogCDrawUtils.getInitBoxSpace())/LogCDrawUtils.getTimeUnit());
	   	   mouseDown=false;
		}
	}
	
	private void moveShadow(int x, int y){
		if(mouseDown){
			int xShift = x-mouseDownX;
			int yShift = y-mouseDownY;
			int xStart=0,xEnd=0,yStart=0,yEnd=0; //area need to be redrawn
			
			Rectangle tBounds = selectorFigure.getTargetBounds();
		
			//calculate the area need to be redrawn for the move event
			if(xShift==0 && yShift==0)
			   return;
			if(xShift>0){
				xStart=tBounds.getLeft().x;
				xEnd=xStart+tBounds.width+xShift;
			}else if(xShift<0){
				xEnd=tBounds.getRight().x;
				xStart=xEnd-tBounds.width+xShift;
			}else if(xShift==0){
			    xStart=tBounds.getLeft().x-1;
			    xEnd=tBounds.getRight().x+1;	
			}
			if(yShift>0){
				yStart=tBounds.getTop().y;
				yEnd=yStart+tBounds.height+yShift;
			}else if(yShift<0){
				yEnd=tBounds.getBottom().y;
				yStart=yEnd-tBounds.height+yShift;
			}else if(yShift==0){
			    yStart=tBounds.getTop().y-1;
			    yEnd=tBounds.getBottom().y+1;	
			}

			selectorFigure.moveFigure(xShift+x-tBounds.x-mouseOffsetX, yShift+y-tBounds.y-mouseOffsetY);
			selectorFigure.paintFigure(new org.eclipse.draw2d.SWTGraphics(gc));
			
			xEnd = Math.max(xEnd,0);
			yEnd = Math.max(yEnd,0);
			xStart = Math.max(xStart,0);
			yStart = Math.max(yStart,0);
			overviewCanvas.redraw(xStart,yStart,(xEnd-xStart),(yEnd-yStart),false);
			sdViewerUI.setScrollbars(tBounds.x*overviewResizeRatio, (tBounds.y*overviewResizeRatio)/LogCDrawUtils.getTimeUnit());
			mouseDownX=x;
			mouseDownY=y;
		}
	}
	
	public void positionShadow(){
		
	   if(expanded){
	   	  int overviewRatio = LogCDrawUtils.getOverviewRatio();
	   	  selectorFigure.positionFigure(graphComposite.getHorizontalBar().getSelection()/overviewRatio, graphComposite.getVerticalBar().getSelection()*LogCDrawUtils.getTimeUnit()/overviewRatio);
	      overviewCanvas.redraw();
	   }
	}
	
	private class SelectorFigure {

	private Image image;
	private Rectangle iBounds = new Rectangle(0, 0, 1, 1);
	Rectangle targetBounds;
	int overviewRatio;
	{
	Display display = Display.getDefault();
	PaletteData pData = new PaletteData(0xFF, 0xFF00, 0xFF0000);
	RGB rgb = new RGB(255, 20, 20);
	int fillColor = pData.getPixel(rgb);
	ImageData iData = new ImageData(1, 1, 24, pData);
	iData.setPixel(0, 0, fillColor);
	iData.setAlpha(0, 0, 40);
	image = new Image(display, iData);
	}
	public SelectorFigure(){
	   overviewRatio = LogCDrawUtils.getOverviewRatio();
	   targetBounds = new Rectangle(0,0,parentSize.x/overviewRatio,parentSize.y/overviewRatio);
	}
	
	protected void dispose()
	{
		image.dispose();
	}
		
	/**
	 * move the selector figure by dx and dy
	 * @param x
	 * @param y
	 */
	public void moveFigure(int x, int y){
		targetBounds.performTranslate(x,y); //move targetBounds by dx and dy
		if(targetBounds.x<0)
		   targetBounds.x=0;
		if(targetBounds.y<0)
		   targetBounds.y=0;
		if(targetBounds.getRight().x>canvasSizeX)
		   targetBounds.x=canvasSizeX-targetBounds.width;
		if(targetBounds.getBottom().y>canvasSizeY)
		    targetBounds.y=canvasSizeY-targetBounds.height;
	}
	
	public void positionFigure(int x, int y){
	
	      targetBounds.x=x;
	      if(targetBounds.x<0)
		     targetBounds.x=0;
		  if(targetBounds.getRight().x>canvasSizeX)
		     targetBounds.x=canvasSizeX-targetBounds.width;

	      targetBounds.y=y;
		  if(targetBounds.y<0)
		     targetBounds.y=0;
		  if(targetBounds.getBottom().y>canvasSizeY)
		     targetBounds.y=canvasSizeY-targetBounds.height;	
	}

	/**
	 * draw the pink overview selection target
	 * @param g
	 */
	public void paintFigure(Graphics g){
		// Avoid drawing images that are 0 in dimension
		if(targetBounds.width < 1 || targetBounds.height < 1)
			return;
		
		g.drawImage(image, iBounds, targetBounds);
		
		// shadow of the selector
//		g.setForegroundColor(ColorConstants.gray);
//		g.drawLine(targetBounds.x + 2, targetBounds.bottom() - 1, targetBounds.right() - 1, targetBounds.bottom() - 1);
//		g.drawLine(targetBounds.right() - 1, targetBounds.y + 2, targetBounds.right() - 1, targetBounds.bottom() - 1);
	}
	
	public boolean positionContained(int x, int y){
		if(x>=targetBounds.x && x<=targetBounds.x+targetBounds.width && y>=targetBounds.y && y<=targetBounds.y+targetBounds.height)
		   return true;
		else
		   return false;
	}
	
	public Rectangle getTargetBounds(){
		
		return targetBounds;
	}

}

}

