package org.eclipse.hyades.ui.sample.svg.generator;
/*******************************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/

import java.io.ByteArrayOutputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.io.Serializable;

import org.apache.xml.serialize.OutputFormat;
import org.apache.xml.serialize.XMLSerializer;
import org.w3c.dom.Document;
/**
 * <code>GraphicDocumentGenerator</code> is an abstract base class for classes which take
 * content data input as Document Object Model (DOM) objects and then generate DOM objects 
 * which represent graphic visualizations of that content data.
 * 
 * @version 1.44.1.10
 */
public abstract class GraphicDocumentGenerator implements IGraphicDocumentGenerator, Serializable {
	/**
	 * Graphic type identifier.
	 */
	protected String graphicType;
	
	/**
	 * @see IGraphicDocumentGenerator#getGraphicType()
	 */
	public String getGraphicType() {
		return graphicType;
	}
	
	/**
     * @see IGraphicDocumentGenerator#generateGraphicDocument(Document)
     */       
	public Document generateGraphicDocument(Document data) throws DataInputProcessingException {	
		GraphicDocumentProperties properties = new GraphicDocumentProperties();
		properties.setDataDocument(data);		
		return generateGraphicDocument(properties);	
	}	
		
	/**
	 * @see IGraphicDocumentGenerator#generateAndSerializeToString(GraphicDocumentProperties)
	 */
	public String generateAndSerializeToString(GraphicDocumentProperties properties) throws DataInputProcessingException, IOException {
		Document generatedDocument = generateGraphicDocument(properties);
		return serializeGeneratedDocumentToString(generatedDocument);
	}
			
	/**
	 * @see IGraphicDocumentGenerator#generateAndSerializeToFile(GraphicDocumentProperties, String)
	 */
	public boolean generateAndSerializeToFile(GraphicDocumentProperties properties, String uri) throws DataInputProcessingException, IOException {
		Document generatedDocument = generateGraphicDocument(properties);
		if (generatedDocument != null) {
			return serializeGeneratedDocumentToFile(generatedDocument, uri);
		}
		else {
			return false;
		}	
	}
			
	/**
	 * @see IGraphicDocumentGenerator#serializeGeneratedDocumentToFile(Document, String)
	 */
	public boolean serializeGeneratedDocumentToFile(Document generatedDocument, String uri) throws IOException {
		if (generatedDocument == null) {
			return false;
		}
		
		//FileWriter fWriter = new FileWriter(uri);
		FileOutputStream fs = new FileOutputStream(uri);
		OutputStreamWriter fWriter = new OutputStreamWriter(fs, "UTF-8");
				
		OutputFormat oFormat = new OutputFormat(generatedDocument);
//		oFormat.setIndenting(true);
//		oFormat.setLineWidth(80);
//		oFormat.setLineSeparator("\n");

		//(new XMLSerializer(fWriter, oFormat)).serialize(generatedDocument);
		XMLSerializer s = new XMLSerializer(fWriter, oFormat);
//		s.setNewLine("\n");
		s.serialize(generatedDocument);
		fWriter.close();
		return true;		
	}		
		
	/**
	 * @see IGraphicDocumentGenerator#serializeGeneratedDocumentToString(Document)
	 */
	public String serializeGeneratedDocumentToString(Document generatedDocument) throws IOException {
		if (generatedDocument == null) {
			return null;
		}
		
		ByteArrayOutputStream stream = new ByteArrayOutputStream();
		OutputStreamWriter writer = new OutputStreamWriter(stream, "UTF-8");
       	(new XMLSerializer(writer, null)).serialize(generatedDocument);        
        return stream.toString();
	}
	
	/**
	 * XML-serializes and returns the generated <code>Document</code> object as a byte array.
	 * 
	 * @param generatedDocument the generated graphic
	 * @return byte[] The XML-serialized form of the <code>Document</code>
	 */	
	public static byte[] serializeGeneratedDocumentToStream(Document generatedDocument) {
        if (generatedDocument == null) {
        	return null;
        }
        	
        ByteArrayOutputStream stream = new ByteArrayOutputStream();
        try {
            OutputStreamWriter writer = new OutputStreamWriter(stream, "UTF-8");
            (new XMLSerializer(writer, null)).serialize(generatedDocument);
        } catch (Exception e) {
      		System.out.println("Serialize to stream failed");
        }
        return stream.toByteArray();
	}		
}
