/*******************************************************************************
 * Copyright (c) 2000, 2003 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.probekit.ui.popup.actions;

import org.eclipse.jface.action.IAction;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.dialogs.ProgressMonitorDialog;
//import org.eclipse.jface.dialogs.ProgressIndicator;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IObjectActionDelegate;
import org.eclipse.ui.IWorkbenchPart;

import java.util.*;
import java.io.*;
import java.lang.reflect.InvocationTargetException;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.Path;
import org.eclipse.hyades.probekit.Compiler;
//import org.eclipse.hyades.probekit.internal.JarWriter;

public class ProbekitUICompileAction implements IObjectActionDelegate {

	/**
	 * Constructor for Action1.
	 */
	public ProbekitUICompileAction() {
		super();
	}

	/**
	 * @see IObjectActionDelegate#setActivePart(IAction, IWorkbenchPart)
	 */
	public void setActivePart(IAction action, IWorkbenchPart targetPart) {
	}

	/**
	 * Compile the selected probe(s) into Java source; compile the Java source into class files;
	 * collect the class files into a jar file; write the jar file and the engine script to the
	 * user's temporary directory, with names Probekit.jar and Probekit.txt.
	 * 
	 */
	public void run(IAction action) {
		// Progress indicator meanings:
		//		Zero ticks: initializing
		//		One tick: just before compiling probes
		//		Two ticks: probes compiled
		//		Three ticks: saved java source
		//		Four ticks: saved engine script
		//		Five ticks: finished (same as above, really)

		Shell shell = new Shell();
		try {
			IRunnableWithProgress op = new IRunnableWithProgress() {
				public void run(IProgressMonitor monitor)
					throws InvocationTargetException, InterruptedException 
				{
					try {
						runWithProgress(monitor);
					}
					catch (Throwable e) {
						throw new InvocationTargetException(e);
					}
				}
			};
			new ProgressMonitorDialog(shell).run(true, /*cancelable =*/ false, op);
		 } catch (InvocationTargetException e) {
			// handle exception
			MessageDialog.openInformation(
				shell,
				"Probekit Compiler",
				"Exception while processing probes: " + e.getTargetException().getMessage());
		 } catch (InterruptedException e) {
			// handle cancelation
			MessageDialog.openInformation(
				shell,
				"Probekit Compiler",
				"Interrupted while processing probes");
		 }
	}
	
	/**
	 * This is is main operation function. It takes a progress monitor
	 * so we can inform the user about long-running operations.
	 * Prior to running this, selectionChanged() should have filled in
	 * the selectedFileList.
	 * 
	 * @param monitor the monitor 
	 * @throws Throwable
	 */
	public void runWithProgress(IProgressMonitor monitor) throws Throwable {
		monitor.beginTask("Compiling probe(s)...", 3);
		try {			
			if (selectedFileList == null || selectedFileList.size() == 0)
				throw new Throwable("No probe files are selected");
	
			// Choose the base name from the first input file
			IFile firstFile = (IFile)selectedFileList.get(0);
			String basename = getBaseName(firstFile.getName());
	
			// Create a compiler, add all selected files to it, compile,
			// save the Java output and the probescript output.
			Compiler c = new Compiler();
			c.setClassPrefix(basename);
	
			for (Iterator i = selectedFileList.iterator(); i.hasNext() ; ) {
				IFile f = (IFile)i.next();
				c.addFile(f.getLocation().toOSString());
			}
	
			monitor.subTask("Generating Java source...");
			monitor.worked(1);
			
			// call getGeneratedSource now, so any exceptions 
			// occur before we create output files.
			String generatedSource = c.getGeneratedSource();
	
			monitor.subTask("Saving Java source...");
			monitor.worked(1);
			
			// Put the generated files in the directory that the (first/only) 
			// probe file was in. When we write the *.java file, it will 
			// automagically be compiled. 
			IFile javaSourceIFile = firstFile.getParent().getFile(new Path(basename + "_probe.java"));
			ByteArrayInputStream newJavaContents = new ByteArrayInputStream(generatedSource.getBytes());
			// TODO: why do I have to test exists() and use setContents vs. create? Doesn't "force" handle that?
			if (javaSourceIFile.exists()) {
				javaSourceIFile.setContents(newJavaContents, true, false, monitor);
			}
			else {
				javaSourceIFile.create(newJavaContents, true, monitor);
			}
	
			monitor.subTask("Saving BCI engine script...");
			monitor.worked(1);
			
			// Write the engine script to Probekit.txt in the project bin directory
			// This overwrites any existing Probekit.jar output file from before.
			String engineScript = c.getEngineScript();
			IFile engineScriptIFile = firstFile.getParent().getFile(new Path(basename + ".probescript"));
			ByteArrayInputStream newScriptContents = new ByteArrayInputStream(engineScript.getBytes());
			// TODO: why do I have to test exists() and use setContents vs. create? Doesn't "force" handle that?
			if (engineScriptIFile.exists()) {
				engineScriptIFile.setContents(newScriptContents, true, false, monitor);
			}
			else {
				engineScriptIFile.create(newScriptContents, true, monitor);
			}
		}
		finally {
			monitor.done();
		}
	}

	/**
	 * Remove the suffix (the last period character and everything after it)
	 * from the input string. Returns null for null; returns the original
	 * string if it has no dots.
	 * @param name the basename of the input string
	 * @return the basename of the input name. 
	 */
	private String getBaseName(String name) {
		if (name == null) return null;
		int lastdot = name.lastIndexOf(".");
		if (lastdot == -1) return name;
		return name.substring(0, lastdot);
	}

	/**
	 * The list of selected files, maintained by selectionChanged() and used by run()
	 */
	List selectedFileList;

	/**
	 * @see IActionDelegate#selectionChanged(IAction, ISelection)
	 */
	
	public void selectionChanged(IAction action, ISelection selection) {
		// This is only called when the selection matches our filter pattern
		// from the plugin.xml file, so we know we should be enabled.
		action.setEnabled(true);

		// The Compile menu item defaults to being a toggle, checked/unchecked.
		// Make it unchecked always.
		action.setChecked(false);

		// Clear out the selection list from last time and populate it
		// with all the *.probe files from this time.
		selectedFileList = new LinkedList();
		if (selection != null && selection instanceof IStructuredSelection) {
			for (Iterator i = ((IStructuredSelection)selection).iterator(); i.hasNext(); ) {
				Object o = i.next();
				if (o instanceof IFile) {
					IFile f = (IFile)o;
					if (f.getName().endsWith(".probe"))
						selectedFileList.add(f);
				}
			}
		}
	}

}
