/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

//==============================================================================
// Java helpers
//------------------------------------------------------------------------------
// JavaHelpers.cpp
//==============================================================================
#include "JavaHelpers.h"

//==============================================================================
// CJavaFlags - java flags
//

//------------------------------------------------------------------------------
// External notation strings for Java flags
//
CSTR CJavaFlags::AccFlagStr[] = 
{
	"public",			// 0x0001
	"private",			// 0x0002
	"protected",		// 0x0004
	"static",			// 0x0008
	"final",			// 0x0010
	"synchronized",		// 0x0020	("super" for interfaces)
	"volotile",			// 0x0040
	"transient",		// 0x0080
	"native",			// 0x0100
	"interface",		// 0x0200
	"abstract",			// 0x0400
};

const int CJavaFlags::MAX_JAVA_FLAGS = SIZE_OF(AccFlagStr);


//------------------------------------------------------------------------------
// ToString
// Convert flags to a string 
//
// In:
//	isClass - Should be true if the flags defined on a class
// Out:
//	-
// Returns: string
//
string
CJavaFlags::ToString(bool isClass) const
{
	unsigned mask = 1;
	string strRet = "";
	for(int i=0; i < MAX_JAVA_FLAGS; i++)
	{
		if(m_u2Flags & mask)
		{
			if(isClass && i == 5)	// thank you, Sun!!!
			{
				strRet += "super";
			}
			else
			{
				strRet += AccFlagStr[i];
			}
			strRet += " ";
		}
		mask <<= 1;
	}
	return strRet;
}

//==============================================================================
// CJavaType
//

//------------------------------------------------------------------------------
// Constructor
//
CJavaType::CJavaType(jtype_t i_jtype, CSTR i_szType, int i_nDim)
{
	m_jtype = i_jtype;
	if(NULL != i_szType)
		m_strType = i_szType;
	m_nDim = i_nDim;
}

//------------------------------------------------------------------------------
// Parse
// Parse a string representing Java type in the internal notation
//
void	
CJavaType::Parse(CSTR i_szTypeSymbol)
{
	const char Sig[] = "BCDFIJLSZV?";
	m_nDim = 0;
	m_jtype = CJavaType::J_UNKNOWN;
	m_strType = i_szTypeSymbol;
	string::iterator itr = m_strType.begin();
	while(*itr == '[')
	{
		m_nDim++;
		itr++;
	}

	for(int n = 0; n < SIZE_OF(Sig); n++)
	{
		if(*itr == Sig[n])
		{
			m_jtype = (jtype_t)n;
			break;
		}
	}
}

//------------------------------------------------------------------------------
// ToString
// Convert into the external notation
//
string
CJavaType::ToString() const
{
	string strRet;
	static const char* szTypes[] = 
	{
		"byte",
		"char",
		"double",
		"float",
		"int",
		"long",
		"",
		"short",
		"boolean",
		"void",
		"[]",
		"???"
	};

	CJavaType::jtype_t jtype = m_jtype;
	if(jtype >= CJavaType::J_LAST)
	{
		jtype = CJavaType::J_UNKNOWN;
	}
	if(IsClass())
	{
		strRet = GetClassName();
	}
	else
	{
		strRet = szTypes[jtype];
	}
	for(int i = 0; i < GetDim(); i++)
	{
		strRet += szTypes[CJavaType::J_ARRAY];
	}
	return strRet.c_str();
}

//------------------------------------------------------------------------------
// GetClassName
CSTR	
CJavaType::GetClassName() const
{
	if(m_jtype == CJavaType::J_CLASS)
	{
		char* szRet = (char*)m_strType.c_str();
		szRet += m_nDim + 1;
		return szRet;
	}
	else
		return NULL;
}

//------------------------------------------------------------------------------
// GetCategory
// Get type category
//
// Returns:
//	1 - category 1	
//	2 - category 2
//	0 - unknown or void
//
int		
CJavaType::GetCategory() const
{
	switch(m_jtype)
	{
	case CJavaType::J_VOID:
	case CJavaType::J_UNKNOWN:
		return 0;
	case CJavaType::J_DOUBLE:
	case CJavaType::J_LONG:
		return 2;
	default:
		return 1;
	}
}

//------------------------------------------------------------------------------
// GetStackSize
// Calculate stack size necessary for passing an argument of this type
//
int		
CJavaType::GetStackSize() const
{
	int nStack;
	if(m_nDim == 0)
		nStack = GetCategory();
	else
		nStack = 1;
	return nStack;
}

//------------------------------------------------------------------------------
// Assignment operator
//
CJavaType&
CJavaType::operator = (const CJavaType& i_type)
{
	m_nDim = i_type.m_nDim;
	m_strType = i_type.m_strType;
	m_jtype = i_type.m_jtype;
	return *this;
}

//==============================================================================
// CJavaMethodName
//

//------------------------------------------------------------------------------
CJavaMethodName::CJavaMethodName(CSTR i_szNameAndSig, bool i_bAutoParse)
{
	m_bAutoParse = i_bAutoParse;
	if(NULL != i_szNameAndSig)
	{
		char* szScan = (char*)i_szNameAndSig;
		while(*szScan != '\0' && *szScan != '(')
		{
			m_strName += *szScan++;
		}
		m_strSignature = szScan;
	}
	if(m_bAutoParse)
	{
		Parse();
	}
}

//------------------------------------------------------------------------------
void 
CJavaMethodName::SetName(CSTR i_szName)
{
	m_strName = i_szName;
}

//------------------------------------------------------------------------------
void 
CJavaMethodName::SetSignature(CSTR i_szSignature)
{
	m_strSignature = i_szSignature;
	if(m_bAutoParse)
	{
		Parse();
	}
}

//------------------------------------------------------------------------------
CSTR 
CJavaMethodName::GetName() const
{
	return m_strName.c_str();
}

//------------------------------------------------------------------------------
CSTR 
CJavaMethodName::GetSignature() const
{
	return m_strSignature.c_str();
}

//------------------------------------------------------------------------------
const CJavaType&  
CJavaMethodName::GetRetType() const
{
	return m_jtRet;
}

//------------------------------------------------------------------------------
int	
CJavaMethodName::GetArgNum() const
{
	return m_args.size();
}

//------------------------------------------------------------------------------
// GetStackSize
// Calculate stack size contribution for the method based on its signature.
// The stack depth is calculated as the stack required for the return type
// minus stack for all arguments.
//
int 
CJavaMethodName::GetStackSize() const
{
	int nRet = m_jtRet.GetStackSize();
	CJavaMethodName::args_t::const_iterator itr;
	for(itr = m_args.begin(); itr != m_args.end(); itr++)
	{
		nRet -= itr->GetStackSize();
	}

	return nRet;
}

//------------------------------------------------------------------------------
void 
CJavaMethodName::Parse()
{
	bool isArg   = false;
	bool isArray = false;
	string strToken;

	string::iterator itr = m_strSignature.begin();
	CJavaType jtype;

	while(itr != m_strSignature.end())
	{
		switch(*itr)
		{
		case ' ':
		case '\t':
			break;
		case '(':
			isArg = true;
			itr++;
			continue;
		case ')':
			isArg = false;
			itr++;
			continue;
		case 'L':
			while(*itr != ';')
				strToken += *itr++;
			break;
		case '[':
			while(*itr == '[')
				strToken += *itr++;
			continue;
		default:
			strToken += *itr;
			break;
		}
		itr++;
		jtype.Parse(strToken.c_str());
		strToken = "";
		if(isArg)
		{
			m_args.push_back(jtype);
		}
		else
		{
			m_jtRet = jtype;
		}
	}
}

//------------------------------------------------------------------------------
string
CJavaMethodName::ToString() const
{
	string strRet = GetRetType().ToString();
	strRet += ' ';
	strRet += m_strName;
	strRet += '(';
	int n = GetArgNum();
	CJavaMethodName::args_t::const_iterator itr;
	for(itr = m_args.begin(); itr != m_args.end(); itr++, n--)
	{
		string strTemp;
		strRet += itr->ToString();
		if(n > 1)
		{
			strRet += ", ";
		}
	}
	strRet += ')';
	return strRet;
}

//= End of JavaHelpers.cpp =====================================================

