/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.test.manual.runner.ui;

import java.awt.Component;
import java.awt.Cursor;
import java.awt.Dimension;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;

import javax.swing.JFrame;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JSplitPane;
import javax.swing.UIManager;
import javax.swing.event.MenuEvent;
import javax.swing.event.MenuListener;
import javax.swing.event.TreeSelectionEvent;
import javax.swing.event.TreeSelectionListener;

import org.eclipse.hyades.test.common.event.TypedEvent;
import org.eclipse.hyades.test.common.event.VerdictEvent;
import org.eclipse.hyades.test.manual.runner.RemoteAppResourceBundle;
import org.eclipse.hyades.test.manual.runner.model.Loop;
import org.eclipse.hyades.test.manual.runner.model.NamedElement;
import org.eclipse.hyades.test.manual.runner.model.TestCase;
import org.eclipse.hyades.test.manual.runner.model.TestInvocation;
import org.eclipse.hyades.test.manual.runner.model.TestSuite;
import org.eclipse.hyades.test.manual.runner.model.util.ModelUtil;
import org.eclipse.hyades.test.manual.runner.ui.panel.BasePanel;
import org.eclipse.hyades.test.manual.runner.ui.panel.EmptyPanel;
import org.eclipse.hyades.test.manual.runner.ui.panel.LoopPanel;
import org.eclipse.hyades.test.manual.runner.ui.panel.TestCasePanel;
import org.eclipse.hyades.test.manual.runner.ui.panel.TestInvocationExecutionPanel;
import org.eclipse.hyades.test.manual.runner.ui.panel.TestInvocationPanel;
import org.eclipse.hyades.test.manual.runner.ui.panel.TestSuitePanel;
import org.eclipse.hyades.test.manual.runner.ui.tree.TestSuiteTreePanel;
import org.eclipse.hyades.test.manual.runner.ui.util.BaseCardLayout;
import org.eclipse.hyades.test.manual.runner.ui.util.EditorItem;
import org.eclipse.hyades.test.manual.runner.ui.util.IRefreshable;
import org.eclipse.hyades.test.manual.runner.ui.util.SwingUtil;


/**
 * @author marcelop
 * @since 1.0.2
 */
public class RemoteAppFrame 
extends JFrame implements TreeSelectionListener
{
	private final static Dimension FRAME_SIZE = new Dimension(750, 500);
	
	private JSplitPane mainSplitPane;
	private JSplitPane detailSplitPane;
	private TestSuiteTreePanel treePanel;	
	private JPanel attributePanel;
	private JPanel executionPanel;
	
	private JMenuItem nodeExpandAllMenuItem;
	private JMenuItem nodeCollapseAllMenuItem;
	
	private int detailSplitPosition = -1;

	
	public RemoteAppFrame()
	{
		try
		{
			UIManager.setLookAndFeel(UIManager.getCrossPlatformLookAndFeelClassName());
		}
		catch(Exception e){}

		setTitle(RemoteAppResourceBundle.getString("frame.title"));
		setJMenuBar(createMenuBar());
		getContentPane().add(createControls());
		setSize(FRAME_SIZE);
		
		setDefaultCloseOperation(DO_NOTHING_ON_CLOSE); 

		addWindowListener(new WindowAdapter()
		{
			public void windowOpened(WindowEvent e)
			{
				doOpen();
			}
			
			public void windowClosing(WindowEvent e)
			{
				doClose(true);
			}
		});		

		System.out.println(RemoteAppResourceBundle.getString("event.frame.Available"));
	}
	
	/**
	 * @see java.awt.Window#dispose()
	 */
	public void dispose()
	{
		treePanel.dispose();
		super.dispose();
	}

	
	private JMenuBar createMenuBar()
	{
		JMenuBar menuBar = new JMenuBar();
		menuBar.add(createFileMenuBar());
		menuBar.add(createViewMenuBar());
		menuBar.add(createTestcaseMenuBar());
		
		return menuBar;		
	}	

	private JMenu createFileMenuBar()
	{
		JMenu fileMenu = new JMenu(RemoteAppResourceBundle.getString("menu.File"));
		SwingUtil.adjustMnemonic(fileMenu);
				
		final JMenuItem closeMenuItem = new JMenuItem(RemoteAppResourceBundle.getString("menu.File.Exit"));
		SwingUtil.adjustMnemonic(closeMenuItem);
		closeMenuItem.addActionListener(new ActionListener()
		{
			public void actionPerformed(ActionEvent e)
			{
				doClose(false);
			}
		});
		
		fileMenu.add(closeMenuItem);
		fileMenu.addMenuListener(new MenuListener()
		{
			/**
			 * @see javax.swing.event.MenuListener#menuSelected(MenuEvent)
			 */
			public void menuSelected(MenuEvent e)
			{
				if(ModelUtil.DEFAULT_EXECUTION_MANAGER.hasNext())
					closeMenuItem.setText(RemoteAppResourceBundle.getString("action.InterruptExecution"));
				else
					closeMenuItem.setText(RemoteAppResourceBundle.getString("action.StopExecution"));
					
				closeMenuItem.setMnemonic(0);
				SwingUtil.adjustMnemonic(closeMenuItem);
			}

			/**
			 * @see javax.swing.event.MenuListener#menuDeselected(MenuEvent)
			 */
			public void menuDeselected(MenuEvent e)
			{
			}

			/**
			 * @see javax.swing.event.MenuListener#menuCanceled(MenuEvent)
			 */
			public void menuCanceled(MenuEvent e)
			{
			}
		});
		return fileMenu;		
	}	

	private JMenu createViewMenuBar()
	{
		JMenu viewMenu = new JMenu(RemoteAppResourceBundle.getString("menu.View"));
		SwingUtil.adjustMnemonic(viewMenu);

		JMenuItem selectTaskToExecuteMenuItem = new JMenuItem(RemoteAppResourceBundle.getString("menu.View.SelectTaskToExecute"));
		SwingUtil.adjustMnemonic(selectTaskToExecuteMenuItem);
		selectTaskToExecuteMenuItem.addActionListener(new ActionListener()
		{
			public void actionPerformed(ActionEvent e)
			{
				selectNextTaskToExecute();
			}
		});
		
		JMenuItem expandAllMenuItem = new JMenuItem(RemoteAppResourceBundle.getString("menu.View.ExpandAll"));
		SwingUtil.adjustMnemonic(expandAllMenuItem);
		expandAllMenuItem.addActionListener(new ActionListener()
		{
			public void actionPerformed(ActionEvent e)
			{
				setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
				try
				{
					SwingUtil.expandTree(getTreePanel().getTree(), -1, true);
				}
				finally
				{
					setCursor(null);
				}
			}
		});

		JMenuItem collapseAllMenuItem = new JMenuItem(RemoteAppResourceBundle.getString("menu.View.CollapseAll"));
		SwingUtil.adjustMnemonic(collapseAllMenuItem);
		collapseAllMenuItem.addActionListener(new ActionListener()
		{
			public void actionPerformed(ActionEvent e)
			{
				setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
				try
				{
					SwingUtil.expandTree(getTreePanel().getTree(), -1, false);
				}
				finally
				{
					setCursor(null);
				}
			}
		});

		nodeExpandAllMenuItem = new JMenuItem(RemoteAppResourceBundle.getString("menu.View.NodeExpandAll"));
		SwingUtil.adjustMnemonic(nodeExpandAllMenuItem);
		nodeExpandAllMenuItem.addActionListener(new ActionListener()
		{
			public void actionPerformed(ActionEvent e)
			{
				expandNode(true);
			}
		});
		nodeExpandAllMenuItem.setEnabled(false);

		nodeCollapseAllMenuItem = new JMenuItem(RemoteAppResourceBundle.getString("menu.View.NodeCollapseAll"));
		SwingUtil.adjustMnemonic(nodeCollapseAllMenuItem);
		nodeCollapseAllMenuItem.addActionListener(new ActionListener()
		{
			public void actionPerformed(ActionEvent e)
			{
				expandNode(false);
			}
		});
		nodeCollapseAllMenuItem.setEnabled(false);

		
		viewMenu.add(selectTaskToExecuteMenuItem);
		viewMenu.addSeparator();
		viewMenu.add(expandAllMenuItem);
		viewMenu.add(collapseAllMenuItem);
		viewMenu.add(nodeExpandAllMenuItem);
		viewMenu.add(nodeCollapseAllMenuItem);
		
		return viewMenu;		
	}
	
	private JMenu createTestcaseMenuBar()
	{
		JMenu executionMenu = new JMenu(RemoteAppResourceBundle.getString("menu.Action"));
		SwingUtil.adjustMnemonic(executionMenu);
		
		JMenuItem sendMessageMenuItem = new JMenuItem(RemoteAppResourceBundle.getString("menu.Execution.SendMessage")+" ...");
		SwingUtil.adjustMnemonic(sendMessageMenuItem);
		sendMessageMenuItem.addActionListener(new ActionListener()
		{
			public void actionPerformed(ActionEvent e)
			{
				sendMessage();
			}
		});

		executionMenu.add(sendMessageMenuItem);
		
		return executionMenu;		
	}	
			
	private Component createControls()
	{
		treePanel = new TestSuiteTreePanel();
		getTreePanel().getTree().addTreeSelectionListener(this);
		
		attributePanel = new JPanel(new BaseCardLayout());
		attributePanel.add(new EmptyPanel(), getClass().getName());
		
		executionPanel = new JPanel(new BaseCardLayout());
		executionPanel.add(new EmptyPanel(), getClass().getName());
		
		detailSplitPane = new JSplitPane(JSplitPane.VERTICAL_SPLIT, attributePanel, executionPanel);
		detailSplitPane.setOneTouchExpandable(true);
				
		mainSplitPane = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT, getTreePanel(), detailSplitPane);
		mainSplitPane.setOneTouchExpandable(true);
		
		return mainSplitPane;
	}
	
	protected TestSuiteTreePanel getTreePanel()
	{
		return treePanel;
	}
	
	protected void doOpen()
	{
		mainSplitPane.setDividerLocation(0.37);
		mainSplitPane.setDividerSize(7);
		detailSplitPane.setDividerLocation(0.65);
		detailSplitPane.setDividerSize(7);
		detailSplitPosition = detailSplitPane.getDividerLocation();
		
		ModelUtil.getEventLogger().logTyped(ModelUtil.DEFAULT_EXECUTION_MANAGER.getRoot(), TypedEvent.START, null);
		selectNextTaskToExecute();
	}

	public void doClose(boolean confirmEnd)
	{
		VerdictEvent verdictEvent = null;		
		String text = null;
		if(ModelUtil.DEFAULT_EXECUTION_MANAGER.hasNext())
		{
			verdictEvent = new VerdictEvent();
			verdictEvent.setOwnerId(ModelUtil.getHierarchyId(ModelUtil.DEFAULT_EXECUTION_MANAGER.getRoot()));
			
			text = RemoteAppResourceBundle.getString("msgbox.close.InterruptText");
			verdictEvent.setVerdict(VerdictEvent.VERDICT_FAIL);
			verdictEvent.setText(RemoteAppResourceBundle.getString("interruptText"));
			verdictEvent.setReason(VerdictEvent.REASON_SEE_DESCRIPTION);
		}
		else
		{
			verdictEvent = ModelUtil.DEFAULT_ARBITER.analyse();
			text = RemoteAppResourceBundle.getString("msgbox.close.StopText");
		}
			
		if(ModelUtil.DEFAULT_EXECUTION_MANAGER.hasNext() || confirmEnd)
		{
			if( JOptionPane.YES_OPTION !=
				JOptionPane.showConfirmDialog(this, text, RemoteAppResourceBundle.getString("word.Question"), JOptionPane.YES_NO_OPTION))
				return;
		}
		
		ModelUtil.DEFAULT_EXECUTION_MANAGER.closeStatedLoops();
		ModelUtil.closeExecutedTests();
		ModelUtil.getEventLogger().log(verdictEvent);
		System.out.println(RemoteAppResourceBundle.getString("event.application.Closed"));
		ModelUtil.getEventLogger().logTyped(ModelUtil.DEFAULT_EXECUTION_MANAGER.getRoot(), TypedEvent.STOP, null);

		try
		{
			ModelUtil.DEFAULT_EXECUTION_MANAGER.dispose();
			ModelUtil.clearVerdicts();
			ModelUtil.setEventLogger(null);
			TestInvocation.removeListeners();
			dispose();
			finalize();
		}
		catch (Throwable e)
		{
		}		
	}
	
	protected void sendMessage()
	{
		Object selection = getTreePanel().getTree().getLastSelectedPathComponent();
		if(selection instanceof NamedElement)
			MessageDialog.show(this, (NamedElement)selection);
	}
	
	protected void selectNextTaskToExecute()
	{
		getTreePanel().select(ModelUtil.DEFAULT_EXECUTION_MANAGER.next());
	}
	
	protected void expandNode(boolean expand)
	{
		if(getTreePanel().getTree().getSelectionCount() == 0)
			return;
			
		Object o = getTreePanel().getTree().getLastSelectedPathComponent();
		int row = getTreePanel().getTree().getSelectionRows()[0];
		
		if((o != null) && ((o instanceof TestSuite) || (o instanceof Loop)))
		{
			setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
			try
			{
				SwingUtil.expandTree(getTreePanel().getTree(), o, row, -1, expand);
			}
			finally
			{
				setCursor(null);
			}
		}		
	}
	
	
	/**
	 * @see TreeSelectionListener#valueChanged(TreeSelectionEvent)
	 */
	public void valueChanged(TreeSelectionEvent e)
	{
		Object selection = getTreePanel().getTree().getLastSelectedPathComponent();			
		adjustAttributePanel(selection);
		adjustExecutionPanel(selection);
		
		boolean notLeafNode = ((selection != null) && ((selection instanceof TestSuite) || (selection instanceof Loop)));
		nodeExpandAllMenuItem.setEnabled(notLeafNode);
		nodeCollapseAllMenuItem.setEnabled(notLeafNode);
	}
	
	private void adjustAttributePanel(Object selection)
	{
		if(selection == null)
			selection = this;
		
		BaseCardLayout cardLayout = (BaseCardLayout)attributePanel.getLayout();
					
		IRefreshable refreshable = (IRefreshable)cardLayout.showAndTell(attributePanel, selection.getClass().getName());
		if(refreshable == null)
		{
			refreshable = createAttributePanel(selection);
			if(refreshable == null)
				selection = this;
				
			refreshable = (IRefreshable)cardLayout.showAndTell(attributePanel, selection.getClass().getName());			
		}
		refreshable.refreshContent(new EditorItem(selection, false));
	}
	
	private IRefreshable createAttributePanel(Object selection)
	{
		BasePanel basePanel = null;
		
		if(selection instanceof TestSuite)
			basePanel = new TestSuitePanel(this);
			
		else if(selection instanceof Loop)
			basePanel = new LoopPanel();

		else if(selection instanceof TestCase)
			basePanel = new TestCasePanel();
			
		else if(selection instanceof TestInvocation)
			basePanel = new TestInvocationPanel();

		if(basePanel != null)
			attributePanel.add(basePanel, selection.getClass().getName());
			
		return basePanel;
	}
	
	private void adjustExecutionPanel(Object selection)
	{
		if(selection == null)
			selection = this;
					
		BaseCardLayout cardLayout = (BaseCardLayout)executionPanel.getLayout();

		IRefreshable refreshable = (IRefreshable)cardLayout.showAndTell(executionPanel, selection.getClass().getName());
		if(refreshable == null)
		{
			refreshable = createExecutionPanel(selection);
			if(refreshable == null)
			{
				selection = this;
			}
				
			refreshable = (IRefreshable)cardLayout.showAndTell(executionPanel, selection.getClass().getName());			
		}
		
		adjustFocus(selection);
		refreshable.refreshContent(new EditorItem(selection, true));
		adjustDetailDividerPosition(selection);
	}

	private void adjustFocus(Object selection)
	{
		if(selection instanceof TestInvocation)
		{
			TestInvocation testInvocation = (TestInvocation)selection;
			if(testInvocation.getExecutionCount() == 0)
			{
				getTreePanel().getTree().transferFocus();
				executionPanel.requestFocus();
			}
		}
	}
	
	private void adjustDetailDividerPosition(Object selection)
	{
		if(selection == this)
		{
			if(detailSplitPane.getRightComponent().isVisible())
			{
				detailSplitPosition = detailSplitPane.getDividerLocation();
				detailSplitPane.getRightComponent().setVisible(false);
			}
		}
		else
		{
			if(!detailSplitPane.getRightComponent().isVisible())
			{
				detailSplitPane.getRightComponent().setVisible(true);
				if(detailSplitPosition >= 0)
					detailSplitPane.setDividerLocation(detailSplitPosition);
			}
		}
	}
	
	private IRefreshable createExecutionPanel(Object selection)
	{
		BasePanel basePanel = null;
		
		if(selection instanceof TestInvocation)
			basePanel = new TestInvocationExecutionPanel();

		if(basePanel != null)
			executionPanel.add(basePanel, selection.getClass().getName());

		return basePanel;
	}
}