package org.eclipse.hyades.logging.java;

/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

import java.net.InetAddress;
import java.net.UnknownHostException;
import java.text.MessageFormat;
import java.util.Locale;
import java.util.MissingResourceException;
import java.util.ResourceBundle;
import java.util.logging.Level;
import java.util.logging.LogRecord;

import org.eclipse.hyades.logging.core.IExternalizableToXml;
import org.eclipse.hyades.logging.events.CbeFormatter;
import org.eclipse.hyades.logging.events.ICommonBaseEvent;
import org.eclipse.hyades.logging.events.IComponentIdentification;
import org.eclipse.hyades.logging.events.IExtendedDataElement;
import org.eclipse.hyades.logging.events.IMsgCatalogToken;
import org.eclipse.hyades.logging.events.IMsgDataElement;
import org.eclipse.hyades.logging.events.IReportSituation;
import org.eclipse.hyades.logging.events.ISimpleEventFactory;
import org.eclipse.hyades.logging.events.ISituation;
import org.eclipse.hyades.logging.events.MsgCatalogTokenImpl;
import org.eclipse.hyades.logging.events.SimpleEventFactoryImpl;

/** 
 * Extension of the Java Logging <code>java.util.logging.LogRecord</code>
 * class used to encapsulate a v 1.0.1 Common Base Event (CBE) or <code>org.eclipse.hyades.logging.events.ICommonBaseEvent</code> 
 * within a <code>java.util.logging.LogRecord</code>.
 * <p>
 * A log record instance is created via the constructor and configured (e.g. setting an ICommonBaseEvent) before 
 * being logged to a logger.
 *  
 * 
 * @author  Paul Slauenwhite
 * @version January 7, 2004
 * @see     java.util.logging.LogRecord
 * @see     org.eclipse.hyades.logging.events.ICommonBaseEvent
 */
public class CommonBaseEventLogRecord extends LogRecord implements IExternalizableToXml {

    /**
     * Instance variable which holds a reference to a <code>org.eclipse.hyades.logging.events.ICommonBaseEvent</code>.
     */
    private ICommonBaseEvent commonBaseEvent = null;

    /**
     * Bit mask for determining when the <code>java.util.logging.LogRecord</code>'s properties have been modified.
     * When a <code>java.util.logging.LogRecord</code> property has been modified, the encapsulated 
     * <code>org.eclipse.hyades.logging.events.ICommonBaseEvent</code> needs to be rebuilt when retrieved.
     * 
     * This bit mask is required for performance reasons.
     * 
     * The 32 bit mask uses the following notation:
     * 
     * [Empty]:              0000 0000 0000 0000 0000 0000 0000 0000 (Binary 0)
     * [Reserved]:           0000 0000 0000 0000 0000 0000 0000 0001 (Binary 1)
     * Level:                0000 0000 0000 0000 0000 0000 0000 0010 (Binary 2)
     * LoggerName:           0000 0000 0000 0000 0000 0000 0000 0100 (Binary 4)
     * Message:              0000 0000 0000 0000 0000 0000 0000 1000 (Binary 8)
     * Millis:               0000 0000 0000 0000 0000 0000 0001 0000 (Binary 16)
     * Parameters:           0000 0000 0000 0000 0000 0000 0010 0000 (Binary 32)
     * Resource Bundle:      0000 0000 0000 0000 0000 0000 0100 0000 (Binary 64)
     * Resource Bundle Name: 0000 0000 0000 0000 0000 0000 1000 0000 (Binary 128)
     * Sequence Number:      0000 0000 0000 0000 0000 0001 0000 0000 (Binary 256)
     * Source Class Name:    0000 0000 0000 0000 0000 0010 0000 0000 (Binary 512)
     * Source Method Name:   0000 0000 0000 0000 0000 0100 0000 0000 (Binary 1024)
     * Thread ID:            0000 0000 0000 0000 0000 1000 0000 0000 (Binary 2048)
     * Thrown:               0000 0000 0000 0000 0001 0000 0000 0000 (Binary 4096)
     */
    private int logRecordMask = 0;

    /**
     * Instance variable which holds a reference to a <code>org.eclipse.hyades.logging.events.ISimpleEventFactory</code> for generating event instances.
     */
    private static ISimpleEventFactory simpleEventFactory = SimpleEventFactoryImpl.getInstance();

    /**
     * Instance variables which hold the Java Logging levels (integer) for performance reasons.
     */
    private static final int LEVEL_ALL = Level.ALL.intValue();
    private static final int LEVEL_FINEST = Level.FINEST.intValue();
    private static final int LEVEL_FINER = Level.FINER.intValue();
    private static final int LEVEL_FINE = Level.FINE.intValue();
    private static final int LEVEL_CONFIG = Level.CONFIG.intValue();
    private static final int LEVEL_INFO = Level.INFO.intValue();
    private static final int LEVEL_WARNING = Level.WARNING.intValue();
    private static final int LEVEL_SEVERE = Level.SEVERE.intValue();
    private static final int LEVEL_OFF = Level.OFF.intValue();

    /**
     * IP address (IPv4) of the local host, otherwise "127.0.0.1". 
     */
    private static String localHostIP = null;

    static {
        try {
            localHostIP = InetAddress.getLocalHost().getHostAddress();
        }
        catch (UnknownHostException u) {
            localHostIP = "127.0.0.1";
        }
    }

    /**
     * Constructor to create a CBE log record with the parameter logging level and message.
     * 
     * @param level The logging level of the newly created log record.
     * @param message The log message of the newly created log record.
     */
    public CommonBaseEventLogRecord(Level level, String message) {

        super(level, message);

        //The following <code>java.util.logging.LogRecord</code> properties are initialized upon object creation:
        // 
        //Level:                0000 0000 0000 0000 0000 0000 0000 0010 (Binary 2)
        //Message:              0000 0000 0000 0000 0000 0000 0000 1000 (Binary 8)
        //Millis:               0000 0000 0000 0000 0000 0000 0001 0000 (Binary 16)
        //Sequence Number:      0000 0000 0000 0000 0000 0001 0000 0000 (Binary 256)
        //Source Class Name:    0000 0000 0000 0000 0000 0010 0000 0000 (Binary 512)
        //Source Method Name:   0000 0000 0000 0000 0000 0100 0000 0000 (Binary 1024)
        //Thread ID:            0000 0000 0000 0000 0000 1000 0000 0000 (Binary 2048)
        //                      -----------------------------------------------------
        //Total:                0000 0000 0000 0000 0000 1111 0001 1010 (Binary 3866)
        logRecordMask |= 3866;
    }

    /**
     * Constructor to create a CBE log record with the parameter logging level.
     * 
     * @param level The logging level of the newly created log record.
     */
    public CommonBaseEventLogRecord(Level level) {
        this(level, null);
    }

    /**
     * Constructor to create a CBE log record with the parameter CBE.
     * 
     * @param commonBaseEvent The CBE of the newly created log record.
     */
    public CommonBaseEventLogRecord(ICommonBaseEvent commonBaseEvent) {

        this(Level.OFF, null);

        setCommonBaseEvent(commonBaseEvent);
    }

    /**
     * Constructor to create a CBE log record with the parameter LogRecord.
     * 
     * @param logRecord The LogRecord of the newly created log record.
     */
    public CommonBaseEventLogRecord(LogRecord logRecord) {

        this(logRecord.getLevel(), logRecord.getMessage());

        this.setMillis(logRecord.getMillis());
        this.setSequenceNumber(logRecord.getSequenceNumber());
        this.setThreadID(logRecord.getThreadID());
        this.setLoggerName(logRecord.getLoggerName());
        this.setParameters(logRecord.getParameters());
        this.setResourceBundle(logRecord.getResourceBundle());
        this.setResourceBundleName(logRecord.getResourceBundleName());
        this.setSourceClassName(logRecord.getSourceClassName());
        this.setSourceMethodName(logRecord.getSourceMethodName());
        this.setThrown(logRecord.getThrown());
    }

    /**
     * Sets value of the commonBaseEvent instance variable to the specified value.
     * 
     * @param commonBaseEvent The new value of the commonBaseEvent instance variable.
     */
    public final void setCommonBaseEvent(ICommonBaseEvent commonBaseEvent) {
        this.commonBaseEvent = commonBaseEvent;
    }

    /**
     * Gets the value of the commonBaseEvent instance variable.
     * 
     * If the commonBaseEvent instance variable needs to be rebuilt, the properties
     * from the parent class (e.g. <code>java.util.logging.LogRecord</code>) are used.
     * 
     * @return The value of the commonBaseEvent instance variable.
     */
    public final ICommonBaseEvent getCommonBaseEvent() {

        //[Empty]:              0000 0000 0000 0000 0000 0000 0000 0000 (Binary 0)
        if ((logRecordMask | 0) != 0) {

            if (commonBaseEvent == null) {
                commonBaseEvent = simpleEventFactory.createCommonBaseEvent();
            }

            //Source Class Name:    0000 0000 0000 0000 0000 0010 0000 0000 (Binary 512)
            //Source Method Name:   0000 0000 0000 0000 0000 0100 0000 0000 (Binary 1024)
            //Thread ID:            0000 0000 0000 0000 0000 1000 0000 0000 (Binary 2048)
            if (((logRecordMask & 512) == 512) || ((logRecordMask & 1024) == 1024) || ((logRecordMask & 2048) == 2048)) {

                IComponentIdentification sourceComponentId = commonBaseEvent.getSourceComponentId();

                if (sourceComponentId == null) {

                    sourceComponentId = simpleEventFactory.createComponentIdentification();
                    sourceComponentId.setLocation(localHostIP);
                    sourceComponentId.setLocationType("IPV4");
                    sourceComponentId.setExecutionEnvironment("Java");
                    sourceComponentId.setComponentIdType("Application");
                    sourceComponentId.setComponent("JavaApplication");
                }

                //Source Class Name:    0000 0000 0000 0000 0000 0010 0000 0000 (Binary 512)
                if ((logRecordMask & 512) == 512) {
                    sourceComponentId.setComponent(super.getSourceClassName());
                }

                //Source Method Name:   0000 0000 0000 0000 0000 0100 0000 0000 (Binary 1024)
                if ((logRecordMask & 1024) == 1024) {
                    sourceComponentId.setSubComponent(super.getSourceMethodName());
                }

                //Thread ID:            0000 0000 0000 0000 0000 1000 0000 0000 (Binary 2048)
                if ((logRecordMask & 2048) == 2048) {
                    sourceComponentId.setThreadId(String.valueOf(super.getThreadID()));
                }

                commonBaseEvent.setSourceComponentId(sourceComponentId);
            }

            //LoggerName:           0000 0000 0000 0000 0000 0000 0000 0100 (Binary 4)
            if ((logRecordMask & 4) == 4) {

                IComponentIdentification reporterComponentId = commonBaseEvent.getReporterComponentId();

                if (reporterComponentId == null) {

                    reporterComponentId = simpleEventFactory.createComponentIdentification();

                    reporterComponentId.setLocation(localHostIP);
                    reporterComponentId.setLocationType("IPV4");
                    reporterComponentId.setExecutionEnvironment("Java");
                    reporterComponentId.setComponent("java.util.logging");
                    reporterComponentId.setSubComponent("java.util.logging.Logger");
                    reporterComponentId.setComponentIdType("Application");
                    reporterComponentId.setComponent("JavaLogging");
                }

                reporterComponentId.setInstanceId(super.getLoggerName());

                commonBaseEvent.setReporterComponentId(reporterComponentId);
            }

            //Parameters:           0000 0000 0000 0000 0000 0000 0010 0000 (Binary 32)
            //Resource Bundle:      0000 0000 0000 0000 0000 0000 0100 0000 (Binary 64)
            //Resource Bundle Name: 0000 0000 0000 0000 0000 0000 1000 0000 (Binary 128)
            if (((logRecordMask & 32) == 32) || ((logRecordMask & 64) == 64) || ((logRecordMask & 128) == 128)) {

                IMsgDataElement msgDataElement = simpleEventFactory.createMsgDataElement();

                //Resource Bundle Name: 0000 0000 0000 0000 0000 0000 1000 0000 (Binary 128)
                if ((logRecordMask & 128) == 128) {
                    msgDataElement.setMsgCatalog(super.getResourceBundleName());
                }

                //Resource Bundle:      0000 0000 0000 0000 0000 0000 0100 0000 (Binary 64)
                if ((logRecordMask & 64) == 64) {

                    ResourceBundle resourceBundle = super.getResourceBundle();

                    if (resourceBundle != null) {

                        Locale bundleLocale = resourceBundle.getLocale();

                        if (bundleLocale != null) {

                            String language = bundleLocale.getLanguage();
                            String country = bundleLocale.getCountry();

                            if ((language.length() > 0) && (country.length() > 0))
                                msgDataElement.setMsgLocale(language.concat("-").concat(country));
                        }
                    }
                }

                //Parameters:           0000 0000 0000 0000 0000 0000 0010 0000 (Binary 32)
                if ((logRecordMask & 32) == 32) {

                    Object[] parameters = super.getParameters();

                    if ((parameters != null) && (parameters.length > 0)) {

                        IMsgCatalogToken[] msgCatalogTokens = new MsgCatalogTokenImpl[parameters.length];
                        IMsgCatalogToken msgCatalogToken = null;

                        for (int counter = 0; counter < parameters.length; counter++) {

                            msgCatalogToken = simpleEventFactory.createMsgCatalogToken();
                            msgCatalogToken.setValue(String.valueOf(parameters[counter]));

                            msgCatalogTokens[counter] = msgCatalogToken;
                        }

                        msgDataElement.setMsgCatalogTokens(msgCatalogTokens);
                    }
                }

                commonBaseEvent.setMsgDataElement(msgDataElement);
            }

            //Thrown:               0000 0000 0000 0000 0001 0000 0000 0000 (Binary 4096)
            if ((logRecordMask & 4096) == 4096) {

                Throwable thrown = super.getThrown();

                if (thrown != null) {

                    IExtendedDataElement extendedDataElement = simpleEventFactory.createExtendedDataElement();
                    extendedDataElement.setName("parm1");
                    extendedDataElement.setType(IExtendedDataElement.TYPE_STRING_VALUE);

                    String thrownMessage = thrown.getMessage();

                    StringBuffer thrownString = new StringBuffer(thrown.getClass().getName());

                    if (thrownMessage != null) {
                        thrownString.append(':');
                        thrownString.append(thrownMessage);
                    }

                    Throwable cause = thrown.getCause();

                    if (cause != null) {

                        thrownString.append(":Cause:");
                        thrownString.append(cause.getClass().getName());

                        thrownMessage = cause.getMessage();

                        if (thrownMessage != null) {
                            thrownString.append(':');
                            thrownString.append(thrownMessage);
                        }
                    }

                    extendedDataElement.setValues(new String[] { thrownString.toString()});

                    commonBaseEvent.addExtendedDataElement(extendedDataElement);

                    StackTraceElement stackTraceElements[] = thrown.getStackTrace();

                    for (int counter = 0; counter < stackTraceElements.length; counter++) {

                        extendedDataElement = simpleEventFactory.createExtendedDataElement();

                        extendedDataElement.setName("parm".concat(String.valueOf(counter + 2)));
                        extendedDataElement.setType(IExtendedDataElement.TYPE_STRING_VALUE);
                        extendedDataElement.setValues(new String[] { stackTraceElements[counter].toString()});

                        commonBaseEvent.addExtendedDataElement(extendedDataElement);
                    }
                }
            }

            //Sequence Number:      0000 0000 0000 0000 0000 0001 0000 0000 (Binary 256)
            if ((logRecordMask & 256) == 256) {

                IExtendedDataElement sequenceNumber = simpleEventFactory.createExtendedDataElement();

                sequenceNumber.setName("SequenceNumber");
                sequenceNumber.setType(IExtendedDataElement.TYPE_LONG_VALUE);
                sequenceNumber.setValues(new String[] { String.valueOf(super.getSequenceNumber())});

                commonBaseEvent.addExtendedDataElement(sequenceNumber);
            }

            //Level:                0000 0000 0000 0000 0000 0000 0000 0010 (Binary 2)
            if ((logRecordMask & 2) == 2) {
                setCBESeverity(super.getLevel());
            }

            //Message:              0000 0000 0000 0000 0000 0000 0000 1000 (Binary 8)
            //Parameters:           0000 0000 0000 0000 0000 0000 0010 0000 (Binary 32)
            //Resource Bundle:      0000 0000 0000 0000 0000 0000 0100 0000 (Binary 64)
            if (((logRecordMask & 8) == 8) || ((logRecordMask & 32) == 32) || ((logRecordMask & 64) == 64)) {
                commonBaseEvent.setMsg(formatMessage(super.getMessage(), super.getResourceBundle(), super.getParameters()));
            }

            //Millis:               0000 0000 0000 0000 0000 0000 0001 0000 (Binary 16)
            if ((logRecordMask & 16) == 16) {
                commonBaseEvent.setCreationTime(super.getMillis());
            }

            logRecordMask = 0; //Binary 0: 0000 0000 0000 0000 0000 0000 0000 0000
        }

        return commonBaseEvent;
    }

    /**
     * Generates the XML fragment representation of the CBE in this log record.
     * 
     * The XML fragment representation of the commonBaseEvent instance variable is generated and not the 
     * XML fragment representation of the complete <code>java.util.logging.LogRecord</code>.
     * 
     * @return The XML fragment representation of the CBE in this log record, or <code>null</code> if the value of the commonBaseEvent instance variable has not been set or is set to <code>null</code>.
     */
    public final String externalizeCanonicalXmlString() {
        return (CbeFormatter.toCanonicalXMLString(getCommonBaseEvent()));
    }

    /**
     * Generates the XML document representation of the CBE in this log record.
     * 
     * The XML document representation of the commonBaseEvent instance variable is generated and not the 
     * XML document representation of the complete <code>java.util.logging.LogRecord</code>.
     * 
     * @return The XML document representation of the CBE in this log record, or <code>null</code> if the value of the commonBaseEvent instance variable has not been set or is set to <code>null</code>.
     */
    public final String externalizeCanonicalXmlDocString() {
        return (CbeFormatter.toCanonicalXMLDocString(getCommonBaseEvent()));
    }

    /**
     * Maps the parameter <code>java.util.logging.Level</code> to a CBE severity, extendedDataElement and situation.
     * 
     * @param level The <code>java.util.logging.Level</code> to be mapped to a CBE severity, extendedDataElement and situation.
     */
    private void setCBESeverity(Level level) {

        IExtendedDataElement extendedDataElement = commonBaseEvent.getExtendedDataElement("LoggingLevel");

        if (extendedDataElement == null) {

            extendedDataElement = simpleEventFactory.createExtendedDataElement();

            extendedDataElement.setType(IExtendedDataElement.TYPE_STRING_VALUE);
            extendedDataElement.setName("LoggingLevel");
        }

        IReportSituation reportSituation = simpleEventFactory.createReportSituation();
        reportSituation.setReasoningScope("INTERNAL");

        ISituation situation = commonBaseEvent.getSituation();

        if (situation == null) {
            situation = simpleEventFactory.createSituation();
        }

        situation.setCategoryName("ReportSituation");
        situation.setSituationType(reportSituation);

        int levelIntValue = level.intValue();

        if (levelIntValue == LEVEL_ALL) {

            commonBaseEvent.setSeverity(((short) (0)));
            extendedDataElement.setValues(new String[] { "ALL" });
            reportSituation.setReportCategory("LOG");
        }

        else if (levelIntValue == LEVEL_FINEST) {

            commonBaseEvent.setSeverity(((short) (10)));
            extendedDataElement.setValues(new String[] { "FINIST" });
            reportSituation.setReportCategory("TRACE");
        }
        else if (levelIntValue == LEVEL_FINER) {

            commonBaseEvent.setSeverity(((short) (10)));
            extendedDataElement.setValues(new String[] { "FINER" });
            reportSituation.setReportCategory("TRACE");
        }
        else if (levelIntValue == LEVEL_FINE) {

            commonBaseEvent.setSeverity(((short) (10)));
            extendedDataElement.setValues(new String[] { "FINE" });
            reportSituation.setReportCategory("TRACE");
        }
        else if (levelIntValue == LEVEL_CONFIG) {

            commonBaseEvent.setSeverity(((short) (10)));
            extendedDataElement.setValues(new String[] { "CONFIG" });
            reportSituation.setReportCategory("TRACE");
        }
        else if (levelIntValue == LEVEL_INFO) {

            commonBaseEvent.setSeverity(((short) (10)));
            extendedDataElement.setValues(new String[] { "INFO" });
            reportSituation.setReportCategory("LOG");
        }
        else if (levelIntValue == LEVEL_WARNING) {

            commonBaseEvent.setSeverity(((short) (30)));
            extendedDataElement.setValues(new String[] { "WARNING" });
            reportSituation.setReportCategory("LOG");
        }
        else if (levelIntValue == LEVEL_SEVERE) {

            commonBaseEvent.setSeverity(((short) (60)));
            extendedDataElement.setValues(new String[] { "SEVERE" });
            reportSituation.setReportCategory("LOG");
        }
        else if (levelIntValue == LEVEL_OFF) {

            commonBaseEvent.setSeverity(((short) (70)));
            extendedDataElement.setValues(new String[] { "OFF" });
            reportSituation.setReportCategory("LOG");
        }
        else {

            commonBaseEvent.setSeverity(((short) (70)));
            extendedDataElement.setValues(new String[] { level.getName()});
            reportSituation.setReportCategory("LOG");
        }
    }

    /**
     * Localizes the parameter property key based on the parameter resource bundle and formats the resultant message 
     * based on the parameter array.
     * 
     * Formatting is done based on <code>java.text</code> style of formating (e.g. {0}).
     * 
     * @param message The string message to be formatted or property key of the localized resource.
     * @param bundle The resource bundle used to retrieved the localized string message.
     * @param parameters The paramenter array used to format the localized/parameter string message.
     * @return Localized and formatted string message.
     */
    private String formatMessage(String message, ResourceBundle bundle, Object[] parameters) {

        if (message != null) {

            String formattedMessage = message;

            if (bundle != null) {

                try {
                    formattedMessage = bundle.getString(message);
                }
                catch (MissingResourceException m) {
                }
            }

            if ((parameters != null) && (parameters.length > 0)) {

                if (formattedMessage.indexOf("{0") != -1) {
                    formattedMessage = MessageFormat.format(formattedMessage, parameters);
                }
            }

            return formattedMessage;
        }

        return null;
    }

    /**
     * @see java.util.logging.LogRecord#setLevel(java.util.logging.Level)
     */
    public final void setLevel(Level level) {

        super.setLevel(level);

        logRecordMask |= 2; //Binary 2: 0000 0000 0000 0000 0000 0000 0000 0010
    }

    /**
     * @see java.util.logging.LogRecord#setLoggerName(java.lang.String)
     */
    public final void setLoggerName(String name) {

        super.setLoggerName(name);

        logRecordMask |= 4; //Binary 4: 0000 0000 0000 0000 0000 0000 0000 0100
    }

    /**
     * @see java.util.logging.LogRecord#setMessage(java.lang.String)
     */
    public final void setMessage(String message) {

        super.setMessage(message);

        logRecordMask |= 8; //Binary 8: 0000 0000 0000 0000 0000 0000 0000 1000
    }

    /**
     * @see java.util.logging.LogRecord#setMillis(long)
     */
    public final void setMillis(long millis) {

        super.setMillis(millis);

        logRecordMask |= 16; //Binary 16: 0000 0000 0000 0000 0000 0000 0001 0000
    }

    /**
     * @see java.util.logging.LogRecord#setParameters(java.lang.Object[])
     */
    public final void setParameters(Object[] parameters) {

        super.setParameters(parameters);

        logRecordMask |= 32; //Binary 32: 0000 0000 0000 0000 0000 0000 0010 0000
    }

    /**
     * @see java.util.logging.LogRecord#setResourceBundle(java.util.ResourceBundle)
     */
    public final void setResourceBundle(ResourceBundle bundle) {

        super.setResourceBundle(bundle);

        logRecordMask |= 64; //Binary 64: 0000 0000 0000 0000 0000 0000 0100 0000
    }

    /**
     * @see java.util.logging.LogRecord#setResourceBundleName(java.lang.String)
     */
    public final void setResourceBundleName(String name) {

        super.setResourceBundleName(name);

        logRecordMask |= 128; //Binary 128: 0000 0000 0000 0000 0000 0000 1000 0000
    }

    /**
     * @see java.util.logging.LogRecord#setSequenceNumber(long)
     */
    public final void setSequenceNumber(long seq) {

        super.setSequenceNumber(seq);

        logRecordMask |= 256; //Binary 256: 0000 0000 0000 0000 0000 0001 0000 0000
    }

    /**
     * @see java.util.logging.LogRecord#setSourceClassName(java.lang.String)
     */
    public final void setSourceClassName(String sourceClassName) {

        super.setSourceClassName(sourceClassName);

        logRecordMask |= 512; //Binary 512: 0000 0000 0000 0000 0000 0010 0000 0000
    }

    /**
     * @see java.util.logging.LogRecord#setSourceMethodName(java.lang.String)
     */
    public final void setSourceMethodName(String sourceMethodName) {

        super.setSourceMethodName(sourceMethodName);

        logRecordMask |= 1024; //Binary 1024: 0000 0000 0000 0000 0000 0100 0000 0000
    }

    /**
     * @see java.util.logging.LogRecord#setThreadID(int)
     */
    public final void setThreadID(int threadID) {

        super.setThreadID(threadID);

        logRecordMask |= 2048; //Binary 2048: 0000 0000 0000 0000 0000 1000 0000 0000
    }

    /**
     * @see java.util.logging.LogRecord#setThrown(java.lang.Throwable)
     */
    public final void setThrown(Throwable thrown) {

        super.setThrown(thrown);

        logRecordMask |= 4096; //Binary 4096: 0000 0000 0000 0000 0001 0000 0000 0000
    }

    /*
        //Test method:
        public static void main(String[] args) {
    
            CommonBaseEventLogRecord cbeLogRecord = new CommonBaseEventLogRecord(Level.FINEST, "loggingMessage");
            cbeLogRecord.setLoggerName("loggingLevel");
            cbeLogRecord.setMillis(1000);
            cbeLogRecord.setParameters(new Object[] { "Parameter1", "Parameter1" });
            cbeLogRecord.setResourceBundle(new ResourceBundle() {
    
                public Object handleGetObject(String key) {
    
                    if (key.equals("ADSM")) {
                        return "This is a test message with two parameters ({0} and {1}).";
                    }
    
                    return null;
                }
    
                public Enumeration getKeys() {
                    return null;
                }
            });
            cbeLogRecord.setResourceBundleName("TestResources");
            cbeLogRecord.setSequenceNumber(1000);
            cbeLogRecord.setSourceClassName("loggingClassName");
            cbeLogRecord.setSourceMethodName("loggingMethodName");
            cbeLogRecord.setThreadID(199);
            cbeLogRecord.setThrown(new Throwable("loggingThrowableMessage", new Throwable("loggingThrowableCauseMessage")));
    
            System.out.println(CbeFormatter.toCanonicalXMLString(cbeLogRecord.getCommonBaseEvent(), true) + "\n");
    
            cbeLogRecord.setLevel(Level.SEVERE);
    
            System.out.println(CbeFormatter.toCanonicalXMLString(cbeLogRecord.getCommonBaseEvent(), true) + "\n");
    
            cbeLogRecord.setMessage("ADSM");
    
            System.out.println(CbeFormatter.toCanonicalXMLString(cbeLogRecord.getCommonBaseEvent(), true) + "\n");
        }
    */
}
