/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.logging.adapter.ui.internal.util;

import java.io.File;

import org.eclipse.emf.common.notify.Notification;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.edit.provider.*;
import org.eclipse.hyades.logging.adapter.model.internal.configuration.ContextInstanceType;
import org.eclipse.hyades.logging.adapter.model.internal.outputter.SingleFileOutputterType;
import org.eclipse.hyades.logging.adapter.ui.AcadEditorPlugin;
import org.eclipse.hyades.logging.adapter.ui.internal.outputter.*;
import org.eclipse.hyades.logging.adapter.ui.internal.views.AcadGuiModelManager;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.*;
import org.eclipse.swt.layout.*;
import org.eclipse.swt.widgets.*;

public class SingleFileOutputterSection extends Composite
								implements IDetails, ModifyListener, SelectionListener
										   , INotifyChangedListener
{
	protected Text directoryText;
	protected Button browseDirectory;			
	protected Text fileIdText;
	protected Button browseFile;
    protected boolean isDisplaying;
    protected SingleFileOutputterType currentSelection;
    protected SingleFileOutputterTypeItemProvider itemProvider;
    protected OutputterItemProviderAdapterFactory itemFactory;
	
	
	public SingleFileOutputterSection(RuleBuilderWidgetFactory factory, Composite composite, int style){

		super(composite, style);
		
		this.itemFactory = new OutputterItemProviderAdapterFactory();

		GridLayout detailLayout = new GridLayout();
		detailLayout.numColumns = 3;
		detailLayout.verticalSpacing = 5;

		this.setLayout(detailLayout);
		this.setLayoutData(GridUtil.createFill());
		
		Label label = factory.createLabel(this, AcadEditorPlugin.getPlugin().getString("STR_SFILEOUT_DIR"), SWT.NONE);	
		directoryText = factory.createText(this, "", SWT.MULTI);
		directoryText.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));		
		browseDirectory = factory.createButton(this, AcadEditorPlugin.getPlugin().getString("BROWSE"), SWT.PUSH);		
		
		label = factory.createLabel(this, AcadEditorPlugin.getPlugin().getString("STR_SFILEOUT_FILE"), SWT.NONE);	
		fileIdText = factory.createText(this, "", SWT.NULL);
		fileIdText.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));		
		browseFile = factory.createButton(this, AcadEditorPlugin.getPlugin().getString("BROWSE2"), SWT.PUSH);		
		

		// add listeners - don't change the order in which the listeners are added
		// because of team support issues modify listeners should always be the last one added
		directoryText.addModifyListener(this);
		fileIdText.addModifyListener(this);
		browseDirectory.addSelectionListener(this);
		browseFile.addSelectionListener(this);
		
		itemProvider = (SingleFileOutputterTypeItemProvider)itemFactory.createSingleFileOutputterTypeAdapter();
		itemProvider.addListener(this);
	}
	
	/**
	 * @see IDetails#display(Object)
	 */
	public void display(EObject object) {
  
		if(object != null)
			AcadGuiModelManager.setCurrentModel((ContextInstanceType)((SingleFileOutputterType)object).eContainer().eContainer());

		if(currentSelection == object)
		  return;
        
		if(directoryText.isFocusControl() || fileIdText.isFocusControl())		   
		  return;
        
		currentSelection = (SingleFileOutputterType)object;
		isDisplaying = true;
			
		directoryText.setText(currentSelection.getDirectory()==null?"":currentSelection.getDirectory());
		fileIdText.setText(currentSelection.getFileName()==null?"":currentSelection.getFileName());
		
		isDisplaying = false;
	}

	/**
	 * @see IDetails#updateCurrentSelection(Object)
	 */
	public boolean updateCurrentSelection(EObject currentSelection) {

 		this.currentSelection = (SingleFileOutputterType)currentSelection;
		
		this.currentSelection.setDirectory(directoryText.getText());
		this.currentSelection.setFileName(fileIdText.getText());
		
		return false;

	}
	
	public void setFocusToText(){
	        directoryText.setFocus();
	        directoryText.selectAll();
	}

	/**
	 * Sent when the text is modified.
	 *
	 * @param e an event containing information about the modify
	 */
	public void modifyText(ModifyEvent e)
	{
		
		if(isDisplaying || currentSelection == null)
		  return;
		  
		if(e.getSource() == directoryText)
		{
			String text = directoryText.getText();
			if(!text.equals(currentSelection.getDirectory()))
			{
				if(text.length() == 0 && currentSelection.getDirectory() == null)
				  return;
				  
				IItemPropertyDescriptor descriptor = 
					itemProvider.getPropertyDescriptor(
						currentSelection,
						SingleFileOutputterTypeItemProvider.DIRECTORY);
						
				if(descriptor != null)
					descriptor.setPropertyValue(currentSelection, text);				
			}			
		}
		else if(e.getSource() == fileIdText)
		{
			String text = fileIdText.getText();
			if(!text.equals(currentSelection.getFileName()))
			{
				if(text.length() == 0 && currentSelection.getFileName() == null)
				  return;
				  
				IItemPropertyDescriptor descriptor = 
					itemProvider.getPropertyDescriptor(
						currentSelection,
						SingleFileOutputterTypeItemProvider.FILE_NAME);
						
				if(descriptor != null)
					descriptor.setPropertyValue(currentSelection, text);				
			}			
		}
	}

	public void dispose() {
		
		itemFactory = null;
		itemProvider = null;
	}

	/**
	 */
	public void notifyChanged(Notification notification)
	{
		Object obj = notification.getNotifier();
		
		if(obj == null || (obj != currentSelection))
		  return;
		  
		display(currentSelection);
	}

	public void widgetDefaultSelected(SelectionEvent e)
	{
	}

	public void widgetSelected(SelectionEvent e)
	{
		if(e.getSource() == browseDirectory)
		{
			DirectoryDialog dialog = new DirectoryDialog(Display.getCurrent().getActiveShell());
			dialog.setMessage("Select a directory");
			String dirName = directoryText.getText();			
			if(dirName.length() > 0)			  
				dialog.setFilterPath(dirName);
			
			String selectedDirectory = dialog.open();
			if (selectedDirectory != null)
				directoryText.setText(selectedDirectory);						
		}
		else if(e.getSource() == browseFile)
		{
			FileDialog dialog = new FileDialog(Display.getCurrent().getActiveShell());
			String dirName = directoryText.getText();
			if(dirName.length() > 0 && (new File(dirName)).exists())			  
				dialog.setFileName(dirName.concat(File.separator).concat("*"));
				
			String selectedFile = dialog.open();
			if (selectedFile != null)
			{
				File file = new File(selectedFile);
				if(file.exists())
				{
					directoryText.setText(file.getParent());
					fileIdText.setText(file.getName());
					 
				}
			}
			
		}
	}

}
