/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.test.common.internal.wizard;

import java.util.ArrayList;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.jface.viewers.AbstractTreeViewer;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.StructuredViewer;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerSorter;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Tree;
import org.eclipse.ui.help.WorkbenchHelp;
import org.eclipse.ui.model.IWorkbenchAdapter;

import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.common.util.UniqueEList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;

import org.eclipse.hyades.models.common.configuration.CFGInstance;
import org.eclipse.hyades.models.common.configuration.CFGLocation;
import org.eclipse.hyades.models.common.configuration.CFGMachine;
import org.eclipse.hyades.models.common.testprofile.Common_TestprofilePackage;
import org.eclipse.hyades.models.common.testprofile.TPFDeployment;
import org.eclipse.hyades.models.common.util.ICommonConstants;
import org.eclipse.hyades.test.common.TestCommonPlugin;
import org.eclipse.hyades.test.common.internal.util.ContextIds;
import org.eclipse.hyades.test.ui.TestUIImages;
import org.eclipse.hyades.test.ui.TestUIPlugin;
import org.eclipse.hyades.test.ui.internal.model.EMFUtil;
import org.eclipse.hyades.test.ui.internal.model.ui.EMFContentProvider;
import org.eclipse.hyades.ui.HyadesUIImages;
import org.eclipse.hyades.ui.internal.util.GridDataUtil;


public class RunLocationPage extends WizardPage implements Listener
{
	private AbstractTreeViewer hostViewer;
	private Button             filterButton;
	private HostsContentProvider contentProvider;
	private CFGLocation          location; 
	private URI initialLocationURI;
	private EnvironmentVariableViewer environmentVariableViewer;
	private boolean showEvironmentVariableViewer;
	
	
	private class HostsContentProvider implements ITreeContentProvider
	{
		public static final int FILTER_FOLDERS = 2;
		public static final int SHOW_ALL = 4;
		
		private int filter;
		private ArrayList list = new ArrayList();
	
		public HostsContentProvider()
		{ 
		}

		public int getFilter()
		{
			return filter;
		}

		public void setFilter(int filter)
		{
			this.filter = filter;
		}
		
		public void dispose()
		{
			list.clear();
		}

		public void inputChanged(Viewer viewer, Object oldInput, Object newInput)
		{
		}

		public Object getParent(Object element)
		{
			if(element instanceof IResource)
				return ((IResource)element).getParent();
			
			if(element instanceof EObject)
			{
				EObject eObject = (EObject)element;
				if(eObject instanceof CFGLocation)
					return eObject.eContainer();
			
				switch(filter)
				{
					case SHOW_ALL:
						IFile file = EMFUtil.getWorkspaceFile(eObject);
						if(file != null)
							return file.getParent();
						break;
				}
			}

			return null;
		}

		public boolean hasChildren(Object element)
		{
			return (getChildren(element).length > 0);
		}	

		public Object[] getElements(final Object inputElement)
		{
			if(inputElement instanceof IWorkspaceRoot)
			{			
				if(getFilter() == FILTER_FOLDERS)
				{			
					list.clear();		
					IProject[] projects = ((IWorkspaceRoot)inputElement).getProjects();					
					
					for(int idx=0; idx<projects.length; idx++)
					{
						Object[] children = getChildren(projects[idx]);
						for(int i=0; i<children.length; i++)
							list.add(children[i]);						
					}
					
					return list.toArray();
				}
				else
				{
					return ((IWorkspaceRoot)inputElement).getProjects();				
				}				
			}
			
			return getChildren(inputElement);
		}

		/**
		 * @see org.eclipse.jface.viewers.ITreeContentProvider#getChildren(java.lang.Object)
		 */
		public Object[] getChildren(Object parentElement)
		{
			if(parentElement instanceof IContainer)
			{
				List children = new UniqueEList(); 
				IResource[] resources = null;
				try
				{
					resources = ((IContainer)parentElement).members();
				}
				catch (CoreException e)
				{
					TestUIPlugin.logError(e);
					return new Object[0];
				}
			 
				for(int i = 0, maxi = resources.length; i < maxi; i++)
				{
					if(resources[i].getType() != IResource.FILE)
					{
						if(filter == FILTER_FOLDERS)
						{
							Object[] tmpArray = getChildren(resources[i]);
							for(int k=0; k<tmpArray.length; k++)
							   children.add(tmpArray[k]);
						}
						else
							children.add(resources[i]);
					}
					else
					{
						    IFile file = (IFile)resources[i];
						    if(ICommonConstants.DEPLOYMENT_FILE_EXTENSION.equals(file.getFileExtension()))
						    {
								EObject[] eObjects = EMFUtil.load(getResourceSet(), (IFile)resources[i]);
								for (int idx=0; idx<eObjects.length; idx++)
								{
									Object eObject = eObjects[idx];
									if(eObject instanceof TPFDeployment)
									   children.add(eObject);
								   
								}						    	
						    }
					}
				}
			
				return children.toArray();
			}
		
			else if(parentElement instanceof Resource)
				return ((Resource)parentElement).getContents().toArray();
		
			else if(parentElement instanceof TPFDeployment)
			{
				List children = new UniqueEList(); 				
				children.addAll(((TPFDeployment)parentElement).getLocations());
				
				return children.toArray();
				
			}
		
			return new Object[0];
		}		
	}
	
	/**
	 * 
	 */
	private class HostsLabelProvider extends LabelProvider
	{
		private Hashtable ovelayCache;
		private boolean useImages;
	
		/**
		 * Constructor for PerspectiveLabelProvider
		 */
		public HostsLabelProvider(boolean useImages)
		{
			super();
			this.useImages = useImages;
			ovelayCache = new Hashtable();
		}
	
		/**
		 * @see LabelProvider#getImage(Object)
		 */
		public Image getImage(Object element)
		{
			if((!useImages) || (element == null))
				return null;

			if(element instanceof TPFDeployment)
				return TestUIImages.INSTANCE.getImage(TestUIImages.IMG_DEPLOYMENT);

			if(element instanceof CFGMachine)
				return HyadesUIImages.INSTANCE.getImage(HyadesUIImages.IMG_UI_NODE);

			if(element instanceof CFGLocation)
				return TestUIImages.INSTANCE.getImage(TestUIImages.IMG_LOCATION);

			if (element instanceof IAdaptable)
			{
				IWorkbenchAdapter adapter =
					(IWorkbenchAdapter)((IAdaptable)element).getAdapter(IWorkbenchAdapter.class);

				if(adapter != null)
					return adapter.getImageDescriptor(element).createImage();
			}

			return null;			
		}

		/**
		 * @see LabelProvider#getText(Object)
		 */
		public String getText(Object element)
		{
			if(element == null)
				return "";

			String text = null;
		
			if(element instanceof TPFDeployment)
				text = ((TPFDeployment)element).getName();
			
			else if(element instanceof CFGLocation)
				text = ((CFGLocation)element).getName();
			
			else if(element instanceof IAdaptable)
			{
				IWorkbenchAdapter adapter =
					(IWorkbenchAdapter)((IAdaptable)element).getAdapter(IWorkbenchAdapter.class);

				if(adapter != null)
					text = adapter.getLabel(element);
			}

			if(text == null)
				text = element.toString();
			
			return text;		
		}
		
	
		/**
		 * @see org.eclipse.jface.viewers.IBaseLabelProvider#dispose()
		 */
		public void dispose()
		{
			if(ovelayCache != null)
			{
				for(Iterator i = ovelayCache.values().iterator(); i.hasNext();)
				{
					Image image = (Image)i.next();
					image.dispose();
				}
				ovelayCache.clear();
			}
		
			super.dispose();
		}
	}	
	
	/**
	 * 
	 * @param required
	 */
	public RunLocationPage(boolean showEvironmentVariableViewer)
	{
		super("node");
		
		setTitle(TestUIPlugin.getString("RUN_WTITLE"));
		setDescription(TestUIPlugin.getString("RUN_NODED"));
		this.showEvironmentVariableViewer = showEvironmentVariableViewer;
	}
	
	/**
	 * @see org.eclipse.jface.dialogs.IDialogPage#dispose()
	 */
	public void dispose()
	{
		if(showEvironmentVariableViewer)
			environmentVariableViewer.dispose();
		super.dispose();
	}

	public void setInitialLocationURI(URI initialLocationURI)
	{
		this.initialLocationURI = initialLocationURI;
	}

	/*
	 * @see IDialogPage#createControl(Composite)
	 */
	public void createControl(Composite parent)
	{
		GridLayout layout = new GridLayout();
		layout.numColumns = 1;
		
		Composite composite = new Composite(parent, SWT.NONE);
		composite.setLayout(layout);
		GridData gridData = GridDataUtil.createFill();
		gridData.heightHint = 100;
		composite.setLayoutData(GridDataUtil.createFill());
	
		filterButton = new Button(composite, SWT.CHECK);
		filterButton.setText(TestUIPlugin.getString("SHOW_FOLDERS"));
		filterButton.addListener(SWT.Selection, this);
		createViewer(composite);
				
		if(showEvironmentVariableViewer)
		{
			Label label = new Label(composite, SWT.NONE);
			label.setText(TestUIPlugin.getString("LBL_ENV_SETS"));
			environmentVariableViewer = new EnvironmentVariableViewer(((RunWizard)getWizard()).getTest());
			environmentVariableViewer.createControl(composite);
		}
			
		hostViewer.setInput(ResourcesPlugin.getWorkspace().getRoot());
		((Tree)hostViewer.getControl()).addListener(SWT.Selection, this);
		
		if(initialLocationURI == null)
		{
			((TreeViewer)hostViewer).expandToLevel(2);
		}
		else
		{
			EObject eObject = null;
			try
			{
				eObject = getResourceSet().getEObject(initialLocationURI, true);
			}
			catch(Exception e)
			{
			}
			
			if((eObject != null) && (eObject instanceof CFGLocation))
			{
				hostViewer.setSelection(new StructuredSelection(eObject));
				setLocation();
			}
		}

		setControl(composite);
		setPageComplete(location != null);
		
		WorkbenchHelp.setHelp(composite, TestCommonPlugin.getID() + ContextIds.RUN_LOC_PAGE);				
		WorkbenchHelp.setHelp(filterButton, TestCommonPlugin.getID() + ContextIds.RUN_LOC_PAGE_FBTN);				
		WorkbenchHelp.setHelp(hostViewer.getControl(), TestCommonPlugin.getID() + ContextIds.RUN_LOC_PAGE_HOST);
	}
	
	/**
	 * @see Listener#handleEvent(Event)
	 */
	public void handleEvent(Event event)
	{	
		if(event.type == SWT.Selection)
		{
			if(event.widget == filterButton)						
			{
				contentProvider.setFilter((filterButton.getSelection()?HostsContentProvider.SHOW_ALL:HostsContentProvider.FILTER_FOLDERS));
				ISelection selection = hostViewer.getSelection();				
				hostViewer.refresh();
				if(!selection.isEmpty())
					hostViewer.setSelection(selection, true);
			}
			
			setLocation();
		}
		
		setPageComplete(getLocation() != null);		
	}
	
    /**
     * 
     * @param parent
     * @return
     */
	protected StructuredViewer createViewer(Composite parent)
	{		
		Tree tree = new Tree(parent, SWT.SINGLE | SWT.FULL_SELECTION | SWT.BORDER );
		GridData gridData = GridDataUtil.createFill();
		gridData.heightHint = 150;
		tree.setLayoutData(gridData);
				
		contentProvider = new HostsContentProvider();
		hostViewer = new TreeViewer(tree);
		hostViewer.setContentProvider(contentProvider);
		hostViewer.setLabelProvider(new HostsLabelProvider(true));

		contentProvider.setFilter(HostsContentProvider.FILTER_FOLDERS);

		ViewerSorter sorter = new ViewerSorter()
		{
			public void sort(Viewer viewer, Object[] elements)
			{
				if(elements.length > 0)
					super.sort(viewer, elements);
			}
			
			public int category(Object element)
			{
				if(element instanceof IResource)
					return 0;

				return 10;
			}		
		};
		hostViewer.setSorter(sorter);
		
		return hostViewer;
	}

    /**
     * 
     *
     */
	protected ITreeContentProvider initializeContentProvider()
	{
		EMFContentProvider contentProvider = new EMFContentProvider(getResourceSet());
		
		EMFContentProvider.ElementDescription elementDescription = new EMFContentProvider.ElementDescription(ICommonConstants.TEST_SUITE_FILE_EXTENSION);
		elementDescription.getEClassifiers().add(Common_TestprofilePackage.eINSTANCE.getTPFSUT());
		contentProvider.addElementDescription(elementDescription); 
				
		return contentProvider;		
	}

	/**
	 * Returns this navigator's resource set.
	 * @return ResourceSet
	 */
	public ResourceSet getResourceSet()
	{
		return ((RunWizard)getWizard()).getResourceSet();
	}

	/**
	 * @return run location
	 */
	public CFGLocation getLocation() {
		return location;
	}
	
	public CFGInstance getTestSuiteInstance()
	{
		if(showEvironmentVariableViewer)
			return environmentVariableViewer.calculateTestSuiteInstance(location);
		return null;
	}
	
	public boolean isCalculatedClasspath()
	{
		if(showEvironmentVariableViewer)
			return environmentVariableViewer.isCalculatedClasspath();
		return true;
	}	

	/**
	 * @param tpfsut
	 */
	public void setLocation() {
		
		location = null;
		
		ISelection selection = hostViewer.getSelection();
		if(selection != null && !selection.isEmpty())
		{ 
			Object sel = ((IStructuredSelection)selection).getFirstElement();
			if(sel != null && sel instanceof CFGLocation)		
				location = (CFGLocation)sel;
		}
		
		if(environmentVariableViewer != null)
			environmentVariableViewer.setInput(getLocation());
	}

}