/*******************************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.ui.internal.editor;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IMarker;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.util.ListenerList;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.CTabFolder;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Label;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IEditorSite;
import org.eclipse.ui.ISelectionListener;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.PartInitException;

import org.eclipse.hyades.ui.HyadesUIPlugin;
import org.eclipse.hyades.ui.editor.IEditorExtension;
import org.eclipse.hyades.ui.editor.IHyadesEditorPart;
import org.eclipse.hyades.ui.extension.IAssociationDescriptor;
import org.eclipse.hyades.ui.internal.editor.action.ActionHandlerListener;
import org.eclipse.hyades.ui.internal.util.UIUtil;
import org.eclipse.hyades.ui.util.IRefreshable;

/**
 * Abstract implementation of the IHyadesEditorPart.  This class has
 * all the code to support the Hyades editor extension mechanism.
 * 
 * <p>Subclasses are supposed to customize the HyadesEditorPart to a 
 * particular type of object or resource file.
 * 
 * @see IHyadesEditorPart
 * @author marcelop
 * @since 0.0.1
 */
public abstract class HyadesEditorPart 
extends PageContainerEditorPart implements IRefreshable, ISelectionListener
{
	private int defaultTabHeight; //Stores the height of the editor's tab.
	private Object editorObject;
	private IAssociationDescriptor editorDescriptor;
	private IEditorExtension editorExtension;
	private ListenerList selectionChangedListeners;
	private EditorSynchronizer editorSynchronizer;
	
	/**
	 * Constructor for HyadesEditorPart
	 */
	public HyadesEditorPart()
	{
		selectionChangedListeners = new ListenerList();
	}
	
	/**
	 * @see org.eclipse.ui.IWorkbenchPart#dispose()
	 */
	public void dispose()
	{
		if(editorSynchronizer != null)
		{
			editorSynchronizer.dispose();
			editorSynchronizer = null;
		}
		
		getSite().setSelectionProvider(null);
		getSite().getPage().removePostSelectionListener(this);
		selectionChangedListeners.clear();
		if(editorExtension != null)
		{
			editorExtension.dispose();
			editorExtension = null;
		}
		editorDescriptor = null;			
		editorObject = null;
		
		super.dispose();
	}

	/**
	 * @see org.eclipse.hyades.ui.editor.IHyadesEditorPart#getEditorExtensionDefinition()
	 */
	public final IAssociationDescriptor getEditorDescriptor()
	{
		return editorDescriptor;
	}

	/**
	 * Returns the editor extension that is used by this editor.  This
	 * method is not supposed to be overwritten.
	 * @return IEditorExtension
	 */
	protected final IEditorExtension getEditorExtension()
	{
		return editorExtension;
	}
	
	/**
	 * @see org.eclipse.hyades.ui.editor.IHyadesEditorPart#setEditorObject(Object)
	 */
	public void setEditorObject(Object editorObject)
	{
		this.editorObject = editorObject;
		firePropertyChange(IHyadesEditorPart.PROP_OBJECT);
	}

	/**
	 * @see org.eclipse.hyades.ui.editor.IHyadesEditorPart#getEditorObject()
	 */
	public Object getEditorObject()
	{
		return editorObject;
	}

	/**
	 * @see org.eclipse.ui.part.MultiPageEditorPart#addPage(org.eclipse.swt.widgets.Control)
	 */
	public int addPage(Control control)
	{
		int ret = super.addPage(control);
		adjustContainer();
		return ret;
	}

	/**
	 * @see org.eclipse.ui.part.MultiPageEditorPart#addPage(org.eclipse.ui.IEditorPart, org.eclipse.ui.IEditorInput)
	 */
	public int addPage(IEditorPart editor, IEditorInput input) 
	throws PartInitException
	{
		int ret = super.addPage(editor, input);
		adjustContainer();
		return ret;
	}

	/**
	 * @see org.eclipse.ui.part.MultiPageEditorPart#pageChange(int)
	 */
	protected void pageChange(int newPageIndex)
	{
		super.pageChange(newPageIndex);
		notifyPageChange(newPageIndex);
	}

	/**
	 * @see org.eclipse.ui.part.MultiPageEditorPart#setActivePage(int)
	 */
	public void setActivePage(int pageIndex)
	{
		boolean pageChanged = (getActivePage() != pageIndex);
		super.setActivePage(pageIndex);
		if(pageChanged)
			notifyPageChange(pageIndex);
	}
	
	/**
	 * Notifies the editor extension that a new page was activated.
	 * @param newPageIndex
	 */
	protected void notifyPageChange(int newPageIndex)
	{
		if(editorExtension == null)
			return;
		
		if(editorExtension.pageActivated(newPageIndex))
			fileSelection(getSelection());		
	}

	/**
	 * @see org.eclipse.ui.part.MultiPageEditorPart#removePage(int)
	 */
	public void removePage(int pageIndex)
	{
		super.removePage(pageIndex);
		adjustContainer();
	}
	
	/**
	 * Adjusts the container by adjusting the height of the tab depending
	 * on the number of available pages - the goal is to hide all
	 * tab if there is only one page.
	 */
	protected void adjustContainer()
	{
		if(getContainer() instanceof CTabFolder)
		{
			int currentTabHeight = ((CTabFolder)getContainer()).getTabHeight();
			if(currentTabHeight > 0)
				defaultTabHeight = currentTabHeight;
			((CTabFolder)getContainer()).setTabHeight((getPageCount()==1)?0:defaultTabHeight);
		}
	}

	/**
	 * @see org.eclipse.ui.part.MultiPageEditorPart#createPages()
	 */
	protected final void createPages()
	{
		if(editorSynchronizer == null)
		{
			//Editor files doesn't exist.
			setTitle(HyadesUIPlugin.getString("EDT_HYADES_TTL"));
			
			String resource = "";
			IFile editorFile = EditorSynchronizer.getEditorFile(getEditorInput());
			if(editorFile != null)
			{
				resource = editorFile.getFullPath().toString();
				try
				{
					editorFile.refreshLocal(IResource.DEPTH_ZERO, null);
				}
				catch (CoreException e)
				{
					HyadesUIPlugin.logError(e);
				}
			}
			
			String text = HyadesUIPlugin.getString("RES_NOT_EXIST", resource);
			HyadesUIPlugin.logError(text);
			
			Label label = new Label(getContainer(), SWT.NONE);
			label.setText(text);
			addPage(label);

			return;
		}
				
		if(editorExtension != null)
		{
			editorExtension.createPages();
		}
		else
		{
			//Hard coded editor extension.  Should never be presented.
			setTitle(HyadesUIPlugin.getString("EDT_HYADES_TTL"));
			
			String text = HyadesUIPlugin.getString("EDT_HYADES_TXT");
			HyadesUIPlugin.logError(text);
			
			Label label = new Label(getContainer(), SWT.NONE);
			label.setText(text);
			label.setBackground(Display.getCurrent().getSystemColor(SWT.COLOR_RED));
			addPage(label);			
		}
		
		if((getTitle() == null) && (editorDescriptor != null) && (editorDescriptor.getName() != null))
			setTitle(editorDescriptor.getName());		
	}
	
	/**
	 * @see org.eclipse.ui.ISaveablePart#doSave(org.eclipse.core.runtime.IProgressMonitor)
	 */
	public void doSave(IProgressMonitor monitor)
	{
		if(editorExtension != null)
		{
			editorSynchronizer.setResourceChangeListenerEnabled(false);
			try
			{
				editorSynchronizer.validateEdit();
				editorExtension.doSave(monitor);
				editorSynchronizer.editorSaved();
			}
			finally
			{
				monitor.setCanceled(isDirty());
				editorSynchronizer.setResourceChangeListenerEnabled(true);
				firePropertyChange(IHyadesEditorPart.PROP_DIRTY);
			}
		}
	}

	/**
	 * @see org.eclipse.ui.ISaveablePart#doSaveAs()
	 */
	public void doSaveAs()
	{
		if(editorExtension != null)
			editorExtension.doSaveAs();
	}

	/**
	 * @see org.eclipse.ui.IEditorPart#gotoMarker(org.eclipse.core.resources.IMarker)
	 */
	public void gotoMarker(IMarker marker)
	{
		if(editorExtension != null)
			editorExtension.gotoMarker(marker);
	}

	/**
	 * @see org.eclipse.ui.ISaveablePart#isSaveAsAllowed()
	 */
	public boolean isSaveAsAllowed()
	{
		if(editorExtension != null)
			editorExtension.isSaveAsAllowed();
			
		return false;
	}

	/**
	 * @see org.eclipse.ui.ISaveablePart#isDirty()
	 */
	public boolean isDirty()
	{
		if(editorExtension != null)
		{
			if(editorExtension.isDirty())
				return true;
		
			if(editorExtension.checkPagesDirtyState())
				return super.isDirty();
		}
		
		return false;
	}

	/**
	 * @see org.eclipse.hyades.ui.util.IRefreshable#refreshContent(java.lang.Object)
	 */
	public void refreshContent(Object data)
	{
		if(editorExtension != null)
		{
			editorExtension.refreshContent(data);
			
			if(editorExtension.refreshPages())
			{
				for(int i=0; i < getPageCount(); i++)
				{
					Object page = getEditor(i);
					if(page == null)
						page = getControl(i);
					
					if(page instanceof IRefreshable)
						((IRefreshable)page).refreshContent(data);
				}
			}
		}
	}

	/**
	 * @see org.eclipse.ui.IEditorPart#init(org.eclipse.ui.IEditorSite, org.eclipse.ui.IEditorInput)
	 */
	public void init(IEditorSite site, IEditorInput input)
	throws PartInitException
	{
		super.init(site, input);
		setTitle(null);
		if(!EditorSynchronizer.exists(EditorSynchronizer.getEditorFile(input)))
			return;
			
		site.setSelectionProvider(this);
		site.getPage().addPostSelectionListener(this);
		
		editorDescriptor = identifyEditorDescriptor(input);
		if(editorDescriptor != null)
		{
			if(cloneExtensionDefinitions())
				editorDescriptor = editorDescriptor.copy();
			
			editorExtension = (IEditorExtension)editorDescriptor.createImplementationClassInstance();
		}
			
		getSite().setSelectionProvider(this);		
		if(editorExtension != null)
			editorExtension.init(this);
			
		enableEditActions();
		enableFileSynchronization();
	}
	
	protected void enableEditActions()
	{
		ActionHandlerListener.DEFAULT.connectPart(this);
	}

	protected void enableFileSynchronization()
	{
		editorSynchronizer = new EditorSynchronizer(this);
	}
	
	/**
	 * Returns true if this editor should clone the extension definitions that are
	 * used in its context.  By cloning those objects the editor shields itself 
	 * against changes that might occur while it is opened - requiring it to be
	 * restarted to apply the changes.
	 * 
	 * <p>Note: The editor extension is always instantiated so it is not necessary
	 * to clone it.
	 * 
	 * <p>Subclasses may overwrite this method.
	 * 
	 * @return boolean
	 */
	protected boolean cloneExtensionDefinitions()
	{
		return true;
	}
	
	/**
	 * Identifies the editor descriptor that should be used with the editor input.
	 * 
	 * <p>Implementors of this method may store the identified extension as a preference 
	 * for the next time this editor is opened.
	 * 
	 * <p><b>Important</b>: This method is supposed to return a valid 
	 * IAssociationDescriptor - ie. not null and related to an editor.
	 * 
	 * @param input
	 * @return IAssociationDescriptor
	 * @throws PartInitException if the editor input is not valid.
	 */
	protected abstract IAssociationDescriptor identifyEditorDescriptor(IEditorInput input)
	throws PartInitException;
		
	/**
	 * @see org.eclipse.jface.viewers.ISelectionProvider#addSelectionChangedListener(org.eclipse.jface.viewers.ISelectionChangedListener)
	 */
	public void addSelectionChangedListener(ISelectionChangedListener listener)
	{
		selectionChangedListeners.add(listener);
	}

	/**
	 * @see org.eclipse.jface.viewers.ISelectionProvider#getSelection()
	 */
	public ISelection getSelection()
	{
		if(getEditorExtension() != null)
			return getEditorExtension().getSelection();
			
		return null;
	}

	/**
	 * @see org.eclipse.jface.viewers.ISelectionProvider#removeSelectionChangedListener(org.eclipse.jface.viewers.ISelectionChangedListener)
	 */
	public void removeSelectionChangedListener(ISelectionChangedListener listener)
	{
		selectionChangedListeners.remove(listener);
	}

	/**
	 * @see org.eclipse.jface.viewers.ISelectionProvider#setSelection(org.eclipse.jface.viewers.ISelection)
	 */
	public void setSelection(ISelection selection)
	{
		if(!UIUtil.areEquals(getSelection(), selection))
		{
			if(selection instanceof IStructuredSelection)
			{
				IStructuredSelection structuredSelection = (IStructuredSelection)selection;
				if(getEditorExtension() != null)
					getEditorExtension().setSelection(structuredSelection);
				updateStatusLine(structuredSelection);
			}
		}
		
		fileSelection(selection);
	}
	
	protected void fileSelection(ISelection selection)
	{
		Object[] listeners = selectionChangedListeners.getListeners();
		for(int i = 0, maxi = listeners.length; i < maxi; i++)
		{
			ISelectionChangedListener listener = (ISelectionChangedListener)listeners[i];
			listener.selectionChanged(new SelectionChangedEvent(this, selection));			
		}		
	}
	
	/**
	 * Updates the message shown in the status line.
	 * @param selection the current selection
	 */
	protected void updateStatusLine(IStructuredSelection structuredSelection)
	{
		String msg = getStatusLineMessage(structuredSelection);
		getEditorSite().getActionBars().getStatusLineManager().setMessage(msg);
	}
	
	/**
	 * Returns the message to show in the status line.  If this method
	 * returns null the status line won't present any message.
	 * 
	 * <p>If a subclass requires a more sophisticated use of the 
	 * status line, it should manipulate the status line direct.  The
	 * status line can be obtained by 
	 * <code>getEditorSite().getActionBars().getStatusLineManager()</code>.
	 *
	 * @param selection the current selection
	 * @return the status line message
	 */
	protected String getStatusLineMessage(IStructuredSelection structuredSelection)
	{
		if(getEditorExtension() != null)
			return getEditorExtension().getStatusLineMessage(structuredSelection);
		return null;
	}
	
	/**
	 * @see org.eclipse.hyades.ui.editor.IHyadesEditorPart#getEditorPart()
	 */
	public IEditorPart getEditorPart()
	{
		return this;
	}
	
	/**
	 * @see org.eclipse.core.runtime.IAdaptable#getAdapter(java.lang.Class)
	 */
	public Object getAdapter(Class adapterClass)
	{
		if(editorExtension != null)
		{
			Object adapter = editorExtension.getAdapter(adapterClass);
			if(adapter != null)
				return adapter;
		}
		return super.getAdapter(adapterClass); 
	}
	
	/**
	 * @see org.eclipse.hyades.ui.editor.IHyadesEditorPart#isReadOnly()
	 */
	public boolean isReadOnly()
	{
		if(editorSynchronizer == null)
			return false;
			
		return EditorSynchronizer.isReadOnly(editorSynchronizer.getEditorFile());
	}
	
	/**
	 * @see org.eclipse.ui.ISelectionListener#selectionChanged(org.eclipse.ui.IWorkbenchPart, org.eclipse.jface.viewers.ISelection)
	 */
	public void selectionChanged(IWorkbenchPart part, ISelection selection)
	{
		if((part != this) && (editorExtension != null))
			editorExtension.selectionChanged(part, selection);
	}
}
