/**********************************************************************
 * Copyright (c) 2003-2004 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.trace.views.internal;

import org.eclipse.emf.ecore.EObject;

import java.util.ArrayList;

import org.eclipse.swt.widgets.*;
import org.eclipse.swt.graphics.*;
import org.eclipse.swt.custom.TableTree;
import org.eclipse.swt.custom.TableTreeItem;

import org.eclipse.jface.viewers.*;
import org.eclipse.jface.action.IMenuManager;

import org.eclipse.hyades.trace.ui.*;
import org.eclipse.hyades.trace.ui.internal.util.TString;
import org.eclipse.hyades.models.trace.*;

import org.eclipse.hyades.trace.views.adapter.internal.CoveragePage;
import org.eclipse.hyades.trace.views.adapter.internal.CoverageViewer;
import org.eclipse.hyades.trace.views.util.internal.*;

import org.eclipse.core.runtime.*;

/**
 * @author hind.hasnaoui@fr.ibm.com  --
 * implements:
 * - the table tree viewer
 * - the filter 
 * - the sorter
 */
public class CoverageStatistic extends StatisticView {

	
	public static final int DEFAULT_LEVEL = 1;
	protected int _level = DEFAULT_LEVEL;
	protected static int _saved_level = DEFAULT_LEVEL;
	//Coverage information can be displayed in three levels:
	// 1: package/class/method level (default)
	// 2: class/method level 
	// 3: method level

	private final int _numberOfColumns = 7;

	/**
	 *  Add a sorter to the coverage view
	 */
	public class CoverageStatisticSorter extends StatisticSorter {

		public CoverageStatisticSorter() {
			super();
		}
		/**
		 * compare between too coverage elements
		*/
		public int compare(Viewer viewer, Object e1, Object e2) {

			Coverage cov1 = (Coverage) e1;
			Coverage cov2 = (Coverage) e2;

			String name1 = null;
			String name2 = null;

			switch (_pos) {
				case 0 : //item names
					if (cov1.getTraceObj() instanceof TRCPackage) {
						if (((TRCPackage) cov1.getTraceObj())
							.getName()
							.equals(""))
							name1 = TraceUIPlugin.getString("DEFAULT_PACKAGE");
						else
							name1 = ((TRCPackage) cov1.getTraceObj()).getName();
					} else if (cov1.getTraceObj() instanceof TRCClass)
						name1 = ((TRCClass) cov1.getTraceObj()).getName();
					else if (cov1.getTraceObj() instanceof TRCMethod)
						name1 = ((TRCMethod) cov1.getTraceObj()).getName();

					if (cov2.getTraceObj() instanceof TRCPackage) {
						if (((TRCPackage) cov2.getTraceObj())
							.getName()
							.equals(""))
							name2 = TraceUIPlugin.getString("DEFAULT_PACKAGE");
						else
							name2 = ((TRCPackage) cov2.getTraceObj()).getName();
					} else if (cov2.getTraceObj() instanceof TRCClass)
						name2 = ((TRCClass) cov2.getTraceObj()).getName();
					else if (cov2.getTraceObj() instanceof TRCMethod)
						name2 = ((TRCMethod) cov2.getTraceObj()).getName();

					return (int) _sortSequence
						* (int) name1.compareToIgnoreCase(name2);

				case 1 : //Class names

					if ((cov1.getTraceObj() instanceof TRCMethod)
						&& (cov2.getTraceObj() instanceof TRCMethod)) {
						TRCMethod m1 = (TRCMethod) cov1.getTraceObj();
						TRCMethod m2 = (TRCMethod) cov2.getTraceObj();

						return (int) _sortSequence
							* (int) (m1
								.getDefiningClass()
								.getName()
								.compareToIgnoreCase(
									m2.getDefiningClass().getName()));
					} else
						return 0;

				case 2 : //Package
					if ((cov1.getTraceObj() instanceof TRCMethod)
						&& (cov2.getTraceObj() instanceof TRCMethod)
						&& (_level == 3)) {
						TRCMethod m1 = (TRCMethod) cov1.getTraceObj();
						TRCMethod m2 = (TRCMethod) cov2.getTraceObj();

						if ((m1.getDefiningClass().getPackage().getName())
							.equals(""))
							name1 = TraceUIPlugin.getString("DEFAULT_PACKAGE");
						else
							name1 =
								m1.getDefiningClass().getPackage().getName();

						if ((m2.getDefiningClass().getPackage().getName())
							.equals(""))
							name2 = TraceUIPlugin.getString("DEFAULT_PACKAGE");
						else
							name2 =
								m2.getDefiningClass().getPackage().getName();

						return (int) _sortSequence
							* (int) (name1.compareToIgnoreCase(name2));

					} else if (
						(cov1.getTraceObj() instanceof TRCClass)
							&& (cov2.getTraceObj() instanceof TRCClass)) {
						if ((((TRCClass) cov1.getTraceObj())
							.getPackage()
							.getName())
							.equals(""))
							name1 = TraceUIPlugin.getString("DEFAULT_PACKAGE");
						else
							name1 =
								((TRCClass) cov1.getTraceObj())
									.getPackage()
									.getName();

						if ((((TRCClass) cov2.getTraceObj())
							.getPackage()
							.getName())
							.equals(""))
							name2 = TraceUIPlugin.getString("DEFAULT_PACKAGE");
						else
							name2 =
								((TRCClass) cov2.getTraceObj())
									.getPackage()
									.getName();

						return (int) _sortSequence
							* (int) (name1.compareToIgnoreCase(name2));
					} else
						return 0;

				case 3 : //Calls
					return (int) _sortSequence
						* (cov1.getCalls() - cov2.getCalls());

				case 4 : //Methods missed
					return (int) _sortSequence
						* (cov1.getMethodsMissed() - cov2.getMethodsMissed());

				case 5 : //Methods hit
					return (int) _sortSequence
						* (cov1.getMethodsHit() - cov2.getMethodsHit());

				case 6 : //% methods hit
					return (int) _sortSequence
						* (int) ((cov1.getPercentage() - cov2.getPercentage()) * 10000);

			}
			//	 if column extended
			if (_listOfColumExtension.size() != 0) {
				ColumnExtensionValue elt =
					(ColumnExtensionValue) _listOfColumExtension.get(_pos-_numberOfColumns);
				return (int) _sortSequence * ((IColumnExtension)
							(elt.getClassOfColumnExtension())).compare(
								elt.getInitialColumnPos(), e1, e2);
			}

			return 0;
		}
	}

	/**
	 * implements a filter
	 *
	 */
	public class CoverageStatisticFilter extends StatisticFilter {

		public CoverageStatisticFilter() {
			super();
		}

		/**
		 * filter the elements according to the pattern
		 */
		public boolean select(Viewer viewer, Object parent, Object element) {

			Coverage cov = (Coverage) element;
			boolean flag = true;
			String compareText = "";

			switch (_level) {
				case 1 : //Package level
					if (!(cov.getTraceObj() instanceof TRCPackage))
						return true;
					if (((TRCPackage) cov.getTraceObj()).getName().equals(""))
						compareText =
							TraceUIPlugin.getString("DEFAULT_PACKAGE");
					else
						compareText =
							((TRCPackage) cov.getTraceObj()).getName();
					break;
				case 2 : //class level
					if (!(cov.getTraceObj() instanceof TRCClass))
						return true;
					compareText = ((TRCClass) cov.getTraceObj()).getName();
					break;

				case 3 : //method level
					if (!(cov.getTraceObj() instanceof TRCMethod))
						return true;
					compareText = ((TRCMethod) cov.getTraceObj()).getName();
					break;

			}

			if (_noPattern)
				return true;

			if (!_caseSensitive) {
				compareText = compareText.toLowerCase();
			}
			if (_exactMatch) {
				return compareText.compareTo(_prefix) == 0;
			}
			if (_prefix != "") {
				flag = compareText.startsWith(_prefix);
			}
			if (flag && _suffix != "") {
				flag = compareText.endsWith(_suffix);
			}
			if (flag) {
				for (int k = 0; k < _textList.size(); k++) {

					String str1 = (String) _textList.get(k);

					int index = compareText.lastIndexOf(str1);
					if (index == -1) {
						flag = false;
						break;
					}

					compareText = compareText.substring(index + str1.length());

				}
			}
			return flag;
		}

	}

	/**
	* Gives the table tree viewer the coverage elements to display 
	*/
	public class CoverageContentProvider implements ITreeContentProvider {

		public void dispose() {
		}

		public Object getParent(Object element) {

			return null;
		}

		public void inputChanged(
			Viewer viewer,
			Object oldInput,
			Object newInput) {
		}

		/**
		 * return the list of elements to display in the table tree
		 */
		public Object[] getElements(Object inputElement) {

			ArrayList tmpList = new ArrayList();
			tmpList.clear();

			CoverageAnalysis covAnalysis =
				((CoveragePage) _page).getCovAnalysis();

			if (covAnalysis.getRoot() != null) {
				switch (_level) {
					case 1 : // Package level
						tmpList.add(covAnalysis.getRoot());
						return tmpList.toArray();

					case 2 : // Class level
						tmpList.add(covAnalysis.getClassLevel());
						return tmpList.toArray();

					case 3 : // Method level
						tmpList.add(covAnalysis.getMethodLevel());
						return tmpList.toArray();

					default :
						return tmpList.toArray();
				}

			}
			/* When we "relaunch" a process the active page have no content until we make
				a "refreh views" */
			else { //No elements to display
				return tmpList.toArray();
			}

		}

		/**
		 * return the children of the coverage element
		 */
		public Object[] getChildren(Object element) {
			ArrayList childs = ((Coverage)element).getChilds();
			if (childs == null) {
				return null;
			}
			return childs.toArray();
		}

		/**
		 * test if the coverage element has children
		 */
		public boolean hasChildren(Object element) {
			return (getChildren(element) != null);
		}
	}

	/**
	* provides the table tree viewer: text and images
	*  for each coverage element to display
	*/
	public class CoverageLabelProvider
		extends LabelProvider
		implements ITableLabelProvider {
		protected StatisticView _viewer;

		/**
		 * 
		 * @param viewer
		 */
		public CoverageLabelProvider(StatisticView viewer) {
			super();
			_viewer = viewer;
		}

		/**
		 *  return for the coverage element to display the associated image 
		 */
		public Image getColumnImage(Object obj, int col) {

			StatisticTableColumnInfo info =
				StatisticTableColumnInfo.getStatisticTableColumnInfo(
					_viewer.getTable().getColumn(col));
			int pos = info.getColumnData().getInitalPos();
			Coverage cov = (Coverage) obj;

			switch (pos) {

				case 1 : //Class names column
					if (cov.getTraceObj() instanceof TRCMethod)
						return TracePluginImages.getImage(
							TracePluginImages.IMG_CLASS);
					else
						return null;

				case 2 : // Package column

					if (cov.getTraceObj() instanceof TRCMethod) {
						if (_level == 1 || _level == 2)
							return null;
						else
							return TracePluginImages.getImage(
								TracePluginImages.IMG_PACKAGE);
					} else if (cov.getTraceObj() instanceof TRCClass)
						return TracePluginImages.getImage(
							TracePluginImages.IMG_PACKAGE);
					else
						return null;

			}
			return null;
		}

		/**
		 * return for the coverage element to display the corresponding text
		 */
		public String getColumnText(Object obj, int col) {

			StatisticTableColumnInfo info =
				StatisticTableColumnInfo.getStatisticTableColumnInfo(
					_viewer.getTable().getColumn(col));
			int pos = info.getColumnData().getInitalPos();
			Coverage cov = (Coverage) obj;

			switch (pos) {
				case 0 : // packages / classes and methods names
					if (cov.getTraceObj() instanceof TRCPackage) {
						if (((TRCPackage) cov.getTraceObj())
							.getName()
							.equals(""))
							return TraceUIPlugin.getString("DEFAULT_PACKAGE");
						else
							return ((TRCPackage) cov.getTraceObj()).getName();
					} else if (cov.getTraceObj() instanceof TRCClass) {
						return ((TRCClass) cov.getTraceObj()).getName();
					}
					if (cov.getTraceObj() instanceof TRCMethod) {
						TRCMethod m = (TRCMethod) cov.getTraceObj();
						return m.getName() + m.getSignature();
					} else {
						return TraceUIPlugin.getString("ST_SUMMARY");
					}

				case 1 : // class names
					if (cov.getTraceObj() instanceof TRCMethod)
						return ((TRCMethod) cov.getTraceObj())
							.getDefiningClass()
							.getName();

					else
						return "";

				case 2 : // package
					if (cov.getTraceObj() instanceof TRCMethod) {
						if (_level == 1 || _level == 2)
							return "";
						else {
							if ((((TRCMethod) cov.getTraceObj())
								.getDefiningClass()
								.getPackage()
								.getName())
								.equals(""))
								return TraceUIPlugin.getString(
									"DEFAULT_PACKAGE");
							else
								return ((TRCMethod) cov.getTraceObj())
									.getDefiningClass()
									.getPackage()
									.getName();
						}
					} else if (cov.getTraceObj() instanceof TRCClass) {
						if ((((TRCClass) cov.getTraceObj())
							.getPackage()
							.getName())
							.equals(""))
							return TraceUIPlugin.getString("DEFAULT_PACKAGE");
						else
							return ((TRCClass) cov.getTraceObj())
								.getPackage()
								.getName();
					} else
						return "";

				case 3 : //Calls
					return String.valueOf(cov.getCalls());

				case 4 : //Method missed
					if (cov.getTraceObj() instanceof TRCMethod) {
						if (cov.getCalls() == 0)
							return TraceUIPlugin.getString("ST_MISSED");

						else
							return "";
					} else {
						return String.valueOf(cov.getMethodsMissed());
					}

				case 5 : //method hits
					if (cov.getTraceObj() instanceof TRCMethod) {
						if (cov.getCalls() != 0)
							return TraceUIPlugin.getString("ST_HIT");

						else
							return "";
					} else {
						return String.valueOf(cov.getMethodsHit());
					}

				case 6 : //%Method hit
					if (cov.getTraceObj() instanceof TRCMethod)
						return "";
					else
						return TString.formatAsPercentage(
								cov.getPercentage());

			}
			//if column extended
			if (_listOfColumExtension.size() != 0) {
				ColumnExtensionValue elt =
					(ColumnExtensionValue) _listOfColumExtension.get(
						pos - _numberOfColumns);
				return (
					(IColumnExtension)
						(elt.getClassOfColumnExtension())).getColumnText(
					obj,
					elt.getInitialColumnPos());
			}

			return "";
		}

	}

	/**
	 * Implements a table tree viewer	
	*/
	class CoverageTreeViewer extends TableTreeViewer {
		public CoverageTreeViewer(Composite parent) {
			super(parent);
		}

		public CoverageTreeViewer(TableTree table) {
			super(table);
		}

		public void expandItem(TableTreeItem item) {

			item.setExpanded(true);
			createChildren(item);

		}

	}

	/**
	 * 
	 * @param parent
	 * @param page
	 */
	public CoverageStatistic(Composite parent, TraceViewerPage page) {
		super(parent, page);
		_level = _saved_level;
		_viewerFilter = new CoverageStatisticFilter();
	}

	public String getColumnsPreferencesKey() {
		// Note this string is not to be translated and must be changed whenever the 
		// default columns template is changed by adding, removing or renaming columns.
		// Changing the version will result in the default preferences being used.
		return TraceUIPlugin.getString("PREFERENCE_KEY");
	}

	/**
	 * Return the column template for each display level
	 * TODO: Don't forget column extensions...
	 */
	public String getDefaultColumnsTemplate() {

		_level = _saved_level;
		String columns = "";

		if (_level == 0
			|| _level == 1) {
			//In the package level the class names column and the package column are hidden				
			columns =
				TraceUIPlugin.getString("STR_COL_ITEM_NAMES")
					+ ":0:"
					+ String.valueOf(
						ColumnData.NONDELETABLE
							| ColumnData.IS_VISIBLE
							| ColumnData.NONMOVABLE)
					+ ":221,"
					+ TraceUIPlugin.getString("STR_COL_CLASS_NAME")
					+ ":1:0"
					+ ":100,"
					+ TraceUIPlugin.getString("STR_COL_PACKAGE")
					+ ":2:0"
					+ ":100,"
					+ TraceUIPlugin.getString("STR_COL_CALLS")
					+ ":3:"
					+ String.valueOf(ColumnData.IS_VISIBLE)
					+ ":right:120,"
					+ TraceUIPlugin.getString("STR_COL_METHOD_MISSED")
					+ ":4:"
					+ String.valueOf(ColumnData.IS_VISIBLE)
					+ ":right:120,"
					+ TraceUIPlugin.getString("STR_COL_METHOD_HIT")
					+ ":5:"
					+ String.valueOf(ColumnData.IS_VISIBLE)
					+ ":right:120,"
					+ TraceUIPlugin.getString("STR_COL_RATE")
					+ ":6:"
					+ String.valueOf(ColumnData.IS_VISIBLE)
					+ ":right:120";

		} else if (
			_level == 2) {
			//In the class level the class names column is hidden
			columns =
				TraceUIPlugin.getString("STR_COL_ITEM_NAMES")
					+ ":0:"
					+ String.valueOf(
						ColumnData.NONDELETABLE
							| ColumnData.IS_VISIBLE
							| ColumnData.NONMOVABLE)
					+ ":200,"
					+ TraceUIPlugin.getString("STR_COL_CLASS_NAME")
					+ ":1:0"
					+ ":100,"
					+ TraceUIPlugin.getString("STR_COL_PACKAGE")
					+ ":2:"
					+ String.valueOf(ColumnData.IS_VISIBLE)
					+ ":150,"
					+ TraceUIPlugin.getString("STR_COL_CALLS")
					+ ":3:"
					+ String.valueOf(ColumnData.IS_VISIBLE)
					+ ":right:60,"
					+ TraceUIPlugin.getString("STR_COL_METHOD_MISSED")
					+ ":4:"
					+ String.valueOf(ColumnData.IS_VISIBLE)
					+ ":right:100,"
					+ TraceUIPlugin.getString("STR_COL_METHOD_HIT")
					+ ":5:"
					+ String.valueOf(ColumnData.IS_VISIBLE)
					+ ":right:100,"
					+ TraceUIPlugin.getString("STR_COL_RATE")
					+ ":6:"
					+ String.valueOf(ColumnData.IS_VISIBLE)
					+ ":right:100";

		} else if (
			_level == 3) { //In the method level all the columns are visible 
			columns =
				TraceUIPlugin.getString("STR_COL_ITEM_NAMES")
					+ ":0:"
					+ String.valueOf(
						ColumnData.NONDELETABLE
							| ColumnData.IS_VISIBLE
							| ColumnData.NONMOVABLE)
					+ ":200,"
					+ TraceUIPlugin.getString("STR_COL_CLASS_NAME")
					+ ":1:"
					+ String.valueOf(ColumnData.IS_VISIBLE)
					+ ":100,"
					+ TraceUIPlugin.getString("STR_COL_PACKAGE")
					+ ":2:"
					+ String.valueOf(ColumnData.IS_VISIBLE)
					+ ":100,"
					+ TraceUIPlugin.getString("STR_COL_CALLS")
					+ ":3:"
					+ String.valueOf(ColumnData.IS_VISIBLE)
					+ ":right:45,"
					+ TraceUIPlugin.getString("STR_COL_METHOD_MISSED")
					+ ":4:"
					+ String.valueOf(ColumnData.IS_VISIBLE)
					+ ":right:90,"
					+ TraceUIPlugin.getString("STR_COL_METHOD_HIT")
					+ ":5:"
					+ String.valueOf(ColumnData.IS_VISIBLE)
					+ ":right:90,"
					+ TraceUIPlugin.getString("STR_COL_RATE")
					+ ":6:"
					+ String.valueOf(ColumnData.IS_VISIBLE)
					+ ":right:90";

		}
		
		return columns + getColumnsTemplateExtensions();

	}

	/**
	 * Create the table tree widget
	 */
	protected Composite createTable(Composite parent, int options) {
		return new TableTree(parent, options);
	}

	/**
	 * Create the table tree viewer
	 */
	protected StructuredViewer createTableViewer(Composite table) {
		CoverageTreeViewer tv = new CoverageTreeViewer((TableTree) table);
		return (StructuredViewer) tv;
	}

	/**
	 * return the table widget
	 */
	public Table getTable() {
		return ((TableTree) getTableViewer().getControl()).getTable();

	}

	/**
	 * Return the Label Provider of the table tree viewer
	 */
	public LabelProvider getTableLabelProvider() {
		return new CoverageLabelProvider(this);
	}

	/**
	 * Update the content of the table tree viewer
	 */
	public void update() {

		if (_firstTime) {

			// Add the filter
			getTableViewer().addFilter(getViewerFilter());
			_firstTime = false;

			//Add the Sorter
			Table table = getTable();
			TableColumn firstColumn = table.getColumn(0);

			_viewerSorter = new CoverageStatisticSorter();

			getViewerSorter().setSortedColumn(firstColumn);
			getTableViewer().setSorter(getViewerSorter());

		}
		// Activate the table tree viewer			
		getTableViewer().setInput(_page.getMOFObject());

		getTable().setRedraw(false);
		getTableViewer().refresh();
		expandFirstElement();
		getTable().setRedraw(true);

		handleSelectionEvent();

	}

	/**
	 * Try to see if the selected object is candidate for openSource
	 */
	protected boolean canOpenSourceOnThisObject() {
		ITraceSelection _model =
			UIPlugin.getDefault().getSelectionModel(_page.getMOFObject());
		if (_model.size() == 0 || _model.getFirstElement() == null) {
			return false;
		}
		Object firstElement = _model.getFirstElement();
		if (firstElement instanceof TRCClass &&
			((TRCClass)firstElement).getSourceInfo() != null &&
			((TRCClass)firstElement).getSourceInfo().getLocation().length() > 0) {
			return true;
		} else if (firstElement instanceof TRCMethod) {
			return true;
		}
		return false;
	}

	/**
	 * Update the action buttons
	 */
	public void updateButtons() {

		switch (_level) {
			case 1 :
				((CoveragePage) getTraceViewerPage())
					.getPackageLevel()
					.setChecked(
					true);
				((CoveragePage) getTraceViewerPage())
					.getClassLevel()
					.setChecked(
					false);
				((CoveragePage) getTraceViewerPage())
					.getMethodLevel()
					.setChecked(
					false);
				break;
			case 2 :
				((CoveragePage) getTraceViewerPage())
					.getPackageLevel()
					.setChecked(
					false);
				((CoveragePage) getTraceViewerPage())
					.getClassLevel()
					.setChecked(
					true);
				((CoveragePage) getTraceViewerPage())
					.getMethodLevel()
					.setChecked(
					false);
				break;
			case 3 :
				((CoveragePage) getTraceViewerPage())
					.getPackageLevel()
					.setChecked(
					false);
				((CoveragePage) getTraceViewerPage())
					.getClassLevel()
					.setChecked(
					false);
				((CoveragePage) getTraceViewerPage())
					.getMethodLevel()
					.setChecked(
					true);
				break;
		}
		
		((CoveragePage) getTraceViewerPage()).openSource().setEnabled(canOpenSourceOnThisObject());
	}
	
	/**
	 * The same element must be selected in all the statistics views
	 */
	protected void handleSelectionEvent() {
		ITraceSelection model =
			UIPlugin.getDefault().getSelectionModel(_page.getMOFObject());
		if (model.size() > 0) {
			Object sel = model.getFirstElement();
			if (sel != null) {
				select(sel);
			}
		}
	}

	/**
	 * select the element (in the coverage table) which is selected in the other 
	 * statistics views
	 */
	private void select(Object obj) {
		TRCClass cls = null;
		TRCPackage pack = null;
		TRCMethod meth = null;
		String parentClsName = null;

		TableTree table = (TableTree) getTableViewer().getControl();

		//Summary element
		TableTreeItem[] items = table.getItems();
		if (items.length != 0) // if we have data
			{
			((CoverageTreeViewer) getTableViewer()).expandItem(items[0]);

			if (obj instanceof TRCPackage)
				pack = (TRCPackage) obj;
			else if (obj instanceof TRCClass) {
				cls = (TRCClass) obj;
				pack = cls.getPackage();
			} else if (obj instanceof TRCMethod) {
				meth = (TRCMethod) obj;
				cls = meth.getDefiningClass();
				pack = cls.getPackage();
			}

			switch (_level) {
				case 1 :
					searchInPackageLevel(
						table,
						items[0],
						pack,
						cls,
						parentClsName,
						meth);
					break;
				case 2 :
					if (cls != null)
						searchInClassLevel(
							table,
							items[0],
							cls,
							parentClsName,
							meth);
					break;
				case 3 :
					if (meth != null)
						searchInMethodLevel(table, items[0], meth);
					break;
			}
		}
	}

	/**
	 * search the element to select in the package level table
	 *
	 */
	private void searchInPackageLevel(
		TableTree table,
		TableTreeItem item,
		TRCPackage pack,
		TRCClass cls,
		String parentClsName,
		TRCMethod meth) {
		TableTreeItem pacItem = null;

		TableTreeItem[] pacItems = item.getItems();

		pacItem = searchItem(pacItems, pack);

		if (pacItem != null) {
			if (cls == null) { //Package selected
				table.setSelection(new TableTreeItem[] { pacItem });
				return;
			}
			((CoverageTreeViewer) getTableViewer()).expandItem(pacItem);

			//TRCClass elements
			searchInClassLevel(
				table,
				pacItem,
				cls,
				parentClsName,
				meth);
			return;
		}
		table.deselectAll();
	}

	/**
	 * search the element to select in the class level table
	 *
	 */
	private void searchInClassLevel(
		TableTree table,
		TableTreeItem item,
		TRCClass cls,
		String parentClsName,
		TRCMethod meth) {
		TableTreeItem[] classItems = item.getItems();
		TableTreeItem classItemBis = null;

		classItemBis = searchItem(classItems, cls);

		if (classItemBis != null) {
			if (meth == null) { // TRCClass selected
				table.setSelection(new TableTreeItem[] { classItemBis });
				return;
			}
			((CoverageTreeViewer) getTableViewer()).expandItem(classItemBis);
			searchInMethodLevel(table, classItemBis, meth);
			return;

		}
		table.deselectAll();

	}

	/**
	 * search the element to select in the method level table
	 *
	 */
	private void searchInMethodLevel(
		TableTree table,
		TableTreeItem item,
		TRCMethod meth) {
		TableTreeItem[] methodItems = item.getItems();

		TableTreeItem methodItem = searchItem(methodItems, meth);

		if (methodItem != null) {
			table.setSelection(new TableTreeItem[] { methodItem });
			return;
		}

		table.deselectAll();
	}

	/**
	 * search in the list of items the corresponding object
	 * 
	 */
	private TableTreeItem searchItem(TableTreeItem[] items, Object obj) {

		for (int j = 0; j < items.length; j++) {
			TableTreeItem item = items[j];
			Coverage cov = (Coverage) item.getData();

			if (cov.getTraceObjects().contains(obj))
				return item;
		}
		return null;
	}

	/**
	 * search in the list of items the corresponding string
	 */
//	private TableTreeItem searchItemByName(
//		TableTreeItem[] items,
//		String itemName) {
//
//		for (int j = 0; j < items.length; j++) {
//			TableTreeItem item = items[j];
//
//			if ((item.getText()).equals(itemName))
//				return item;
//		}
//		return null;
//	}

	/**
	 * Return the content provider of the table tree viewer
	 */
	public IContentProvider getContentProvider() {

		return new CoverageContentProvider();

	}

	/**
	 * Update the "model selection" when an element is selected in the table
	 */
	public void updateModelSelection() {

		ISelection selection = getTableViewer().getSelection();
		if (selection != null && !selection.isEmpty()) {
			Object sel = ((IStructuredSelection) selection).getFirstElement();
			EObject obj = ((Coverage)sel).getTraceObj();
			UIPlugin.getDefault().getSelectionModel(_page.getMOFObject()).add(obj);
			ViewSelectionChangedEvent event =
				UIPlugin.getDefault().getViewSelectionChangedEvent();
			event.setSource(_page.getMOFObject());
			UIPlugin.getDefault().notifyViewSelectionChangedListener(event);
		}
	}

	/**
	 * if the selection event is sent by the active table: discard the event
	 * else handle it
	 */
	public void handleViewSelectionChangedEvent(ViewSelectionChangedEvent event) {
		if (!getTraceViewerPage().getTraceViewer().isProcessRefreshEvents()
			|| getTable().isFocusControl()) 
			return;

		handleSelectionEvent();

	}

	/**
	 * expand the fisrt element of the table which is "Summary"
	 */
	protected void expandFirstElement() {
		TableTree table = ((TableTreeViewer) getTableViewer()).getTableTree();
		TableTreeItem[] items = table.getItems();

		if (items.length != 0)
			 ((CoverageTreeViewer) getTableViewer()).expandItem(items[0]);

	}

	/**
	 * Called when the context menu is about to open.
	 * @see IFillMenuTarget#fillContextMenu
	 */
	public void menuAboutToShow(IMenuManager menu) {

		menu.add(getUpdateAction());
		menu.add(fSeparator);
		resetChooseColumnsAction();
		menu.add(
			getChooseColumnsAction(
				getColumnDataList(),
				getColumnsPreferencesKey()));
		if (canOpenSourceOnThisObject()) {
			menu.add(((CoverageViewer)getTraceViewerPage().getTraceViewer()).openSource());
		}
	}

	/**
	 * @return the level of information to display (package/class or method)
	 */
	public int getLevel() {
		return _level;
	}

	/**
	 * @param i:the level of information to display (package/class or method)
	 */
	public void setLevel(int i) {
		_level = i;
		_saved_level = _level;
	}

	/**
	 * Redraw the table to display the three levels
	 * (package level, class level and method level)
	 * 
	 */
	public void redrawTable() {
		getTable().setRedraw(false);
		String template = getDefaultColumnsTemplate();

		ArrayList list = ColumnData.createColumnDataFromString(template);
		resetColumns(list);
		_currentColumns = list;
		getTable().setRedraw(true);
	}

	/**
	 * handle the selection event on:
	 * - A column for sorting
	 * - An element of the table tree or
	 * - on the "Case-sensitive" button
	 */
	public void widgetSelected(org.eclipse.swt.events.SelectionEvent arg0) {
		super.widgetSelected(arg0);
		if (arg0.widget instanceof TableColumn) {
			expandFirstElement();
			handleSelectionEvent();
		}
	}

	/**
	 * Filter the elements
	 */
	protected void filterNames() {
		super.filterNames();
		expandFirstElement();
		handleSelectionEvent();
	}

	/**
	 * dispose the active columns and create others
	 */
	protected void resetColumns(ArrayList list) {
		super.resetColumns(list);
		expandFirstElement();
		handleSelectionEvent();
	}

	/**
	 * reset the choose column Action
	 *
	 */
	protected void resetChooseColumnsAction() {
		_chooseColumnAction = null;
	}

	/**
	 * Return the column template for each display level
	 */
	public String getColumnsTemplateExtensions()
	{
		String columnName = "";
		String pos = "";
		String style = "";
		String width = "";
		String align = "";
		int colNumber = _numberOfColumns;
		
		_templateOfColumExtension = "";
		_listOfColumExtension.clear();

		IExtensionPoint extPoint =
		Platform.getPluginRegistry().getExtensionPoint(
		"org.eclipse.hyades.trace.views.columnExtension");

		if (extPoint != null) {
			IConfigurationElement[] members =
			extPoint.getConfigurationElements();

			if (members.length != 0) {
				//For each column
				for (int i = 0; i < members.length; i++) {
					columnName =
					members[i].getAttribute(
							TraceUIPlugin.getString("STR_COL_NAME"));
					pos =
					members[i].getAttribute(
							TraceUIPlugin.getString("STR_COL_POS"));
					style =
					members[i].getAttribute(
							TraceUIPlugin.getString("STR_COL_STYLE"));
					width =
					members[i].getAttribute(
							TraceUIPlugin.getString("STR_COL_WIDTH"));
					align =
					members[i].getAttribute(
							TraceUIPlugin.getString("STR_COL_ALIGN"));
					_templateOfColumExtension +=
					","
					+ columnName
					+ ":"
					+ colNumber
					+ ":"
					+ style
					+ ":"
					+ align
					+ ":"
					+ width;
					colNumber++;

					try {
						ColumnExtensionValue columnVal = new ColumnExtensionValue();
						columnVal.setClassOfColumnExtension(
							(IColumnExtension) members[i].createExecutableExtension("class"));
						columnVal.setInitialColumnPos((new Integer(pos)).intValue());
						_listOfColumExtension.add(columnVal);
					} catch (CoreException e) {
						System.err.println(e);
					}
				}
			}
		}
		return _templateOfColumExtension;
	}
	
	public boolean isEmpty()
	{
		CoverageAnalysis covAnalysis =
			((CoveragePage) _page).getCovAnalysis();

		return (covAnalysis.getRoot() == null ||
				covAnalysis.getRoot().getMethods() == 0);
	}

	public void dispose() {
		super.dispose();
		_saved_level = _level;
	}

}
