/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.test.manual.runner.model.util;

import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;

import org.eclipse.hyades.test.common.event.TypedEvent;
import org.eclipse.hyades.test.manual.runner.model.Action;
import org.eclipse.hyades.test.manual.runner.model.IActionOwner;
import org.eclipse.hyades.test.manual.runner.model.IVerdictListener;
import org.eclipse.hyades.test.manual.runner.model.Loop;
import org.eclipse.hyades.test.manual.runner.model.NamedElement;
import org.eclipse.hyades.test.manual.runner.model.Test;
import org.eclipse.hyades.test.manual.runner.model.TestCase;
import org.eclipse.hyades.test.manual.runner.model.TestInvocation;
import org.eclipse.hyades.test.manual.runner.model.TestSuite;

/**
 * @author marcelop
 * @since 1.0.2
 */
public class ExecutionManager
implements IVerdictListener
{
	private TestInvocation lastExecuted;
	private Set testInvocations;
	private Set executed;
	private Set startedLoopHierarchyIds;
	private int currentExecutionId;

	private TestSuite rootTestSuite;
	
	public void dispose()
	{		
		ModelUtil.dispose(rootTestSuite);
		
		lastExecuted = null;
		rootTestSuite = null;		
		if(testInvocations != null)
			testInvocations.clear();
		if(executed != null)
			executed.clear();
		if(startedLoopHierarchyIds != null)
			startedLoopHierarchyIds.clear();		
	}
	
	/**
	 * @see org.eclipse.hyades.test.manual.runner.model.IVerdictListener#handleVerdict(org.eclipse.hyades.test.manual.runner.model.TestInvocation)
	 */
	public void handleVerdict(TestInvocation testInvocation)
	{
		lastExecuted = testInvocation;
		if(testInvocation.getVerdictEvent() == null)
		{
			if(executed != null)
				executed.remove(testInvocation);
		}
		else
		{
			if(executed == null)
				executed = new HashSet();
			executed.add(testInvocation);
		}
	}

	/**
	 * Returns the last element that had its verdict set.
	 * @return NamedElement
	 */
	public TestInvocation getLast()
	{
		return lastExecuted;
	}	
	
	public void setRoot(TestSuite testSuite)
	{
		rootTestSuite = testSuite;
	}
	
	public TestSuite getRoot()
	{
		return rootTestSuite;
	}
	
	public void registerTestInvocations(Collection tis)
	{
		if(tis != null)
			testInvocations = new HashSet(tis);
	}

	public int getTestInvocationCount()
	{
		if(testInvocations == null)
			return 0;		
		return testInvocations.size();
	}
	
	public Collection getExecutedTestInvocations()
	{
		if(executed == null)
			executed = new HashSet();
		return executed;
	}
	
	public int getExecutionCount()
	{
		if(executed == null)
			return 0;	
		return executed.size();		
	}
	
	public void startLoop(Loop loop)
	{
		if(loop == null)
			return;
		
		String loopId = ModelUtil.getHierarchyId(loop);
		if(startedLoopHierarchyIds == null)
			startedLoopHierarchyIds = new HashSet();
		else if(startedLoopHierarchyIds.contains(loopId))
			return;
		
		startedLoopHierarchyIds.add(loopId);
		ModelUtil.getEventLogger().logTyped(loop, TypedEvent.START, null);
	}
	
	public void closeStatedLoops()
	{
		if(startedLoopHierarchyIds != null)
		{
			for (Iterator i = startedLoopHierarchyIds.iterator(); i.hasNext();)
			{
				String loopId = (String)i.next();
				TypedEvent event = new TypedEvent();
				event.setOwnerId(loopId);
				event.setType(TypedEvent.STOP);
				ModelUtil.getEventLogger().log(event);
			}
		}
	}
	
	/**
	 * Returns the next named element to be executed or {@link #getRoot()}
	 * if all elements were already executed.
	 * @return NamedElement
	 */
	public NamedElement next()
	{
		if(hasNext())
		{
			NamedElement next = getNext(getRoot());
			if(next != null)
			{
				currentExecutionId = next.getExecutionId();
				return next;
			}
		}
		
		currentExecutionId = Integer.MIN_VALUE;
		return getRoot();
	}
	
	private TestInvocation getNext(IActionOwner actionOwner)
	{
		if(actionOwner != null)
		{
			for (Iterator i = actionOwner.getActions().iterator(); i.hasNext();)
			{
				Action action = (Action)i.next();
				if(action instanceof Loop)
				{
					TestInvocation testInvocation = getNext((Loop)action);
					if(testInvocation != null)
						return testInvocation;
				}
				else if(action instanceof TestInvocation)
				{
					TestInvocation testInvocation =(TestInvocation)action;
					if(testInvocation.getTest() != null)
					{
						if(testInvocation.getTest() instanceof TestSuite)
						{
							TestInvocation ti = getNext((TestSuite)testInvocation.getTest());
							if(ti != null)
								return ti;
						}
						else if(testInvocation.getVerdictEvent() == null)
							return testInvocation;
					}
				}
			}
		}
		
		return null;
	}
	
	public boolean hasNext()
	{
		if(getTestInvocationCount() == getExecutionCount())
		{
			currentExecutionId = Integer.MAX_VALUE;
			return false;
		}
		return true;
	}
	
	public int getCurrentExecutionId()
	{
		return currentExecutionId;	
	}
	
	/**
	 * Returns whether the specified named element is "atomically" executabled, ie,
	 * if it is executable by it self.
	 * @param namedElement
	 * @return
	 */
	public boolean isExecutable(NamedElement namedElement)
	{
		if((namedElement != null) && (namedElement instanceof TestInvocation))
		{
			Test test = ((TestInvocation)namedElement).getTest();
			if((test != null) && (test instanceof TestCase))
				return true;
		}
		
		return false;
	}
}
