/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.execution.harness.util;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.util.ArrayList;
import java.util.HashMap;

import org.eclipse.hyades.execution.core.IExecutionComponentFactory;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;

/**
 * This class provides utility methods for standalone test execution. 
 * 
 * @author jtoomey
 *
 */
public class StandaloneExecutionUtilities {


	/**
	 * <p>This method parses all of the plugin.xml files specified in the 
	 * standaloneConfigurationFiles parameter searching for the following three
	 * extension points:</p>
	 * <ul><li>org.eclipse.hyades.execution.harness.RegisteredExecutionComponentImpl</li>
	 * <li>org.eclipse.hyades.execution.harness.ExecutionEnvironmentAdapter</li>
	 * <li>org.eclipse.hyades.execution.harness.ExecutableObjectAdapter</li></ul>
	 * 
	 * <p>For all found implementations of these extension points, the factory and
	 * testTypeMap are appropriately populated.</p>
	 *   
	 * @param standaloneConfigurationFiles
	 * @param testTypeMap
	 * @param factory
	 * @throws ClassNotFoundException
	 * @throws FileNotFoundException
	 */
	public static void initializeRegisteredExecutionComponents(
		String[] standaloneConfigurationFiles, HashMap testTypeMap, 
		IExecutionComponentFactory factory) throws ClassNotFoundException, FileNotFoundException
	{
		for ( int i=0; i < standaloneConfigurationFiles.length; i++)
		{
			parseConfigFile(standaloneConfigurationFiles[i], testTypeMap, factory);
		}
	}
	
	private static void parseConfigFile( String configFile, HashMap testTypeMap, 
		IExecutionComponentFactory factory) throws ClassNotFoundException, FileNotFoundException
	{
		FileInputStream stream = new FileInputStream(configFile);
		Element plugin = XMLUtil.loadDom(stream, "plugin");
		
		NodeList nodeList = XMLUtil.getChildrenByName(plugin, "extension");
		if(nodeList != null)
		{
			for(int i=0; i<nodeList.getLength(); i++)
			{
				if(nodeList.item(i) instanceof Element)
				{
					Element extensionElement = (Element)nodeList.item(i);
					if(extensionElement != null)
					{
						if ( extensionElement.getAttribute("point").equals("org.eclipse.hyades.execution.harness.RegisteredExecutionComponentImpl"))
						{
							NodeList execComps = XMLUtil.getChildrenByName(extensionElement, "RegisteredExecutionComponentImpl");
							for ( int j=0; j<execComps.getLength(); j++ )
							{
								if ( execComps.item(j) instanceof Element )
									parseRegisteredExecutionComponents((Element)execComps.item(j), testTypeMap, factory);
							}
						}

						if ( extensionElement.getAttribute("point").equals("org.eclipse.hyades.execution.harness.ExecutionEnvironmentAdapter"))
						{
							NodeList envAdapters = XMLUtil.getChildrenByName(extensionElement, "ExecutionEnvironmentAdapter");
							for ( int j=0; j<envAdapters.getLength(); j++ )
							{
								if ( envAdapters.item(j) instanceof Element )
									parseAdapters((Element)envAdapters.item(j), testTypeMap, factory, "EXECUTION_ENVIRONMENT_ADAPTER");
							}
						}

						if ( extensionElement.getAttribute("point").equals("org.eclipse.hyades.execution.harness.ExecutableObjectAdapter"))
						{
							NodeList execObjAdapters = XMLUtil.getChildrenByName(extensionElement, "ExecutableObjectAdapter");
							for ( int j=0; j<execObjAdapters.getLength(); j++ )
							{
								if ( execObjAdapters.item(j) instanceof Element )
									parseAdapters((Element)execObjAdapters.item(j), testTypeMap, factory, "EXECUTABLE_OBJECT_ADAPTER");
							}
						}
					}
				}
			}
		}
		
	}

	private static void parseRegisteredExecutionComponents(Element extensionElement, HashMap testTypeMap, 
		IExecutionComponentFactory factory) throws ClassNotFoundException
	{
		String type = extensionElement.getAttribute("type");
		String executionComponentName = extensionElement.getAttribute("name");
		factory.addExecutionComponent(executionComponentName, extensionElement.getAttribute("implClass"));
		factory.addStub(executionComponentName, extensionElement.getAttribute("stubClass"));
		factory.addSkeleton(executionComponentName, extensionElement.getAttribute("skeletonClass"));
					
		Object map = testTypeMap.get(type);
		if ( map == null )
		{
			map = new HashMap();
			testTypeMap.put(type, map);
		}
		HashMap execCompMap = (HashMap)map;
		
		NodeList supportedTestTypes = XMLUtil.getChildrenByName(extensionElement, "SupportedTestType");
		for ( int j=0; j<supportedTestTypes.getLength(); j++ )
		{
			if (supportedTestTypes.item(j) instanceof Element )
			{
				Element supportedType = (Element) supportedTestTypes.item(j);
				execCompMap.put(supportedType.getAttribute("name"), executionComponentName);
			}
		}
	}


	private static void parseAdapters(Element adapterElement, HashMap testTypeMap, 
		IExecutionComponentFactory factory, String type) throws ClassNotFoundException
	{

		String adapterClass = adapterElement.getAttribute("class");
		
		Object map = testTypeMap.get(type);
		if ( map == null )
		{
			map = new HashMap();
			testTypeMap.put(type, map);
		}
		HashMap execCompMap = (HashMap)map;
			
		NodeList supportedTestTypes = XMLUtil.getChildrenByName(adapterElement, "SupportedTestType");
		for ( int j=0; j<supportedTestTypes.getLength(); j++ )
		{
			if (supportedTestTypes.item(j) instanceof Element )
			{
				Element supportedType = (Element) supportedTestTypes.item(j);
				execCompMap.put(supportedType.getAttribute("name"), adapterClass);
			}
		}
	}

	/**
	 * This method takes a directory which is structured as (and may actually be)
	 * an Eclipse directory.  This method will walk all subdirectories in the plugins subdir of
	 * the specified configDir, and will return an array containing the fully 
	 * qualified paths to any plugin.xml files found in those subdirectories.  If
	 * no plugin.xml files are found, or if the configDir does not exist, this method
	 * will return null. 
	 * 
	 * @param configDir
	 * @return
	 */
	public static String[] getConfigFilesFromConfigDir(String configDir)
	{
		File configDirFile = new File(configDir);
		if ( !configDirFile.isDirectory() )
			return null;

		configDir = configDirFile.getAbsolutePath();
		configDir += File.separator + "plugins";
		configDirFile = new File(configDir);
		if ( !configDirFile.isDirectory() )
			return null;

		ArrayList pluginPaths = new ArrayList(); 
			 
		// Get the list of all elements of configDir's plugin directory
		String[] elements = configDirFile.list();
		String element;
		File elementFile;
		
		for ( int i=0; i<elements.length; i++ )
		{
			element = configDir + File.separator + elements[i];
			elementFile = new File(element);
			if ( !elementFile.isDirectory())
				continue;
			
			// Found a subdirectory -- check to see if it contains
			// a file called plugin.xml
			element = element + File.separator + "plugin.xml";
			elementFile = new File(element);
			if ( elementFile.exists() )
				pluginPaths.add(element);
		}
		
		return (String[])pluginPaths.toArray(new String[0]);
		
	}

	/**
	 * This method takes a directory which is structured as (and may actually be)
	 * an Eclipse plugins directory.  This method attempts to locate the 
	 * execution harness plugin directory (org.eclipse.hyades.execution.harness)
	 * within the specified directory.  If the execution harness plugin 
	 * directory can not be found in the specified directory, or if the 
	 * specified directory does not exist, this method will return null. 
	 * 
	 * @param configDir
	 * @return 
	 */
	public static String getHarnessDir(String configDir)
	{
		File configDirFile = new File(configDir);
		if ( !configDirFile.isDirectory() )
			return null;
			
		configDir = configDirFile.getAbsolutePath();
		
		String harnessDir = configDir + File.separator + "plugins" + File.separator + "org.eclipse.hyades.execution.harness";
		
		File harnessFile = new File(harnessDir);
		
		if ( harnessFile.isDirectory())
			return harnessDir;
		else
			return null;
		
	}
	
}

