/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v0.5
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v05.html
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.loaders.trace;

import org.eclipse.hyades.loaders.hierarchy.IgnoredXMLFragmentLoader;
import org.eclipse.hyades.loaders.util.HierarchyContext;
import org.eclipse.hyades.loaders.util.LoadersUtils;
import org.eclipse.hyades.loaders.util.LookupServiceExtensions;
import org.eclipse.hyades.models.hierarchy.TRCCollectionMode;
import org.eclipse.hyades.models.trace.TRCArrayClass;
import org.eclipse.hyades.models.trace.TRCClass;
import org.eclipse.hyades.models.trace.TRCFullTraceObject;
import org.eclipse.hyades.models.trace.TRCHeapDump;
import org.eclipse.hyades.models.trace.TRCHeapObject;
import org.eclipse.hyades.models.trace.TRCMethod;
import org.eclipse.hyades.models.trace.TRCObject;
import org.eclipse.hyades.models.trace.TRCPrimitiveType;
import org.eclipse.hyades.models.trace.TRCProcess;
import org.eclipse.hyades.models.trace.TRCThread;
import org.eclipse.hyades.models.trace.TraceFactory;
import org.eclipse.hyades.models.trace.impl.TRCArrayClassImpl;
import org.eclipse.hyades.models.trace.impl.TRCClassImpl;
import org.eclipse.hyades.models.trace.impl.TRCFullTraceObjectImpl;
import org.eclipse.hyades.models.trace.impl.TRCHeapObjectImpl;
import org.eclipse.hyades.models.trace.impl.TRCMethodImpl;
import org.eclipse.hyades.models.trace.impl.TRCThreadImpl;


/**
 * @author slavescu
 *
 */
public class TraceXMLFragmentLoader extends IgnoredXMLFragmentLoader {
    //~ Static fields/initializers -----------------------------------------------------------------

    protected static final String NAME = "name";
    protected static final String OBJ_ID_REF = "objIdRef";
    protected static final String METHOD_ID_REF = "methodIdRef";
    protected static final String CLASS_ID_REF = "classIdRef";
    protected static final String THREAD_ID_REF = "threadIdRef";
    protected static final String TIME = "time";
    protected static final String CLASS_CLASS1 = "java/lang/Class";
    protected static final String CLASS_CLASS2 = "java.lang.Class";
    protected static final String UNKNOWN = "unknown";

    //~ Instance fields ----------------------------------------------------------------------------

    protected TRCClass theClass;
    protected TRCHeapDump theHeapDump;
    protected TRCMethod theMethod;
    protected TRCObject theObject;
    protected TRCProcess theProcess;
    protected TRCThread theThread;
    protected VirtualObjectInfo virtualObject;
    protected boolean loadToModel;
    protected double time = 0;
//    protected double zeroTime = 0;
    protected int classIdRef;
    protected int deltaSize = 0;
    protected int methodIdRef;
    protected int threadIdRef = 0;
    protected long objIdRef;
    private TRCClass aClass;

    //~ Methods ------------------------------------------------------------------------------------

    public void addAttribute(String name, String value) {
        switch (LoadersUtils.getHashCode(name)) {
        case TraceConstants.TIME_int:
            time = Double.parseDouble(value);

            break;

        case TraceConstants.THREAD_ID_REF_int:
            threadIdRef = Integer.parseInt(value);

            break;

        case TraceConstants.CLASS_ID_REF_int:
            classIdRef = Integer.parseInt(value);

            break;

        case TraceConstants.METHOD_ID_REF_int:
            methodIdRef = Integer.parseInt(value);

            break;

        case TraceConstants.OBJ_ID_REF_int:
            objIdRef = Long.parseLong(value);

            break;

        default:
            break;
        }
    }

    public void initialize(HierarchyContext context, String name) {
        super.initialize(context, name);
        threadIdRef = 0;
        classIdRef = 0;
        methodIdRef = 0;
        objIdRef = 0;
        time = 0;
        theProcess = null;
        theThread = null;
        theClass = null;
        theMethod = null;
        theObject = null;
        virtualObject = null;
    }

    protected TRCClass getClassByIdRef(TRCThread thread, Class theObjectClass, int isArray) {
        Object key;
        TRCClass targetClass = null;
		TRCObject anObject=null;

        if (isArray > 0) {
			key = LoadersUtils.getLookUpKey(-classIdRef);
            theClass = (TRCClass) LookupServiceExtensions.getInstance().locate(context, TRCArrayClassImpl.class, key);

            if (theClass != null) {
                return theClass;
            }

			key = LoadersUtils.getLookUpKey(classIdRef);
            targetClass = (TRCClass) LookupServiceExtensions.getInstance().locate(context, TRCClassImpl.class, key);
            if(targetClass==null)
            {
            	// support for multi dimensional arrays
				targetClass = (TRCClass) LookupServiceExtensions.getInstance().locate(context, TRCArrayClassImpl.class, key);
            }
        } else {
			key = LoadersUtils.getLookUpKey(classIdRef);
            theClass = (TRCClass) LookupServiceExtensions.getInstance().locate(context, TRCClassImpl.class, key);

            if (theClass != null) {
                return theClass;
            }
        }

        //        TRCProcess p = thread.getProcess();
        if (isArray > 0) {
            theClass = TraceFactory.eINSTANCE.createTRCArrayClass();
			theClass.setId((int) -classIdRef);
            ((TRCArrayClass) theClass).setArrayType(TRCPrimitiveType.get(isArray));
        } else {
            theClass = TraceFactory.eINSTANCE.createTRCClass();
			theClass.setId((int) classIdRef);
        }

        if (targetClass != null) {
            theClass.setName("[" + targetClass.getName());
            theClass.setLoadTime(createDeltaTime());
            theClass.setLoadedBy(thread);
            theClass.setPackage(targetClass.getPackage());

			// now add a class object
			anObject = getClassObjectByIdRef(-classIdRef, theObjectClass);
			theClass.getClassObjects().add(anObject);
			anObject.setSize(theClass.getSize());
			
			//adds the class object to the Class class
			TraceUtils.addClassObjectToClassClass(context, theProcess, anObject);

            return theClass;
        }

        String name = UNKNOWN + classIdRef;
        String sourceName = name;

        theClass.setName(TraceUtils.className(name));
        theClass.setLoadTime(createDeltaTime());

        TraceUtils.addClassToPackage(context, theClass, name, theProcess);

        TraceUtils.setSourceInfo(context, theProcess, theClass, sourceName);

        if (theProcess.getClassClass() == null) // if class Class not set
         {
         	int h = LoadersUtils.getHashCode(name);
            if (h==TraceConstants.CLASS_CLASS1_int || h==TraceConstants.CLASS_CLASS2_int) //if class Class
             {
                theProcess.setClassClass(theClass);
            }

            //	else do nothing because a fake one will be created automaticaly
        }

        //        /* Look for the class object in the model if we have been given an object Id*/
        //        theObject = (TRCObjectEntry) LookupServiceExtensions.getInstance().locate(context, TRCObjectEntry.class, LoadersUtils.getLookUpKey(-classIdRef));
        //        if (theObject == null) {
        // now add a class object
        anObject = getClassObjectByIdRef(-classIdRef, theObjectClass);

		anObject.setSize(theClass.getSize());
        
        theClass.getClassObjects().add(anObject);
        theClass.setLoadedBy(thread);

        //adds the class object to the Class class
        TraceUtils.addClassObjectToClassClass(context, theProcess, anObject);

        return theClass;
    }

    protected TRCObject getClassObjectByIdRef(long id, Class theObjectClass) {
        aClass = theProcess.getClassClass();
		if(aClass==null)
		{
			TraceUtils.addClassObjectToClassClass(context,theProcess,null);
			aClass = theProcess.getClassClass();
		}
        return createObjectAndClassObject(id, theObjectClass);
    }

    protected TRCMethod getMethodByIdRef(TRCClass theClass) {
        TRCMethod method = (TRCMethod) LookupServiceExtensions.getInstance().locate(context, TRCMethodImpl.class, LoadersUtils.getLookUpKey(methodIdRef));

        if (method == null) {
            method = TraceFactory.eINSTANCE.createTRCMethod();
            method.setId(methodIdRef);
            method.setName(UNKNOWN + methodIdRef);
            method.setSignature("()");
            method.setDefiningClass(theClass);
        }

        return method;
    }

    protected TRCObject getObjectByIdRef(long id, Class theObjectClass) {
        aClass = theClass;
		if(aClass==null)
		{
			theProcess = getProcess();
			aClass = theProcess.getClassClass();
			if(aClass==null)
			{
				TraceUtils.addClassObjectToClassClass(context,theProcess,null);
				aClass = theProcess.getClassClass();
			}
		}
        return createObjectAndClassObject(id, theObjectClass);
    }

    protected TRCProcess getProcess() {
        if (context.getAgent() == null) {
            LoadersUtils.createAgent(context);
        }

        TRCProcess p = TraceUtils.getProcess(context.getAgent());

        if (p == null) {
            //create the process
            p = TraceFactory.eINSTANCE.createTRCProcess();
            p.setId(context.getProcessProxy().getRuntimeId());
            p.setPid(context.getProcessProxy().getPid());
            p.setAgent(context.getAgent());
            p.setStartTime(createDeltaTime());
        }

        return p;
    }

    protected TRCThread getThreadByIdRef(TRCProcess p) {
        TRCThread thread = (TRCThread) LookupServiceExtensions.getInstance().locate(context, TRCThreadImpl.class, LoadersUtils.getLookUpKey(threadIdRef));

        if (thread == null) {
            thread = TraceFactory.eINSTANCE.createTRCThread();
            thread.setId(threadIdRef);
            thread.setName(UNKNOWN + threadIdRef);
            thread.setStartTime(createDeltaTime());
            thread.setProcess(p);
        }

        return thread;
    }

    protected double createDeltaTime() {
        return time - getZeroTime();
    }

    protected TRCObject createObject(long id, Class theObjectClass) {
        if (theObjectClass == TRCHeapObjectImpl.class) {
            return createTRCHeapObject(id);
        }

        if (theObjectClass == TRCFullTraceObjectImpl.class) {
            return createTRCFullTraceObject(id);
        }

        return createTRCObject(id);
    }

    protected TRCObject createObject(long id, Class theObjectClass, TRCObjectEntry objectEntry) {
        if (theObjectClass == TRCHeapObjectImpl.class) {
            return createTRCHeapObject(id, objectEntry);
        }

        if (theObjectClass == TRCFullTraceObjectImpl.class) {
            return createTRCFullTraceObject(id, objectEntry);
        }

        return createTRCObject(id);
    }

    protected TRCObject createTRCFullTraceObject(long oId) {
        TRCFullTraceObject object = TraceFactory.eINSTANCE.createTRCFullTraceObject();

        object.setId(oId);
        object.setCreateTime(createDeltaTime());
        object.setThread(theClass.getLoadedBy());
        object.setIsA(aClass);
        object.setProcess(getTRCObjectProcess());

        return object;
    }

    protected TRCObject createTRCFullTraceObject(long oId, TRCObjectEntry objectEntry) {
        TRCFullTraceObject object = TraceFactory.eINSTANCE.createTRCFullTraceObject();

        object.setId(oId);

        TRCHeapObject o = (TRCHeapObject) objectEntry.get(TRCHeapObjectImpl.class);

        if (o != null) {
            object.setIsA(o.getIsA());
        } else {
            object.setIsA(theClass);
        }

        object.setCreateTime(createDeltaTime());
        object.setThread(theClass.getLoadedBy());
        object.setProcess(getTRCObjectProcess());

        return object;
    }

    protected TRCObject createTRCHeapObject(long oId) {
        TRCHeapObject object = TraceFactory.eINSTANCE.createTRCHeapObject();

        object.setId(oId);

        //		theObject.setCreateTime(createDeltaTime());
        //		theObject.setThread(theClass.getLoadedBy());
        object.setIsA(aClass);
        object.setProcess(getTRCObjectProcess());

        return object;
    }

    protected TRCObject createTRCHeapObject(long oId, TRCObjectEntry objectEntry) {
        TRCHeapObject object = TraceFactory.eINSTANCE.createTRCHeapObject();

        object.setId(oId);

        //		theObject.setCreateTime(createDeltaTime());
        //		theObject.setThread(theClass.getLoadedBy());
        TRCFullTraceObject o = (TRCFullTraceObject) objectEntry.get(TRCFullTraceObjectImpl.class);

        if (o != null) {
            object.setIsA(o.getIsA());
        } else {
            object.setIsA(theClass);
        }

        object.setProcess(getTRCObjectProcess());

        return object;
    }

    protected TRCObject createTRCObject(long oId) {
        TRCObject object = TraceFactory.eINSTANCE.createTRCObject();

        object.setId(oId);

        //		theObject.setCreateTime(createDeltaTime());
        //		theObject.setThread(theClass.getLoadedBy());
        object.setIsA(aClass);
        object.setProcess(getTRCObjectProcess());

        return object;
    }

    protected void dispatchProcessMode(int step) {
        switch (context.getCollectionMode().getValue()) {
        case TRCCollectionMode.HEAP_STATISTICS_ONLY:
            processHS(step);

            break;

        case TRCCollectionMode.HEAP_FULL:
            processHF(step);

            break;

        case TRCCollectionMode.HEAP_AND_EXECUTION_FULL:
            processHF_EF(step);

            break;

        case TRCCollectionMode.HEAP_AND_EXECUTION_STATISTICS_ONLY:
            processHS_ES(step);

            break;

        case TRCCollectionMode.EXECUTION_NO_INSTANCES:
            processENI(step);

            break;

        case TRCCollectionMode.EXECUTION_FULL:
            processEF(step);

            break;

        case TRCCollectionMode.HEAP_FULL_AND_EXECUTION_STATISTICS_ONLY:
            processHF_ES(step);

            break;

        case TRCCollectionMode.EXECUTION_STATISTICS_ONLY:
            processES(step);

            break;

        default:
            dynamicProcess(this, step);

            break;
        }
    }

    protected void dynamicProcess(TraceXMLFragmentLoader loader, int step) {
    }

    protected void processEF(int step) {
    }

    protected void processENI(int step) {
        processES(step);
    }

    protected void processES(int step) {
    }

    protected void processHF(int step) {
    }

    protected void processHF_EF(int step) {
        processHF(step);
        processEF(step);
    }

    protected void processHF_ES(int step) {
        processHF(step);
        processES(step);
    }

    protected void processHS(int step) {
    }

    protected void processHS_ES(int step) {
        processHS(step);
        processES(step);
    }

    protected void updateVirtualObject(long oId, int size) {
        virtualObject = (VirtualObjectInfo) LookupServiceExtensions.getInstance().locate(context, VirtualObjectInfo.class, LoadersUtils.getLookUpKey(oId));

        if (virtualObject == null) {
            virtualObject = new VirtualObjectInfo();
            LoadersUtils.registerGenericLookUpEntry(context, LoadersUtils.getLookUpKey(oId), virtualObject);
        }

        //        size = TraceUtils.getObjectRealSize(isArray, size);
        virtualObject.myClass = theClass;
        virtualObject.myThread = theThread;
        virtualObject.createTime = createDeltaTime();
        virtualObject.size = size;
    }

    private TRCProcess getTRCObjectProcess() {
        return getProcess();
    }

    private double getZeroTime() {
        return getProcess().getAgent().getStartTime();
    }

    private TRCObject createObjectAndClassObject(long id, Class theObjectClass) {
        Object oKey = LoadersUtils.getLookUpKey(id);
        TRCObjectEntry objectEntry = (TRCObjectEntry) LookupServiceExtensions.getInstance().locate(context, TRCObjectEntry.class, oKey);
        TRCObject object = null;

        if (objectEntry != null) {
            object = objectEntry.get(theObjectClass);

            if (object == null) {
                object = createObject(id, theObjectClass, objectEntry);
                objectEntry.put(object.getClass(), object);
            }
        } else {
            object = createObject(id, theObjectClass);

            //          objectEntry = new TRCObjectEntry();
            //			objectEntry.put(object.getClass(),object);
            //			LoadersUtils.registerGenericLookUpEntry(context,oKey,objectEntry);
        }

        return object;
    }
}
