/*******************************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.ui.test.util;

import java.io.UnsupportedEncodingException;
import java.util.Arrays;

import junit.framework.Test;
import junit.framework.TestCase;
import junit.framework.TestSuite;

import org.eclipse.hyades.ui.internal.util.StringUtil;

/**
 * Constains the methods to test the 
 * {@link org.eclipse.hyades.ui.internal.util.StringUtil} class.
 *  
 * @author marcelop
 * @since 0.0.1
 */
public class StringUtilTest 
extends TestCase
{
	/**
	 * Constructor for BaseStringTest.
	 * @param name
	 */
	public StringUtilTest(String name)
	{
		super(name);
	}
	
	/**
	 * Returns a suite with the tests defined in this class.
	 * @return Test
	 */
	public static Test suite()
	{
		TestSuite testSuite = new TestSuite("Hyades UI String Util");
		testSuite.addTestSuite(StringUtilTest.class);
		return testSuite;
	}

	/**
	 * Test for {@link StringUtil#decode(String, byte[])}
	 */
	public void testDecode()
	throws UnsupportedEncodingException
	{
		String encoding = "UTF-8";
		String ret = "\u4E13a\u4E13\u4E13";
		byte[] data = ret.getBytes(encoding);
		assertEquals(ret, StringUtil.decode(encoding, data));

		encoding = "ISO-8859-1";
		ret = "a";
		data = ret.getBytes(encoding);
		assertEquals(ret, StringUtil.decode(encoding, data));

		encoding = "Test";
		assertNull(StringUtil.decode(encoding, data));
	}

	/**
	 * Test for {@link StringUtil#encode(String, String)}
	 */
	public void testEncode()
	throws UnsupportedEncodingException
	{
		String encoding = "UTF-8";
		String text = "\u4E13a\u4E13\u4E13";
		assertTrue(Arrays.equals(text.getBytes(encoding), StringUtil.encode(encoding, text)));
		
		encoding = "US-ASCII";
		assertTrue(Arrays.equals(text.getBytes(encoding), StringUtil.encode(encoding, text)));
		
		encoding = "Test";
		assertNull(StringUtil.encode(encoding, text));
	}

	/**
	 * Test for {@link StringUtil#indexOf(StringBuffer, String, int)}
	 */
	public void testIndexOf()
	{
		String str = "TesT";
		StringBuffer in = new StringBuffer(str);
		for(int fromIndex=-1; fromIndex<=str.length()+1; fromIndex++)
			assertEquals("fromIndex: " + fromIndex, in.toString().indexOf(str, fromIndex), StringUtil.indexOf(in, str, fromIndex));

		str = "T";
		in = new StringBuffer(str).append("a").append(str);
		for(int fromIndex=-1; fromIndex<=str.length()+1; fromIndex++)
			assertEquals("fromIndex: " + fromIndex, in.toString().indexOf(str, fromIndex), StringUtil.indexOf(in, str, fromIndex));

		str = "aa";
		in = new StringBuffer("a");
		for(int fromIndex=-1; fromIndex<=str.length()+1; fromIndex++)
			assertEquals("fromIndex: " + fromIndex, in.toString().indexOf(str, fromIndex), StringUtil.indexOf(in, str, fromIndex));

		str = "aa";
		in = new StringBuffer("aabbaaaaaccaaaaaadaaaa");
		for(int fromIndex=-1; fromIndex<=str.length()+1; fromIndex++)
			assertEquals("fromIndex: " + fromIndex, in.toString().indexOf(str, fromIndex), StringUtil.indexOf(in, str, fromIndex));
	}

	/**
	 * Test for {@link StringUtil#replace(String, String, String)}
	 */
	public void testReplace()
	{
		String in = "Test";
		String oldString = "es";
		String newString = "##";
		assertEquals("T##t", StringUtil.replace(in, oldString, newString));

		in = "Test1111111Test11111111Test111111";
		oldString = "11111";
		newString = "a";
		assertEquals("Testa11Testa111Testa1", StringUtil.replace(in, oldString, newString));		

		in = "Testesst";
		oldString = "s";
		newString = "";
		assertEquals("Tetet", StringUtil.replace(in, oldString, newString));		

		in = "Testesst";
		oldString = "s";
		newString = null;
		assertEquals("Tetet", StringUtil.replace(in, oldString, newString));		

		in = "Testest";
		oldString = "est";
		newString = "";
		assertEquals("T", StringUtil.replace(in, oldString, newString));		

		in = "Testest";
		oldString = "";
		newString = "TT";
		assertEquals("Testest", StringUtil.replace(in, oldString, newString));		

		in = "TesT";
		oldString = "T";
		newString = "TOT";
		assertEquals("TOTesTOT", StringUtil.replace(in, oldString, newString));		


		in = "TesT";
		oldString = "T";
		newString = "U";
		assertEquals("UesU", StringUtil.replace(in, oldString, newString));		

		in = "Test";
		oldString = "es";
		newString = "es#Testt#es";
		assertEquals("Tes#Testt#est", StringUtil.replace(in, oldString, newString));		

		in = "Testest";
		oldString = "Testest";
		newString = "test";
		assertEquals("test", StringUtil.replace(in, oldString, newString));		

		in = "Testest";
		oldString = "Testest";
		newString = "testtesttest";
		assertEquals("testtesttest", StringUtil.replace(in, oldString, newString));		

		in = "Testest";
		oldString = "123";
		newString = "456";
		assertEquals("Testest", StringUtil.replace(in, oldString, newString));
	}

	/**
	 * Test for {@link StringUtil#tokenizer(String, String, boolean)}
	 */
	public void testTokenizer()
	{
		String text = "1a2ab3abc4abc";
		String token = "a";
		boolean includeToken = true;
		String[] expected = new String[]{"1","a","2","a","b3","a","bc4","a","bc"};
		assertTrue(Arrays.equals(expected, StringUtil.tokenizer(text, token, includeToken)));
			
		includeToken = false;
		expected = new String[]{"1","2","b3","bc4","bc"};
		assertTrue(Arrays.equals(expected, StringUtil.tokenizer(text, token, includeToken)));

		token = "ab";
		includeToken = true;
		expected = new String[]{"1a2","ab","3","ab","c4","ab","c"};
		assertTrue(Arrays.equals(expected, StringUtil.tokenizer(text, token, includeToken)));

		includeToken = false;
		expected = new String[]{"1a2","3","c4","c"};
		assertTrue(Arrays.equals(expected, StringUtil.tokenizer(text, token, includeToken)));

		token = "abc";
		includeToken = true;
		expected = new String[]{"1a2ab3","abc","4","abc"};
		assertTrue(Arrays.equals(expected, StringUtil.tokenizer(text, token, includeToken)));

		includeToken = false;
		expected = new String[]{"1a2ab3","4"};
		assertTrue(Arrays.equals(expected, StringUtil.tokenizer(text, token, includeToken)));

		token = "1a2";
		includeToken = true;
		expected = new String[]{"1a2","ab3abc4abc"};
		assertTrue(Arrays.equals(expected, StringUtil.tokenizer(text, token, includeToken)));

		includeToken = false;
		expected = new String[]{"ab3abc4abc"};
		assertTrue(Arrays.equals(expected, StringUtil.tokenizer(text, token, includeToken)));

		token = "1x";
		includeToken = true;
		expected = new String[]{"1a2ab3abc4abc"};
		assertTrue(Arrays.equals(expected, StringUtil.tokenizer(text, token, includeToken)));

		includeToken = false;
		expected = new String[]{"1a2ab3abc4abc"};
		assertTrue(Arrays.equals(expected, StringUtil.tokenizer(text, token, includeToken)));
	}
	
	/**
	 * Test for {@link BaseString#isDigits(String)}
	 */
	public void testIsDigits()
	{
		String srcString = "0123456789";
		assertTrue(StringUtil.isDigits(srcString));

		srcString = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz";
		assertFalse(StringUtil.isDigits(srcString));
		
		srcString = "0a1b2c3d4e5f6g7h8i9j";
		assertFalse(StringUtil.isDigits(srcString));
	}	
}
