/*******************************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.internal.dialog;

import java.util.Vector;

import org.eclipse.hyades.models.common.datapool.DPLDatapool;
import org.eclipse.hyades.models.common.datapool.DPLVariable;
import org.eclipse.hyades.test.ui.TestUIPlugin;
import org.eclipse.hyades.ui.internal.util.GridDataUtil;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.KeyListener;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;

/**
 * Dialog to collect information about a variable/column that is being
 * inserted or edited by the datapool table.  
 * 
 * @author psun
 *
 */
public class DataPoolColumnDialog extends Dialog implements KeyListener, SelectionListener{

	private static final String TAG_VARIABLES = "variables"; //$NON-NLS-1$

	
	private static final int HEIGHT = 80;
	private static final int WIDTH = 300;
	private Shell shell = null;
	private DPLDatapool datapool = null;
	private String title = null;
	private DPLVariable selectedVariable = null;
	private DPLVariable previousVariable = null;

	private Text nameField = null;
	private Text typeField = null;
	private Color nameFieldForeground = null;
	private Combo insertionVariables = null;

	private String name = null;
	private String type = null;
	private String insertionVariableID = null;
	
	public DataPoolColumnDialog(Shell parentShell, 
									     DPLDatapool datapool,
									     DPLVariable selectedVariable,
									     DPLVariable previousVariable,
									     String title)
	{
		super(parentShell);
		this.shell = parentShell;
		this.datapool = datapool;
		this.title = title; 
		this.selectedVariable = selectedVariable;
		this.previousVariable = previousVariable;
	}

	/**
	 * @see org.eclipse.jface.window.Window#createContents(org.eclipse.swt.widgets.Composite)
	 */
	protected Control createContents(Composite parent)
	{
		Control control = super.createContents(parent);
		enableOK();
		return control;
	}

	/**
	 * @see org.eclipse.jface.dialogs.Dialog#createDialogArea(org.eclipse.swt.widgets.Composite)
	 */
	protected Control createDialogArea(Composite parent)
	{
		getShell().setText(title);
		Composite superComposite = (Composite)super.createDialogArea(parent);
		GridData gridData = GridDataUtil.createFill();
		gridData.heightHint = HEIGHT;
		gridData.widthHint = WIDTH;
		superComposite.setLayoutData(gridData);		

		GridLayout gridLayout = new GridLayout();
		gridLayout.numColumns = 2;
		superComposite.setLayout(gridLayout);

		Label nameLabel = new Label(superComposite, SWT.NONE);
		nameLabel.setText(TestUIPlugin.getString("DATA_COL_DLG_NAME")); //$NON-NLS-1$

		nameField = new Text(superComposite, SWT.SINGLE | SWT.BORDER);
		gridData = new GridData(GridData.HORIZONTAL_ALIGN_FILL);
		gridData.horizontalSpan = 1;
		gridData.widthHint = WIDTH - 100;
		nameField.setLayoutData(gridData);
		nameField.addKeyListener(this);
		nameFieldForeground = nameField.getForeground();

		Label typeLabel = new Label(superComposite, SWT.NONE);
		typeLabel.setText(TestUIPlugin.getString("DATA_COL_DLG_TYPE")); //$NON-NLS-1$

		typeField = new Text(superComposite, SWT.SINGLE | SWT.BORDER);
		gridData = new GridData(GridData.HORIZONTAL_ALIGN_FILL);
		gridData.horizontalSpan = 1;
		typeField.setLayoutData(gridData);
		typeField.addKeyListener(this);

		DPLVariable[] variables = getVariables(null);
		if(selectedVariable != null)
		{
			name = selectedVariable.getName();
			nameField.setText(name);
			nameField.setSelection(name.length() + 1);
			if(selectedVariable.getType() != null)
			{
				type = selectedVariable.getType();
				typeField.setText(type);
				typeField.setSelection(type.length() + 1);
			}
		}
		String[] variableLabels = getVariableLabels(variables);

		Label insertionLabel = new Label(superComposite, SWT.NONE);
		if(title.equals(TestUIPlugin.getString("DATA_COL_DLG_TITLE_INS"))) //$NON-NLS-1$
			insertionLabel.setText(TestUIPlugin.getString("DATA_COL_DLG_INS")); //$NON-NLS-1$
		else
			insertionLabel.setText(TestUIPlugin.getString("DATA_COL_DLG_MOVE")); //$NON-NLS-1$

		insertionVariables = new Combo(superComposite, SWT.DROP_DOWN | SWT.READ_ONLY);
		insertionVariables.setItems(variableLabels);
		insertionVariables.setData(TAG_VARIABLES, variables);
	    insertionVariables.setLayoutData(new GridData(GridData.HORIZONTAL_ALIGN_FILL));
		insertionVariables.addSelectionListener(this);
		
		if(previousVariable != null)
		{
			int selectIndex = findVariableIndexInCombo(variables, previousVariable);
			insertionVariables.select(selectIndex);
			insertionVariableID = variables[selectIndex].getId();				
		}
		else
		{
			insertionVariables.select(insertionVariables.getItemCount() - 1);
			insertionVariableID = new String();
		}

		return superComposite;
	}

	private boolean enableOK()
	{
		if(nameField.getText().length() > 0)
		{
			if(isNameUnique(nameField.getText()))
			{		
				nameField.setForeground(nameFieldForeground);
				getButton(IDialogConstants.OK_ID).setEnabled(true);
				return true;
			}
			else
			{
				Display display= Display.getDefault();
				Color color= display.getSystemColor(SWT.COLOR_RED);
				nameField.setForeground(color);
			}
		}
		getButton(IDialogConstants.OK_ID).setEnabled(false);		
		return false;
	}
	
	private DPLVariable[] getVariables(String exclusionID)
	{
		int variableCount = datapool.getVariables().size();
		Vector tempList = new Vector();
		for(int i = 0; i < variableCount; i++)
		{
			DPLVariable variable = (DPLVariable)datapool.getVariables().get(i);
			String variableID = variable.getId();
			if(variableID.equals(exclusionID))
				continue;
			else
				tempList.add(variable);
		}

		int listSize = tempList.size();
		DPLVariable[] returnList = new DPLVariable[listSize];
		tempList.toArray(returnList);
		return returnList;
	}
	
	private String[] getVariableLabels(DPLVariable[] variables)
	{
		if(variables == null)
			return null;
		int listSize = variables.length;
		String[] returnList = new String[listSize + 1];
		for(int i = 0; i < listSize; i++)
		{
			DPLVariable variable = variables[i];
			returnList[i] = variable.getName() + TestUIPlugin.getString("DATA_EDT_DIVIDER") + variable.getType(); //$NON-NLS-1$
		}	
		returnList[listSize] = TestUIPlugin.getString("DATA_COL_DLG_HEAD_VAR"); //$NON-NLS-1$
		return returnList;
	}

	public String getName()
	{
		return name;
	}

	public String getType()
	{
		return type;
	}
	
	public String getInsertionVariableID()
	{
		return insertionVariableID;
	}

	private int findVariableIndexInCombo(DPLVariable[] variables, DPLVariable variable)
	{
		if(variables == null || variable == null)
			return -1;
		for(int i = 0; i < variables.length; i++)
		{
			if(variables[i].equals(variable))
				return i;	
		}
		return -1;
	}

	private boolean isNameUnique(String text)
	{
		if(datapool == null)
			return true;
		for(int i = 0; i < datapool.getVariables().size(); i++)
		{
			DPLVariable variable = (DPLVariable)datapool.getVariables().get(i);
			if(!variable.equals(selectedVariable) && text.equals(variable.getName()))
				return false;
		}
		return true;			
	}

	// ISelectionListener

	/* 
	 * @see org.eclipse.swt.events.SelectionListener#widgetSelected(org.eclipse.swt.events.SelectionEvent)
	 */
	public void widgetSelected(SelectionEvent e) {
		Combo insertVariableCombo = (Combo)e.widget;
		int selectionIndex = insertVariableCombo.getSelectionIndex();
		DPLVariable[] variables = (DPLVariable[])insertVariableCombo.getData(TAG_VARIABLES);
		if(selectionIndex < variables.length)
		{
			DPLVariable variable = variables[selectionIndex];
			insertionVariableID = variable.getId();
		}
		else
			insertionVariableID = new String();
	}

	/* 
	 * @see org.eclipse.swt.events.SelectionListener#widgetDefaultSelected(org.eclipse.swt.events.SelectionEvent)
	 */
	public void widgetDefaultSelected(SelectionEvent e) 
	{
		return;	
	}

	// IKeyListener

	/* 
	 * @see org.eclipse.swt.events.KeyListener#keyPressed(org.eclipse.swt.events.KeyEvent)
	 */
	public void keyPressed(KeyEvent e) 
	{		
		if(nameField != null)
			name = nameField.getText();
		if(typeField != null)
			type = typeField.getText();
		enableOK();
	}

	/* 
	 * @see org.eclipse.swt.events.KeyListener#keyReleased(org.eclipse.swt.events.KeyEvent)
	 */
	public void keyReleased(KeyEvent e) 
	{
		return;
	}

}
