/*******************************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.internal.editor.extension;

import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.StructuredSelection;

import org.eclipse.hyades.models.common.configuration.CFGLocation;
import org.eclipse.hyades.models.common.testprofile.TPFDeployment;
import org.eclipse.hyades.test.ui.TestUIPlugin;
import org.eclipse.hyades.test.ui.internal.editor.form.DeploymentForm;
import org.eclipse.hyades.test.ui.internal.editor.form.LocationForm;
import org.eclipse.hyades.test.ui.internal.editor.form.util.WidgetFactory;
import org.eclipse.hyades.ui.editor.IHyadesEditorPart;

/**
 * Hyades deployment editor extension that is suitable for all the
 * deployment objects.
 * 
 * <p>This class is used whenever it is not possible neither to identify nor
 * to locate the appropriate extension editor.
 * 
 * @author marcelop
 * @since 0.0.1
 */
public class DeploymentEditorExtension
extends BaseEditorExtension
{
	private static final int PAGE_OVERVIEW = 0;
	private static final int PAGE_LOCATIONS = 1;

	public static final String EXTENSION_ID = "org.eclipse.hyades.test.ui.editor.extension.Deployment";
	
	private DeploymentForm deploymentForm;
	private LocationForm locationsForm;
	
	/**
	 * @see org.eclipse.hyades.ui.util.IDisposable#dispose()
	 */
	public void dispose()
	{
		deploymentForm.getWidgetFactory().dispose();
		deploymentForm.dispose();
		locationsForm.dispose();
		
		super.dispose();
	}

	/**
	 * Returns the deployment that is manipulated by this editor
	 * extension.
	 * @return TPFDeployment 
	 */
	public TPFDeployment getDeployment()
	{
		return (TPFDeployment)getHyadesEditorPart().getEditorObject();
	}
	
	/**
	 * @see org.eclipse.hyades.ui.editor.IEditorExtension#createPages()
	 */
	public void createPages()
	{
		IHyadesEditorPart hyadesEditorPart = getHyadesEditorPart();
		WidgetFactory widgetFactory = new WidgetFactory();
		
		deploymentForm = new DeploymentForm(this, widgetFactory);
		deploymentForm.setLocationPageIndex(PAGE_LOCATIONS);	
		hyadesEditorPart.addPage(deploymentForm.createControl());
		hyadesEditorPart.setPageText(0, TestUIPlugin.getString("W_OVERVIEW"));

		locationsForm = new LocationForm(this, widgetFactory);
		hyadesEditorPart.addPage(locationsForm.createControl());
		hyadesEditorPart.setPageText(PAGE_LOCATIONS, TestUIPlugin.getString("W_LOCS"));
		
		deploymentForm.updateTitle();
	}
	
	/**
	 * @see org.eclipse.hyades.ui.util.IRefreshable#refreshContent(java.lang.Object)
	 */
	public void refreshContent(Object data)
	{
		deploymentForm.load();
		deploymentForm.updateTitle();
		
		locationsForm.load();
	}

	/**
	 * @see org.eclipse.hyades.ui.editor.IEditorExtension#getSelection()
	 */
	public IStructuredSelection getSelection()
	{
		switch(getHyadesEditorPart().getActivePage())
		{
			case PAGE_OVERVIEW:
				return new StructuredSelection(getDeployment());
				
			case PAGE_LOCATIONS:
				return (IStructuredSelection)locationsForm.getSelection();
		}
		
		return StructuredSelection.EMPTY;
	}
	
	/**
	 * @see org.eclipse.hyades.ui.editor.IEditorExtension#pageActivated(int)
	 */
	public boolean pageActivated(int index)
	{
		switch(index)
		{
			case PAGE_OVERVIEW:
				return deploymentForm.activated();
				
			case PAGE_LOCATIONS:
				return locationsForm.activated();
		}
		
		return false;
	}

	/**
	 * @see org.eclipse.hyades.ui.editor.IEditorExtension#setSelection(org.eclipse.jface.viewers.IStructuredSelection)
	 */
	public void setSelection(IStructuredSelection structuredSelection)
	{
		if(structuredSelection.size() == 1)
		{
			Object object = structuredSelection.getFirstElement();
			if(object == getDeployment())
			{
				// don't do anything.
			}
			else if(object instanceof CFGLocation)
			{
				getHyadesEditorPart().setActivePage(PAGE_LOCATIONS);
				locationsForm.selectReveal(structuredSelection);
			}
		}
	}	
}
