/*******************************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.internal.editor.form.base;

import org.eclipse.hyades.models.common.datapool.Common_DatapoolFactory;
import org.eclipse.hyades.models.common.datapool.DPLCell;
import org.eclipse.hyades.models.common.datapool.DPLDatapool;
import org.eclipse.hyades.models.common.datapool.DPLEquivalenceClass;
import org.eclipse.hyades.models.common.datapool.DPLRecord;
import org.eclipse.hyades.models.common.datapool.DPLVariable;
import org.eclipse.hyades.test.ui.TestUIPlugin;
import org.eclipse.hyades.test.ui.internal.action.DeleteColumnAction;
import org.eclipse.hyades.test.ui.internal.action.DeleteRowAction;
import org.eclipse.hyades.test.ui.internal.action.DeleteRowGroupAction;
import org.eclipse.hyades.test.ui.internal.action.EditColumnAction;
import org.eclipse.hyades.test.ui.internal.action.EditRowAction;
import org.eclipse.hyades.test.ui.internal.action.EditRowGroupAction;
import org.eclipse.hyades.test.ui.internal.action.InsertColumnAction;
import org.eclipse.hyades.test.ui.internal.action.InsertRowAction;
import org.eclipse.hyades.test.ui.internal.action.InsertRowGroupAction;
import org.eclipse.hyades.test.ui.internal.dialog.DataPoolColumnDialog;
import org.eclipse.hyades.test.ui.internal.dialog.DataPoolRowDialog;
import org.eclipse.hyades.test.ui.internal.dialog.DataPoolRowGroupDialog;
import org.eclipse.hyades.test.ui.internal.editor.extension.DatapoolEditorExtension;
import org.eclipse.jface.action.IMenuListener;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.viewers.ColumnPixelData;
import org.eclipse.jface.viewers.ICellEditorListener;
import org.eclipse.jface.viewers.ISelectionProvider;
import org.eclipse.jface.viewers.TableLayout;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.ControlEditor;
import org.eclipse.swt.custom.ScrolledComposite;
import org.eclipse.swt.custom.TableCursor;
import org.eclipse.swt.events.ControlAdapter;
import org.eclipse.swt.events.ControlEvent;
import org.eclipse.swt.events.ControlListener;
import org.eclipse.swt.events.FocusListener;
import org.eclipse.swt.events.KeyAdapter;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.KeyListener;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.events.MouseListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Cursor;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.swt.widgets.TableItem;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.IWorkbenchPartSite;

/**
 * A Datapool Table is a Excel like table editor that wraps the JFace 
 * TableViewer and SWT Table widget and edits the DPLDatapool object.  It 
 * provides the ability to insert, delete, and edit variables, equivalence 
 * classes, and records.  A varible is represented as a column in the 
 * table.  An equivalence class is a grouping of rows with the same name,
 * and a record is a single row in the table.  The table enforces that an 
 * equivalence class always has one or more records. 
 * 
 * @author psun
 */
public class DatapoolTable implements IMenuListener, SelectionListener {

	private static final String START_LITERAL = "<Literal>"; //$NON-NLS-1$
	private static final String END_LITERAL = "</Literal>"; //$NON-NLS-1$
	private static final String TAG_DATA = "data"; //$NON-NLS-1$
	private static final String TAG_RECORD = "record"; //$NON-NLS-1$
	private static final String TAG_VARIABLE = "variable"; //$NON-NLS-1$
	private static final String TAG_VARIABLE_INDEX = "variableIndex"; //$NON-NLS-1$
	private static final String TAG_EQUIVALENCE_CLASS = "equivalenceClass"; //$NON-NLS-1$
	
	private static final int MAX_TABLE_WIDTH = 60000;
	private static final int MAX_TABLE_HEIGHT = 120000;
		 
	private DPLDatapool datapool = null;
	private Table table = null;
	private DatapoolTableUtil tableUtil = null;
	private TableViewer viewer = null;
	private Composite parent = null;
	private Menu contextMenu = null;
	private TableCursor cursor = null;
	private ControlEditor controlEditor = null;
	private DatapoolEditorExtension extension = null;
	private Text textEditor = null;

	private FocusListener focusListener;
	private ICellEditorListener	cellEditorListener;

	/*
	 * Handles the selection of the header area.
	 * <p>Clicking on the header area bring up the dialog to change and/or 
	 * move the selected variable/column.
	 */
	private SelectionListener headerListener = new SelectionAdapter()
	{
		public void widgetSelected(SelectionEvent event)
		{
			if(extension.getHyadesEditorPart().isReadOnly())
				return;
			if(textEditor != null && !textEditor.isDisposed())
				textEditor.dispose();
			TableColumn tableColumn = (TableColumn) event.widget;
			editColumnAux(tableColumn);
		}
	};

	/*
	 * Handles the resizing of scroll bars when the ScrollableComposite 
	 * containing the table is resized.  The datapool table uses the 
	 * composite's scroll bars. 
	 */
	private ControlListener resizeListener = new ControlAdapter()
	{
		public void controlMoved(ControlEvent e) 
		{
		}

		public void controlResized(ControlEvent e) 
		{
			resetScrollBarSize(true);
		}				
	};

	/*
	 * Handles the resizing of scroll bars when columns in the table are 
	 * resized.  
	 */
	private ControlListener resizeColumnListener = new ControlAdapter()
	{
		public void controlMoved(ControlEvent e) 
		{
		}

		public void controlResized(ControlEvent e) 
		{
			if(cursor != null && !cursor.isDisposed())
				cursor.setSelection(table.getSelectionIndex(), cursor.getColumn());
			resetScrollBarSize(false);
		}				
	};


	/*
	 * Handles the keystrokes of the text editor when it is in edit mode.
	 */
	class TableCellKeyAdaptor implements KeyListener
	{
		private Text text = null;
		private TableItem row = null;
		private int rowIndex = -1;
		private int column = -1;
		
		public TableCellKeyAdaptor(Text text, TableItem row, int column)
		{
			this.text = text;
			this.row = row;
			this.column = column;
			if(row != null)
				this.rowIndex = table.indexOf(row);
		}
		
		/* 
		 * @see org.eclipse.swt.events.KeyListener#keyReleased(org.eclipse.swt.events.KeyEvent)
		 */		
		public void keyPressed(KeyEvent e) {
			if(extension.getHyadesEditorPart().isReadOnly())
				return;
			// Close the text editor and copy the data over when the user hits
			// "ENTER" or any of the arrow keys.
			if (e.character == SWT.CR ||
				e.keyCode == SWT.ARROW_DOWN ||
				e.keyCode == SWT.ARROW_UP ||
				e.keyCode == SWT.ARROW_LEFT ||
				e.keyCode == SWT.ARROW_RIGHT) 
			{
				row.setText(column, text.getText());
				applyEditingValue(text.getText());
			}
			// Move the cursor if any of the arrow keys are hit.  This is to
			// simulate excel behavior.
			if(cursor != null && !cursor.isDisposed())
			{
				if(e.keyCode == SWT.ARROW_DOWN)
				{
					if(rowIndex + 1 < table.getItemCount())
					{
						cursor.setSelection(rowIndex + 1, column);	
						table.setSelection(rowIndex + 1);
						setScrollBars();
					}
				}
				if(e.keyCode == SWT.ARROW_UP)
				{
					if(rowIndex > 0)
					{
						cursor.setSelection(rowIndex - 1, column);	
						table.setSelection(rowIndex - 1);						
						setScrollBars();
					}
				}
				if(e.keyCode == SWT.ARROW_RIGHT)
				{
					if(column + 1 < table.getColumnCount())
					{
						cursor.setSelection(rowIndex, column + 1);
						setScrollBars();							
					}
				}
				if(e.keyCode == SWT.ARROW_LEFT)
				{
					if(column > 0)
					{
						cursor.setSelection(rowIndex, column - 1);
						setScrollBars();							
					}
				}
			}
			// Close the text editor when the user hits "ESC"
			if (e.character == SWT.ESC) {
				text.dispose();
			}
		}

		/* 
		 * @see org.eclipse.swt.events.KeyListener#keyReleased(org.eclipse.swt.events.KeyEvent)
		 */
		public void keyReleased(KeyEvent e) 
		{
		}		
	}

	/*
	 * Default constructor for the Datapool Table.  Registers various key, 
	 * menu, and selection listeners.  Creates context menu and populates 
	 * the table.
	 */
	public DatapoolTable(Composite parent, DatapoolEditorExtension datapoolEditorExtension, DPLDatapool datapool)
	{
		if(parent == null || datapool == null || datapoolEditorExtension == null)
			return;
		this.datapool = datapool;
		this.parent = parent;
		
		setWaitCursor();
		
		extension = datapoolEditorExtension;
		createTable(parent);
		tableUtil = new DatapoolTableUtil(table);

		MenuManager menuMgr= new MenuManager("#DatapoolEditor"); //$NON-NLS-1$
		menuMgr.setRemoveAllWhenShown(true);
		menuMgr.addMenuListener(this);
		
		contextMenu = menuMgr.createContextMenu(table);
		table.setMenu(contextMenu);
		table.addSelectionListener(this);

		IWorkbenchPartSite site = extension.getHyadesEditorPart().getEditorPart().getSite();

		site.setSelectionProvider(viewer);

		makeActions();
		
		try
		{
			parent.getParent().getParent().addControlListener(resizeListener);
		}
		catch(Exception e)
		{
			// do nothing
		}
		
		unsetWaitCursor();
	}

	/* 
	 * Creates the table widget and viewer.  Populates the table from the datapool.
	 */
	private void createTable(Composite parent)
	{					
		table = new Table(parent, SWT.BORDER | SWT.FULL_SELECTION);
		viewer = new TableViewer(table);

		table.setLinesVisible(true);
		table.setHeaderVisible(true);
		GridData gd = new GridData();
		gd.horizontalAlignment = GridData.FILL;
		gd.grabExcessHorizontalSpace = true;
		gd.verticalAlignment = GridData.FILL;
		gd.grabExcessVerticalSpace = true;
		gd.widthHint = MAX_TABLE_WIDTH;
		gd.heightHint = MAX_TABLE_HEIGHT;
		table.setLayoutData(gd);
		
		TableLayout layout = new TableLayout();
		layout.addColumnData(new ColumnPixelData(100, false));
		table.setLayout(layout);
				
		// Populate table		
		if(isWithoutColumns())
			makeHeaderColumn();	
		else
			makeColumns();
		
		if(!isWithoutRows())
			makeRows();

		createCursor();
			
		// Show the TableCursor when the user releases the "SHIFT" or "CTRL" key.
		// This signals the end of the multiple selection task.
		table.addKeyListener(new KeyAdapter() {
			
			public void keyReleased(KeyEvent e) {
				if (e.keyCode == SWT.CONTROL && (e.stateMask & SWT.SHIFT) != 0) return;
				if (e.keyCode == SWT.SHIFT && (e.stateMask & SWT.CONTROL) != 0) return;
				if (e.keyCode != SWT.CONTROL && (e.stateMask & SWT.CONTROL) != 0) return;
				if (e.keyCode != SWT.SHIFT && (e.stateMask & SWT.SHIFT) != 0) return;
			
				if(table.getItemCount() != 0)
				{
					TableItem[] selection = table.getSelection();
					TableItem row = (selection.length == 0) ? table.getItem(table.getTopIndex()) : selection[0];
					table.showItem(row);
					cursor.setSelection(row, 0);
					cursor.setVisible(true);
					cursor.setFocus();
				}
			}
		});
	}

	/*
	 * Creates the table cursor and its assicated event listeners.
	 */
	private void createCursor() {
		if(datapool.getEquivalenceClasses().size() == 0)
			return;
		cursor = new TableCursor(table, SWT.NONE);
		// Create an editor to edit the cell when the user hits "ENTER" 
		// while over a cell in the table
		controlEditor = new ControlEditor(cursor);
		controlEditor.grabHorizontal = true;
		controlEditor.grabVertical = true;
		
		cursor.setMenu(contextMenu);
		
		cursor.addSelectionListener(new SelectionAdapter() {
			// When the TableEditor is over a cell, select the corresponding row in 
			// the table.
			public void widgetSelected(SelectionEvent e) 
			{
				if(extension.getHyadesEditorPart().isReadOnly())
					return;

				if(textEditor != null && !textEditor.isDisposed())
					textEditor.dispose();	
				table.setSelection(new TableItem[] {cursor.getRow()});
				setScrollBars();
			}
			
			// Create an editor when the user hits "ENTER".
			public void widgetDefaultSelected(SelectionEvent e){
				if(extension.getHyadesEditorPart().isReadOnly())
					return;
				textEditor = new Text(cursor, SWT.NONE);
				TableItem row = cursor.getRow();
				int column = cursor.getColumn();
				// No editing of equivalence class/record name;
				if(column == 0)
				{
					editRowAux(cursor.getRow());
					return;					
				}
				textEditor.setText(row.getText(column));				
				textEditor.addKeyListener(new TableCellKeyAdaptor(textEditor, row, column));
				controlEditor.setEditor(textEditor);
				textEditor.setFocus();
			}
		});

		// Handles keyboard shortcuts for most operations.  Kicks the cell into
		// editing mode if the user hits a non-excluded key.
		cursor.addKeyListener(new KeyAdapter() {
			
			public void keyPressed(KeyEvent e) {
				if(extension.getHyadesEditorPart().isReadOnly())
					return;
				if (e.keyCode == SWT.INSERT && (e.stateMask & SWT.CTRL) != 0)
				{
					int column = cursor.getColumn();
					if(column == 0)
						insertRowGroup();
					if(column > 0)
						insertColumn();
					return;
				}
				if (e.keyCode == SWT.DEL && (e.stateMask & SWT.CTRL) != 0)
				{
					int column = cursor.getColumn();
					if(column == 0)
						deleteRowGroup();
					if(column > 0)
						deleteColumn();
					return;
				}
				if(e.keyCode == SWT.ESC || 
					e.keyCode == SWT.CR ||
					e.keyCode == SWT.ALT ||
					e.keyCode == SWT.PAGE_DOWN ||
					e.keyCode == SWT.PAGE_UP ||
					e.keyCode == SWT.HOME ||
					e.keyCode == SWT.END ||
					e.keyCode == SWT.CTRL ||
					e.keyCode == SWT.SHIFT ||
					((e.stateMask & SWT.CTRL) != 0) ||
					((e.stateMask & SWT.ALT) != 0) ||
					(e.keyCode >= SWT.F1 && e.keyCode <= SWT.F12))
				{
					return;
				}
				if(e.keyCode == SWT.INSERT)
				{
					int column = cursor.getColumn();
					if(column == 0)
						insertRow();
					if(column <= 0)
						return;
				}
				if(e.keyCode == SWT.DEL)
				{
					int column = cursor.getColumn();
					if(column == 0)
						deleteRow();
					if(column > 0)
						clearCell();
					return;					
				}
				if(e.keyCode == SWT.ARROW_DOWN ||
					e.keyCode == SWT.ARROW_UP ||
					e.keyCode == SWT.ARROW_LEFT ||
					e.keyCode == SWT.ARROW_RIGHT)
				{
					if(textEditor != null && !textEditor.isDisposed())
						textEditor.dispose();
					int selectedColumnIndex = cursor.getColumn();
					int selectedRowIndex = table.getSelectionIndex();
					// Code to prevent the cursor from moving too far to the
					// right.  This is to accomidate the fact that when a 
					// table column deleted, it is really hidden as a zero sized
					// column to the right
					if(e.keyCode == SWT.ARROW_RIGHT)
					{
						int visibleColumnCount = tableUtil.getColumnCount();
						if(selectedColumnIndex > visibleColumnCount - 1)
							cursor.setSelection(selectedRowIndex, selectedColumnIndex - 1);
					}
					return;	
				}
				else 
				{
					char inputCharacter = e.character;
					textEditor = new Text(cursor, SWT.NONE);
					TableItem row = cursor.getRow();
					int column = cursor.getColumn();
					// No editing of equivalence class/record name;
					if(column == 0)
					{
						editRowAux(cursor.getRow());
						return;					
					}
					if(e.keyCode == SWT.BS)
						textEditor.setText(new String());
					else
						textEditor.setText(row.getText(column) + String.valueOf(inputCharacter));
					textEditor.setSelection(textEditor.getText().length() + 1);						
					textEditor.addKeyListener(new TableCellKeyAdaptor(textEditor, row, column));
					controlEditor.setEditor(textEditor);
					textEditor.setFocus();
				}
				
			}
		});
		
		cursor.addMouseListener(new MouseListener() {
			
			// Double clicking in a cell creates an editor and begins editing 
			// for that cell
			public void mouseDoubleClick(MouseEvent event) 
			{
				if(extension.getHyadesEditorPart().isReadOnly())
					return;				
				textEditor = new Text(cursor, SWT.NONE);
				TableItem row = cursor.getRow();
				int column = cursor.getColumn();
				// No editing of equivalence class/record name;
				if(column == 0)
				{
					editRowAux(row);
					return;					
				}
				textEditor.setText(row.getText(column));
				textEditor.setSelection(textEditor.getText().length() + 1);						
				textEditor.addKeyListener(new TableCellKeyAdaptor(textEditor, row, column));
				controlEditor.setEditor(textEditor);
				textEditor.setFocus();												
			}
		
			public void mouseDown(MouseEvent event) 
			{
				// Bring up the context menu if the user hits the right mouse
				// button on a cursor.
				if(event.button == 3)
				{
					contextMenu.setEnabled(true);
					contextMenu.setVisible(true);
				}
				if(event.button != 1)				
					return;
				if(textEditor != null && !textEditor.isDisposed())
					textEditor.dispose();
			}
			
			public void mouseUp(MouseEvent event) 
			{
			}
		});

		if(table.getItemCount() > 0)
		{
			table.select(0);
			cursor.setSelection(0,0);
		}
	}
	
	/*
	 * Creates and displays a wait cursor.
	 */
	private void setWaitCursor()
	{
		if(parent == null)
			return;
		Display display = parent.getShell().getDisplay();
		Cursor waitCursor = new Cursor(display, SWT.CURSOR_WAIT);
		setDisplayCursor(waitCursor);		
	}

	/*
	 * Cancels the display of the wait cursor.
	 */
	private void unsetWaitCursor()
	{
		setDisplayCursor(null);
	}

	/*
	 * Creates and sets the cursor to c.
	 */
	private void setDisplayCursor(Cursor c) {
		if(parent == null)
			return;
		Display display = parent.getShell().getDisplay();
		Shell[] shells = display.getShells();
		for (int i = 0; i < shells.length; i++)
			shells[i].setCursor(c);
	}

	/*
	 * Takes the value text from the editor and sets it to the table item as
	 * well as the datapool cell.
	 */
	private void applyEditingValue(String text)
	{
		if (cursor != null && cursor.getColumn() > 0)
		{
			TableItem selectedTableItem = cursor.getRow();
			int selectedColumnIndex = cursor.getColumn();
			if(selectedTableItem == null || selectedTableItem.isDisposed())
				return;
			DPLCell[] rowData = (DPLCell[])selectedTableItem.getData(TAG_DATA); 
			DPLCell cell = rowData[cursor.getColumn() - 1];
			if(cell == null)
			{
				// If the text has not changed, don't make any changes.
				if(text.length() == 0)
				{
					if(textEditor != null && !textEditor.isDisposed())
						textEditor.dispose();
					return;
				}
				cell = Common_DatapoolFactory.eINSTANCE.createDPLCell();
				cell.setRecord((DPLRecord)selectedTableItem.getData(TAG_RECORD)); 
				TableColumn tableColumn = table.getColumn(selectedColumnIndex);
				cell.setVariable((DPLVariable)tableColumn.getData(TAG_VARIABLE));
				rowData[cursor.getColumn() - 1] = cell;
			}
			else
				// If the text has not changed, don't make any changes.
				if(wrapAsLiterl(text).equals(cell.getValue()))
				{
					if(textEditor != null && !textEditor.isDisposed())
						textEditor.dispose();
					return;
				}
			cell.setValue(wrapAsLiterl(text));
			selectedTableItem.setText(cursor.getColumn(), text);
			cursor.setSelection(table.getSelectionIndex(), cursor.getColumn());
			extension.markDirty();
			if(textEditor != null && !textEditor.isDisposed())
				textEditor.dispose();
		}
					
	}

	/*
	 * Creates TableColumn objects from the DPLVariable objects in the 
	 * datapool.  The indexes are shifted by one because there is a 
	 * column for the row headers.
	 */
	private void makeColumns()
	{
		makeHeaderColumn();
		for(int i = 0; i < datapool.getVariables().size(); i++)
		{
			TableColumn tableColumn = new TableColumn(table, SWT.NONE, i + 1);
			DPLVariable variable = (DPLVariable)datapool.getVariables().get(i);
			tableColumn.setResizable(true);
			tableColumn.setText(variable.getName() + TestUIPlugin.getString("DATA_EDT_DIVIDER") + variable.getType()); //$NON-NLS-1$
			tableColumn.setData(TAG_VARIABLE, variable); 
			tableColumn.setData(TAG_VARIABLE_INDEX, new Integer(i)); 
			tableColumn.setWidth(100);
			tableColumn.addSelectionListener(headerListener);	
			tableColumn.addControlListener(resizeColumnListener);		
		}		
    }

	/*
	 * Creates the column header.
	 */
	private void makeHeaderColumn() {
		TableColumn recordTableColumn = new TableColumn(table, SWT.NONE, 0);
		recordTableColumn.setResizable(true);
		recordTableColumn.setWidth(100);
		recordTableColumn.setData(TAG_VARIABLE, null);
		recordTableColumn.addControlListener(resizeColumnListener); 
	}

	/*
	 * Creates TableItem objects from the DPLEquivalenceClass and the DPLRecord
	 * objects in the datapool.  Records do not need to have cells for every
	 * variable.
	 */
	private void makeRows()
	{
		for(int i = 0; i < datapool.getEquivalenceClasses().size(); i++)
		{
			DPLEquivalenceClass equivalenceClass = (DPLEquivalenceClass)datapool.getEquivalenceClasses().get(i);
			for(int j = 0; j < equivalenceClass.getRecord().size(); j++)
			{
				DPLRecord record = (DPLRecord)equivalenceClass.getRecord().get(j);
				int cellCount = record.getCells().size();
								
				String rowContents[] = new String[datapool.getVariables().size() + 1];
				DPLCell[] rowData = new DPLCell[datapool.getVariables().size()];
				rowContents[0] = equivalenceClass.getName() + TestUIPlugin.getString("DATA_EDT_DIVIDER") + String.valueOf(j);  //$NON-NLS-1$

				for(int k = 0; k < cellCount; k++)
				{
					DPLCell cell = (DPLCell)record.getCells().get(k);
					DPLVariable cellVariable = cell.getVariable();
					int index = findColumnIndex(cellVariable.getId());
					rowContents[index] = unwrapLiteral(cell.getValue());
					rowData[index - 1] = cell;
				}				
							
				TableItem item = new TableItem(table, SWT.NULL);
				item.setText(rowContents);
				item.setData(TAG_DATA, rowData);
				item.setData(TAG_RECORD, record); 
				item.setData(TAG_EQUIVALENCE_CLASS, equivalenceClass); 
			}	
		}
	}

	/*
	 * Checks to see if the datapool is empty of equivalence classes causing
	 * the table to be without rows.
	 */
	private boolean isWithoutRows()
	{
		if(datapool.getEquivalenceClasses().size() > 0)
		{
			if(datapool.getEquivalenceClasses().size() == 1)
			{
				DPLEquivalenceClass equivalenceClass = (DPLEquivalenceClass)datapool.getEquivalenceClasses().get(0);
				if(equivalenceClass.getRecord().size() == 0)
					return true;
				else
					return false;
			}
			else
				return false;
			
		}
		else
		{
			return true;
		}
	}

	/*
	 * Checks to see if the datapool is empty of variables causing the table 
	 * to be without columns.
	 */
	private boolean isWithoutColumns()
	{
		if(datapool.getVariables().size() == 0)
			return true;
		else
			return false;
	}
	
	/*
	 * Finds the column index from the varible's ID.
	 */
	private int findColumnIndex(String variableID)
	{
		for(int i = 0; i < table.getColumnCount(); i++)
		{
			TableColumn tableColumn = table.getColumn(i);
			DPLVariable variable = (DPLVariable)tableColumn.getData(TAG_VARIABLE); 
			if(variable != null && variableID.equals(variable.getId()))
				return i;	
		}	
		return -1;
	}

	/*
	 * Sets the scroll bars location when the cursor or selection moves.
	 */
	private void setScrollBars() {
		Point cursorPoint = cursor.getLocation();		
		Composite greatGrandParent = parent.getParent().getParent();
		if(greatGrandParent != null && greatGrandParent instanceof ScrolledComposite) 
		{
			int scrollableComponentHeight = greatGrandParent.getBounds().height;
			int scrollableComponentWidth = greatGrandParent.getBounds().width;
			
			int xAdjustment = cursorPoint.x - scrollableComponentWidth + cursor.getBounds().width + 30;
			int yAdjustment = cursorPoint.y - scrollableComponentHeight + cursor.getBounds().height + 53;
			((ScrolledComposite)greatGrandParent).setOrigin(xAdjustment, yAdjustment);
		}
	}

	/*
	 * Resize the scrollbar by xChange and yChange.
	 */
	private void resizeScrollBars()
	{
		try
		{
			Composite greatGrandParent = parent.getParent().getParent();
						
			int currentHorizontalMaximum = greatGrandParent.getHorizontalBar().getMaximum();
			int currentVerticalMaximum = greatGrandParent.getVerticalBar().getMaximum();
	
			int scrollableComponentHeight = greatGrandParent.getBounds().height;
			int scrollableComponentWidth = greatGrandParent.getBounds().width;
	
			int tableWidth = table.getBounds().width;
			int tableHeight = table.getBounds().height;
			
			int horizontalAdjust = 0;
			int verticalAdjust = 68;
			for(int i = 0; i < table.getColumnCount(); i++)
				horizontalAdjust += table.getColumn(i).getWidth();
			for(int i = 0; i < table.getItemCount(); i++)
			{
				TableItem item = table.getItem(i);
				verticalAdjust += item.getBounds(0).height + 1;
			}
	
			if(horizontalAdjust < scrollableComponentWidth)
				greatGrandParent.getHorizontalBar().setVisible(false);
			else
			{
				greatGrandParent.getHorizontalBar().setVisible(true);
				
				if(currentHorizontalMaximum < 119)
					greatGrandParent.getHorizontalBar().setMaximum(horizontalAdjust - scrollableComponentWidth + 44);
				else
					greatGrandParent.getHorizontalBar().setMaximum(horizontalAdjust + 12);
			}

			if(verticalAdjust < scrollableComponentHeight)
				greatGrandParent.getVerticalBar().setVisible(false);
			else			
			{
				greatGrandParent.getVerticalBar().setVisible(true);

				if(currentVerticalMaximum < 119)
					greatGrandParent.getVerticalBar().setMaximum(verticalAdjust - scrollableComponentHeight + 58);
				else
					greatGrandParent.getVerticalBar().setMaximum(verticalAdjust - 12);
			}

		}
		catch(Exception e)
		{
		}
	}

	/*
	 * Resets the scroll bar upon changes in control size.  The boolean 
	 * indicates if the resize request is from the container of the table or 
	 * from an internal table element.
	 */
	private void resetScrollBarSize(boolean container)
	{
		try
		{
			Composite greatGrandParent = table.getParent().getParent().getParent();	
			int horizontalMax = greatGrandParent.getHorizontalBar().getMaximum();
			int verticalMax = greatGrandParent.getVerticalBar().getMaximum();
	
			int scrollableComponentHeight = greatGrandParent.getBounds().height;
			int scrollableComponentWidth = greatGrandParent.getBounds().width;
	
			int horizontalAdjust = 0;
			int verticalAdjust = 0;
			for(int i = 0; i < table.getColumnCount(); i++)
				horizontalAdjust += table.getColumn(i).getWidth();
			for(int i = 0; i < table.getItemCount(); i++)
			{
				TableItem item = table.getItem(i);
				verticalAdjust += item.getBounds(0).height + 1;
			}
	
			if(horizontalAdjust + 14 < scrollableComponentWidth)
				horizontalAdjust = 0;
			if(verticalAdjust < scrollableComponentHeight)
				verticalAdjust = 0;
	
			if(container)
			{
				greatGrandParent.getHorizontalBar().setMaximum(horizontalMax + horizontalAdjust - MAX_TABLE_WIDTH - 26);
				greatGrandParent.getVerticalBar().setMaximum(verticalMax + verticalAdjust - MAX_TABLE_HEIGHT - 4);
			}
			else
			{
				if(horizontalMax < 119)
					greatGrandParent.getHorizontalBar().setMaximum(horizontalAdjust - scrollableComponentWidth + 44);
				else
					greatGrandParent.getHorizontalBar().setMaximum(horizontalAdjust + 12);

				if(verticalMax < 119)
					greatGrandParent.getVerticalBar().setMaximum(verticalAdjust - scrollableComponentHeight + 58);
				else
					greatGrandParent.getVerticalBar().setMaximum(verticalAdjust - 12);

//				greatGrandParent.getHorizontalBar().setMaximum(horizontalAdjust + 14);
//				greatGrandParent.getVerticalBar().setMaximum(verticalAdjust);				
			}
		}
		catch(Exception e)
		{
		}
	}

	// SelectionListener
	
	/* 
	 * @see org.eclipse.swt.events.SelectionListener#widgetSelected(org.eclipse.swt.events.SelectionEvent)
	 */
	public void widgetSelected(SelectionEvent e) 
	{
		int selectedRowIndex = table.getSelectionIndex();
		if(cursor != null && !cursor.isDisposed())
		{
			int selectedColumnIndex = cursor.getColumn();
			if(selectedColumnIndex >= 0)
				cursor.setSelection(selectedRowIndex, selectedColumnIndex);
		}		
	}

	/* 
	 * @see org.eclipse.swt.events.SelectionListener#widgetDefaultSelected(org.eclipse.swt.events.SelectionEvent)
	 */
	public void widgetDefaultSelected(SelectionEvent e) 
	{
	}
	

	// Context Menu

	private InsertRowAction insertRowAction = null;	
	private DeleteRowAction deleteRowAction = null;
 	private EditRowAction editRowAction = null;
	private InsertColumnAction insertColumnAction = null;
	private DeleteColumnAction deleteColumnAction = null;
 	private EditColumnAction editColumnAction = null;
	private InsertRowGroupAction insertRowGroupAction = null;
	private DeleteRowGroupAction deleteRowGroupAction = null;
	private EditRowGroupAction editRowGroupAction = null;
	
	/*
	 * Creates all the actions
	 */
	private void makeActions()
	{
		ISelectionProvider provider = viewer;
		
		insertRowAction = new InsertRowAction(provider, this);
		deleteRowAction = new DeleteRowAction(provider, this);
		editRowAction = new EditRowAction(provider, this);
		insertColumnAction = new InsertColumnAction(provider, this);
		deleteColumnAction = new DeleteColumnAction(provider, this);
		editColumnAction = new EditColumnAction(provider, this);		
		insertRowGroupAction = new InsertRowGroupAction(provider, this);
		deleteRowGroupAction = new DeleteRowGroupAction(provider, this);
		editRowGroupAction = new EditRowGroupAction(provider, this);

	}
	
		
	// IMenuListener
		
	/**
 	* see menuAboutToShow(org.eclipse.jface.action.IMenuManager) in org.eclipse.jface.action.IMenuListener
	*/
	public void menuAboutToShow(IMenuManager imenuMgr)
	{
		MenuManager menuMgr = (MenuManager)imenuMgr;

		// Add actions to menu.
		menuMgr.add(insertRowGroupAction);				
		menuMgr.add(deleteRowGroupAction);				
		menuMgr.add(editRowGroupAction);				
		menuMgr.add(insertRowAction);				
		menuMgr.add(deleteRowAction);				
		menuMgr.add(editRowAction);				
		menuMgr.add(insertColumnAction);				
		menuMgr.add(deleteColumnAction);
		menuMgr.add(editColumnAction);

		// Set the mode.
		int menuMode = 0;
		if(cursor != null && !cursor.isDisposed())
			if(cursor.getColumn() == 0)
				menuMode = 0;
			else
				menuMode = 1;
		if(table.getSelectionIndex() == -1)
			menuMode = 2;
		if(datapool.getEquivalenceClasses().size() == 0)
			menuMode = 3;		
		if(datapool.getEquivalenceClasses().size() == 0 && datapool.getVariables().size() == 0)
			menuMode = 4;		
		if(extension.getHyadesEditorPart().isReadOnly())
			menuMode = 5;

		// Based on the mode, enable certain menu items.
		switch(menuMode)
		{
			case 0: 
				insertRowAction.aboutToShow(true);
				deleteRowAction.aboutToShow(true);
				editRowAction.aboutToShow(true);
				insertRowGroupAction.aboutToShow(true);
				deleteRowGroupAction.aboutToShow(true);
				editRowGroupAction.aboutToShow(true);
				insertColumnAction.aboutToShow(true);
				deleteColumnAction.aboutToShow(false);
				editColumnAction.aboutToShow(true);
				break;
			case 1:
				insertRowAction.aboutToShow(true);
				deleteRowAction.aboutToShow(true);
				editRowAction.aboutToShow(true);
				insertRowGroupAction.aboutToShow(true);
				deleteRowGroupAction.aboutToShow(true);
				editRowGroupAction.aboutToShow(true);
				insertColumnAction.aboutToShow(true);
				deleteColumnAction.aboutToShow(true);
				editColumnAction.aboutToShow(true);
				break;
			case 2:
			case 3:
			case 4:
				insertRowAction.aboutToShow(false);
				deleteRowAction.aboutToShow(false);
				editRowAction.aboutToShow(false);
				insertRowGroupAction.aboutToShow(true);
				deleteRowGroupAction.aboutToShow(false);
				editRowGroupAction.aboutToShow(false);
				insertColumnAction.aboutToShow(true);
				deleteColumnAction.aboutToShow(false);
				editColumnAction.aboutToShow(false);
				break;
			case 5:
				insertRowAction.aboutToShow(false);
				deleteRowAction.aboutToShow(false);
				editRowAction.aboutToShow(false);
				insertRowGroupAction.aboutToShow(false);
				deleteRowGroupAction.aboutToShow(false);
				editRowGroupAction.aboutToShow(false);
				insertColumnAction.aboutToShow(false);
				deleteColumnAction.aboutToShow(false);
				editColumnAction.aboutToShow(false);
				break;
			default:
				insertRowAction.aboutToShow(true);
				deleteRowAction.aboutToShow(true);
				editRowAction.aboutToShow(true);
				insertRowGroupAction.aboutToShow(true);
				deleteRowGroupAction.aboutToShow(true);
				editRowGroupAction.aboutToShow(true);
				insertColumnAction.aboutToShow(true);
				deleteColumnAction.aboutToShow(true);
				editColumnAction.aboutToShow(true);
				break;
		}

	}
	
	// Action handlers
	
	/*
	 * Inserts a new equivalence class/row group to the table.  The newly
	 * created equivalence class will have one record.  Brings up a dialog to 
	 * gather information about the equivalence class/row group.  Adds the 
	 * equivalence class to the datapool object.  Inserts a row in the table.
	 */
	public void insertRowGroup()
	{
		TableItem previousTableItem = null;
		if(cursor != null && !cursor.isDisposed())
			previousTableItem = cursor.getRow();
		DPLEquivalenceClass previousEquivalenceClass = null;
		if(previousTableItem != null)
			previousEquivalenceClass = (DPLEquivalenceClass)previousTableItem.getData(TAG_EQUIVALENCE_CLASS); 

		DataPoolRowGroupDialog dialog = new DataPoolRowGroupDialog(Display.getCurrent().getActiveShell(), datapool, table, null, previousEquivalenceClass, TestUIPlugin.getString("DATA_ROW_GRP_DLG_TITLE_INS"));  //$NON-NLS-1$
		if (dialog.open() == IDialogConstants.OK_ID)
		{
			setWaitCursor();			
			String name = dialog.getName();
			if(name != null && name.length() != 0)
			{
				DPLEquivalenceClass equivalenceClass = Common_DatapoolFactory.eINSTANCE.createDPLEquivalenceClass();
				equivalenceClass.setName(name);	
				DPLRecord record = Common_DatapoolFactory.eINSTANCE.createDPLRecord();
				record.setEquivalenceClass(equivalenceClass);
				int insertionIndex = dialog.getInsertionIndex();
				datapool.getEquivalenceClasses().add(insertionIndex + 1, equivalenceClass);
				equivalenceClass.setDatapool(datapool);
				equivalenceClassAdded(datapool, insertionIndex);
			}
			unsetWaitCursor();
		}
	}

	/*
	 * Deletes the currently selected equivalence class/row group.  Removes 
	 * the equivalence class and all assocaited records from the datapool 
	 * object.  Removes the rows from the table.
	 */
	public void deleteRowGroup()
	{
		TableItem tableItem = null;
		if(cursor != null && !cursor.isDisposed())
			tableItem = cursor.getRow();
		else
			return;
			
		setWaitCursor();
		DPLEquivalenceClass equivalenceClass = (DPLEquivalenceClass)tableItem.getData(TAG_EQUIVALENCE_CLASS);
		int equivalenceClassIndex = datapool.getEquivalenceClasses().indexOf(equivalenceClass);
		datapool.getEquivalenceClasses().remove(equivalenceClass);
		equivalenceClassRemoved(datapool, equivalenceClassIndex);
		unsetWaitCursor();
	}

	/*
	 * Edits the currently selected equivalence class/row group.  Renames 
	 * and/or moves the equivalence class in the datapool object.  Updates the
	 * table to reflect the edit and/or move.
	 */
	public void editRowGroup()
	{
		TableItem tableItem = null;
		if(cursor != null && !cursor.isDisposed())
			tableItem = cursor.getRow();
		else
			return;
		editRowGroupAux(tableItem);
	}

	/*
	 * Function that actually does the editing of the equivalence class/row group.
	 */
	private void editRowGroupAux(TableItem tableItem)
	{
		DPLEquivalenceClass selectedEquivalenceClass = null;
		if(tableItem != null)
			selectedEquivalenceClass = (DPLEquivalenceClass)tableItem.getData(TAG_EQUIVALENCE_CLASS);
		int selectedEquivalenceClassIndex = -1;
		if(selectedEquivalenceClass != null)
			selectedEquivalenceClassIndex = datapool.getEquivalenceClasses().indexOf(selectedEquivalenceClass);
		DPLEquivalenceClass previousEquivalenceClass = null;
		if(selectedEquivalenceClassIndex != -1 && selectedEquivalenceClassIndex != 0)
			previousEquivalenceClass = (DPLEquivalenceClass)datapool.getEquivalenceClasses().get(selectedEquivalenceClassIndex - 1);

		DataPoolRowGroupDialog dialog = new DataPoolRowGroupDialog(Display.getCurrent().getActiveShell(), datapool, table,  selectedEquivalenceClass, previousEquivalenceClass, TestUIPlugin.getString("DATA_ROW_GRP_DLG_TITLE_EDIT")); //$NON-NLS-1$
		if (dialog.open() == IDialogConstants.OK_ID)
		{
			setWaitCursor();
			String name = dialog.getName();
			if(!selectedEquivalenceClass.getName().equals(name))
			{
				selectedEquivalenceClass.setName(name);	
				equivalenceClassChanged(datapool, selectedEquivalenceClassIndex);
			}
			int insertionIndex = dialog.getInsertionIndex();
			if(insertionIndex == selectedEquivalenceClassIndex - 1)
			{
				unsetWaitCursor();
				return;
			}
			equivalenceClassMoved(datapool, selectedEquivalenceClassIndex, insertionIndex);
			if(insertionIndex < selectedEquivalenceClassIndex)
				datapool.getEquivalenceClasses().move(insertionIndex + 1, selectedEquivalenceClassIndex);
			else
				datapool.getEquivalenceClasses().move(insertionIndex, selectedEquivalenceClassIndex);
			unsetWaitCursor();				
		}
	}

	/*
	 * Inserts a new record/row to the table.  Brings up a dialog to gather 
	 * information about the where to place the row.  Adds the record to the 
	 * datapool object.  Inserts a row in the table.
	 */
	public void insertRow()
	{
		TableItem previousTableItem = null;
		if(cursor != null && !cursor.isDisposed())
			previousTableItem = cursor.getRow();
		else
			return;

		DataPoolRowDialog dialog = new DataPoolRowDialog(Display.getCurrent().getActiveShell(), datapool, table, null, previousTableItem, TestUIPlugin.getString("DATA_ROW_DLG_TITLE_INS"));  //$NON-NLS-1$
		if (dialog.open() == IDialogConstants.OK_ID)
		{
			setWaitCursor();
			int insertionEquivalenceClassIndex = dialog.getInsertionEquivalenceClassIndex();
			int insertionRecordIndex = dialog.getInsertionRecordIndex();
			if(insertionEquivalenceClassIndex != -1 && insertionRecordIndex != -1)
			{
				DPLEquivalenceClass equivalenceClass = (DPLEquivalenceClass)datapool.getEquivalenceClasses().get(insertionEquivalenceClassIndex);
				DPLRecord record = Common_DatapoolFactory.eINSTANCE.createDPLRecord();
				equivalenceClass.getRecord().add(insertionRecordIndex + 1, record);
				record.setEquivalenceClass(equivalenceClass);
				recordAdded(datapool, insertionEquivalenceClassIndex, insertionRecordIndex);
			}
			else
			{
				DPLEquivalenceClass equivalenceClass = (DPLEquivalenceClass)datapool.getEquivalenceClasses().get(insertionEquivalenceClassIndex);
				DPLRecord record = Common_DatapoolFactory.eINSTANCE.createDPLRecord();
				equivalenceClass.getRecord().add(0, record);
				record.setEquivalenceClass(equivalenceClass);
				recordAdded(datapool, insertionEquivalenceClassIndex, insertionRecordIndex);
			}
			unsetWaitCursor();		
		}
	}

	/*
	 * Deletes the currently selected record/row.  Removes the record from the 
	 * datapool object.  Removes the row from the table.
	 */
	public void deleteRow()
	{
		TableItem tableItem = null;
		if(cursor != null && !cursor.isDisposed())
			tableItem = cursor.getRow();
		else
			return;

		setWaitCursor();
		DPLEquivalenceClass equivalenceClass = (DPLEquivalenceClass)tableItem.getData(TAG_EQUIVALENCE_CLASS);
		DPLRecord record = (DPLRecord)tableItem.getData(TAG_RECORD);
		int equivalenceClassIndex = datapool.getEquivalenceClasses().indexOf(equivalenceClass);
		int recordIndex = equivalenceClass.getRecord().indexOf(record);		
		equivalenceClass.getRecord().remove(record);
		if(equivalenceClass.getRecord().size() == 0)
		{
			datapool.getEquivalenceClasses().remove(equivalenceClass);
			equivalenceClassRemoved(datapool, equivalenceClassIndex);			
		}
		else
			recordRemoved(datapool, equivalenceClassIndex, recordIndex);			
		unsetWaitCursor();
	}

	/*
	 * Edits the currently selected record/row.  Moves the record in the 
	 * datapool object.  Updates the table to reflect move.
	 */
	public void editRow()
	{
		TableItem tableItem = null;
		if(cursor != null && !cursor.isDisposed())
			tableItem = cursor.getRow();
		else
			return;
		editRowAux(tableItem);		
	}

	/*
	 * Function that actually does the editing of the record/row.
	 */
	private void editRowAux(TableItem tableItem)
	{
		int rowIndex = table.getSelectionIndex();
		TableItem previousTableItem = null;
		if(rowIndex != 0)
			previousTableItem = table.getItem(rowIndex - 1);

		DataPoolRowDialog dialog = new DataPoolRowDialog(Display.getCurrent().getActiveShell(), datapool, table,  tableItem, previousTableItem, TestUIPlugin.getString("DATA_ROW_DLG_TITLE_EDIT"));  //$NON-NLS-1$
		if (dialog.open() == IDialogConstants.OK_ID)
		{
			setWaitCursor();
			DPLEquivalenceClass equivalenceClass = (DPLEquivalenceClass)tableItem.getData(TAG_EQUIVALENCE_CLASS);
			int equivalenceClassIndex = datapool.getEquivalenceClasses().indexOf(equivalenceClass);
			DPLRecord record = (DPLRecord)tableItem.getData(TAG_RECORD);
			int recordIndex = equivalenceClass.getRecord().indexOf(record);

			int insertionEquivalenceClassIndex = dialog.getInsertionEquivalenceClassIndex();
			int insertionRecordIndex = dialog.getInsertionRecordIndex();
			if(equivalenceClassIndex == insertionEquivalenceClassIndex)
			{
				if(recordIndex != insertionRecordIndex + 1)
				{
					if(insertionRecordIndex > recordIndex)
					{
						recordMoved(datapool, equivalenceClassIndex, recordIndex, insertionRecordIndex);
						equivalenceClass.getRecord().move(insertionRecordIndex, recordIndex);	
					}
					else
					{
						recordMoved(datapool, equivalenceClassIndex, recordIndex, insertionRecordIndex + 1);
						equivalenceClass.getRecord().move(insertionRecordIndex + 1, recordIndex);							
					}

				}
			}
			else
			{
				if(insertionEquivalenceClassIndex != -1 && insertionRecordIndex != -1)
				{
					DPLEquivalenceClass targetEquivalenceClass = (DPLEquivalenceClass)datapool.getEquivalenceClasses().get(insertionEquivalenceClassIndex);
					targetEquivalenceClass.getRecord().add(insertionRecordIndex + 1, record);
					if(equivalenceClassIndex > insertionEquivalenceClassIndex)
					{
						recordAdded(datapool, insertionEquivalenceClassIndex, insertionRecordIndex);
						recordRemoved(datapool, equivalenceClassIndex, recordIndex);				
					}
					else
					{
						recordRemoved(datapool, equivalenceClassIndex, recordIndex);				
						recordAdded(datapool, insertionEquivalenceClassIndex, insertionRecordIndex);
					}
					if(equivalenceClass.getRecord().size() == 0)
						datapool.getEquivalenceClasses().remove(equivalenceClass);	
				}
				else
				{
					DPLEquivalenceClass targetEquivalenceClass = (DPLEquivalenceClass)datapool.getEquivalenceClasses().get(insertionEquivalenceClassIndex);
					targetEquivalenceClass.getRecord().add(0, record);
					if(equivalenceClassIndex > insertionEquivalenceClassIndex)
					{
						recordAdded(datapool, insertionEquivalenceClassIndex, insertionRecordIndex);							
						recordRemoved(datapool, equivalenceClassIndex, recordIndex);				
					}
					else
					{
						recordRemoved(datapool, equivalenceClassIndex, recordIndex);				
						recordAdded(datapool, insertionEquivalenceClassIndex, insertionRecordIndex);							
					}
					if(equivalenceClass.getRecord().size() == 0)
						datapool.getEquivalenceClasses().remove(equivalenceClass);	
				}
			}
			unsetWaitCursor();		
		}
	}

	/*
	 * Inserts a new variable/column to the table.  Brings up a dialog to 
	 * gather information about the variable/column.  Adds the variable to the 
	 * datapool object.  Inserts a column in the table.
	 */
	public void insertColumn()
	{
		int selectedColumnIndex = -1;
		if(cursor != null && !cursor.isDisposed())
			selectedColumnIndex = cursor.getColumn();
		TableColumn previousTableColumn = null;
		if(selectedColumnIndex > 0)
			previousTableColumn = table.getColumn(selectedColumnIndex);
		DPLVariable previousVariable = null;
		if(previousTableColumn != null)
			previousVariable = (DPLVariable)previousTableColumn.getData(TAG_VARIABLE);

		DataPoolColumnDialog dialog = new DataPoolColumnDialog(Display.getCurrent().getActiveShell(), datapool, null, previousVariable, TestUIPlugin.getString("DATA_COL_DLG_TITLE_INS"));  //$NON-NLS-1$
		if ( dialog.open() == IDialogConstants.OK_ID)
		{
			setWaitCursor();
			DPLVariable variable = Common_DatapoolFactory.eINSTANCE.createDPLVariable();
			variable.setName(dialog.getName());
			variable.setType(dialog.getType());
			
			int insertionIndex = findColumnIndex(dialog.getInsertionVariableID());
			if(insertionIndex == -1)
				insertionIndex = 0;

			datapool.getVariables().add(insertionIndex, variable);
			variable.setDatapool(datapool);				
			variableAdded(datapool, insertionIndex);
			unsetWaitCursor();		
		}
	}

	/*
	 * Deletes the currently selected variable/column.  Removes the column and 
	 * all assocaited cells from the datapool object.  Removes the column from 
	 * the table.
	 */
	public void deleteColumn()
	{
		if(cursor == null || cursor.isDisposed())
			return;
		int selectedColumnIndex = cursor.getColumn();
		int selectedRowIndex = table.getSelectionIndex();
		// Don't delete the row name column.
		if (selectedColumnIndex < 1)	
			return;
		
		setWaitCursor();
		TableColumn tableColumn = table.getColumn(selectedColumnIndex);
		DPLVariable variable = (DPLVariable)tableColumn.getData(TAG_VARIABLE);
		datapool.getVariables().remove(variable);
		// loop through and remove cells
		TableItem[] rows = table.getItems();
		for(int i = 0; i < rows.length; i++)
		{
			DPLCell[] rowData = (DPLCell[])rows[i].getData(TAG_DATA);
			for(int j = 0; j < rowData.length; j++)
			{
				if(j == selectedColumnIndex -1)
				{
					DPLRecord record = (DPLRecord)rows[i].getData(TAG_RECORD);
					record.getCells().remove(rowData[j]);
				}
			}
		}
		variableRemoved(datapool, selectedColumnIndex);
		unsetWaitCursor();
	}
	
	/*
	 * Edits the currently selected variable/column.  Renames and/or moves the 
	 * variable in the datapool object.  Updates the table to reflect the edit 
	 * and/or move.
	 */
	public void editColumn()
	{
		int selectedColumnIndex = cursor.getColumn();
		TableColumn tableColumn = (TableColumn) table.getColumn(selectedColumnIndex);
		editColumnAux(tableColumn);		
	}

	/*
	 * Function that actually does the editing of the variable/column.
	 */	
	private void editColumnAux(TableColumn tableColumn)
	{
		int columnIndex = table.indexOf(tableColumn);
		TableColumn previousTableColumn = null;
		if(columnIndex != 0)
			previousTableColumn = table.getColumn(columnIndex-1);
		DPLVariable previousVariable = null;
		if(previousTableColumn != null)
			previousVariable = (DPLVariable)previousTableColumn.getData(TAG_VARIABLE); 
		DPLVariable variable = (DPLVariable)tableColumn.getData(TAG_VARIABLE); 
		DataPoolColumnDialog dialog = new DataPoolColumnDialog(Display.getCurrent().getActiveShell(), datapool, variable, previousVariable, TestUIPlugin.getString("DATA_COL_DLG_TITLE_EDIT"));  //$NON-NLS-1$
		if ( dialog.open() == IDialogConstants.OK_ID)
		{
			setWaitCursor();
			String name = dialog.getName();
			String type = dialog.getType();
			String insertionVariableID = dialog.getInsertionVariableID();
			String previousVariableID = new String();
			if(previousVariable != null)
				previousVariableID = previousVariable.getId();
			if(name.equals(variable.getName()) && 
				type.equals(variable.getType()) &&
				insertionVariableID.equals(previousVariableID))
			{
				unsetWaitCursor();
				return;	
			}
			variable.setName(dialog.getName());
			variable.setType(dialog.getType());
			Integer variableIndex = (Integer)tableColumn.getData(TAG_VARIABLE_INDEX); 
			variableChanged(datapool, variableIndex.intValue());

			int insertionIndex = findColumnIndex(dialog.getInsertionVariableID());
			if(insertionIndex == columnIndex - 1)
			{
				unsetWaitCursor();
				return;
			}
			if(insertionIndex == -1)
				datapool.getVariables().move(0, columnIndex - 1);
			else
				if(insertionIndex > columnIndex)
					datapool.getVariables().move(insertionIndex - 1, columnIndex - 1);
				else
					datapool.getVariables().move(insertionIndex, columnIndex - 1);

			if(insertionIndex > columnIndex)
				variableMoved(datapool, columnIndex, insertionIndex);
			else
			{
				if(insertionIndex == -1)
					insertionIndex = 0;	
				variableMoved(datapool, columnIndex, insertionIndex + 1);
			}
			unsetWaitCursor();				
		}
	}
	
	/* 
	 * Clears the value in cell.
	 */
	public void clearCell()
	{
		applyEditingValue(new String());
	}

	/*
	 * Wraps the text in tags that indicate the text is a literal.  All cell 
	 * values in this release are literals.  Returns the wrapped string.
	 */	
	private String wrapAsLiterl(String text)
	{
		if(text != null)
			return START_LITERAL + text + END_LITERAL;
		else
			return START_LITERAL + END_LITERAL;
	}
	
	/*
	 * Takes a literal wrapped string and stips the literal tags.  Returns the
	 * unwrapped string.
	 */
	private String unwrapLiteral(String text)
	{
		text = text.trim();
		if(text.startsWith(START_LITERAL) && text.endsWith(END_LITERAL))
		{
			text = text.substring(START_LITERAL.length(), text.length());
			int endIndex = text.lastIndexOf(END_LITERAL);
			text = text.substring(0, endIndex);
			return text;	
		}
		else
			return new String();
		
	}
	
	// IDatapoolListener
	
	/*
	 * The following methods follow a spec that will eventually correspond to a
	 * listener interface that will make callbacks to the Datapool table.  
	 * Based on the callback, the table will update accordingly.  
	 */

	public void variableAdded(DPLDatapool datapool, int variableIndex) 
	{	
		TableColumn tableColumn = new TableColumn(table, SWT.NONE, variableIndex + 1);
		DPLVariable variable = (DPLVariable)datapool.getVariables().get(variableIndex);
		tableColumn.setResizable(true);
		tableColumn.setText(variable.getName() + TestUIPlugin.getString("DATA_EDT_DIVIDER") + variable.getType());  //$NON-NLS-1$
		tableColumn.setData(TAG_VARIABLE, variable);
		tableColumn.setData(TAG_VARIABLE_INDEX, new Integer(variableIndex));
		tableColumn.addSelectionListener(headerListener);
		tableColumn.setWidth(100);
		tableColumn.addSelectionListener(headerListener);
		tableColumn.addControlListener(resizeColumnListener);		
		tableUtil.insertColumn(tableColumn, variableIndex);	

		if(cursor != null && !cursor.isDisposed())
		{
			int selectedRowIndex = table.getSelectionIndex();
			if(selectedRowIndex >= 0)
			{
				table.setSelection(selectedRowIndex);
				cursor.setSelection(selectedRowIndex, variableIndex + 1);
				resizeScrollBars();
				setScrollBars();
			}

		}
		extension.markDirty();		
	}	
	
	public void variableRemoved(DPLDatapool datapool, int variableIndex)
	{
		tableUtil.deleteColumn(variableIndex);

		if(cursor != null && !cursor.isDisposed())
		{
			if(variableIndex > 0)
			{
				int selectedRowIndex = table.getSelectionIndex();
				if(selectedRowIndex >= 0)
				{
					table.select(selectedRowIndex);
					cursor.setSelection(selectedRowIndex, variableIndex -1);
					resizeScrollBars();
					setScrollBars();
				}
			}
			else
				table.deselectAll();

		}
		extension.markDirty();		
	}
	
	public void variableMoved(DPLDatapool datapool, int beforeVariableIndex, int afterVariableIndex)
	{
		tableUtil.moveColumn(beforeVariableIndex, afterVariableIndex);
		if(cursor != null && !cursor.isDisposed())
			if(afterVariableIndex > 0)
			{
				int selectedIndex = table.getSelectionIndex();
				if(selectedIndex >= 0)
				{
					table.select(selectedIndex);
					cursor.setSelection(selectedIndex, afterVariableIndex);
					setScrollBars();
				}
			}
		extension.markDirty();		
	}
	
	public void variableChanged(DPLDatapool datapool, int variableIndex)
	{
		DPLVariable variable = (DPLVariable)datapool.getVariables().get(variableIndex);
		int columnIndex = findColumnIndex(variable.getId());
		TableColumn tableColumn = table.getColumn(columnIndex);
		tableColumn.setText(variable.getName() + TestUIPlugin.getString("DATA_EDT_DIVIDER") + variable.getType());  //$NON-NLS-1$
		tableColumn.setData(TAG_VARIABLE, variable);
		tableColumn.setData(TAG_VARIABLE_INDEX, new Integer(variableIndex));
		extension.markDirty();		
	}
	
	public void equivalenceClassChanged(DPLDatapool datapool, int equivalenceClassIndex)
	{
		DPLEquivalenceClass equivalenceClass = (DPLEquivalenceClass)datapool.getEquivalenceClasses().get(equivalenceClassIndex);
		
		for(int i = 0; i < equivalenceClass.getRecord().size(); i++)
		{
			// Assumes rows are synced with datapool
			int tableRowIndex = findRowIndex(equivalenceClassIndex, i);
			TableItem tableItem = table.getItem(tableRowIndex);
			tableItem.setText(0, equivalenceClass.getName() + TestUIPlugin.getString("DATA_EDT_DIVIDER") + String.valueOf(i)); //$NON-NLS-1$
			extension.markDirty();		
		}
	}
	
	public void equivalenceClassAdded(DPLDatapool datapool, int afterEquivalenceClassIndex)
	{
		DPLEquivalenceClass equivalenceClass = (DPLEquivalenceClass)datapool.getEquivalenceClasses().get(afterEquivalenceClassIndex + 1);
		DPLEquivalenceClass previousEquivalenceClass = null;
		int insertionIndex = 0;
		if(afterEquivalenceClassIndex >= 0)
		{
			previousEquivalenceClass = (DPLEquivalenceClass)datapool.getEquivalenceClasses().get(afterEquivalenceClassIndex);
			insertionIndex = findRowIndex(afterEquivalenceClassIndex, previousEquivalenceClass.getRecord().size() - 1) + 1;
		}

		for(int j = 0; j < equivalenceClass.getRecord().size(); j++)
		{
			DPLRecord record = (DPLRecord)equivalenceClass.getRecord().get(j);
			String rowContents[] = new String[table.getColumnCount()];
			DPLCell[] rowData = new DPLCell[table.getColumnCount() - 1];
			rowContents[0] = equivalenceClass.getName() + TestUIPlugin.getString("DATA_EDT_DIVIDER") + String.valueOf(j); //$NON-NLS-1$
			for(int k = 0; k < record.getCells().size(); k++)
			{
				DPLCell cell = (DPLCell)record.getCells().get(k);
				DPLVariable cellVariable = cell.getVariable();
				int index = findColumnIndex(cellVariable.getId());
				rowContents[index] = unwrapLiteral(cell.getValue());	
				rowData[index - 1] = cell;
			}
												
			TableItem item = new TableItem(table, SWT.NULL, insertionIndex + j);
			item.setText(rowContents);
			item.setData(TAG_DATA, rowData);
			item.setData(TAG_RECORD, record); 
			item.setData(TAG_EQUIVALENCE_CLASS, equivalenceClass);
		}
		if(table.getItemCount() == 1)
			createCursor();		
		if(cursor != null && !cursor.isDisposed() && table != null)
		{
			table.setSelection(insertionIndex);
			cursor.setSelection(insertionIndex, 0);
			resizeScrollBars();
			setScrollBars();	
		}
		extension.markDirty();	
	}

	public void equivalenceClassRemoved(DPLDatapool datapool, int equivalenceClassIndex)
	{
		// Assumes rows are synced with datapool
		int tableRowIndex = findRowIndex(equivalenceClassIndex, 0);
		TableItem tableItem = table.getItem(tableRowIndex);
		DPLEquivalenceClass equivalenceClass = (DPLEquivalenceClass)tableItem.getData(TAG_EQUIVALENCE_CLASS);
		boolean removeRow = true;
		while(removeRow)
		{
			table.remove(tableRowIndex);
			if(tableRowIndex < table.getItemCount())
			{
				tableItem = table.getItem(tableRowIndex);
				DPLEquivalenceClass nextEquivalenceClass = (DPLEquivalenceClass)tableItem.getData(TAG_EQUIVALENCE_CLASS);
				if(!nextEquivalenceClass.equals(equivalenceClass))
					removeRow = false;
			}
			else
				removeRow = false;
		}
		extension.markDirty();		
		if(cursor != null && !cursor.isDisposed() && table != null)
		{
			int tableSize = table.getItemCount();
			if(tableSize == 0)
			{
				table.deselectAll();
				cursor.dispose();	
				return;								
			}
			if(tableSize > tableRowIndex)
			{
				table.setSelection(tableRowIndex);
				cursor.setSelection(tableRowIndex, 0);
				resizeScrollBars();				
				setScrollBars();
				return;
			}
			if(tableSize > tableRowIndex - 1)	
			{
				table.setSelection(tableRowIndex - 1);
				cursor.setSelection(tableRowIndex - 1, 0);
				resizeScrollBars();
				setScrollBars();
			}
		}		
	}
	
	public void equivalenceClassMoved(DPLDatapool datapool, int beforeEquivalenceClassIndex, int afterEquivalenceClassIndex)
	{
		if(beforeEquivalenceClassIndex == afterEquivalenceClassIndex)
			return;

		int targetIndex = 0;
		if(afterEquivalenceClassIndex != -1)
		{
			DPLEquivalenceClass targetEquivalenceClass = (DPLEquivalenceClass)datapool.getEquivalenceClasses().get(afterEquivalenceClassIndex);
			int targetRecordCount = targetEquivalenceClass.getRecord().size();
			if(beforeEquivalenceClassIndex < afterEquivalenceClassIndex)
				targetRecordCount -= 1;
			targetIndex = findRowIndex(afterEquivalenceClassIndex, targetRecordCount);
		}
		
		DPLEquivalenceClass sourceEquivalenceClass = (DPLEquivalenceClass)datapool.getEquivalenceClasses().get(beforeEquivalenceClassIndex);
		int sourceRecordCount = sourceEquivalenceClass.getRecord().size();
		int sourceIndex = findRowIndex(beforeEquivalenceClassIndex, 0);
		for(int i = 0; i < sourceEquivalenceClass.getRecord().size(); i++)
		{
			if(beforeEquivalenceClassIndex > afterEquivalenceClassIndex)
				tableUtil.moveRow(sourceIndex + sourceRecordCount - 1, targetIndex);
			else
				tableUtil.moveRow(sourceIndex, targetIndex);
		}
		if(cursor != null && !cursor.isDisposed() && table != null)
		{
			table.setSelection(targetIndex);
			cursor.setSelection(targetIndex, 0);
			setScrollBars();
		}		
		extension.markDirty();		
	}

	public void recordAdded(DPLDatapool datapool, int equivalenceClassIndex, int afterRecordIndex)
	{
		DPLEquivalenceClass equivalenceClass = (DPLEquivalenceClass)datapool.getEquivalenceClasses().get(equivalenceClassIndex);
		DPLRecord record = (DPLRecord)equivalenceClass.getRecord().get(afterRecordIndex + 1);
		String rowContents[] = new String[table.getColumnCount()];
		DPLCell[] rowData = new DPLCell[table.getColumnCount() - 1];
		rowContents[0] = equivalenceClass.getName() + TestUIPlugin.getString("DATA_EDT_DIVIDER") + String.valueOf(afterRecordIndex + 1); //$NON-NLS-1$
		for(int i = 0; i < record.getCells().size(); i++)
		{
			DPLCell cell = (DPLCell)record.getCells().get(i);
			DPLVariable cellVariable = cell.getVariable();
			int index = findColumnIndex(cellVariable.getId());
			rowContents[index] = unwrapLiteral(cell.getValue());	
			rowData[index - 1] = cell;			
		}
		int insertionIndex = findRowIndex(equivalenceClassIndex, afterRecordIndex + 1);
		TableItem tableItem = new TableItem(table, SWT.NULL, insertionIndex);
		tableItem.setText(rowContents);
		tableItem.setData(TAG_DATA, rowData);
		tableItem.setData(TAG_RECORD, record);
		tableItem.setData(TAG_EQUIVALENCE_CLASS, equivalenceClass);
		repopulateRowLabels(equivalenceClassIndex);
		if(cursor != null && !cursor.isDisposed() && table != null)
		{
			table.setSelection(insertionIndex);
			cursor.setSelection(insertionIndex, 0);
			resizeScrollBars();			
			setScrollBars();	
		}		
		extension.markDirty();		
	}
	
	public void recordRemoved(DPLDatapool datapool, int equivalenceClassIndex, int recordIndex)
	{
		int tableRowIndex = findRowIndex(equivalenceClassIndex, recordIndex);
		table.remove(tableRowIndex);
		repopulateRowLabels(equivalenceClassIndex);

		extension.markDirty();				
		if(cursor != null && !cursor.isDisposed() && table != null)
		{
			int tableSize = table.getItemCount();
			if(tableSize == 0)
			{
				table.deselectAll();
				cursor.dispose();	
				return;								
			}
			if(tableSize > tableRowIndex)
			{
				table.setSelection(tableRowIndex);
				cursor.setSelection(tableRowIndex, 0);
				resizeScrollBars();
				setScrollBars();
				return;
			}
			if(tableSize > tableRowIndex - 1)	
			{
				table.setSelection(tableRowIndex - 1);
				cursor.setSelection(tableRowIndex - 1, 0);
				resizeScrollBars();
				setScrollBars();
			}
		}		
	}
	
	public void recordMoved(DPLDatapool datapool, int equivalenceClassIndex, int beforeRecordIndex, int afterRecordIndex)
	{
		int sourceRowIndex = findRowIndex(equivalenceClassIndex, beforeRecordIndex);
		int targetRowIndex = findRowIndex(equivalenceClassIndex, afterRecordIndex);
		tableUtil.moveRow(sourceRowIndex, targetRowIndex);
		repopulateRowLabels(equivalenceClassIndex);
		extension.markDirty();		
	}
	
	public void cellChanged(DPLDatapool datapool, int equivalenceClassIndex, int recordIndex, int variableIndex)
	{
		DPLEquivalenceClass equivalenceClass = (DPLEquivalenceClass)datapool.getEquivalenceClasses().get(equivalenceClassIndex);
		DPLRecord record = (DPLRecord)equivalenceClass.getRecord().get(recordIndex);
		DPLVariable variable = (DPLVariable)datapool.getVariables().get(variableIndex);

		DPLCell cell = null;		
		for(int i = 0; i < record.getCells().size(); i++)
		{
			cell = (DPLCell)record.getCells().get(i);
			if(cell.getVariable().equals(variable))
				break;
			else
				cell = null;

		}

		if(cell != null)
		{
			String value = unwrapLiteral(cell.getValue());
			int tableRowIndex = findRowIndex(equivalenceClassIndex, recordIndex);
			TableItem tableItem = table.getItem(tableRowIndex);
			tableItem.setText(equivalenceClassIndex + 1, value);
			cursor.setSelection(tableRowIndex, equivalenceClassIndex + 1);
			extension.markDirty();
		}
		extension.markDirty();		
	}

	public void datapoolReordered(DPLDatapool datapool)
	{
		extension.markDirty();		
	}

	/*
	 * Finds the index of the row in the table that corresponds to the record
	 * at the given equivalence class and record index in the datapool object.
	 */
	private int findRowIndex(int equivalenceClassIndex, int recordIndex)
	{
		int count = recordIndex;
		for(int i = 0; i < equivalenceClassIndex; i++)
		{
			DPLEquivalenceClass equivalenceClass = (DPLEquivalenceClass)datapool.getEquivalenceClasses().get(i);
			count += equivalenceClass.getRecord().size();
		}
		return count;
	}

	/*
	 * Recreates the row labels for a row group when a memeber changes or
	 * moves.
	 */	
	private void repopulateRowLabels(int equivalenceClassIndex)
	{
		DPLEquivalenceClass equivalenceClass = (DPLEquivalenceClass)datapool.getEquivalenceClasses().get(equivalenceClassIndex);
		String name = equivalenceClass.getName();
		int tableRowIndex = findRowIndex(equivalenceClassIndex, 0);
		for(int i = 0; i < equivalenceClass.getRecord().size(); i++)
		{
			if(table.getItemCount() <= tableRowIndex + 1)
				continue;
			TableItem tableItem = table.getItem(tableRowIndex + i);
			tableItem.setText(0, name + TestUIPlugin.getString("DATA_EDT_DIVIDER") + String.valueOf(i)); //$NON-NLS-1$
		}
	}
}
