/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.sd.ui.internal.model;

import java.util.*;

import org.eclipse.hyades.sd.ui.*;
import org.eclipse.hyades.sd.ui.internal.util.*;
import org.eclipse.hyades.sd.ui.internal.views.*;
import org.eclipse.jface.action.*;
import org.eclipse.swt.*;
import org.eclipse.swt.events.*;
import org.eclipse.swt.graphics.*;
import org.eclipse.swt.widgets.*;

public class ColumnFigure extends ColumnNode implements NodeContainer,
														PaintListener,
														MouseMoveListener,
														MouseListener,
														MouseTrackListener,
														Listener {

	private int width;
	private int tempWidth;
	private int height;
	private int number;
	private Rectangle parentBounds;
	private Point originalLocation;
	private SDViewerUI sdViewerUI;
	private Image headerImage = null;
	private boolean firstTime = true;
	private boolean selected = false;
	private Canvas canvas=null;
	private Composite parentComposite;
	private boolean collapsed = false;
	
	public ColumnFigure() {}
		
	public int init(Composite c, String name, int xLoc, int no, Canvas canvas){
		parentComposite = c;
		sdViewerUI = (SDViewerUI)c.getParent();
		
		//determine OS
		String whichOS = System.getProperty("os.name");
		whichOS = whichOS.toLowerCase();
		int index = whichOS.indexOf("linux");
		
		//if linux OS, need to use 6.6, otherwise use 5.3
		if (index < 0) width = Math.max(LogCDrawUtils.getMinColumnWidthExpanded(),(int)(name.length()*5.3));
		else width = Math.max(LogCDrawUtils.getMinColumnWidthExpanded(),(int)(name.length()*6.6));
        
        setXOffset(xLoc);
		number = no;
        this.canvas = canvas;			
        originalLocation = new Point(xLoc, 0);
		parentBounds = sdViewerUI.getParent().getClientArea();
	    LogCDrawUtils.setBottomMax(parentBounds.height / LogCDrawUtils.getTimeUnit());
	    this.height = parentBounds.height;
	    setHeaderImage();
	    addListeners();
	    
	    return width;
	}
		    
	private void setHeaderImage(){
		switch(getType()){
		   case(NodeContainer.UNKNOWN):
			  break;
		   case(NodeContainer.MONITOR):
			  headerImage = SDPluginImages.getImage(SDPluginImages.IMG_MONITOR);
			  break;
		   case(NodeContainer.CLUSTER):
			  headerImage = SDPluginImages.getImage(SDPluginImages.IMG_MONITOR);
			  break;
		   case(NodeContainer.NODE):
			  headerImage = SDPluginImages.getImage(SDPluginImages.IMG_NODE);
			  break;
		   case(NodeContainer.PROCESS):
			  headerImage = SDPluginImages.getImage(SDPluginImages.IMG_PROCESS);
			  break;
		   case(NodeContainer.AGENT):
			  headerImage = SDPluginImages.getImage(SDPluginImages.IMG_AGENT);
			  break;
		   case(NodeContainer.THREAD):
			  headerImage = SDPluginImages.getImage(SDPluginImages.IMG_THREAD);
			  break;
		   case(NodeContainer.OBJECT):
			  headerImage = SDPluginImages.getImage(SDPluginImages.IMG_OBJECT);
			  break;
		   case(NodeContainer.CLASS):
			  headerImage = SDPluginImages.getImage(SDPluginImages.IMG_CLASS);
			  break;
		   case(NodeContainer.METHOD):
			  headerImage = SDPluginImages.getImage(SDPluginImages.IMG_METHOD);
			  break;
		   case(NodeContainer.LOG):
			  headerImage = SDPluginImages.getImage(SDPluginImages.IMG_LOG);
			  break;              
		   default:
			  break;
		}
	}    

	private void addListeners(){
		canvas.addDisposeListener(new DisposeListener() {
			public void widgetDisposed(DisposeEvent e) {
				ColumnFigure.this.widgetDisposed(e);
			}
		});
		canvas.addPaintListener(this);
		canvas.addMouseTrackListener(this);
	}
	
	public void paintControl(PaintEvent e) {
		Rectangle clientArea = canvas.getClientArea();
		if(originalLocation.x+width<=clientArea.x || originalLocation.x>=(clientArea.x+clientArea.width)){
		   return;
		}
		int nodeWidth = LogCDrawUtils.getNodeWidth();
		int boxSpace = LogCDrawUtils.getBoxSpace();
		int boxHeight = LogCDrawUtils.getBoxHeight();
		int sideSpace = LogCDrawUtils.getSideSpace();
		int upperSpace = LogCDrawUtils.getUpperSpace();
		int startPosition = LogCDrawUtils.getStartPosition();
		int timeUnit = LogCDrawUtils.getTimeUnit();
		int bottomMax = LogCDrawUtils.getBottomMax();
		int viewportHeight = sdViewerUI.getSize().y;
		GC gc = e.gc;
		int leftNameSpace=0;
		
		if(selected)
		   gc.setForeground(LogCDrawUtils.getForegroundSelectedColor());
		else
		   gc.setForeground(LogCDrawUtils.getForegroundColor());
		if (upperSpace + boxHeight - startPosition > 0) {
			gc.drawRectangle(getXOffset()+sideSpace, upperSpace, width - 2 * sideSpace, boxHeight);
			if(getType()==NodeContainer.NODE)
			   leftNameSpace = sideSpace+(width-getShortName().length()* gc.getCharWidth('M'))/2;
			else
			   leftNameSpace = 2 * sideSpace;
			if(headerImage!=null){
			   gc.drawImage(headerImage,getXOffset()+sideSpace+1, upperSpace+1);
			   gc.drawString(getShortName(), getXOffset()+leftNameSpace, upperSpace + boxHeight / 2);
			}else
			   gc.drawString(getShortName(), getXOffset()+leftNameSpace, upperSpace + boxHeight / 4);
			if(secondaryName!=null)
			   gc.drawString(secondaryName,getXOffset()+2*sideSpace+16, upperSpace+1);
		}
		gc.setLineStyle(SWT.LINE_DOT);
		gc.drawLine(getXOffset()+width / 2, upperSpace + boxHeight, getXOffset()+width / 2, height);
		gc.setLineStyle(SWT.LINE_SOLID);
		if(selected)
		   gc.setForeground(LogCDrawUtils.getForegroundColor());
		ColumnNode node;
		int correctedBegin;
		int correctedEnd;
		int bottomMaxTime = bottomMax*timeUnit;
		for (int i = 0; i < getInternalNodeCount(); i++){
			node = (ColumnNode) internalNodes[i];
			correctedBegin = (node.getStartPosition() - startPosition)*timeUnit + upperSpace + boxHeight;
			correctedEnd = (node.getStopPosition() - startPosition)*timeUnit + upperSpace + boxHeight;
			if((correctedBegin<0 && correctedEnd<0)||(correctedBegin>viewportHeight && correctedEnd>viewportHeight));
		    else 
		       node.drawSelf(gc, width, timeUnit, boxSpace, startPosition, viewportHeight, nodeWidth, getXOffset(),sdViewerUI.getColumnPassagesMap());
		     
		    
		    if (graph.getType()==Graph.LOGoverRECORD||graph.getType()==Graph.THREADoverRECORD)
		    {
				NodeLink nodeConnection;
				NodeConnection[] targetConnections = node.getTargetConnections();
				for (int t = 0; t < targetConnections.length; t++) {
					if(targetConnections[t]!=null){
						nodeConnection = (NodeLink) targetConnections[t];
						if (nodeConnection.getSource().getStartIncrementValue()!=nodeConnection.getTarget().getStartIncrementValue()
						    && nodeConnection.getTarget().getContainer().getIndexInContainer()==getIndexInContainer())
							 nodeConnection.drawSelf(gc, width, this, timeUnit, boxSpace, startPosition, nodeWidth, getXOffset(),sdViewerUI.getColumnPassagesMap());
					}
				}			
		    }
		}
	
	
		ColumnPassage passage;
		int position=0;
		Iterator columnPassages = sdViewerUI.getColumnPassages();
		while (columnPassages.hasNext()){
		   passage = (ColumnPassage)columnPassages.next();
		   if(passage.isOver(indexInContainer)){
		      position = (passage.getPassagePosition() - startPosition)*timeUnit + upperSpace + boxHeight;
			  if (position >= 0 && position <= viewportHeight)
		         passage.drawSelf(gc, width, timeUnit, boxSpace, startPosition, getXOffset());
		   }
		}

		
		int index = LogCDrawUtils.getSelectedWeightIndex();
		if(index!=-1){
		   int weightPosition = boxSpace + timeUnit*(index-startPosition);
		   //need to add the timeUnit/2 since the weight position is of the upper edge of the time period.
		   if(getColumnNodeAtPosition(weightPosition)!=null)
		      drawSelectedWeight(gc,timeUnit,weightPosition);	
		}
		
		if(firstTime){
			canvas.addMouseMoveListener(this);
			canvas.addMouseListener(this);
			firstTime = false;
		}

        Color c = e.gc.getBackground();
        
		e.gc.setForeground(Display.getDefault().getSystemColor(SWT.COLOR_WIDGET_FOREGROUND));
		e.gc.setBackground(Display.getDefault().getSystemColor(SWT.COLOR_WIDGET_BACKGROUND));      		
		e.gc.drawString(getShortName(), originalLocation.x, 0, false);
		e.gc.setBackground(c);

	}
	
	private void drawSelectedWeight(GC gc,int timeUnit,int position){
	   Color originalColor = gc.getBackground();
	   int nodeWidth = LogCDrawUtils.getNodeWidth();
	   gc.setBackground(LogCDrawUtils.getSelectedWeightColor());
	   gc.fillRectangle(getXOffset()+width/2-nodeWidth/2+1,position+1,nodeWidth-1,timeUnit-1);
	   gc.setBackground(originalColor);
	 
	}
	
	//draw in overview
	public int drawSelf(GC gcX, Point space, boolean drawLifeLines, int boxHeight, int overviewRatio){
		int widthX = width/overviewRatio;
		int xPos = originalLocation.x/overviewRatio;
		xPos=xPos+widthX / 2;
		if(drawLifeLines) {
		   gcX.setLineStyle(SWT.LINE_DOT);
		   gcX.drawLine(xPos, 0, xPos, space.y+boxHeight);
		   gcX.setLineStyle(SWT.LINE_SOLID);
		}		
		return xPos;
	}


	public void setName(String inputName) {
		super.setName(inputName);
	}
	public String getFullName() {
		return getName();
	}
	
	public void setWidth(int w) {
		width = w;
	}
	public int getWidth() {
		return width;
	}

	public void setHeight(int pos) {		
		height = pos;
	}
	public int getHeight() {
		return height;
	}

	public int getNumber() {
		return number;
	}

	public int getNodesNumber() {
		return getInternalNodeCount();
	}
	public ColumnNode getNode(int no) {
		return (ColumnNode) internalNodes[no];
	}

	void widgetDisposed(DisposeEvent e) {
		   disposeMe();
	}
	
	public void disposeMe(){
		if (canvas!=null) {
			canvas.removeMouseListener(this);
			canvas.removePaintListener(this);
			canvas.removeMouseTrackListener(this);
			canvas.removeMouseMoveListener(this);
		}
	}
	
	//MouseMoveListener -> begin
	public void mouseMove(MouseEvent e){
        if(!((e.x >= getXOffset()) && e.x <=(getXOffset()+width)))
 	          return;
		sdViewerUI.setToolTip(e.x,e.y,this);
	}
	//MouseMoveListener ->end
	
	//MouseTrackListener ->begin
	public void mouseEnter(MouseEvent e){}
	public void mouseExit(MouseEvent e){
	   sdViewerUI.removeToolTip();
	}	
	public void mouseHover(MouseEvent e){
	   canvas.forceFocus();
	}	
	//MouseTrackListener ->end
	
	//MenuListener ->begin
	public void menuAboutToShow(IMenuManager menu){
		sdViewerUI.fillContextMenu(menu);
    }
    //MenuListener ->end
	
	//MouseListener->begin
	public void mouseDoubleClick(MouseEvent e){	
	//	sdViewerUI.mouseDoubleClick(e);
	}
	
	public void mouseDown(MouseEvent e){
		if(e.x<getXOffset() || e.x>getXOffset()+width)
		   return;

		Rectangle clientArea = canvas.getClientArea();
		if(originalLocation.x+width<=clientArea.x || originalLocation.x>=(clientArea.x+clientArea.width))
		   return;
		   
		sdViewerUI.getGraphComposite().forceFocus();
		sdViewerUI.setIdEvent(e);
		
		//selection made within the column top name box
       if((LogCDrawUtils.getBoxSpace()!=0 
       			&& e.y<=LogCDrawUtils.getBoxSpace())) 
       {
			  sdViewerUI.setSelectedColumnFigure(this);
	          return;
       }
       
       //selection made on a passage
       ColumnPassage columnPassage = getColumnPassageAtPosition(e.y);
       if(columnPassage != null){
			sdViewerUI.setSelectedNodeLink(this,(NodeLink)columnPassage.getNodeConnection());
       		return;
       }
	
	   //selection made on a node
	   ArrayList columnNodes = getColumnNodesAtPosition(e.y);
	  
	   int nodeWidth = LogCDrawUtils.getNodeWidth();
	   if (columnNodes!=null && columnNodes.size()>0)
	   {
	   	 for (int count=0;count<columnNodes.size();count++) {
	   	   ColumnNode columnNode = (ColumnNode)(columnNodes.get(count));
		  
		   if(columnNode != null){
		      if(e.x>=(getXOffset()+(width-nodeWidth)/2) && e.x<=(getXOffset()+(width+nodeWidth)/2)){
				sdViewerUI.setSelectedNode(columnNode, this);
		      	return;
		      }
		      else {
		      	//selection made on a link
		         NodeLink nodeLink = getNodeLinkAtPosition(columnNode,e.x,e.y);	
		      	 if(nodeLink != null){
					sdViewerUI.setSelectedNodeLink(this,nodeLink);
		      	 	return;
		      	 }
		      }
		   	
	   		}
	   	 }
	   }
	   
	   //selection made on the figure dot line
	   if(e.x >= (getXOffset()+width/2-1) && e.x <=(getXOffset()+width/2+1)){
			  sdViewerUI.setSelectedColumnFigure(this);
	          return;
	   }
	   sdViewerUI.setSelectedColumnFigure(null);
	}
	
	public void mouseUp(MouseEvent e){}
	//MouseListener ->end
	
	public void handleEvent (Event event){
		if(collapsed){
			width = tempWidth;
			tempWidth = LogCDrawUtils.getMinColumnWidth();
		} else if(!collapsed){
	    	tempWidth = width;
	        width = LogCDrawUtils.getMinColumnWidth();
	    }
	    collapsed = !collapsed;
		sdViewerUI.adjustColumns(width-tempWidth, number);		
	}

	public ArrayList getColumnNodesAtPosition(int y) {
		ArrayList columnNodes = new ArrayList();
		int correctedY = (y-LogCDrawUtils.getBoxSpace())/LogCDrawUtils.getTimeUnit()+LogCDrawUtils.getStartPosition();        
		ColumnNode columnNode;
		for (int i = getInternalNodeCount() - 1; i>=0; i--){
			columnNode = (ColumnNode) internalNodes[i];
			
			if (correctedY >= columnNode.getStartPosition()
				&& correctedY <= columnNode.getStopPosition()) 
			{
				columnNodes.add(columnNode);
			}
		}
		return columnNodes;
	}
	
	public ColumnNode getColumnNodeAtPosition(int y) {
        int correctedY = (y-LogCDrawUtils.getBoxSpace())/LogCDrawUtils.getTimeUnit()+LogCDrawUtils.getStartPosition();        
		ColumnNode columnNode;
		for (int i = getInternalNodeCount() - 1; i>=0; i--){
			columnNode = (ColumnNode) internalNodes[i];
			if (correctedY >= columnNode.getStartPosition()
				&& correctedY <= columnNode.getStopPosition()) {
				return columnNode;
			}
		}
		return null;
	}
	

	public NodeLink getNodeLinkAtPosition(ColumnNode columnNode, int x, int y) {
		if (x<getXOffset() || x>getXOffset()+width) return null;
		
		int timeUnit = LogCDrawUtils.getTimeUnit();
		NodeLink nodeConnection;
		int connectionPosition;
		for (int i = 0; i < columnNode.getSourceConnections().length; i++) {
			if(columnNode.getSourceConnections()[i]!=null){
			   nodeConnection = (NodeLink) columnNode.getSourceConnections()[i];
			   
			   if(nodeConnection.isLogRecordConection()
			   && (nodeConnection.getTarget().getStartIncrement().getValue()<=nodeConnection.getSource().getStartIncrement().getValue()
		       || nodeConnection.getTarget().getStartIncrement().getValue()>=nodeConnection.getSource().getEndIncrement().getValue()))
			   {
			   	   connectionPosition = LogCDrawUtils.getBoxSpace()+timeUnit* (nodeConnection.getSource().getStartIncrementValue() - LogCDrawUtils.getStartPosition());
	//		   	   System.out.println("log="+connectionPosition+":y="+y); 
			   } 
			   
			   else
			    {
			      connectionPosition = LogCDrawUtils.getBoxSpace()+timeUnit* (nodeConnection.getStartIncrementValue() - LogCDrawUtils.getStartPosition());
	//			  System.out.println("src="+connectionPosition+":y="+y);
			    }
			   if(y>=connectionPosition-1 && y<=connectionPosition+1){
			   
//				if((x>getXOffset()+(width/2) //link towards right
//										&& nodeConnection.getTarget().getContainer().getIndexInContainer()>=columnNode.getContainer().getIndexInContainer() 
//										&& nodeConnection.getSource().getContainer().getIndexInContainer()>=columnNode.getContainer().getIndexInContainer()))
//				System.out.println("source->");
//				if ((x<getXOffset()+(width/2) //link towards left
//				&& nodeConnection.getTarget().getContainer().getIndexInContainer()<=columnNode.getContainer().getIndexInContainer()
//				&& nodeConnection.getSource().getContainer().getIndexInContainer()<=columnNode.getContainer().getIndexInContainer()))
//				System.out.println("source<-");
			
			   	
				  if((x>getXOffset()+(width/2) //link towards right
				  		&& nodeConnection.getTarget().getContainer().getIndexInContainer()>=columnNode.getContainer().getIndexInContainer() 
				  		&& nodeConnection.getSource().getContainer().getIndexInContainer()>=columnNode.getContainer().getIndexInContainer())
				  	|| (x<getXOffset()+(width/2) //link towards left
				  		&& nodeConnection.getTarget().getContainer().getIndexInContainer()<columnNode.getContainer().getIndexInContainer()
				  		&& nodeConnection.getSource().getContainer().getIndexInContainer()<=columnNode.getContainer().getIndexInContainer()))
						return nodeConnection;
			   }
			}
		}
		for (int i = 0; i < columnNode.getTargetConnections().length; i++) {
			if(columnNode.getTargetConnections()[i]!=null){
			   nodeConnection = (NodeLink) columnNode.getTargetConnections()[i];
		       
		       connectionPosition = LogCDrawUtils.getBoxSpace()+timeUnit* (nodeConnection.getStartIncrementValue() - LogCDrawUtils.getStartPosition());
	//		   System.out.println("tar="+connectionPosition+":y="+y); 
			   
			   if(y>=connectionPosition-1 && y<=connectionPosition+1){

//				if((x>getXOffset()+(width/2) //link towards right
//						&& nodeConnection.getTarget().getContainer().getIndexInContainer()>=columnNode.getContainer().getIndexInContainer() 
//						&& nodeConnection.getSource().getContainer().getIndexInContainer()>=columnNode.getContainer().getIndexInContainer()))
//			  System.out.println("target->");
//			
//				if ((x<getXOffset()+(width/2) //link towards left
//				&& nodeConnection.getTarget().getContainer().getIndexInContainer()<=columnNode.getContainer().getIndexInContainer()
//				&& nodeConnection.getSource().getContainer().getIndexInContainer()<=columnNode.getContainer().getIndexInContainer()))
//				System.out.println("target<-");

				  if((x>getXOffset()+(width/2) //link towards right
				  		&& nodeConnection.getTarget().getContainer().getIndexInContainer()>=columnNode.getContainer().getIndexInContainer() 
				  		&& nodeConnection.getSource().getContainer().getIndexInContainer()>=columnNode.getContainer().getIndexInContainer())
				  	|| (x<getXOffset()+(width/2) //link towards left
				  		&& nodeConnection.getTarget().getContainer().getIndexInContainer()<=columnNode.getContainer().getIndexInContainer()
				  		&& nodeConnection.getSource().getContainer().getIndexInContainer()<columnNode.getContainer().getIndexInContainer()))
			       		return nodeConnection;
			   }
			}
		}
		return null;
	}

	public ColumnPassage getColumnPassageAtPosition(int y){
		int timeUnit = LogCDrawUtils.getTimeUnit();
		ColumnPassage columnPassage;
		int passagePosition;
		Iterator columnPassages = sdViewerUI.getColumnPassages();
		while (columnPassages.hasNext()){
			columnPassage = (ColumnPassage)columnPassages.next();
			passagePosition = LogCDrawUtils.getBoxSpace()+timeUnit* (columnPassage.getPassagePosition() - LogCDrawUtils.getStartPosition());
		    if(columnPassage.isOver(indexInContainer) && y>=passagePosition-1 && y<=passagePosition+1)
		        return columnPassage;
		}
		return null;
	}
	
	public void adjustLocation(int shift) {
		originalLocation.x+=shift;
	}
	
	public void setOriginalLocation(int x, int y) {
		originalLocation.x = x;
		originalLocation.y = y;
		setXOffset(x);
	}
	public Point getOriginalLocation(){
	   return originalLocation;	
	}
	
	public NodeContainer getNodeContainer(){
	   return this;	
	}
	
	public void setSelected(boolean selection){
	   selected = selection;	
	}
	
	public Canvas getMyCanvas(){
	   return canvas;	
	}
	
	//Implement NodeContainer -> begin	
	public GraphNode lastReadNode = null;
    public GraphNode internalNodes[] = new GraphNode[3];
    public int nextNewInternalNode = 0;
	
  	public GraphNode[] getInternalNodes() {
 	 	return internalNodes;
  	}

  	public void setInternalNodes(GraphNode[] nodes) {
	    internalNodes = nodes;
	}

 	public void addInternalNode(GraphNode node) {
    	try {
      		internalNodes[nextNewInternalNode] = node;
    	}
    	catch (ArrayIndexOutOfBoundsException e) {
      		GraphNode[] tempArray = new GraphNode[2 * internalNodes.length];
      		System.arraycopy(internalNodes, 0, tempArray, 0, internalNodes.length);
      		internalNodes = tempArray;
      		internalNodes[nextNewInternalNode] = node;
    	}
    	node.setIndexInContainer(nextNewInternalNode);
    	nextNewInternalNode++;
  	}

  	public int getInternalNodeCount() {
  		return nextNewInternalNode;
 	 }
  	//Implement NodeContainer -> end	
}
