/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.internal.execution.recorder.remote.appadapters;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.io.UnsupportedEncodingException;
import java.util.StringTokenizer;


/**
 * This object is the client object for a RecorderApplication extension point (point org.eclipse.hyades.execution.recorder.RecorderApplication)
 * it contains logic for initializing, starting, and stopping Internet Explorer.  In particular, this object is able
 * to modify system registry settings to cause Internet Explorer to use a particular host and port for proxy recording.
 * This adapter only works on Win32.
 * @author Ernest Jessee
 */
public class InternetExplorerAdapter implements IRecorderApplicationAdapter
{	
	/**
	 * the Internet Explorer Process
	 */
	private Process ieProcess;
		
	/**
	 * filename for storing the unmodified registry settings 
	 */
	private String originalSettingsFileName = "IEConfig.reg";//$NON-NLS-1$
	
	/**
	 * filname for storing the registry settings to be used while recording
	 */
	private String tempSettingsFileName = "tempIEConfig.reg";//$NON-NLS-1$

	/**
	 * init routine called by the recorder to have internet explorer prepared for recording
	 * @param String initInfo -- this is a comma delimeted string indicating which port to point IE at. (ie. port,450)
	 */
	public void init(String initInfo) throws RecorderApplicationAdapterException
	{
			
		StringBuffer inputFileContents = getRegistryInfo();
		
		StringBuffer outputFileContents = new StringBuffer();
		
		StringTokenizer fileTokenizer = new StringTokenizer(inputFileContents.toString(),"\r\n");//$NON-NLS-1$
		while(fileTokenizer.hasMoreTokens())
		{
			String line=fileTokenizer.nextToken();
			String newLine = line;
			if(line.startsWith("\"ProxyEnable"))//$NON-NLS-1$
			{
				StringTokenizer lineTokenizer = new StringTokenizer(line,":");//$NON-NLS-1$
				newLine=lineTokenizer.nextToken();
				newLine+=":00000001";//$NON-NLS-1$
				newLine+="\r\n\"ProxyServer\"=\"socks=localhost:";//$NON-NLS-1$
				newLine+=initInfo.substring(initInfo.indexOf(',')+1,initInfo.length())+"\"";//$NON-NLS-1$				
			}
			else if(line.startsWith("\"ProxyHttp1.1"))//$NON-NLS-1$
			{
				StringTokenizer lineTokenizer = new StringTokenizer(line,":");//$NON-NLS-1$
				newLine=lineTokenizer.nextToken();
				newLine+=":00000001";//$NON-NLS-1$
								
			}
			else if(line.startsWith("\"ProxyServer"))//$NON-NLS-1$
			{
				//it has been found that if the target computer has never been pointed to a proxy.  This 
				//entry may not be present.  Therefore, we will ignore it if it is there and always create it from scratch.
				newLine=null;				
			}
			if(newLine!=null)
				outputFileContents.append(newLine+"\r\n");//$NON-NLS-1$
		}
			
		
		createRegSettingsFile(outputFileContents);
		
		try
		{
			pushRegFile(tempSettingsFileName);
		}
		catch (Exception e)
		{
			throw new RecorderApplicationAdapterException(e.getMessage());
		}
		
	}


	/**
	 * causes a registry file to be pushed into the registry using regedit.
	 * @param String fileName
	 */
	private void pushRegFile(String fileName) throws InterruptedException, IOException
	{
		String pushFileCommandLine = "regedit /s "+fileName;//$NON-NLS-1$
		
		Process pusher = Runtime.getRuntime().exec(pushFileCommandLine,null);
		pusher.waitFor();
		
	}

	/**
	 * method which creates a registry settings file (*.reg)
	 * @param StringBuffer ouputFileContents
	 */
	private void createRegSettingsFile(StringBuffer outputFileContents)
	{
		
		File newFile = new File(tempSettingsFileName);		
				
		try
		{
			OutputStreamWriter outputWriter = new OutputStreamWriter(new FileOutputStream(newFile),"UnicodeLittle");//$NON-NLS-1$
			outputWriter.write(outputFileContents.toString());
			outputWriter.close();
		}
		catch (UnsupportedEncodingException e1)
		{
			e1.printStackTrace();
		}
		catch (FileNotFoundException e1)
		{
			e1.printStackTrace();
		} 
		catch (IOException e)
		{
			e.printStackTrace();
		}
		
	}


	/**
	 * method which exports the registry keys of interested into a file using regedit and then reads the file
	 * @return StringBuffer
	 */
	private StringBuffer getRegistryInfo()
	{
		
		String getOriginalInfoCommandLine = "regedit /e "+originalSettingsFileName+" \"HKEY_CURRENT_USER\\Software\\Microsoft\\Windows\\CurrentVersion\\Internet Settings\"";//$NON-NLS-1$//$NON-NLS-2$
		InputStreamReader inputReader=null;
		StringBuffer inputFileContents = new StringBuffer();
		try
		{
			Process registryChange = Runtime.getRuntime().exec(getOriginalInfoCommandLine,null);
			registryChange.waitFor();
			File file = new File(originalSettingsFileName);
			inputReader= new InputStreamReader(new FileInputStream(file),"UTF-16");//$NON-NLS-1$
			inputFileContents = new StringBuffer();
			char[] inputBuf = new char[1000];
			int bytesRead = inputReader.read(inputBuf);
								
			while(bytesRead!=-1)
			{				
				String buf = new String(inputBuf);				
				inputFileContents.append(inputBuf,0,bytesRead);
				bytesRead = inputReader.read(inputBuf);
				
			}			
			
		}
		catch (IOException e)
		{
			e.printStackTrace();
		} 
		catch (InterruptedException e)
		{
			e.printStackTrace();
		}
		finally
		{
			if(inputReader!=null)
				try
				{
					inputReader.close();
				}
				catch (IOException e1)
				{
					e1.printStackTrace();
				}
		}
		return inputFileContents;
	}

	
	/**
	 * method called by the recorder to start the target application (Internet Explorer)
	 * @return Process
	 */
	public Process start() throws RecorderApplicationAdapterException 	
	{
		ieProcess = null;
		try
		{
			//TODO derive path to iexplore.exe using registry.
			ieProcess = Runtime.getRuntime().exec("C:\\Program Files\\Internet Explorer\\iexplore.exe about:blank");//$NON-NLS-1$
		}
		catch (IOException e)
		{
			throw new RecorderApplicationAdapterException("unable to start Internet Explorer Process: "+e.getMessage());//$NON-NLS-1$
		}
		return ieProcess;
	}

	
	/**
	 * method called to restore the process back to its original state
	 */
	public void cleanup()throws RecorderApplicationAdapterException
	{
		try
		{
			pushRegFile(originalSettingsFileName);
		}
		catch (Exception e)
		{
			throw new RecorderApplicationAdapterException(e.getMessage());
		}
	}

	/**
	 * method called to stop the target process.
	 */
	public void stop() throws RecorderApplicationAdapterException
	{
		ieProcess.destroy();
	}

}
