/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v0.5
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v05.html
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.loaders.common;

import org.eclipse.emf.ecore.EObject;
import org.eclipse.hyades.loaders.hierarchy.IgnoredXMLFragmentLoader;
import org.eclipse.hyades.loaders.util.HierarchyContext;
import org.eclipse.hyades.loaders.util.LoadersUtils;
import org.eclipse.hyades.models.common.interactions.BVRInteractionFragment;
import org.eclipse.hyades.models.common.testprofile.TPFExecutionEvent;
import org.eclipse.hyades.models.common.testprofile.TPFExecutionHistory;
import org.eclipse.hyades.models.common.testprofile.TPFExecutionResult;
import org.eclipse.hyades.models.common.testprofile.TPFInvocationEvent;
import org.eclipse.hyades.models.common.testprofile.impl.Common_TestprofileFactoryImpl;

/**
 * @author amathur
 *
 * This is the base class for the execution event loaders and handles 
 * attributes of the base event.
 */
public abstract class XMLexecutionEventLoader extends IgnoredXMLFragmentLoader {

	// Final static debug variables
	public static final boolean DEBUG = false;		// set 'true' to turn n debugging in fragment loaders
	public static final String ROOT_PARENT = "ROOT";
	
	// Static variables -----------------------------
	protected final static String EVENT_ID = "id";
	protected final static String EVENT_OWNERID = "ownerId";
	protected final static String EVENT_TIMESTAMP = "timestamp";
	protected final static String EVENT_TEXT = "text";
	protected final static String EVENT_PARENTID = "parentId";
	
	// Instance variables
	protected String 			id = null;
	protected String 			ownerId = null;
	protected long 				timestamp = -1;
	protected String 			text;
	protected TPFExecutionEvent event = null;
	protected String			parentId = null;
	
	/* (non-Javadoc)
	 * @see org.eclipse.hyades.loaders.util.XMLFragmentLoader#initialize(org.eclipse.hyades.loaders.util.HierarchyContext, java.lang.String)
	 */
	public void initialize(HierarchyContext context, String name) {
		super.initialize(context, name);
		event = null;
		id = null;
		ownerId = null;
		timestamp = 0;
		text = null;
		parentId = null;
		
		printTag(name);
	}

	/* (non-Javadoc)
	 * @see org.eclipse.hyades.loaders.util.XMLFragmentLoader#addAttribute(java.lang.String, java.lang.String)
	 */
	public void addAttribute(String name, String value) {
		/* The attributes belonging to the base execution event are:
		 * id
		 * ownerId
		 * timestamp
		 * text
		 */
		switch (LoadersUtils.getHashCode(name)) {
			case ExecutionConstants.EVENT_ID_int :
				id = value;
				break;

			case ExecutionConstants.EVENT_OWNERID_int :
				ownerId = value;
				break;

			case ExecutionConstants.EVENT_PARENTID_int :
				parentId = value;
				break;

			case ExecutionConstants.EVENT_TIMESTAMP_int :
				timestamp = Long.parseLong(value);
				break;

			case ExecutionConstants.EVENT_TEXT_int :
				text = value;
				break;

			default :
				break;
		}
	}

	/* (non-Javadoc)
	 * @see org.eclipse.hyades.loaders.util.XMLFragmentLoader#addYourselfInContext()
	 */
	public void addYourselfInContext() {
		if (event == null) {
			// Cast the specialized event to the base event before calling this method
			return;
		}
		
		printEndTag();
		
		// Set the base event attributes
		event.setId(id);
		event.setOwnerId(ownerId);
		event.setTimestamp(timestamp);
		event.setText(text);
		
		if ( parentId == null )
		{
			// This condition is here to accommodate the old XML fragment grammar for 
			// execution history in which there is no parent ID, and in which the
			// limited expressible hierarchy is encoded in the ownerId string (and
			// must be parsed out to determine the containing event element and
			// the related interaction fragment (if any)).
			ExecutionContext eContext = (ExecutionContext) context.getCustomData().get(ExecutionContext.root);
			eContext.logEvent(event);
		}
		else
		{
			// ownerId (if populated) specifies the GUID of the test model element whose
			// execution gave rise to this event.  Find that element in the test model
			// and setthe reference from this execution event to that model element.
			if ( ownerId != null && ownerId.length() != 0 )
			{
				ExecutionContext eContext = (ExecutionContext) context.getCustomData().
					get(ExecutionContext.root);
				EObject testElement = ExecutionResultData.getObjectFromTest(
					eContext.getTestSuite(), ownerId);
				
				if ( testElement != null && testElement instanceof BVRInteractionFragment )
				{
					event.setInteractionFragment((BVRInteractionFragment)testElement);
				}
			}
		}
		
	}
	
	
	/**
	 * This method adds the event to the correct container within the execution result.
	 * The container may be specified by a parentId, or may be the root container
	 * (which is the outermost execution result, and is signified by an empty parentId.)
	 * 
	 * If the parentId specifies an Invocation Event, then the invocation event is
	 * not actually the container.  The real container is the execution history that is
	 * contained by the execution result contained by the invocation event.  (Got that?)
	 * To make more sense of this, look at the data model, and see that an execution event
	 * can be contained by either another execution event, or by an execution history.  
	 * If the parentId specifies an invocation event, then the container is really the
	 * execution history.
	 */
	public void addYourselfToContainer()
	{
		ExecutionContext eContext = (ExecutionContext) context.getCustomData().get(ExecutionContext.root);
		
		if ( parentId != null && parentId.equals(ROOT_PARENT))
		{
			// This event has no specified parent, so it must be a child of
			// the root execution result.
			TPFExecutionResult exResult = eContext.rootResult.getResult();
			if ( exResult != null )
			{
				TPFExecutionHistory history = exResult.getExecutionHistory();
				if ( history == null )
				{
					// TODO: Determine if this condition can ever be met and
					// remove the code if it can not.
					history = Common_TestprofileFactoryImpl.eINSTANCE
						.createTPFExecutionHistory();
					exResult.setExecutionHistory(history);
				}
				event.setExecutionHistory(history);				
			}
		}
		else if ( parentId != null )
		{
			TPFExecutionResult result = eContext.rootResult.getResult();
			EObject obj = result.eResource().getEObject(parentId);
			if ( obj != null && obj instanceof TPFInvocationEvent )
			{
				// If the event identified as the parent of this event is 
				// an invocation event, then add this event as a child of the
				// invoked execution result's execution history object.

				TPFExecutionHistory history = ((TPFInvocationEvent) obj).getInvokedExecutionResult().getExecutionHistory();
				if ( history != null )
				{
					event.setExecutionHistory(history);
				}
				
			}
			else if ( obj != null )
			{
				// Otherwise simply add this event to its parent.
				TPFExecutionEvent parentEvent = (TPFExecutionEvent) obj; 
				event.setParent(parentEvent);
				
			}
		}
	}
	
	protected void printTag(String str) {
		if (DEBUG) {
			System.out.println("<" + str);
		}
	}
	
	protected void printAttribute(String name, String value) {
		if (DEBUG) {
			System.out.println(name + "=\"" + value + "\"");
		}
	}
	
	protected void printEndTag() {
		if (DEBUG) {
			System.out.println("/>");
		}
	}
}
